/** @file imgswell.c
 *  @brief Inflate the dimensions of image.
 *  @copyright (c) Turku PET Centre
 *  @author Vesa Oikonen
 */
/// @cond
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <string.h>
#include <math.h>
#include <time.h>
/*****************************************************************************/
#include "libtpcmisc.h"
#include "libtpcimgio.h"
#include "libtpcimgp.h"
/*****************************************************************************/

/*****************************************************************************/
static char *info[] = {
  "Inflate the dimensions of PET image by making 8 voxels of each original",
  "pixel. ECAT 6.3, 7.x, and Analyze 7.5 and NIfTI-1 formats are accepted.",
  " ",
  "Usage: @P [Options] image bigimage",
  " ",
  "Options:",
  " -z=<yes|no>",
  "     Image is inflated in z dimension or not inflated. By default, 3D",
  "     is inflated in all dimensions, but 2D image only in x,y dimensions.",
  " -stdoptions", // List standard options like --help, -v, etc
  " ",
  "Example:",
  "     @P i5998dy1.v i5998dy1_swollen.v",
  " ",
  "See also: imgshrink, imgdim, imgbox, imgfiltg, imgmask",
  " ",
  "Keywords: image, simulation, tool, software testing",
  0};
/*****************************************************************************/

/*****************************************************************************/
/* Turn on the globbing of the command line, since it is disabled by default in
   mingw-w64 (_dowildcard=0); in MinGW32 define _CRT_glob instead, if necessary;
   In Unix&Linux wildcard command line processing is enabled by default. */
/*
#undef _CRT_glob
#define _CRT_glob -1
*/
int _dowildcard = -1;
/*****************************************************************************/

/*****************************************************************************/
/**
 *  Main
 */
int main(int argc, char **argv)
{
  int      ai, help=0, version=0, verbose=1;
  char     petfile[FILENAME_MAX], outfile[FILENAME_MAX];
  int      inflate_planes=-1; 
  char    *cptr;
  int      ret;

  
  /*
   *  Get arguments
   */
  if(argc==1) {tpcPrintUsage(argv[0], info, stderr); return(1);}
  petfile[0]=outfile[0]=(char)0;
  /* Get options */
  for(ai=1; ai<argc; ai++) if(*argv[ai]=='-') {
    if(tpcProcessStdOptions(argv[ai], &help, &version, &verbose)==0) continue;
    cptr=argv[ai]+1; if(*cptr=='-') cptr++; if(cptr==NULL) continue;
    if(strncasecmp(cptr, "Z=", 2)==0) {
      cptr+=2;
      if(strncasecmp(cptr, "YES", 1)==0) {inflate_planes=1; continue;}
      if(strncasecmp(cptr, "NO", 1)==0) {inflate_planes=0; continue;}
    }
    fprintf(stderr, "Error: invalid option '%s'.\n", argv[ai]);
    return(1);
  } else break;
  
  /* Print help or version? */
  if(help==2) {tpcHtmlUsage(argv[0], info, ""); return(0);}
  if(help) {tpcPrintUsage(argv[0], info, stdout); return(0);}
  if(version) {tpcPrintBuild(argv[0], stdout); return(0);}

  /* Process other arguments, starting from the first non-option */
  if(ai<argc) {strlcpy(petfile, argv[ai++], FILENAME_MAX);}
  if(ai<argc) {strlcpy(outfile, argv[ai++], FILENAME_MAX);}
  if(ai<argc) {
    fprintf(stderr, "Error: invalid argument '%s'.\n", argv[ai]);
    return(1);
  }

  /* Did we get all the information that we need? */
  if(!outfile[0]) {
    fprintf(stderr, "Error: missing command-line argument; use option --help\n");
    return(1);
  }

  /* In verbose mode print arguments and options */
  if(verbose>1) {
    printf("petfile := %s\n", petfile);
    printf("outfile := %s\n", outfile);
    if(inflate_planes>=0) printf("inflate_planes := %d\n", inflate_planes);
  }
  if(verbose>9) IMG_TEST=verbose-10; else IMG_TEST=0;


  /*
   *  Read image
   */
  if(verbose>0) fprintf(stdout, "reading image %s\n", petfile);
  IMG img; imgInit(&img);
  ret=imgRead(petfile, &img);
  if(ret) {
    fprintf(stderr, "Error: %s\n", img.statmsg); if(verbose>1) imgInfo(&img);
    return(2);
  }
  if(verbose>1) {
    printf("image dimensions: %d %d %d\n", img.dimz, img.dimy, img.dimx);
    printf("image frame nr: %d\n", img.dimt);
  }
  /* Check if PET data is raw or image */
  if(img.type!=IMG_TYPE_IMAGE) {
    fprintf(stderr, "Error: %s is not an image.\n", petfile);
    imgEmpty(&img); return(2);
  }
  /* Decide whether to inflate in z dimension, if user did not tell what to do */
  if(inflate_planes<0) {
    if(img.dimz==1) inflate_planes=0; else inflate_planes=1;
    if(verbose>1) printf("inflate_planes := %d\n", inflate_planes);
  }

  /*
   *  Inflate the image
   */
  if(verbose>0) fprintf(stdout, "inflating...\n");
  IMG out; imgInit(&out); 
  ret=imgSwell(&img, &out, inflate_planes);
  if(ret) {
    fprintf(stderr, "Error: cannot inflate image.\n");
    if(verbose>1) printf("ret=%d\n", ret);
    imgEmpty(&img); imgEmpty(&out); return(3);
  }

  /* Free memory allocated for original image */
  imgEmpty(&img);


  /*
   *  Save the swollen image
   */
  if(verbose>1) fprintf(stdout, "writing swollen image in %s\n", outfile);
  ret=imgWrite(outfile, &out);
  if(ret) {
    fprintf(stderr, "Error: %s\n", out.statmsg);
    imgEmpty(&out); return(11);
  }
  if(verbose>0) fprintf(stdout, "Inflated image was written in %s\n", outfile);

  /* Free memory allocated for output image */
  imgEmpty(&out);

  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/// @endcond
