#!/bin/bash
#: Title      : test_imgcbvp
#: Date       : 2023-03-14
#: Author     : "Vesa Oikonen" <vesa.oikonen@utu.fi>
#: Options    : None

# Set the name of executable to test
if [ -n "${OS+1}" ] && [ "$OS" = "Windows_NT" ]; then
EXT=.exe;
else
EXT=;
fi

PROGRAM=../../imgcbvp$EXT;
if [ ! -f $PROGRAM ]; then
  printf "Failed: executable does not exist.\n"
  exit 1
fi


printf "=====================================================================\n"
printf "creating test data for %s\n" $PROGRAM
printf "=====================================================================\n"

if [ ! -f plasma.fit ] || [ ! -f p2rbc.par ] || [ ! -f cpet.par ] || [ ! -f frames.sif ]; then
  printf "Failed: test data does not exist.\n"
  exit 1
fi

if [ ! -f plasma.bld ]; then
  rm -f blood.bld cpet.tac dynamic.v
  fit2dat -c=0,11,0.02 plasma.fit plasma.bld
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi

if [ ! -f plasmac.bld ] || [ ! -f rbcc.bld ] || [ ! -f blood.bld ]; then
  rm -f cpet.tac blood.tac dynamic.v
# Computing PTAC as component of blood as (1-HCT)*Cp where HCT=0.39
  taccalc plasma.bld x 0.61 plasmac.bld
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
# Simulating RBC concentration curve dCrbc=kc*(1-HCT)*Cp
# where kc=0.60 and thus k2=kc*HCT=0.234
  sim_3tcm p2rbc.par plasmac.bld plasmac.bld rbc.bld
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
# Calculating the blood radioactivity residing in RBCs
  taccalc rbc.bld x 0.39 rbcc.bld
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
# Calculating the blood curve as Cb=(1-HCT)*Cp+HCT*Crbc
  taccalc plasmac.bld + rbcc.bld blood.bld
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
# Setting file header information for plotting
  tacren blood.bld 1 Blood
  tacren plasma.bld 1 Plasma
  tacren plasmac.bld 1 Plasma
  tacren rbc.bld 1 RBC
  tacren rbcc.bld 1 RBC
  tacstudy --force blood.bld sim1
  tacstudy --force plasma.bld sim1
  tacstudy --force plasmac.bld sim1
  tacstudy --force rbc.bld sim1
  tacstudy --force rbcc.bld sim1
# Plotting blood components
  tac2svg -legend=y -yt="SUV" -mt="" -x2=10.2 blood.svg -l blood.bld plasmac.bld rbcc.bld
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi


if [ ! -f cpet.tac ] || [ ! -f blood.tac ] || [ ! -f ct.tac ]; then
  rm dynamic.v ct.v
# Simulating regional muscle concentration curves (Cpet)
  sim_3tcm cpet.par plasmac.bld blood.bld cpet.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacstudy --force cpet.dat sim1
  simframe cpet.dat frames.sif cpet.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tac2svg -legend=y cpet.svg -s cpet.tac -l cpet.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
# Frames for BTAC
  simframe blood.bld frames.sif blood.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
# Simulate regional (1-Vb)*Ct data
  taccalc blood.bld x 0 zero.bld
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  sim_3tcm cpet.par plasmac.bld zero.bld ct.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacstudy --force ct.dat sim1
  simframe ct.dat frames.sif ct.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi

if [ ! -f dynamic.v ] || [ ! -f ct.v ]; then
# Simulating dynamic image
  simcirc -iring=1 -segm=4 -format=float -diam2=180 256 220 ring.bin
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  flat2img -scanner=HR+ ring.bin mask.v 1 1 256 256
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  dft2img -tpl=mask.v -scanner=HR+ cpet.tac dynamic.v
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  img2tif -rb -s dynamic.v
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
# Simulating dynamic (1-Vb)Ct image
  dft2img -tpl=mask.v -scanner=HR+ ct.tac ct.v
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  img2tif -rb -s ct.v
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi

if [ ! -f truevb.v ]; then
  printf "start[minutes]	end[g/cc]	LVcav	Global	ROI01	ROI02	ROI03\n" > vb.tac
  printf "0	0	1	0.25	0.15	0.20	0.35\n" >> vb.tac
  dft2img -tpl=mask.v -scanner=HR+ vb.tac truevb.v
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  img2tif -rb -s truevb.v
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi


if [ ! -f rv.tac ]; then
  rm -f dynamicrv.v
  tactime blood.bld -0.1 temp.bld
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  taccalc temp.bld x 10 rv.bld
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  simframe rv.bld frames.sif rv.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi


if [ ! -f dynamicrv.v ] || [ ! -f maskrv.v ]; then
# Add injection site into dynamic image and make mask
  simcirc -format=float 256 20 temp.bin
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  flat2img -scanner=HR+ temp.bin maskrv.v 1 1 256 256
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  imgmove maskrv.v 110 110
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  img2tif -rb -s maskrv.v
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  dft2img -tpl=maskrv.v -scanner=HR+ rv.tac rv.v
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  imgcalc dynamic.v + rv.v dynamicrv.v
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  img2tif -rb -s dynamicrv.v
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi


printf "\n\n"
printf "=====================================================================\n"
printf "testing %s\n" $PROGRAM
printf "=====================================================================\n"

printf "\n 1.0.0 \n"
printf " Test case 1: Estimate Vb and calculate corrected image. \n"
printf " Note: Test image only contains arterial Vb.\n"
printf " Expected result: reasonable results are obtained. \n\n"

rm -f v100.v d100.v
$PROGRAM -vb=v100.v dynamic.v blood.tac d100.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
img2tif -rb -s d100.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
img2tif -rb -s v100.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

img2dft v100.v mask.v v100.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.05 -rel=5 vb.tac v100.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

img2dft d100.v mask.v d100.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tac2svg -legend=y d100.svg -s ct.tac -l d100.tac
tacmatch -abs=1 -rel=1 ct.tac d100.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 1.0.1 \n"
printf " Test case 1: Option -LV. \n"
printf " Expected result: reasonable results are obtained. \n\n"

rm -f v101.v d101.v
$PROGRAM -LV -vb=v101.v dynamic.v blood.tac d101.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
img2tif -rb -s d101.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
img2tif -rb -s v101.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

img2dft v101.v mask.v v101.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.05 -rel=5 vb.tac v101.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

img2dft d101.v mask.v d101.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tac2svg -legend=y d101.svg -s ct.tac -l d101.tac
tacmatch -abs=1 -rel=1 ct.tac d101.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 1.0.2 \n"
printf " Test case 1: Option -RV. \n"
printf " Test case 2: Data contains no RV activity. \n"
printf " Expected result: No data is written. \n\n"

rm -f v102.v d102.v
$PROGRAM -RV -vb=v102.v dynamic.v blood.tac d102.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

if [ -f v102.v ]; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
if [ -f d102.v ]; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n 2.0.0 \n"
printf " Test case 1: Test image contains injection site. \n"
printf " Test case 2: Remove injection activity with option -RV.\n"
printf " Expected result: RV site is zeroed. \n\n"

rm -f v200.v d200.v
$PROGRAM -RV -vb=v200.v dynamicrv.v blood.tac d200.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
img2tif -rb -s d200.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
img2tif -rb -s v200.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

img2dft d200.v maskrv.v d200.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
dftmax d200.tac > d200.ift
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
iftisval -abs=0.01 d200.ift d200.tac 0
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n 2.0.1 \n"
printf " Test case 1: Test image contains injection site. \n"
printf " Test case 2: Option -LV.\n"
printf " Expected result: Other image regions are corrected normally. \n\n"

rm -f v201.v d201.v
$PROGRAM -LV -vb=v201.v dynamicrv.v blood.tac d201.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
img2tif -rb -s d201.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
img2tif -rb -s v201.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

img2dft d201.v mask.v d201.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tac2svg -legend=y d201.svg -s ct.tac -l d201.tac
tacmatch -abs=1 -rel=1 ct.tac d201.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n 2.0.2 \n"
printf " Test case 1: Test image contains injection site. \n"
printf " Test case 2: Default mode with both corrections.\n"
printf " Expected result: Resulting images reasonably close to correct. \n\n"

rm -f v202.v d202.v
$PROGRAM -vb=v202.v dynamicrv.v blood.tac d202.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
img2tif -rb -s d202.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
img2tif -rb -s v202.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

img2dft d202.v mask.v d202.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tac2svg -legend=y d202.svg -s ct.tac -l d202.tac
tacmatch -abs=1 -rel=1 ct.tac d202.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


img2dft v202.v mask.v v202.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.05 -rel=5 vb.tac v202.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"


imgmatch -abs=1 -rel=1 ct.v d202.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n===================================================================\n"
printf "   All passed! \n"
printf "===================================================================\n"
exit 0

