/** @file pbconv.c
 *  @brief Functions for computing RBC/plasma or plasma/blood curve based on published or
 *   measured functions or population averages.
 *  @copyright (c) Turku PET Centre
 *  @author Vesa Oikonen
 */
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <string.h>
#include <strings.h>
/*****************************************************************************/
#include "pbconv.h"
/*****************************************************************************/
/** Plasma water content */
#define PLASMA_WATER 0.94
/** Blood water content */
#define BLOOD_WATER 0.81
/** RBC water content */
#define RBC_WATER 0.63
/*****************************************************************************/

/*****************************************************************************/
/** Identify the tracer based on given string.
    Tracer code enums (tracerCode) can be found in pbconv.h.
    @return Returns the tracer code or a negative number if tracer is unknown.
    @sa rbc_plasma_ratio, plasma_to_blood_ratio
 */
int tracer_code(
  /** String containing the name of the tracer */
  char *tracer
) {
  //printf("tracer='%s'\n", tracer);
  if(strcasecmp(tracer, "NORBC")==0) return(NORBC);
  if(strcasecmp(tracer, "INWATER")==0) return(INWATER);
  if(strcasecmp(tracer, "FTHA")==0) return(NORBC);
  if(strcasecmp(tracer, "FDG")==0) return(FDG);
  if(strcasecmp(tracer, "FLUORIDE")==0) return(FLUORIDE);
  if(strcasecmp(tracer, "FBPA")==0) return(FBPA);
  if(strcasecmp(tracer, "FDOPA")==0) return(FDOPA);
  if(strcasecmp(tracer, "MEAIB")==0) return(MEAIB);
  if(strcasecmp(tracer, "6OHBTA1")==0) return(PIB);
  if(strcasecmp(tracer, "PIB")==0)     return(PIB);
  if(strncasecmp(tracer, "FLUMAZENIL", 3)==0) return(FLUMAZENIL);
  if(strncasecmp(tracer, "AH690", 5)==0) return(AH690);
  if(strncasecmp(tracer, "AH691", 5)==0) return(AH691);
  if(strcasecmp(tracer, "FMPEP-D2")==0) return(FMPEPD2);
  if(strncasecmp(tracer, "FMPEPD2", 5)==0) return(FMPEPD2);
  if(strncasecmp(tracer, "PALMITATE", 4)==0) return(PALMITATE);
  if(strcasecmp(tracer, "PK11195")==0) return(PK11195);
  if(strcasecmp(tracer, "PE2I")==0) return(PE2I);
  if(strcasecmp(tracer, "CARFENTANIL")==0) return(CARFENTANIL);
  if(strcasecmp(tracer, "METOMIDATE")==0) return(METOMIDATE);
  if(strcasecmp(tracer, "MTO")==0) return(METOMIDATE);
  if(strcasecmp(tracer, "ORM-B")==0) return(ORMB);
  if(strcasecmp(tracer, "ORMB")==0) return(ORMB);
  if(strcasecmp(tracer, "PBR28HAB")==0) return(PBR28HAB);
  if(strcasecmp(tracer, "PBR28MAB")==0) return(PBR28MAB);
  if(strcasecmp(tracer, "PBR28LAB")==0) return(PBR28LAB);
  if(strcasecmp(tracer, "SMW139")==0) return(SMW139);
  if(strcasecmp(tracer, "UCB-J")==0) return(UCBJ);
  if(strcasecmp(tracer, "UCBJ")==0) return(UCBJ);
  if(strcasecmp(tracer, "MOUSEFDG2")==0) return(MOUSEFDG2);
  if(strcasecmp(tracer, "MOUSEFDG1")==0) return(MOUSEFDG1);
  if(strcasecmp(tracer, "MOUSEFDG")==0) return(MOUSEFDG1);
  if(strcasecmp(tracer, "FDGMOUSE")==0) return(MOUSEFDG1);
  if(strcasecmp(tracer, "RATFDG")==0) return(RATFDG);
  if(strcasecmp(tracer, "FDGRAT")==0) return(RATFDG);
  return(-1);
}
/******************************************************************************/

/******************************************************************************/
/** Calculates RBC/PL ratio curve using specific functions and population average parameters.
    @return Returns non-zero in case an error is encountered, and specifically -1 if tracer code is
     unknown.
    @author Vesa Oikonen
    @sa plasma_to_blood_ratio, tracer_code
 */
int rbc_plasma_ratio(
  /** tracerCode */
  int tracer,
  /** Sample times (minutes from tracer injection) */
  double *t,
  /** Pointer to preallocated array where RBC/plasma ratio will be written */
  double *r,
  /** Nr of samples */
  int nr
) {
  int i;
  double K, A, B, C, D, Rmax, Th, dt;

  /* Check the arguments */
  if(t==NULL || r==NULL || nr<1) return(1);

  /* Calculate the population average RBC/PL ratio curve */
  switch(tracer) {
    case NORBC:
      for(i=0; i<nr; i++)
        r[i]= 0.0;
      break;
    case INWATER:
    case METOMIDATE:
      r[0]=(RBC_WATER/PLASMA_WATER);
      for(i=1; i<nr; i++)
        r[i]=r[0];
      break;
    case FDG:
      K=0.0012; B=0.80;
      for(i=0; i<nr; i++)
        r[i]= K*t[i] + B;
      break;
    case FBPA:
      K=0.00888; B=0.0;
      for(i=0; i<nr; i++)
        r[i]= K*t[i] + B;
      break;
    case FDOPA:
      Rmax=1.446; Th=83.56;
      for(i=0; i<nr; i++)
        r[i]= (Rmax*t[i]) / (Th + t[i]);
      break;
    case MEAIB:
      K=0.00398; B=0.0;
      for(i=0; i<nr; i++)
        r[i]= K*t[i] + B;
      break;
    case PIB:
      /*A=9.470E-01; B=1.9200E+00; C=8.2395E+01;*/
      A=7.4150E-01; B=4.0618E+00; C=1.2182E+04;
      for(i=0; i<nr; i++) {
        r[i]= 1.0 - (A*pow(t[i], B)/(pow(t[i], B) + C));
        r[i]*= (RBC_WATER/PLASMA_WATER);
      }
      break;
    case FLUMAZENIL:
      A=6.89E-01; B=6.89E-01; C=4.57E+01;
      for(i=0; i<nr; i++) {
        r[i]= 1.0 - (A*pow(t[i], B)/(pow(t[i], B) + C));
        r[i]*= (RBC_WATER/PLASMA_WATER);
      }
      break;
    case AH690:
      A=7.6467E-01; B=2.0124E+00; C=5.1609E+01; D=1.1381E-03;
      for(i=0; i<nr; i++)
        r[i]= 1.0 - (A*pow(t[i], B)/(pow(t[i], B) + C) + D*t[i]);
      break;
    case AH691:
      A=8.1894E-01; B=1.1967E+00; C=1.7223E+00; D=1.1873E-03;
      for(i=0; i<nr; i++)
        r[i]= 1.0 - (A*pow(t[i], B)/(pow(t[i], B) + C) + D*t[i]);
      break;
    case FMPEPD2: // based on measurements in TPC
      A=1.07741e+001; B=1.29526e+001; C=1.28; D=3.7e-002;
      for(i=0; i<nr; i++) {
        dt=t[i]-0.5;
        if(dt<=0.0) r[i]=0.0; else {
          r[i]= (C*pow(dt, C-1.0))/(pow(B, C)+pow(dt, C)) -
                (C*pow(dt, 2.0*C-1.0))/pow(pow(B, C)+pow(dt, C), 2.0) +
                (D*pow(dt, C))/(pow(B, C)+pow(dt, C));
          r[i]*=A;
        }
      }
      break;
    case PALMITATE:
      A=3.906489E-003; B=-9.939087E-002; C=2.713401E-004; D=4.740762E-003;
      for(i=0; i<nr; i++) {
        dt=t[i]-5.0;
        if(dt<0.0) r[i]=0.0; else
        r[i]= (A*dt + C*dt*dt) / (1.0 + B*dt + D*dt*dt);
      }
      break;
    case PK11195:
      A=1.9087E-01; B=3.0773E+00; C=2.8266E+02;
      for(i=0; i<nr; i++) {
        r[i]= A*pow(t[i], B)/(pow(t[i], B) + C);
      }
      break;
    case PE2I:
      A=2.063642; B=7.460386E-001; C=6.812810E+001;
      for(i=0; i<nr; i++) {
        r[i]= A*pow(t[i], B)/(pow(t[i], B) + C);
      }
      break;
    case CARFENTANIL: // based on 8 studies in TPC
      A=1.857965E-001; B=5.388415; C=1.344971E+007; D=3.015209E-001;
      for(i=0; i<nr; i++) {
        r[i]= A*pow(t[i], B)/(pow(t[i], B) + C) + D;
      }
      break;
    case ORMB:
      A=0.4281961; B=1.14880; C=4.013195; D=1.863528;
      for(i=0; i<nr; i++) {
        dt=t[i]-D;
        if(dt<=0.0) r[i]=0.0;
        else r[i]= A*pow(t[i], B)/(pow(t[i], B) + C);
      }
/*
      A=0.428495; B=1.307704; C=7.609658;
      for(i=0; i<nr; i++) {
        r[i]= A*pow(t[i], B)/(pow(t[i], B) + C);
      }
*/
      break;
    case SMW139:
      // From TPC analysis, n=14
      A=0.006459; B=0.06913; C=0.1278;
      for(i=0; i<nr; i++) {
        double e=exp(-B*t[i]);
        r[i]=A*(t[i]*e + (C/(B*B))*(1.0-(B*t[i]+1.0)*e));
      }
      break;
    case UCBJ:
      // Median parameters from TPC analysis, n=7
      {
      double A1=0.691911;
      double L1=0.9;
      double A2=0.174026;
      double L2=0.029;
      double A3=0.357205;
      double L3=-0.004;
      for(i=0; i<nr; i++) {
        if(!(t[i]>0.0)) r[i]=0.0;
        else r[i]=(A1*t[i] - A2 - A3)*exp(-L1*t[i]) + A2*exp(-L2*t[i]) + A3*exp(-L3*t[i]);
      }
      }
      break;

    default:
      return(-1);
  }

  return(0);
}
/******************************************************************************/

/******************************************************************************/
/** Calculates plasma/blood ratio curve using specific functions and population average parameters.
    This function can only be used when the impact of variable hematocrit can be ignored.
    @return Returns non-zero in case an error is encountered, and specifically -1 if tracer code 
     is unknown.
    @author Vesa Oikonen
    @sa rbc_plasma_ratio, tracer_code
 */
int plasma_to_blood_ratio(
  /** tracerCode */
  int tracer,
  /** Sample times (minutes from tracer injection) */
  double *t,
  /** Pointer to preallocated array where plasma/blood ratio will be written */
  double *r,
  /** Nr of samples */
  int nr
) {
  int i;
  double K, K2, A, B, C, D;

  /* Check the arguments */
  if(t==NULL || r==NULL || nr<1) return(1);

  /* Calculate the plasma/blood ratio curve */
  switch(tracer) {
    case INWATER:
    case METOMIDATE:
      r[0]=(PLASMA_WATER/BLOOD_WATER);
      for(i=1; i<nr; i++)
        r[i]=r[0];
      break;
    case FLUORIDE:
      // From Hawkins et al., JNM 1992;33:633-642.
      A=1.23; B=-0.00123;
      for(i=0; i<nr; i++) r[i]= A+B*t[i];
      break;
    case MOUSEFDG1:
      // From Yu et al JNM 2009;50(6):966-973, referring to Wu et al JNM 2007;48(5):837-845.
      A=0.39; K=0.19; C=1.17;  
      for(i=0; i<nr; i++) r[i]= A*exp(-K*t[i]) + C;
      break;
    case MOUSEFDG2:
      // From Huang et al JNM 2017;58(4):611-616.
      A=0.619; K=0.0842; C=1.051;  
      for(i=0; i<nr; i++) r[i]= A*exp(-K*t[i]) + C;
      break;
    case RATFDG:
      // From Weber et al Eur J Nucl Med 2002;29:319–323.
      A=0.51; K=4.79; B=0.3; K2=337.; C=0.8;
      for(i=0; i<nr; i++)
        r[i]= A*exp(-(M_LN2/K)*t[i]) + B*exp(-(M_LN2/K2)*t[i]) + C;
      break;
    case PBR28HAB:
      // From TPC analysis
      A=-4.374460E-001; B=2.515638E+000; C=2.945167E+003; D=1.154175E+000;
      for(i=0; i<nr; i++) {K=pow(t[i], B); r[i]=1.0/((A*K/(K+C))+D);}
      break;
    case PBR28MAB:
      // From TPC analysis
      A=-2.516136E-001; B=1.713713E+000; C=2.349197E+002; D=9.116462E-001;
      for(i=0; i<nr; i++) {K=pow(t[i], B); r[i]=1.0/((A*K/(K+C))+D);}
      break;
    case PBR28LAB:
      // From TPC analysis
      A=8.020889E-002; B=1.835577E+000; C=1.908775E+001; D=5.329330E-001;
      for(i=0; i<nr; i++) {K=pow(t[i], B); r[i]=1.0/((A*K/(K+C))+D);}
      break;
    default:
      return(-1);
  }

  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
