#!/bin/bash
#: Title      : test_fit_ppf
#: Date       : 2023-04-21
#: Author     : "Vesa Oikonen" <vesa.oikonen@utu.fi>
#: Options    : None

# Set the name of executable to test
if [ -n "${OS+1}" ] && [ "$OS" = "Windows_NT" ]; then
EXT=.exe;
else
EXT=;
fi
PROGRAM=../../fit_ppf$EXT;
if [ ! -f $PROGRAM ]; then
  printf "Failed: executable does not exist.\n"
  exit 1
fi

printf "=====================================================================\n"
printf "creating test data for %s\n" $PROGRAM
printf "=====================================================================\n"


printf "\n verifying that required data exists \n"

if [ ! -f orig_hill.fit ] || [ ! -f orig_hilldelay.fit ]; then
  printf "Failed: required data does not exist.\n"
  exit 1
fi
if [ ! -f orig_hillext.fit ] || [ ! -f orig_hillm3.fit ] ; then
  printf "Failed: required data does not exist.\n"
  exit 1
fi
if [ ! -f orig_hillm3level.fit ] || [ ! -f orig_hillm3delay.fit ] ; then
  printf "Failed: required data does not exist.\n"
  exit 1
fi
if [ ! -f orig_ppf.fit ] || [ ! -f orig_ppf2.fit ] ; then
  printf "Failed: required data does not exist.\n"
  exit 1
fi
if [ ! -f orig_gcdf.fit ] ; then
  printf "Failed: required data does not exist.\n"
  exit 1
fi
if [ ! -f orig_ep.fit ] ; then
  printf "Failed: required data does not exist.\n"
  exit 1
fi



if [ ! -f data1.dat ] || [ ! -f data1.bld ]; then
  fit2dat -x=1,5,10,20,30,45,60 orig_hill.fit data1.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacformat -format=PMOD -hdr=no -mid data1.dat data1.bld
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi
if [ ! -f data1.csv ] || [ ! -f data1.tsv ]; then
  tacformat -format=CSV-UK -hdr=yes -mid data1.dat data1.csv
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacformat -format=TSV-INT -hdr=yes -mid data1.dat data1.tsv
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi

if [ ! -f data1_multi.dat ] ; then
  cp -f data1.dat data1_multi.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  fit2dat -x=1,5,10,20,30,45,60 orig_ppf.fit temp.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacadd -force data1_multi.dat temp.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi

if [ ! -f data1_delay1.dat ] ; then
  fit2dat -x=0.5,1,5,10,20,30,45,60 orig_hilldelay.fit data1_delay.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi

if [ ! -f data1_na.dat ] ; then
  cp -f data1.dat data1_na.dat
  echo 100 . >> data1_na.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi

if [ ! -f data1_snr1.dat ] ; then
  fit2dat -x=6 orig_hilldelay.fit data1_snr1.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi

if [ ! -f data1_snr2.dat ] ; then
  fit2dat -x=6,20 orig_hilldelay.fit data1_snr2.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi

if [ ! -f data1_snr3.dat ] ; then
  fit2dat -x=2,6,20 orig_hilldelay.fit data1_snr3.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi

if [ ! -f data1_snr4.dat ] ; then
  fit2dat -x=0.5,2,6,20 orig_hilldelay.fit data1_snr4.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi

if [ ! -f data1_snr5.dat ] ; then
  fit2dat -x=0.5,2,6,20,30 orig_hilldelay.fit data1_snr5.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi

if [ ! -f data1_snr6.dat ] ; then
  fit2dat -x=0.5,2,6,10,20,30 orig_hilldelay.fit data1_snr6.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi

if [ ! -f data1_snr7.dat ] ; then
  fit2dat -x=0.5,1.5,5,10,20,30,45 orig_hilldelay.fit data1_snr7.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi

if [ ! -f data1_catenated1.dat ] ; then
  fit2dat -x=0.5,5,10,30,60 orig_hilldelay.fit temp1.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacformat -f=simple -mid -hdr=no temp1.dat data1_catenated1.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  fit2dat -x=1,15,20,45 orig_hilldelay.fit temp2.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacformat -f=simple -mid -hdr=no temp2.dat temp2.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  cat temp2.dat >> data1_catenated1.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi

if [ ! -f data1_catenated2.dat ] ; then
  fit2dat -x=0.5,1,5,10,15,20,30,45,60 orig_hilldelay.fit data1_catenated2.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi

if [ ! -f data1_duplicated.dat ] ; then
  dftcat -both data1.dat data1.dat data1_duplicated.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacformat -f=simple -mid -hdr=no data1_duplicated.dat data1_duplicated.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi

if [ ! -f data1_perc.dat ] ; then
  tacformat -f=simple -mid -hdr=no data1.dat data1_perc.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  taccalc data1_perc.dat x 100 data1_perc.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi

if [ ! -f data1_onehigh.dat ] ; then
  echo 0.5 1.05 > data1_onehigh.dat
  fit2dat -x=1,5,10,15,20,30,45,60 orig_hilldelay.fit temp.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacformat -f=simple -mid -hdr=no temp.dat temp.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  cat temp.dat >> data1_onehigh.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi

if [ ! -f data2.dat ] ; then
  fit2dat -x=0.5,1,2.5,5,10,15,20,30,40,50,60 orig_hillext.fit data2.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacformat -f=simple -mid -hdr=no data2.dat data2.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi

if [ ! -f data3.dat ] ; then
  fit2dat -x=0.5,1,2.5,5,10,15,20,30,40,50,60 orig_ppf.fit data3.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacformat -f=simple -mid -hdr=no data3.dat data3.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi

if [ ! -f data4.dat ] || [ ! -f data4z.dat ]; then
  fit2dat -x=0.5,1,2.5,5,10,15,20,30,40,50,60 orig_ppf2.fit data4.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacformat -f=simple -mid -hdr=no data4.dat data4.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  fit2dat -x=0,0.5,1,2.5,5,10,15,20,30,40,50,60 orig_ppf2.fit data4z.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacformat -f=simple -mid -hdr=no data4z.dat data4z.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi

if [ ! -f data2_bad.dat ] ; then
  cp -f data2.dat data2_bad.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  echo 80 0.6 >> data2_bad.dat
  echo 90 0.6 >> data2_bad.dat
  echo 100 0.6 >> data2_bad.dat
  echo 110 0.6 >> data2_bad.dat
  echo 120 0.6 >> data2_bad.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi

if [ ! -f data4_bad.dat ] ; then
  cp -f data4.dat data4_bad.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  echo 80 0.6 >> data4_bad.dat
  echo 90 0.6 >> data4_bad.dat
  echo 100 0.6 >> data4_bad.dat
  echo 110 0.6 >> data4_bad.dat
  echo 120 0.6 >> data4_bad.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi

if [ ! -f data5_weights.dat ] ; then
  fit2dat -x=0.5,1.5,2.5,3.5,4.5,5.5,7,9 orig_ppf.fit data5_weights.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacweigh -wf data5_weights.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi

if [ ! -f data5_noweights.dat ] ; then
  tacformat -f=simple -mid -hdr=no data5_weights.dat data5_noweights.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi

if [ ! -f data_hillm3.dat ] ; then
  fit2dat -a=60 orig_hillm3.fit data_hillm3.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi

if [ ! -f data_hillm3level.dat ] ; then
  fit2dat -a=60 orig_hillm3level.fit data_hillm3level.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi

if [ ! -f data_hillm3delay.dat ] ; then
  fit2dat -a=60 orig_hillm3delay.fit data_hillm3delay.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi

if [ ! -f data_pfm3.dat ] ; then
  fit2dat -a=60 orig_pfm3.fit data_pfm3.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi

if [ ! -f data_pfm3level.dat ] ; then
  fit2dat -a=60 orig_pfm3level.fit data_pfm3level.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi

if [ ! -f data_pfm3delay.dat ] ; then
  fit2dat -a=60 orig_pfm3delay.fit data_pfm3delay.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi

if [ ! -f data_gcdf.dat ] ; then
  fit2dat -x=0.5,1,1.5,2.5,5,10,15,20,30,40,50,60 orig_gcdf.fit data_gcdf.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi

if [ ! -f data_ep.dat ] ; then
  fit2dat -x=0.5,1,1.5,2.5,5,7.5,10,15,20,30,40,50,60 orig_ep.fit data_ep.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi



printf "\n\n"
printf "=====================================================================\n"
printf "testing %s\n" $PROGRAM
printf "=====================================================================\n"



printf "\n 0.1.0 \n"
printf " Test case: Unknown option. \n"
printf " Expected result: Error. \n"

rm -f stdout.txt
$PROGRAM -stupidoption data1.dat &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error "invalid option '-stupidoption'."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n 0.2.0 \n"
printf " Test case: Defaults. \n"
printf " Expected result: Default values of variables are correct. \n"

rm -f stdout.txt
$PROGRAM -d2 data1.dat &> stdout.txt
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

iftisval stdout.txt noDivide "0"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt weighting "0"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt last_col_is_weight "0"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt model "2"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt type "846"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt MRL_check "0"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt min_delay
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt mdelay_joint "1"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt frwgt[0] "1"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
iftisval stdout.txt frwgt[1] "1"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
iftisval stdout.txt frwgt[2] "1"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
iftisval stdout.txt frwgt[3] "1"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n 0.2.1 \n"
printf " Test case: Options are correctly interpreted. \n"
printf " Expected result: Values of variables are correct. \n"

rm -f stdout.txt
$PROGRAM -d2 -model=HILL -wf -mrl -nd -delaymin=0,5 -svg=test.svg -mdelay=separated -wp=2 -wm1=1.5 -wm2=1 -wm3=0.5 data1.dat &> stdout.txt
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

iftisval stdout.txt noDivide "1"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt weighting "2"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt last_col_is_weight "0"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt model "2"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt MRL_check "1"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt min_delay "0.5"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt mdelay_joint "0"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt svgfile "test.svg"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt frwgt[0] "2"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
iftisval stdout.txt frwgt[1] "1.5"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
iftisval stdout.txt frwgt[2] "1"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
iftisval stdout.txt frwgt[3] "0.5"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"




printf "\n 0.3.0 \n"
printf " Test case: Missing argument. \n"
printf " Expected result: Error. \n"

rm -f stdout.txt
$PROGRAM -d -model=HILL -wf -mrl -nd &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error "missing command-line argument; use option --help"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 0.3.1 \n"
printf " Test case: Appropriate arguments entered. \n"
printf " Expected result: Arguments are correctly identified. \n"

rm -f stdout.txt
$PROGRAM -d2 data1.dat test.fit &> stdout.txt
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt dfile "data1.dat"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt ffile "test.fit"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 0.3.2 \n"
printf " Test case: Extra argument entered. \n"
printf " Expected result: Error. \n"

rm -f stdout.txt
$PROGRAM -d2 data1.dat test.fit extra &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error "too many arguments: 'extra'."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n\n---------------------------------------------------------------\n\n"

printf "\n 1.0.0 \n"
printf " Test case: Input file does not exist. \n"
printf " Expected result: Error. \n"

rm -f stdout.txt
$PROGRAM -d2 nonexistingfile.dat test.fit &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Error in reading 'nonexistingfile.dat'" "cannot open file"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 1.1.0 \n"
printf " Test case: Input file in wrong format. \n"
printf " Expected result: Error. \n"

rm -f test.dat stdout.txt
echo meh ngh      >  test.dat
echo zzzzhhhh bop >> test.dat
$PROGRAM -d2 test.dat test.fit &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Error in reading 'test.dat'" "wrong format"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 1.2.0 \n"
printf " Test case: Input file contains more than one fraction curve. \n"
printf " Expected result: First is used, and warning is printed. \n"

rm -f output.fit output2.fit stdout.txt
rm -f output.dat output2.dat

$PROGRAM -model=HILL data1.dat output.fit
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
$PROGRAM -model=HILL data1_multi.dat output2.fit &> stdout.txt
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Warning "extra columns in data1_multi.dat are ignored."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

fit2dat -f=data1.dat output.fit output.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fit2dat -f=data1.dat output2.fit output2.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
tacmatch -abs=0.01 -rel=1 -x=y -y=y output.dat output2.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n 1.3.0 \n"
printf " Test case: Fraction file must contain at least one sample more \n"
printf "            than is the number of fitted parameters. \n"
printf " Expected result: Error or success, depending on data and model. \n\n"

printf "\n One sample more than parameters \n\n"

$PROGRAM -model=HILL data1_snr6.dat output.fit
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n 1.3.1 \n"
printf "\n Equal number of parameters and samples \n\n"

rm -f stdout.txt
$PROGRAM -model=HILL data1_snr5.dat output2.fit &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error "too few samples for fitting in data1_snr5.dat"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n 1.3.2 \n"
printf "\n ... fixing one parameter helps ... \n\n"

rm -f stdout.txt
$PROGRAM -model=HILL -d=1 data1_snr5.dat output2.fit &> stdout.txt
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n 1.3.3 \n"
printf "\n with one sample less the same error \n\n"

rm -f stdout.txt
$PROGRAM -model=HILL -d=1 data1_snr4.dat output2.fit &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error "too few samples for fitting in data1_snr4.dat"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n 1.3.4 \n"
printf "\n ... fixing one more parameter helps ... \n\n"

rm -f stdout.txt
$PROGRAM -model=HILL -d=1 -e=0.5 data1_snr4.dat output2.fit &> stdout.txt
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n 1.3.5 \n"
printf "\n One free parameter can be fitted to two samples \n\n"

rm -f stdout.txt
$PROGRAM -model=HILL -a=0.6002 -b=2.9029 -d=1 -e=0.5 data1_snr2.dat output2.fit &> stdout.txt
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n 1.3.6 \n"
printf "\n ... but not to one sample \n\n"

rm -f stdout.txt
$PROGRAM -model=HILL -a=0.6002 -b=2.9029 -d=1 -e=0.5 data1_snr1.dat output2.fit &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error "too few samples for fitting in data1_snr1.dat"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 1.3.7 \n"
printf "\n All parameters can not be fixed \n\n"

rm -f stdout.txt
$PROGRAM -model=HILL -a=0.6002 -b=2.9029 -c=1012.7 -d=1 -e=0.5 data1_snr1.dat output2.fit &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error "too many parameters were fixed."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n 1.4.0 \n"
printf " Test case: NA in file. \n"
printf " Expected result: sample gets zero weight. \n\n"
printf " Expected result: NA is not counted in sample nr. \n\n"

rm -f stdout.txt
$PROGRAM -d3 -model=HILL data1_na.dat output.fit &> stdout.txt
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt data_weights "1, 1, 1, 1, 1, 1, 1, 0"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

rm -f stdout.txt
$PROGRAM -d3 -wf -model=HILL data1_na.dat output.fit &> stdout.txt
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt data_weights "0.264463, 0.297521, 0.495868, 0.661157, 0.826446, 0.991736, 1.81818, 0"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

fit2res output.fit
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval output.res "Data nr" 7
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 1.5.0 \n"
printf " Test case: Fraction file is catenated from two studies. \n"
printf " Expected result: Results are the same whether data is sorted or not. \n\n"

rm -f output.res output2.res

$PROGRAM -model=HILL data1_catenated1.dat output.fit
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
$PROGRAM -model=HILL data1_catenated2.dat output2.fit
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
fit2res output.fit output2.fit
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval output.res "Data nr" 9
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval output2.res "Data nr" 9
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
resmatch -header=n -param=y output.res output2.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 1.5.1 \n"
printf " Test case: Fraction file contains duplicate samples. \n"
printf " Expected result: Results are the same but sample nr is double. \n\n"

rm -f output.res output2.res

$PROGRAM -model=HILL data1.dat output.fit
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
$PROGRAM -model=HILL data1_duplicated.dat output2.fit
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
fit2res output.fit output2.fit
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval output.res "Data nr" 7
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval output2.res "Data nr" 14
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
resmatch -res=2-6 -abs=1.0E-06 output.res output2.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 1.6.0 \n"
printf " Test case: File contains percentages instead of fractions. \n"
printf " Expected result: Automatically converted to fractions with warning. \n\n"

rm -f output.res output2.res stdout.txt

$PROGRAM -model=HILL data1.dat output.fit
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

$PROGRAM -model=HILL data1_perc.dat output2.fit &> stdout.txt
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Warning" "converting percentages to fractions."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

fit2res output.fit output2.fit
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

resmatch -res=2-6 -abs=1.0E-06 output.res output2.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n 1.6.1 \n"
printf " Test case: File contains a fraction that exceeds 1. \n"
printf " Expected result: Assumed to be percentages, leads to stupid results, \n"
printf "                  unless option -nd is used. \n\n"

rm -f stdout.txt
$PROGRAM -model=HILL data1_onehigh.dat output.fit &> stdout.txt
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Warning" "converting percentages to fractions."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

rm -f stdout.txt
$PROGRAM -nd -model=HILL data1_onehigh.dat output.fit &> stdout.txt
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Warning" "converting percentages to fractions."
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n\n---------------------------------------------------------------\n\n"

printf "\n 1.7.0 \n"
printf " Test case 1: Result file is written in correct format. \n"
printf " Test case 2: Backup is made of existing output file. \n"
printf " Expected result: Result file can be processed by other program. \n\n"

rm -f output.fit output.fit.bak output.dat

$PROGRAM -model=HILL data1_delay.dat output.fit
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
fit2dat -f=data1_delay.dat output.fit output.dat 
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.01 -rel=2 data1_delay.dat output.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

$PROGRAM -model=HILL data1.dat output.fit
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
fit2dat -f=data1.dat output.fit output.dat 
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.01 -rel=2 data1.dat output.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

fit2dat -f=data1_delay.dat output.fit.bak output.dat 
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.01 -rel=2 data1_delay.dat output.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n 1.7.1 \n"
printf " Test case: Output filename not given. \n"
printf " Expected result: Fit results written to stdout. \n\n"

rm -f output.fit output.fit.bak output.dat

$PROGRAM -model=HILL data1_delay.dat > output.fit
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
fit2dat -f=data1_delay.dat output.fit output.dat 
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.01 -rel=2 data1_delay.dat output.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n===================================================================\n"
printf "   Testing that reasonable parent fraction data is well fitted \n"
printf "===================================================================\n"

printf "\n 2.0.0 \n"
printf " Test case: Extended Hill function. \n"
printf " Expected result: Fitted curve and parameters are close to correct. \n\n"

rm -f output.fit output.res output.dat

$PROGRAM -model=HILL data2.dat output.fit
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
fit2dat -f=data2.dat output.fit output.dat 
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.01 -rel=2 data2.dat output.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

fit2res output.fit orig_hillext.fit
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
resmatch -around -res=2-6 orig_hillext.res output.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 2.0.1 \n"
printf " Test case: Extended Hill function and ODR. \n"
printf " Expected result: Fitted curve and parameters are close to correct. \n\n"

rm -f output.fit output.res output.dat

$PROGRAM -model=HILL -min=ODR data2.dat output.fit
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
fit2dat -f=data2.dat output.fit output.dat 
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.01 -rel=2 data2.dat output.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

fit2res output.fit orig_hillext.fit
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
resmatch -around -res=2-6 orig_hillext.res output.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n\n---------------------------------------------------------------\n\n"


printf "\n 2.1.0 \n"
printf " Test case: Extended Power function. \n"
printf " Expected result: Fitted curve and parameters are close to correct. \n\n"

rm -f output.fit output.res output.dat

$PROGRAM -model=PF data3.dat output.fit
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
fit2dat -f=data3.dat output.fit output.dat 
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.01 -rel=2 data3.dat output.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

fit2res output.fit orig_ppf.fit
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
resmatch -roughly -res=2,5,6 orig_ppf.res output.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
resmatch -abs=0.01 -res=4 orig_ppf.res output.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
resmatch -abs=0.0001 -res=7 orig_ppf.res output.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 2.1.1 \n"
printf " Test case: Extended Power function and ODR. \n"
printf " Expected result: Fitted curve and parameters are close to correct. \n\n"

rm -f output.fit output.res output.dat

$PROGRAM -min=ODR -model=PF data3.dat output.fit
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
fit2dat -f=data3.dat output.fit output.dat 
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.01 -rel=2 data3.dat output.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

fit2res output.fit orig_ppf.fit
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
resmatch -roughly -res=2,5,6 orig_ppf.res output.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
resmatch -abs=0.01 -res=4 orig_ppf.res output.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
resmatch -abs=0.0001 -res=7 orig_ppf.res output.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n 2.1.2 \n"
printf " Test case: Extended Power function. \n"
printf " Expected result: Fitted curve and parameters are close to correct. \n\n"

rm -f output.fit output.res output.dat

$PROGRAM -model=PF data4.dat output.fit
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
fit2dat -f=data4.dat output.fit output.dat 
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.01 -rel=2 data4.dat output.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

fit2res output.fit orig_ppf2.fit
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
resmatch -around -res=2-6 orig_ppf2.res output.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 2.1.3 \n"
printf " Test case: Extended Power function and ODR. \n"
printf " Expected result: Fitted curve and parameters are close to correct. \n\n"

rm -f output.fit output.res output.dat

$PROGRAM -model=PF -min=ODR data4.dat output.fit
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
fit2dat -f=data4.dat output.fit output.dat 
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.01 -rel=2 data4.dat output.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

fit2res output.fit orig_ppf2.fit
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
resmatch -around -res=2-6 orig_ppf2.res output.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n\n---------------------------------------------------------------\n\n"


printf "\n 2.2.0 \n"
printf " Test case: Gamma CDF function. \n"
printf " Expected result: Fitted curve and parameters are close to correct. \n\n"

rm -f output.fit output.res output.dat

$PROGRAM -model=GCDF data_gcdf.dat output.fit
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
fit2dat -f=data_gcdf.dat output.fit output.dat 
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.01 -rel=2 data_gcdf.dat output.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

fit2res output.fit orig_gcdf.fit
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
resmatch -roughly -res=2,5,6 orig_gcdf.res output.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
resmatch -abs=0.01 -res=4 orig_gcdf.res output.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
resmatch -abs=0.0001 -res=7 orig_gcdf.res output.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n\n---------------------------------------------------------------\n\n"


printf "\n 2.3.0 \n"
printf " Test case: Exp/Power function. \n"
printf " Expected result: Fitted curve and parameters are close to correct. \n\n"

rm -f output.fit output.res output.dat

$PROGRAM -model=EP data_ep.dat output.fit
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
fit2dat -f=data_ep.dat output.fit output.dat 
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.01 -rel=2 data_ep.dat output.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

fit2res output.fit orig_ep.fit
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
resmatch -roughly -res=2,5,6 orig_ep.res output.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
resmatch -abs=0.01 -res=4 orig_ep.res output.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
resmatch -abs=0.0001 -res=7 orig_ep.res output.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



# MHILL and MPF too slow to test every time, and MPF does not work well
if [ 0 -eq 1 ]; then

printf "\n===================================================================\n"
printf "   Testing that metabolite fraction models provide reasonable fits \n"
printf "===================================================================\n"

printf "\n 3.0.0 \n"
printf " Test case: Hill metabolite model, parent level initially 1. \n"
printf " Expected result: Fitted curve and parameters are close to correct. \n\n"

rm -f output.fit output.res output.dat

$PROGRAM -model=mhill -d=1 -svg=hillm3.svg data_hillm3.dat output.fit
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

fit2dat -f=data_hillm3.dat output.fit output.dat 
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.01 -rel=2 data_hillm3.dat output.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

fit2res output.fit orig_hillm3.fit
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
resmatch -around -res=2-16 orig_hillm3.res output.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 3.0.1 \n"
printf " Test case: Hill metabolite model, parent level initially less than 1. \n"
printf " Expected result: Fitted curve and parameters are close to correct. \n\n"

rm -f output.fit output.res output.dat

$PROGRAM -model=mhill -e=1 -svg=hillm3level.svg data_hillm3level.dat output.fit
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

fit2dat -f=data_hillm3level.dat output.fit output.dat 
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.01 -rel=2 data_hillm3level.dat output.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

fit2res output.fit orig_hillm3level.fit
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
resmatch -around -res=2-16 orig_hillm3level.res output.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 3.0.2 \n"
printf " Test case: Hill metabolite model, different delay times. \n"
printf " Expected result: Fitted curve and parameters are close to correct. \n\n"

rm -f output.fit output.res output.dat

$PROGRAM -model=mhill -d=1 -mdelay=separated -svg=hillm3delay.svg data_hillm3delay.dat output.fit
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

fit2dat -f=data_hillm3delay.dat output.fit output.dat 
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.01 -rel=2 data_hillm3delay.dat output.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

fit2res output.fit orig_hillm3delay.fit
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
resmatch -around -res=2-16 orig_hillm3delay.res output.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n\n---------------------------------------------------------------\n\n"



printf "\n 3.1.0 \n"
printf " Test case: PF metabolite model, parent fraction starts from 1. \n"
printf " Expected result: Fitted curve and parameters are close to correct. \n\n"

rm -f output.fit output.res output.dat

$PROGRAM -model=mpf -d=1 -svg=pfm3.svg data_pfm3.dat output.fit
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

fit2dat -f=data_pfm3.dat output.fit output.dat 
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.01 -rel=2 data_pfm3.dat output.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

fit2res output.fit orig_pfm3.fit
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
resmatch -abs=1.0E-04 -res=2-16 orig_pfm3.res output.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 3.1.1 \n"
printf " Test case: PF metabolite model, parent fraction less than 1. \n"
printf " Expected result: Fitted curve and parameters are close to correct. \n\n"

rm -f output.fit output.res output.dat

$PROGRAM -model=mpf -e=1 -svg=pfm3level.svg data_pfm3level.dat output.fit
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

fit2dat -f=data_pfm3level.dat output.fit output.dat 
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.01 -rel=2 data_pfm3level.dat output.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

fit2res output.fit orig_pfm3level.fit
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
resmatch -abs=1.0E-04 -res=2-16 orig_pfm3level.res output.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 3.1.2 \n"
printf " Test case: PF metabolite model, different delay times. \n"
printf " Expected result: Fitted curve and parameters are close to correct. \n\n"

rm -f output.fit output.res output.dat

$PROGRAM -model=mpf -d=1 -mdelay=separated -svg=pfm3delay.svg data_pfm3delay.dat output.fit
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

fit2dat -f=data_pfm3delay.dat output.fit output.dat 
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.01 -rel=2 data_pfm3delay.dat output.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

fit2res output.fit orig_pfm3delay.fit
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
resmatch -abs=1.0E-03 -res=7-16 orig_pfm3delay.res output.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
resmatch -abs=0.8 -res=2-6 orig_pfm3delay.res output.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

fi
# Continue testing from here




printf "\n===================================================================\n"
printf "   Fitting data that cannot be well fitted. \n"
printf "===================================================================\n"

printf "\n 4.0.0 \n"
printf " Test case: Extended Hill function without option -MRL \n"
printf " Expected result: Warning of bad fit. \n\n"

rm -f output.fit stdout.txt

$PROGRAM -model=Hill -svg=data2_bad.svg data2_bad.dat output.fit &> stdout.txt
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
if [ ! -f output.fit ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Warning" "bad fit."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n 4.0.1 \n"
printf " Test case: Extended Hill function with option -MRL \n"
printf " Expected result: Error of bad fit. \n\n"

rm -f output.fit stdout.txt

$PROGRAM -MRL -model=Hill -svg=data2_bad.svg data2_bad.dat output.fit &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
if [ -f output.fit ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Error" "bad fit."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n 4.1.0 \n"
printf " Test case: Extended Power function without option -MRL \n"
printf " Expected result: Warning of bad fit. \n\n"

rm -f output.fit stdout.txt

$PROGRAM -model=PF -svg=data4_bad.svg data4_bad.dat output.fit &> stdout.txt
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
if [ ! -f output.fit ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Warning" "bad fit."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n 4.1.1 \n"
printf " Test case: Extended Power function with option -MRL \n"
printf " Expected result: Error of bad fit. \n\n"

rm -f output.fit stdout.txt

$PROGRAM -mrl -model=PF -svg=data4_bad.svg data4_bad.dat output.fit &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
if [ -f output.fit ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Error" "bad fit."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n===================================================================\n"
printf "   Fixed parameters. \n"
printf "===================================================================\n"


printf "\n 5.0.0 \n"
printf " Test case: Parameter A is fixed. \n"
printf " Expected result 1: Test print shows correct parameter limits. \n"
printf " Expected result 2: Fitted parameter is as required. \n\n"

rm -f output.fit stdout.txt

$PROGRAM -d2 -a=0.6 -model=PF data4.dat output.fit &> stdout.txt
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "limit[1]" "0.6 - 0.6"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "par[1]" "0.6"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n 5.0.1 \n"
printf " Test case: Parameter B is fixed. \n"
printf " Expected result 1: Test print shows correct parameter limits. \n"
printf " Expected result 2: Fitted parameter is as required. \n\n"

rm -f output.fit stdout.txt

$PROGRAM -d2 -b=1,2 -model=PF data4.dat output.fit &> stdout.txt
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "limit[2]" "1.2 - 1.2"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "par[2]" "1.2"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n 5.0.2 \n"
printf " Test case: Parameter C is fixed. \n"
printf " Expected result 1: Test print shows correct parameter limits. \n"
printf " Expected result 2: Fitted parameter is as required. \n\n"

rm -f output.fit stdout.txt

$PROGRAM -d2 -c=0.35 -model=PF data4.dat output.fit &> stdout.txt
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "limit[3]" "0.35 - 0.35"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "par[3]" "0.35"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n 5.0.3 \n"
printf " Test case: Parameter D is fixed. \n"
printf " Expected result 1: Test print shows correct parameter limits. \n"
printf " Expected result 2: Fitted parameter is as required. \n\n"

rm -f output.fit stdout.txt

$PROGRAM -d2 -d=0.3 -model=PF data4.dat output.fit &> stdout.txt
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "limit[4]" "0.3 - 0.3"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "par[4]" "0.3"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n 5.0.4 \n"
printf " Test case: Parameter E is fixed. \n"
printf " Expected result 1: Test print shows correct parameter limits. \n"
printf " Expected result 2: Fitted parameter is as required. \n\n"

rm -f output.fit stdout.txt

$PROGRAM -d2 -e=2 -model=PF data4.dat output.fit &> stdout.txt
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "limit[5]" "2 - 2"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "par[5]" "2"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n 5.1.0 \n"
printf " Test case 1: Parameter D is fixed with both -d and -fix0. \n"
printf " Test case 2: Data does not contain zero sample. \n"
printf " Expected result 1: Option -fix0 is ignored. \n"
printf " Expected result 2: No warning related to these options. \n\n"

rm -f output.fit stdout.txt

$PROGRAM -d2 -d=0.3 -fix0 -model=PF data4.dat output.fit &> stdout.txt
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "limit[4]" "0.3 - 0.3"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "par[4]" "0.3"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Warning" "initial fraction is not fixed to sample zero."
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n 5.1.1 \n"
printf " Test case 1: Parameter D is fixed with both -d and -fix0. \n"
printf " Test case 2: Data contains zero sample. \n"
printf " Expected result 1: Option -d is ignored because of option -fix0. \n"
printf " Expected result 2: Warning that -d is not effective. \n\n"

rm -f output.fit stdout.txt

$PROGRAM -d2 -d=0.3 -fix0 -model=PF data4z.dat output.fit &> stdout.txt
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "limit[4]" "0.25 - 0.25"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "par[4]" "0.25"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Note" "initial fraction fixed to 0.25"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"
iftisval stdout.txt "Warning" "non-effective option -d for initial fraction."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n 5.1.2 \n"
printf " Test case 1: Parameter D is fixed with only -fix0. \n"
printf " Test case 2: Data contains zero sample. \n"
printf " Expected result 1: Option -fix0 is effective. \n\n"

rm -f output.fit stdout.txt

$PROGRAM -d2 -fix0 -model=PF data4z.dat output.fit &> stdout.txt
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "limit[4]" "0.25 - 0.25"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "par[4]" "0.25"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Note" "initial fraction fixed to 0.25"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n 5.1.3 \n"
printf " Test case 1: Parameter D is fixed with -fix0. \n"
printf " Test case 2: Data contains duplicate samples including zeroes. \n"
printf " Expected result 1: D is correctly fixed. \n\n"

rm -f temp.dat output.fit stdout.txt
tacblend --force temp.dat data4z.dat data4z.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi

$PROGRAM -d2 -fix0 -model=PF temp.dat output.fit &> stdout.txt
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "limit[4]" "0.25 - 0.25"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "par[4]" "0.25"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Note" "initial fraction fixed to 0.25"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n 5.1.4 \n"
printf " Test case 1: Parameter D is fixed with -fix0. \n"
printf " Test case 2: Hill metabolite model, parent level initially 1. \n"
printf " Expected result: Fitted curve and parameters are close to correct. \n\n"

rm -f output.fit output.res output.dat stdout.txt

$PROGRAM -d2 -model=mhill -fix0 -e=1 -svg=hillm3z.svg data_hillm3.dat output.fit &> stdout.txt
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

fit2dat -f=data_hillm3.dat output.fit output.dat 
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.01 -rel=2 data_hillm3.dat output.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

fit2res output.fit orig_hillm3.fit
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
resmatch -around -res=2-16 orig_hillm3.res output.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

iftisval stdout.txt "Note" "initial fraction fixed to 1"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n\n---------------------------------------------------------------\n\n"


printf "\n 5.2.0 \n"
printf " Test case: Minimum delay can be set with option -delaymin \n"
printf " Expected result 1: Test print shows correct parameter limits. \n"
printf " Expected result 2: Fitted parameter is as required. \n\n"

rm -f output.fit stdout.txt

$PROGRAM -d2 -delaymin=5 -model=PF data4.dat output.fit &> stdout.txt
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "limit[5]" "5 - 7.5"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "par[5]" "5"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 5.2.1 \n"
printf " Test case: Minimum delay can be set to zero with option -delaymin \n"
printf " Expected result 1: Test print shows correct parameter limits. \n"
printf " Expected result 2: Fitted parameter is as required. \n\n"

rm -f output.fit stdout.txt

$PROGRAM -d2 -delaymin=0 -model=PF data3.dat output.fit &> stdout.txt
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "limit[5]" "0 - 5"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval -lt stdout.txt "par[5]" 3
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n===================================================================\n"
printf "   Weights \n"
printf "===================================================================\n"


printf "\n 6.0.0 \n"
printf " Test case: No weights in data and not set with options. \n"
printf " Expected result: Test print shows correct weights. \n\n"

rm -f output.fit stdout.txt

$PROGRAM -d2 -model=PF data5_noweights.dat output.fit &> stdout.txt
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt data_weights "1, 1, 1, 1, 1, 1, 1, 1"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 6.0.1 \n"
printf " Test case: Weights in data but not set with options. \n"
printf " Expected result: Test print shows correct weights. \n\n"

rm -f output.fit stdout.txt

$PROGRAM -d2 -model=PF data5_weights.dat output.fit &> stdout.txt
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt data_weights "0.8, 0.8, 0.8, 0.8, 0.8, 0.8, 1.6, 1.6"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n\n---------------------------------------------------------------\n\n"

printf "\n 6.1.0 \n"
printf " Test case: Option -wc, when file only contains times and parent fractions. \n"
printf " Expected result: Error. \n"

rm -f output.fit stdout.txt
tacformat -f=simple -hdr=no -mid data5_weights.dat temp.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi

$PROGRAM -d2 -wc -model=PF temp.dat output.fit &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error "no column for weights in temp.dat"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 6.1.1 \n"
printf " Test case: Option -wc, when file contains times and 2 data cols. \n"
printf " Expected result: Weights are correctly set. \n"

rm -f output.fit stdout.txt
tacformat -f=simple -hdr=no -mid data5_weights.dat temp.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
taccalc -f temp.dat : temp.dat temp2.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
taccalc -f temp2.dat x 6 temp2.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
tacadd -force temp.dat temp2.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi

$PROGRAM -d2 -wc -model=PF temp.dat output.fit &> stdout.txt
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Warning "extra columns in temp.dat are ignored."
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt data_weights "6, 6, 6, 6, 6, 6, 6, 6"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 6.1.2 \n"
printf " Test case: Option -wc, when file contains times and 3 data cols. \n"
printf " Expected result: Weights are correctly set. \n"

rm -f output.fit stdout.txt
tacadd -force temp.dat temp2.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi

$PROGRAM -d2 -wc -model=PF temp.dat output.fit &> stdout.txt
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Warning "extra columns in temp.dat are ignored."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt data_weights "6, 6, 6, 6, 6, 6, 6, 6"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n\n---------------------------------------------------------------\n\n"


printf "\n 6.2.0 \n"
printf " Test case: Option -w1 overrides weights in file. \n"
printf " Expected result: Weights are correctly set. \n"

rm -f output.fit stdout.txt

$PROGRAM -d2 -w1 -model=PF data5_weights.dat output.fit &> stdout.txt
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt data_weights "1, 1, 1, 1, 1, 1, 1, 1"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 6.2.1 \n"
printf " Test case: Option -wf overrides weights in file. \n"
printf " Expected result: Weights are correctly set. \n"

rm -f output.fit stdout.txt

$PROGRAM -d2 -wf -model=PF data5_weights.dat output.fit &> stdout.txt
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt data_weights "0.8, 0.8, 0.8, 0.8, 0.8, 1, 1.4, 1.6"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 6.2.2 \n"
printf " Test case: Option -wf when samples are not ordered by time. \n"
printf " Expected result: Weights are correctly set. \n"

rm -f output.fit output2.fit stdout.txt

$PROGRAM -d2 -model=HILL -wf data1_catenated1.dat output.fit &> stdout.txt
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt data_weights "0.0669145, 0.301115, 0.60223, 0.669145, 0.669145, 1.00372, 1.67286, 2.00743, 2.00743"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

$PROGRAM -d2 -model=HILL -wf data1_catenated2.dat output2.fit &> stdout.txt
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt data_weights "0.0669145, 0.301115, 0.60223, 0.669145, 0.669145, 1.00372, 1.67286, 2.00743, 2.00743"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 6.2.3 \n"
printf " Test case: Option -wf when duplicate samples. \n"
printf " Expected result: Weights are correctly set. \n"

rm -f output.fit output2.fit stdout.txt

$PROGRAM -d2 -model=HILL -wf data1_duplicated.dat output.fit &> stdout.txt
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt data_weights "0.408759, 0.408759, 0.459854, 0.459854, 0.766423, 0.766423, 1.0219, 1.0219, 1.27737, 1.27737, 1.53285, 1.53285, 1.53285, 1.53285"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n===================================================================\n"
printf "   File formats \n"
printf "===================================================================\n"


printf "\n 7.0.0 \n"
printf " Test case: PMOD BLD format. \n"
printf " Expected result 1: Data is correctly fitted. \n\n"
printf " Expected result 2: Fit file contains time unit. \n\n"

rm -f output.fit output.dat stdout.txt

$PROGRAM -model=HILL -wf data1.bld output.fit
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval output.fit "Time unit" "min"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
fit2dat -f=data1.dat output.fit output.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
tacmatch -abs=0.01 -rel=1 -x=y -y=y -xunit=y data1.dat output.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n 7.1.0 \n"
printf " Test case: CSV format. \n"
printf " Expected result 1: Data is correctly fitted. \n\n"
printf " Expected result 2: Fit file contains time unit. \n\n"

rm -f output.fit output.dat stdout.txt

$PROGRAM -model=HILL -wf data1.csv output.fit
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval output.fit "Time unit" "min"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
fit2dat -f=data1.dat output.fit output.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
tacmatch -abs=0.01 -rel=1 -x=y -y=y -xunit=y data1.dat output.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n 7.2.0 \n"
printf " Test case: TSV format. \n"
printf " Expected result 1: Data is correctly fitted. \n\n"
printf " Expected result 2: Fit file contains time unit. \n\n"

rm -f output.fit output.dat stdout.txt

$PROGRAM -model=HILL -wf data1.tsv output.fit
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval output.fit "Time unit" "min"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
fit2dat -f=data1.dat output.fit output.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
tacmatch -abs=0.01 -rel=1 -x=y -y=y -xunit=y data1.dat output.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n===================================================================\n"
printf "   All passed! \n"
printf "===================================================================\n"
exit 0

