#!/bin/bash
#: Title      : test_fitdelay
#: Date       : 2020-01-24
#: Author     : "Vesa Oikonen" <vesa.oikonen@utu.fi>
#: Options    : None

# Set the name of executable to test
if [ -n "${OS+1}" ] && [ "$OS" = "Windows_NT" ]; then
EXT=.exe;
else
EXT=;
fi
PROGRAM=../../fitdelay$EXT;

if [ ! -f $PROGRAM ]; then
  printf "Failed: executable does not exist.\n"
  exit 1
fi


printf "=====================================================================\n"
printf "creating test data for %s\n" $PROGRAM
printf "=====================================================================\n"


printf "\n verifying that required data exists \n"

if [ ! -f radiowater.fit ] || [ ! -f wframes.sif ]; then
  printf "Failed: required data does not exist.\n"
  exit 1
fi

if [ ! -f flumazenil.bld ] || [ ! -f headcurve.sif ]; then
  printf "Failed: required data does not exist.\n"
  exit 1
fi

printf "\n simulating radiowater data for tests \n\n"

if [ ! -f radiowater.bld ] || [ ! -f radiowater_delayed.bld ]; then
  rm -f radiowater.tac radiowater_idi.tac radiowater.svg
  fit2dat -c=0,400,1 radiowater.fit temp.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacformat -f=PMOD -hdr=no temp.dat radiowater.bld
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacren radiowater.bld 1 Blood__
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tactime -nogap radiowater.bld +50 radiowater_delayed.bld
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacren radiowater.bld 1 Blood_delay_
fi

if [ ! -f radiowater_idi.tac ]; then
  simframe radiowater.bld wframes.sif radiowater_idi.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi

if [ ! -f radiowater.tac ]; then
  rm -f radiowater.svg
  b2t_h2o radiowater.bld  3.0 0.98 1.0 4.0 30 tissue1.dat
  b2t_h2o radiowater.bld 20.0 0.80 1.0 5.0 30 tissue2.dat
  b2t_h2o radiowater.bld 60.0 0.95 1.0 4.0 30 tissue3.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tactime -nogap tissue1.dat +15 tissue1_delayed.dat
  tactime -nogap tissue2.dat +10 tissue2_delayed.dat
  tactime -nogap tissue3.dat +5  tissue3_delayed.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  simframe tissue1_delayed.dat wframes.sif wtissue1_delayed.dat
  simframe tissue2_delayed.dat wframes.sif wtissue2_delayed.dat
  simframe tissue3_delayed.dat wframes.sif wtissue3_delayed.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacadd -ovr radiowater.dat wtissue1_delayed.dat
  tacadd      radiowater.dat wtissue2_delayed.dat
  tacadd      radiowater.dat wtissue3_delayed.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacformat -f=PMOD -hdr=no radiowater.dat radiowater.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacren radiowater.tac 1 tac1_p5_
  tacren radiowater.tac 2 tac2_0_
  tacren radiowater.tac 3 tac3_m5_
fi

if [ ! -f radiowater.svg ]; then
  tac2svg -legend=y -mt="Simulated [O-15]H2O" radiowater.svg radiowater_idi.tac -l radiowater_delayed.bld -b radiowater.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi



printf "\n simulating flumazenil data for tests \n\n"

if [ ! -f flumazenil_apc.bld ] || [ ! -f flumazenil_ab.bld ]; then
  rm -f flumazenil.tac flumazenil.svg flumazenil_delayed_ap.bld flumazenil_delayed_apc.bld flumazenil_apm.bld
  tacadd -ovr flumazenil_apc.bld flumazenil.bld Plasma_parent_
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacadd -ovr flumazenil_ab.bld flumazenil.bld Blood
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi
printf "\n ok \n"
if [ ! -f flumazenil_apm.bld ] || [ ! -f flumazenil_ap.bld ]; then
  tacadd -ovr flumazenil_apm.bld flumazenil.bld Plasma_metabolite_
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacadd -ovr flumazenil_ap.bld flumazenil.bld Plasma_total_
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi
printf "\n ok \n"
if [ ! -f flumazenil_delayed_apc.bld ] || [ ! -f flumazenil_delayed_ab.bld ]; then
  rm -f flumazenil.svg
  tactime -nogap flumazenil.bld +0.8 flumazenil_delayed.bld
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacadd -ovr flumazenil_delayed_apc.bld flumazenil_delayed.bld Plasma_parent_
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacadd -ovr flumazenil_delayed_ab.bld flumazenil_delayed.bld Blood
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi
printf "\n ok \n"
if [ ! -f flumazenil_delayed_apm.bld ] || [ ! -f flumazenil_delayed_ap.bld ]; then
  tactime -nogap flumazenil.bld +0.8 flumazenil_delayed.bld
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacadd -ovr flumazenil_delayed_apm.bld flumazenil_delayed.bld Plasma_metabolite_
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacadd -ovr flumazenil_delayed_ap.bld flumazenil_delayed.bld Plasma_total_
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi
printf "\n ok \n"



# Simulating a head curve
# k1=0.32 k2=0.35 BP=0-7
printf "\n simulating head curve \n"
if [ ! -f flumazenil.tac ] || [ ! -f flumazenil.head ]; then
  rm -f flumazenil.svg
  p2t_v3c -nosub flumazenil_apc.bld flumazenil_ab.bld 0.32 0.35 0.70 0.10 0 0 0.8 4.5 25 temp1.dat
  p2t_v3c -nosub flumazenil_apc.bld flumazenil_ab.bld 0.32 0.35 0.50 0.10 0 0 0.8 4.5 25 temp2.dat
  p2t_v3c -nosub flumazenil_apc.bld flumazenil_ab.bld 0.32 0.35 0.30 0.10 0 0 0.8 4.5 25 temp3.dat
  p2t_v3c -nosub flumazenil_apc.bld flumazenil_ab.bld 0.32 0.35 0.30 0.30 0 0 0.8 4.5 25 temp4.dat
  p2t_v3c -nosub flumazenil_apc.bld flumazenil_ab.bld 0.10 0.10 0.10 0.10 0 0 0.8 4.5 25 temp5.dat
  avgttac flumazenil.dat temp1.dat temp2.dat temp3.dat temp4.dat temp5.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacunit -xconv=sec flumazenil.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  simframe -mid flumazenil.dat headcurve.sif head.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacformat -f=PMOD head.dat flumazenil.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacformat -f=simple head.dat temp.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  taccalc --force temp.dat x 10000 flumazenil.head
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacunit -yset=cps flumazenil.head
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi

if [ ! -f flumazenil.svg ]; then
  tac2svg -legend=y -mt="Simulated flumazenil" flumazenil.svg -l flumazenil_apc.bld flumazenil_delayed_apc.bld flumazenil.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi


printf "\n\n"
printf "=====================================================================\n"
printf "testing %s\n" $PROGRAM
printf "=====================================================================\n"


printf "\n 1.0.0 \n"
printf " Test case 1: Simulated radiowater data. \n"
printf " Test case 2: All data in PMOD format. \n"
printf " Test case 3: Sample times in sec and units in kBq/ml. \n"
printf " Test case 4: Default settings, except -log. \n"
printf " Expected result 1: BTAC corrected to the mean delay. \n"
printf " Expected result 2: BTAC saved with default filename. \n\n"

rm -f radiowater_delayed.delay.bld
tactime radiowater_delayed.bld -40 correct.bld
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi

$PROGRAM -log radiowater_delayed.bld radiowater.tac 360
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval -abs=0.5 radiowater_delayed.delay.bld time_delay -40
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.1 -rel=1 correct.bld radiowater_delayed.delay.bld
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 1.0.1 \n"
printf " Test case 1: 1TCM model is selected with option -model=1. \n"
printf " Test case 2: Output filename given with option -o. \n"
printf " Expected result: Corrected BTAC saved with given filename. \n\n"

rm -f output.bld
tactime radiowater_delayed.bld -40 correct.bld
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi

$PROGRAM -log -model=1 -o=output.bld radiowater_delayed.bld radiowater.tac 360
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval -abs=0.5 output.bld time_delay -40
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.1 -rel=1 correct.bld output.bld
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 1.1.0 \n"
printf " Test case 1: BTAC sample times are in min, TTAC in sec. \n"
printf " Test case 2: BTAC units are in kBq/ml, TTAC in Bq/ml. \n"
printf " Expected result: Corrected BTAC saved. \n\n"

rm -f output.bld
tactime radiowater_delayed.bld -40 correct.bld
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
cp -f radiowater_delayed.bld temp.bld
cp -f radiowater.tac temp.tac
tacunit -xconv=min correct.bld temp.bld
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
tacunit -yconv=Bq/ml temp.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi

$PROGRAM -log -model=1 -o=output.bld temp.bld temp.tac 360
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval -abs=0.5 output.bld time_delay -40
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.1 -rel=1 correct.bld output.bld
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 1.1.1 \n"
printf " Test case 1: BTAC sample times are in sec, TTAC in min. \n"
printf " Test case 2: BTAC units are in Bq/ml, TTAC in kBq/ml. \n"
printf " Expected result: Corrected BTAC saved. \n\n"

rm -f output.bld
tactime radiowater_delayed.bld -40 correct.bld
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
cp -f radiowater_delayed.bld temp.bld
cp -f radiowater.tac temp.tac
tacunit -xconv=min temp.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
tacunit -yconv=Bq/ml correct.bld temp.bld
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi

$PROGRAM -log -model=1 -o=output.bld temp.bld temp.tac 360
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval -abs=0.5 output.bld time_delay -40
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.1 -rel=1 correct.bld output.bld
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n 1.2.0 \n"
printf " Test case: Simulated image-derived input. \n"
printf " Expected result: Corrected BTAC saved in given file. \n\n"

rm -f corrected.tac
tactime radiowater_idi.tac +10 correct.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi

$PROGRAM -log -o=corrected.tac  radiowater_idi.tac radiowater.tac 360
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval -abs=1.1 corrected.tac time_delay +10
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=2 -rel=5 correct.tac corrected.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "=====================================================================\n"


printf "\n 2.0.0 \n"
printf " Test case 1: Simulated flumazenil data. \n"
printf " Test case 2: Simulated head curve in plain data format. \n"
printf " Test case 3: Default settings, except -log. \n"
printf " Expected result: Corrected BTAC saved with default filename. \n\n"

rm -f flumazenil_delayed_apc.delay.bld

$PROGRAM -log flumazenil_delayed_apc.bld flumazenil.head 600
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval -abs=1.5 flumazenil_delayed_apc.delay.bld time_delay -48
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
interpol -y -f=flumazenil_apc.bld flumazenil_delayed_apc.delay.bld temp.bld
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=2 -rel=2 flumazenil_apc.bld temp.bld
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n 2.1.0 \n"
printf " Test case 1: Additional input TACs are corrected simultaneously. \n"
printf " Test case 2: Output filenames given with options. \n"
printf " Expected result: Corrected BTACs saved with given filenames. \n\n"

rm -f corrected_apc.bld corrected_ab.bld corrected_apm.bld corrected_ap.bld

$PROGRAM -log -o=corrected_apc.bld -O2=corrected_ab.bld -o3=corrected_apm.bld -o4=corrected_ap.bld flumazenil_delayed_apc.bld flumazenil.head 600 flumazenil_delayed_ab.bld flumazenil_delayed_apm.bld flumazenil_delayed_ap.bld
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval -abs=1.5 corrected_apc.bld time_delay -48
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

interpol -y -f=flumazenil_apc.bld corrected_apc.bld temp.bld
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=2 -rel=2 flumazenil_apc.bld temp.bld
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

interpol -y -f=flumazenil_ab.bld corrected_ab.bld temp.bld
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=2 -rel=2 flumazenil_ab.bld temp.bld
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

interpol -y -f=flumazenil_apm.bld corrected_apm.bld temp.bld
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=2 -rel=2 flumazenil_apm.bld temp.bld
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

interpol -y -f=flumazenil_ap.bld corrected_ap.bld temp.bld
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=2 -rel=2 flumazenil_ap.bld temp.bld
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "=====================================================================\n"

printf "\n 2.2.0 \n"
printf " Test case: Option -matrix to save NNLS matrix. \n"
printf " Expected result: Matrix file saved. \n\n"

tacadd -ovr matrix_tissue.tac radiowater.tac 2
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
tactime -nogap radiowater.bld +20 matrix_input.bld
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi

rm -f matrix.csv matrix_input.delay.bld

$PROGRAM -matrix=matrix.csv -LOG matrix_input.bld matrix_tissue.tac 420
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval -abs=0.5 matrix_input.delay.bld time_delay -10
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
if [ ! -f matrix.csv ]; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n\n"
printf "=====================================================================\n"
printf "testing errors\n"
printf "=====================================================================\n"


printf "\n 3.0.0 \n"
printf " Test case: Unknown option. \n"
printf " Expected result: Error. \n\n"

rm -f stdout.txt
$PROGRAM -stupidoption flumazenil_delayed_apc.bld flumazenil.head 300 &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error "invalid option '-stupidoption'."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 3.1.0 \n"
printf " Test case: Non-existing input file. \n"
printf " Expected result: Error. \n\n"

rm -f stdout.txt
$PROGRAM nonexistingfile.dat flumazenil.head 300 &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 3.1.1 \n"
printf " Test case: Non-existing tissue file. \n"
printf " Expected result: Error. \n\n"

rm -f stdout.txt
$PROGRAM flumazenil_delayed_apc.bld nonexistingfile.dat 600 &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 3.1.2 \n"
printf " Test case: Fittime missing. \n"
printf " Expected result: Error. \n\n"

rm -f stdout.txt
$PROGRAM flumazenil_delayed_apc.bld flumazenil.head &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 3.1.3 \n"
printf " Test case: Too many input files. \n"
printf " Expected result: Error. \n\n"

rm -f stdout.txt
$PROGRAM flumazenil_delayed_apc.bld flumazenil.head 900 flumazenil_delayed_ab.bld flumazenil_delayed_ab.bld flumazenil_delayed_ab.bld flumazenil_delayed_ab.bld flumazenil_delayed_ab.bld &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 3.1.4 \n"
printf " Test case: Nonexisting extra file. \n"
printf " Expected result: Error. \n\n"

rm -f stdout.txt
$PROGRAM flumazenil_delayed_apc.bld flumazenil.head 900 nonexistingfile.dat &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n 3.2.0 \n"
printf " Test case: Input peak is missed. \n"
printf " Expected result: Error. \n\n"

rm -f stdout.txt nopeak.bld
tactime -nogap flumazenil_apc.bld -3 nopeak.bld 
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi

$PROGRAM nopeak.bld flumazenil.head 900 &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error "missed the ascending phase of plasma/blood data."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n 3.2.1 \n"
printf " Test case: Tissue peak is missed. \n"
printf " Expected result: Error. \n\n"

rm -f stdout.txt
$PROGRAM radiowater_delayed.bld nopeak.bld 360 &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error "missed the ascending phase of tissue data."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n===================================================================\n"
printf "   All passed! \n"
printf "===================================================================\n"
exit 0

