/******************************************************************************
 * This file is not compiled into the library, but it contains main()
 * which is compiled to an executable, used to test the library functions. 
 *****************************************************************************/

/*****************************************************************************/
#include "libtpcidi.h"
/*****************************************************************************/

/*****************************************************************************/
/* Test function declarations */
int test_heartcor1(int verbose);
int test_idiSimulateTubeVol1(int verbose);
int test_imgCircleMask1(int verbose);
int test_imgRingMask1(int verbose);
/*****************************************************************************/

/*****************************************************************************/
static char *info[] = {
  "Usage: @P [options]",
  " ",
  "Options:",
  " -stdoptions", // List standard options like --help, -v, etc
  " -t, --test",
  "     Run all tests for library functions.",
  0};
/*****************************************************************************/

/*****************************************************************************/
/** Run unit tests to the library functions
    @author Vesa Oikonen
    @return 0 if all tests pass, otherwise >0.
 */
int main(
  /** Nr of arguments */
  int argc,
  /** Pointer to arrays of argument string */
  char *argv[ ]
) {
  int i, help=0, version=0, verbose=1, error=0, test=0;
  int ret;
  char *cptr;

  if(argc==1) {tpcPrintUsage(argv[0], info, stdout); return(0);}
  for(i=1; i<argc; i++) {
    if(tpcProcessStdOptions(argv[i], &help, &version, &verbose)==0) continue;
    cptr=argv[i]; if(*cptr=='-') cptr++; if(*cptr=='-') cptr++;
    if(strncasecmp(cptr, "TEST", 1)==0) {
      test=1; continue;
    } else {
      error++; break;
    }
  }
  if(error>0) {
    fprintf(stderr, "Error: specify --help for usage.\n");
    return(1);
  }
  /* Print help or version? */
  if(help) {tpcPrintUsage(argv[0], info, stdout); return(0);}
  if(version) {tpcPrintBuild(argv[0], stdout); return(0);}

  if(test==0) return(0);

  if(verbose>0) printf("running tests for library functions...\n");

  i=10;
  i++; if((ret=test_heartcor1(verbose-1))!=0) {
    fprintf(stderr, "failed (%d).\n", ret); return(i);}
  i++; if((ret=test_idiSimulateTubeVol1(verbose-1))!=0) {
    fprintf(stderr, "failed (%d).\n", ret); return(i);}
  i++; if((ret=test_imgCircleMask1(verbose-1))!=0) {
    fprintf(stderr, "failed (%d).\n", ret); return(i);}
  i++; if((ret=test_imgRingMask1(verbose-1))!=0) {
    fprintf(stderr, "failed (%d).\n", ret); return(i);}

  if(verbose>0) printf("\nAll tests passed.\n\n");
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
int test_heartcor1(int VERBOSE)
{
  int error_code = 0;

  if(VERBOSE>0) {
    printf("\n=====================================\n");
    printf("\nTesting heartcor #1...\n");
    printf("\n=====================================\n");
  }
  
  if(error_code)
    printf("\n Test FAILED: test_heartcor1 failed with error code: %i\n",
           error_code);
  else
    printf("\n Test SUCCESFULL: test_heartcor1 exited with: %i\n",
           error_code); 

  return(error_code);
}
/******************************************************************************/

/******************************************************************************/
int test_idiSimulateTubeVol1(int VERBOSE)
{
  char *funcname="idiSimulateTubeVol1";
  int error_code = 0;

  VOL vol;
  int ret;

  printf("\n=====================================\n");
  printf("\nTesting %s #1...\n", funcname);
  printf("\n=====================================\n");

  volInit(&vol);
  ret=volAllocate(&vol, 5, 10, 10);
  if(ret!=0) return(-1);

  /* Error, if pixel sizes are missing */
  vol.sizex=vol.sizey=vol.sizez=0.0;
  ret=idiSimulateTubeVol(&vol, 0, 5.0, 5.0, 2.0, 2.0, 0.0, 10.0);
  if(ret==0) error_code+=1;
  if(VERBOSE) printf("error_code=%d\n", error_code);
  
  vol.sizex=vol.sizey=vol.sizez=1.0;

  /* Must be able to provide correct image even with extreme parameters */
  // all zero
  ret=idiSimulateTubeVol(&vol, 0, 5.0, 5.0, 2.0, 2.0, 0.0, 0.0);
  if(ret!=0) error_code+=10;
  if(VERBOSE) printf("error_code=%d\n", error_code);
  // all the same >0
  ret=idiSimulateTubeVol(&vol, 1, 5.0, 5.0, 2.0, 2.0, 10.0, 10.0);
  if(ret!=0) error_code+=10;
  if(VERBOSE) printf("error_code=%d\n", error_code);
  // vessel covers whole volume
  ret=idiSimulateTubeVol(&vol, 2, 5.0, 5.0, 100.0, 2.0, 0.0, 10.0);
  if(ret!=0) error_code+=10;
  if(VERBOSE) printf("error_code=%d\n", error_code);
  // vessel is outside volume
  ret=idiSimulateTubeVol(&vol, 3, 50.0, 50.0, 2.0, 2.0, 1.0, 10.0);
  if(ret!=0) error_code+=10;
  if(VERBOSE) printf("error_code=%d\n", error_code);
  // FWHM is very high
  ret=idiSimulateTubeVol(&vol, 4, 5.0, 5.0, 2.0, 1.0E+3, 1.0, 10.0);
  if(ret!=0) error_code+=10;
  if(VERBOSE) printf("error_code=%d\n", error_code);
  /* check results */
  if(error_code==0) {
    float avg;
    VOL_RANGE r;
    r.x1=r.y1=1; r.x2=r.y2=vol.dimx;

    r.z1=r.z2=1; volAvg(&vol, &r, &avg); if(VERBOSE) printf("avg := %E\n", avg);
    if(avg!=0.0) error_code+=10;
    if(VERBOSE) printf("error_code=%d\n", error_code);

    r.z1=r.z2=2; volAvg(&vol, &r, &avg); if(VERBOSE) printf("avg := %E\n", avg);
    if(avg!=10.0) error_code+=10;
    if(VERBOSE) printf("error_code=%d\n", error_code);

    r.z1=r.z2=3; volAvg(&vol, &r, &avg); if(VERBOSE) printf("avg := %E\n", avg);
    if(avg!=10.0) error_code+=10;
    if(VERBOSE) printf("error_code=%d\n", error_code);

    r.z1=r.z2=4; volAvg(&vol, &r, &avg); if(VERBOSE) printf("avg := %E\n", avg);
    if(avg!=1.0) error_code+=10;
    if(VERBOSE) printf("error_code=%d\n", error_code);

    r.z1=r.z2=5; volAvg(&vol, &r, &avg); if(VERBOSE) printf("avg := %E\n", avg);
    if(avg<=1.0 || avg>2.0) error_code+=10;
    if(VERBOSE) printf("error_code=%d\n", error_code);
  }

  /* Correct image with reasonable parameters */
  // vessel > bkg
  ret=idiSimulateTubeVol(&vol, 0, 5.0, 5.0, 2.0, 2.0, 10.0, 20.0);
  if(ret!=0) error_code+=100;
  if(VERBOSE) printf("error_code=%d\n", error_code);
  // vessel < bkg
  ret=idiSimulateTubeVol(&vol, 1, 5.0, 5.0, 2.0, 2.0, 20.0, 10.0);
  if(ret!=0) error_code+=100;
  if(VERBOSE) printf("error_code=%d\n", error_code);
  // small FWHM and high radius will give full peak value
  ret=idiSimulateTubeVol(&vol, 2, 5.0, 5.0, 3.0, 1.0, 5.0, 10.0);
  if(ret!=0) error_code+=100;
  if(VERBOSE) printf("error_code=%d\n", error_code);
  // vessel is not at center
  ret=idiSimulateTubeVol(&vol, 3, 1.0, 10.0, 2.0, 2.0, 1.0, 10.0);
  if(ret!=0) error_code+=100;
  if(VERBOSE) printf("error_code=%d\n", error_code);
  // peak value is not the same as given by RC formula (Germano et al)
  // but somewhat higher
  ret=idiSimulateTubeVol(&vol, 4, 5.0, 5.0, 2.0, 3.0, 0.0, 1.0);
  if(ret!=0) error_code+=100;
  if(VERBOSE) printf("error_code=%d\n", error_code);
  /* check results */
  if(error_code==0) {
    float avg, max, min;
    VOL_RANGE r;
    VOL_PIXEL maxp, minp;
    r.x1=r.y1=1; r.x2=r.y2=vol.dimx;

    r.z1=r.z2=1;
    volAvg(&vol, &r, &avg); if(VERBOSE) printf("avg := %E\n", avg);
    volMax(&vol, &r, &maxp, &max, &minp, &min);
    if(VERBOSE) printf("max := %E at (%d,%d,%d)\n", max, maxp.x, maxp.y, maxp.z);
    if(VERBOSE) printf("min := %E at (%d,%d,%d)\n", min, minp.x, minp.y, minp.z);
    if(max<=10.0 || max>=20.0 || avg<=10) error_code+=100;
    if(maxp.x!=5 || maxp.y!=5) error_code+=100;

    r.z1=r.z2=2;
    volAvg(&vol, &r, &avg); if(VERBOSE) printf("avg := %E\n", avg);
    volMax(&vol, &r, &maxp, &max, &minp, &min);
    if(VERBOSE) printf("max := %E at (%d,%d,%d)\n", max, maxp.x, maxp.y, maxp.z);
    if(VERBOSE) printf("min := %E at (%d,%d,%d)\n", min, minp.x, minp.y, minp.z);
    if(max>20.0 || avg<=10.0 || avg>=20.0) error_code+=100;
    if(minp.x!=5 || minp.y!=5) error_code+=100;

    r.z1=r.z2=3;
    volAvg(&vol, &r, &avg); if(VERBOSE) printf("avg := %E\n", avg);
    volMax(&vol, &r, &maxp, &max, &minp, &min);
    if(VERBOSE) printf("max := %E at (%d,%d,%d)\n", max, maxp.x, maxp.y, maxp.z);
    if(VERBOSE) printf("min := %E at (%d,%d,%d)\n", min, minp.x, minp.y, minp.z);
    if(max<9.99999) error_code+=100;

    r.z1=r.z2=4;
    volAvg(&vol, &r, &avg); if(VERBOSE) printf("avg := %E\n", avg);
    volMax(&vol, &r, &maxp, &max, &minp, &min);
    if(VERBOSE) printf("max := %E at (%d,%d,%d)\n", max, maxp.x, maxp.y, maxp.z);
    if(VERBOSE) printf("min := %E at (%d,%d,%d)\n", min, minp.x, minp.y, minp.z);
    if(maxp.x!=1 || maxp.y!=10) error_code+=100;

    r.z1=r.z2=5;
    volAvg(&vol, &r, &avg); if(VERBOSE) printf("avg := %E\n", avg);
    volMax(&vol, &r, &maxp, &max, &minp, &min);
    if(VERBOSE) printf("max := %E at (%d,%d,%d)\n", max, maxp.x, maxp.y, maxp.z);
    if(VERBOSE) printf("min := %E at (%d,%d,%d)\n", min, minp.x, minp.y, minp.z);
    double rc, rad=2.0, fwhm=3.0, s;
    s=fwhm/2.355; rc=1.0-exp(-rad*rad/(2.0*s*s));
    if(VERBOSE) printf("rc := %g\n", rc);
    if(max<rc || max>=1.0) error_code+=100;
  }

  volEmpty(&vol);

  if(error_code)
    printf("\n Test FAILED: %s failed with error code: %i\n",
           funcname, error_code);
  else
    printf("\n Test SUCCESFULL: %s exited with: %i\n",
           funcname, error_code); 

  return(error_code);
}
/******************************************************************************/

/******************************************************************************/
int test_imgCircleMask1(int VERBOSE)
{
  char *funcname="imgCircleMask1";
  int error_code = 0;

  IMG img;
  int ret, i;
  double mv1, mv2, a, d;
  float f;

  printf("\n=====================================\n");
  printf("\nTesting %s #1...\n", funcname);
  printf("\n=====================================\n");


  /* Allocate memory for test image with odd dimensions */
  imgInit(&img); ret=imgAllocate(&img, 2, 5, 5, 1);
  if(ret!=0) {
    fprintf(stderr, "Error: cannot allocate memory for image.\n");
    return(1000+ret);
  }

  /* Simulate circle at the centre of image */
  {
  if(VERBOSE) printf("simulation 1\n");
  img.sizex=img.sizey=img.sizez=20.0;
  float correct[]={
    0,   0,   0,   0,   0,
    0,  52, 100,  52,   0,
    0, 100, 100, 100,   0,
    0,  52, 100,  52,   0,
    0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,
    0,  24,  72,  24,   0,
    0,  72, 100,  72,   0,
    0,  24,  72,  24,   0,
    0,   0,   0,   0,   0
  };
  ret=imgCircleMask(&img, 0, 50.0, 50.0, 30.0, 100.0, &mv1, VERBOSE);
  if(ret==0) ret=imgCircleMask(&img, 1, 50.0, 50.0, 25.0, 100.0, &mv2, VERBOSE);
  if(ret!=0) {
    fprintf(stderr, "Error %d from imgCircleMask().\n", ret);
    imgEmpty(&img); return(1100+ret);
  }
  f=100;
  tiffWriteImg(&img, -1, -1, &f, PET_GRAYSCALE, "test_imgCircleMask1a.tif",
               0, 0, 0, NULL);
  for(i=0; i<img.dimz*img.dimx*img.dimy; i++) {
    d=fabs(img.pixel[i]-correct[i]);
    if(d>0.0000001) error_code++;
    //printf("%3g, ", img.pixel[i]); if((i+1)%5==0) printf("\n");
    //printf(" %g vs %g\n", img.pixel[i], correct[i]);
  }
  a=100.0*M_PI*1.5*1.5; d=fabs(mv1-a)/a;
  if(VERBOSE) printf("  mv1 := %g vs %g (%g%%)\n", mv1, a, 100.*d);
  if(d>0.05) error_code++;
  a=100.0*M_PI*1.25*1.25; d=fabs(mv2-a)/a;
  if(VERBOSE) printf("  mv2 := %g vs %g (%g%%)\n", mv2, a, 100.*d);
  if(d>0.05) error_code++;

  if(error_code) printf("\nsimulated image 1 is not correct.\n");
  }

  /* Simulate circle NOT at the centre of image */
  if(error_code==0) {

  if(VERBOSE) printf("simulation 2\n");
  img.sizex=img.sizey=img.sizez=20.0;
  float correct[]={
    0,   0,   0,   0,   0,
    0,  36,  36,   0,   0,
   36, 100, 100,  36,   0,
   36, 100, 100,  36,   0,
    0,  36,  36,   0,   0,
    0,   0,  24,  72,  24,
    0,   0,  72, 100,  72,
    0,   0,  24,  72,  24,
    0,   0,   0,   0,   0,
    0,   0,   0,   0,   0
  };
  for(i=0; i<img.dimz*img.dimx*img.dimy; i++) img.pixel[i]=0;
  ret=imgCircleMask(&img, 0, 40.0, 60.0, 30.0, 100.0, &mv1, VERBOSE);
  if(ret==0) ret=imgCircleMask(&img, 1, 70.0, 30.0, 25.0, 100.0, &mv2, VERBOSE);
  if(ret!=0) {
    fprintf(stderr, "Error %d from imgCircleMask().\n", ret);
    imgEmpty(&img); return(1200+ret);
  }
  f=100;
  tiffWriteImg(&img, -1, -1, &f, PET_GRAYSCALE, "test_imgCircleMask1b.tif",
               0, 0, 0, NULL);
  for(i=0; i<img.dimz*img.dimx*img.dimy; i++) {
    d=fabs(img.pixel[i]-correct[i]);
    if(d>0.0000001) error_code++;
    //printf("%3g, ", img.pixel[i]); if((i+1)%5==0) printf("\n");
    //printf(" %g vs %g\n", img.pixel[i], correct[i]);
  }
  a=100.0*M_PI*1.5*1.5; d=fabs(mv1-a)/a;
  if(VERBOSE) printf("  mv1 := %g vs %g (%g%%)\n", mv1, a, 100.*d);
  if(d>0.05) error_code++;
  a=100.0*M_PI*1.25*1.25; d=fabs(mv2-a)/a;
  if(VERBOSE) printf("  mv2 := %g vs %g (%g%%)\n", mv2, a, 100.*d);
  if(d>0.05) error_code++;

  if(error_code) printf("\nsimulated image 2 is not correct.\n");
  }



  /* Allocate memory for large test image with even dimensions */
  imgEmpty(&img); ret=imgAllocate(&img, 2, 100, 100, 1);
  if(ret!=0) {
    fprintf(stderr, "Error: cannot allocate memory for image.\n");
    return(1000+ret);
  }

  /* Simulate circle */
  if(error_code==0) {
  if(VERBOSE) printf("simulation 3\n");
  img.sizex=img.sizey=img.sizez=1.0;
  for(i=0; i<img.dimz*img.dimx*img.dimy; i++) img.pixel[i]=0;
  ret=imgCircleMask(&img, 0, 50.0, 50.0, 40.0, 100.0, &mv1, VERBOSE);
  if(ret==0) ret=imgCircleMask(&img, 1, 70.0, 30.0, 18.3, 100.0, &mv2, VERBOSE);
  if(ret!=0) {
    fprintf(stderr, "Error %d from imgCircleMask().\n", ret);
    imgEmpty(&img); return(1300+ret);
  }
  f=100;
  tiffWriteImg(&img, -1, -1, &f, PET_GRAYSCALE, "test_imgCircleMask1c.tif",
               0, 0, 0, NULL);
  a=100.0*M_PI*40.0*40.0; d=fabs(mv1-a)/a;
  if(VERBOSE) printf("  mv1 := %g vs %g (%g%%)\n", mv1, a, 100.*d);
  if(d>0.001) error_code++;
  a=100.0*M_PI*18.3*18.3; d=fabs(mv2-a)/a;
  if(VERBOSE) printf("  mv2 := %g vs %g (%g%%)\n", mv2, a, 100.*d);
  if(d>0.001) error_code++;

  if(error_code) printf("\nsimulated image 3 is not correct.\n");
  }


  /* Simulate a ring */
  if(error_code==0) {
  if(VERBOSE) printf("simulation 4\n");
  img.sizex=img.sizey=img.sizez=1.0;
  for(i=0; i<img.dimz*img.dimx*img.dimy; i++) img.pixel[i]=0;
  ret=imgCircleMask(&img, 0, 50.0, 50.0, 40.0, 100.0, &mv1, VERBOSE);
  if(ret==0) ret=imgCircleMask(&img, 0, 50.0, 50.0, 30.0, -100.0, &mv2, VERBOSE);
  if(ret==0) ret=imgCircleMask(&img, 1, 70.0, 30.0, 18.3, 100.0, &mv2, VERBOSE);
  if(ret==0) ret=imgCircleMask(&img, 1, 70.0, 30.0, 17.3, -100.0, &mv2, VERBOSE);
  if(ret!=0) {
    fprintf(stderr, "Error %d from imgCircleMask().\n", ret);
    imgEmpty(&img); return(1300+ret);
  }
  f=100;
  tiffWriteImg(&img, -1, -1, &f, PET_GRAYSCALE, "test_imgCircleMask1d.tif",
               0, 0, 0, NULL);
  }


  if(error_code)
    printf("\n Test FAILED: %s failed with error code: %i\n",
           funcname, error_code);
  else
    printf("\n Test SUCCESFULL: %s exited with: %i\n",
           funcname, error_code); 

  imgEmpty(&img);
  return(error_code);
}
/******************************************************************************/

/******************************************************************************/
int test_imgRingMask1(int VERBOSE)
{
  char *funcname="imgRingMask1";
  int error_code = 0;

  IMG img;
  int ret, i;
  double mv, a, d;
  float f;

  printf("\n=====================================\n");
  printf("\nTesting %s #1...\n", funcname);
  printf("\n=====================================\n");


  /* Allocate memory for test image */
  imgInit(&img); ret=imgAllocate(&img, 1, 200, 200, 1);
  if(ret!=0) {
    fprintf(stderr, "Error: cannot allocate memory for image.\n");
    return(1000+ret);
  }

  /* Simulate ring */
  {
  if(VERBOSE) printf("simulation 1\n");
  img.sizex=img.sizey=img.sizez=1.0;
  ret=imgRingMask(&img, 0, 95.0, 105.0, 75.0, 85.0, 1.0, &mv, VERBOSE);
  if(ret!=0) {
    fprintf(stderr, "Error %d from imgRingMask().\n", ret);
    imgEmpty(&img); return(1100+ret);
  }
  f=1;
  tiffWriteImg(&img, -1, -1, &f, PET_RAINBOW, "test_imgRingMask1a.tif",
               0, 0, 0, NULL);
  for(i=0; i<img.dimz*img.dimx*img.dimy; i++) {
#if(0)
    d=fabs(img.pixel[i]-correct[i]);
    if(d>0.0000001) error_code++;
    printf("%3g, ", img.pixel[i]); if((i+1)%10==0) printf("\n");
#endif
  }

  a=M_PI*(85.0*85.0 - 75.0*75.0); d=fabs(mv-a)/a;
  if(VERBOSE) printf("  mv := %g vs %g (%g%%)\n", mv, a, 100.*d);
  if(d>0.001) error_code++;

  if(error_code) printf("\nsimulated image 1 is not correct.\n");
  }

  if(error_code)
    printf("\n Test FAILED: %s failed with error code: %i\n",
           funcname, error_code);
  else
    printf("\n Test SUCCESFULL: %s exited with: %i\n",
           funcname, error_code); 

  imgEmpty(&img);
  return(error_code);
}
/*****************************************************************************/

/*****************************************************************************/
