/** @file mask2pxl.c
 *  @brief List the pixel coordinates in a mask image.
 *  @copyright (c) Turku PET Centre
 *  @author Vesa Oikonen
 */
/// @cond
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <string.h>
#include <unistd.h>
#include <time.h>
/*****************************************************************************/
#include "libtpcmisc.h"
#include "libtpcimgio.h"
#include "libtpcimgp.h"
/*****************************************************************************/

/*****************************************************************************/
static char *info[] = {
  "List the pixel coordinates in a mask image.",
  "Pixels with |value| >= 0.5 in mask image are considered as part of mask,",
  "and their coordinates (x,y,z,f) are written in specified file or stdout.",
  " ",
  "Usage: @P [Options] maskfile [pxlfile]",
  " ",
  "Options:",
  " -stdoptions", // List standard options like --help, -v, etc
  " ",
  "See also: pxl2mask, maskinv, maskconj, maskdila, pxl2tac, imgprofi",
  " ",
  "Keywords: image, mask, pixel",
  0};
/*****************************************************************************/

/*****************************************************************************/
/* Turn on the globbing of the command line, since it is disabled by default in
   mingw-w64 (_dowildcard=0); in MinGW32 define _CRT_glob instead, if necessary;
   In Unix&Linux wildcard command line processing is enabled by default. */
/*
#undef _CRT_glob
#define _CRT_glob -1
*/
int _dowildcard = -1;
/*****************************************************************************/

/*****************************************************************************/
/**
 *  Main
 */
int main(int argc, char **argv)
{
  int     ai, help=0, version=0, verbose=1;
  int     ret;
  char    maskfile[FILENAME_MAX], pxlfile[FILENAME_MAX];
  char   *cptr=NULL;


  /*
   *  Get arguments
   */
  if(argc==1) {tpcPrintUsage(argv[0], info, stderr); return(1);}
  maskfile[0]=pxlfile[0]=(char)0;
  /* Options */
  for(ai=1; ai<argc; ai++) if(*argv[ai]=='-') { /* options */
    cptr=argv[ai]+1; if(*cptr=='-') cptr++; if(cptr==NULL) continue;
    if(tpcProcessStdOptions(argv[ai], &help, &version, &verbose)==0) continue;
    fprintf(stderr, "Error: invalid option '%s'.\n", argv[ai]); 
    return(1);
  } else break;
  
  /* Print help or version? */
  if(help==2) {tpcHtmlUsage(argv[0], info, ""); return(0);}
  if(help) {tpcPrintUsage(argv[0], info, stdout); return(0);}
  if(version) {tpcPrintBuild(argv[0], stdout); return(0);}

  /* Process other arguments, starting from the first non-option */
  if(ai<argc) {strlcpy(maskfile, argv[ai], FILENAME_MAX); ai++;}
  if(ai<argc) {strlcpy(pxlfile, argv[ai], FILENAME_MAX); ai++;}
  if(ai<argc) {fprintf(stderr, "Error: too many arguments.\n"); return(1);}

  /* Did we get all the information that we need? */
  if(!maskfile[0]) {
    fprintf(stderr, "Error: missing command-line argument; use option --help\n");
    return(1);
  }


  /* In verbose mode print options */
  if(verbose>1) {
    printf("maskfile := %s\n", maskfile);
    if(pxlfile[0]) printf("pxlfile := %s\n", pxlfile);
    fflush(stdout);
  }


  /*
   *  Read mask image
   */
  IMG mask; imgInit(&mask);

  if(verbose>0) {printf("reading %s\n", maskfile); fflush(stdout);}
  ret=imgRead(maskfile, &mask);
  if(ret) {
    fprintf(stderr, "Error: %s\n", mask.statmsg);
    if(verbose>1) printf("ret := %d\n", ret);
    return(2);
  }
  if(mask.dimt>1) {
    fprintf(stderr, "Error: mask cannot be dynamic image.\n");
    imgEmpty(&mask); return(2);
  }


  /*
   *  Get pixel list from mask image
   */
  if(verbose>0) {printf("processing the mask\n"); fflush(stdout);}
  IMG_PIXELS pxl; pxlInit(&pxl);
  ret=pxlAddFromMask(&pxl, &mask);
  if(ret==0) {
    fprintf(stderr, "Error: no voxels are part of mask; empty list not saved.\n");
    imgEmpty(&mask);
    return(3);
  }
  if(verbose>1) {
    printf("nr_of_mask_voxels := %u\n", ret);
  }
  imgEmpty(&mask);


  /*
   *  Save or print the pixel list
   */
  if(!pxlfile[0]) {
    pxlWrite(&pxl, stdout, NULL);
    pxlFree(&pxl);
    return(0);
  }
  FILE *fp; fp=fopen(pxlfile, "w");
  if(fp==NULL) {
    fprintf(stderr, "Error: cannot open file for writing (%s).\n", pxlfile);
    pxlFree(&pxl); return(11);
  }
  ret=pxlWrite(&pxl, fp, NULL);
  fclose(fp); pxlFree(&pxl);
  if(ret) {
    fprintf(stderr, "Error: cannot write in file %s.\n", pxlfile);
    return(12);
  }
  if(verbose>0) printf("Mask pixel coordinates written in %s\n", pxlfile);

  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/// @endcond
