/** @file maskcloak.c
 *  @brief Cload mask with a new mask.
 *  @copyright (c) Turku PET Centre
 *  @author Vesa Oikonen
 */
/// @cond
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <string.h>
#include <unistd.h>
#include <time.h>
/*****************************************************************************/
#include "libtpcmisc.h"
#include "libtpcimgio.h"
#include "libtpcimgp.h"
/*****************************************************************************/

/*****************************************************************************/
static char *info[] = {
  "Cloak a mask image in ECAT 6.3 or 7.x, NIfTI-1, or Analyze 7.5 format.",
  "Original mask image file is overwritten if name for new file is not given.",
  " ",
  "Usage: @P [Options] maskfile [newmaskfile]",
  " ",
  "Options:",
  " -xyz | -xy",
  "     Cloak the mask in 3D (default) or 2D.",
  " -stdoptions", // List standard options like --help, -v, etc
  " ",
  "See also: maskdila, maskeros, imgmask, masksize, imgthrs",
  " ",
  "Keywords: image, mask, dilation",
  0};
/*****************************************************************************/

/*****************************************************************************/
/* Turn on the globbing of the command line, since it is disabled by default in
   mingw-w64 (_dowildcard=0); in MinGW32 define _CRT_glob instead, if necessary;
   In Unix&Linux wildcard command line processing is enabled by default. */
/*
#undef _CRT_glob
#define _CRT_glob -1
*/
int _dowildcard = -1;
/*****************************************************************************/

/*****************************************************************************/
#if(0)
/** Cloak mask.
    @sa imgMaskDilate, imgMaskErode
    @return Returns 0 when successful.
 */
int imgMaskCloak(
  /** Pointer to 2D/3D mask image. */
  IMG *img,
  /** Cloak the existing mask in xy (2) or xyz (3) dimensions. */
  int dim,
  /** Id number for the cloak mask. Should be different from existing IDs. */
  float id
) {
  if(img==NULL) return(1);
  int dimx=img->dimx, dimy=img->dimy, dimz=img->dimz;
  if(dimx<1 || dimy<1 || dimz<1) return(2);

  for(int z=0; z<dimz; z++) {
    for(int y=0; y<dimy; y++) {
      for(int x=0; x<dimx; x++) if(!(img->m[z][y][x][0]<0.5) && img->m[z][y][x][0]!=id) {
        int x1=x, x2=x, y1=y, y2=y, z1=z, z2=z;
        if(x>0) x1=x-1;
        if(x<dimx-1) x2=x+1;
        if(y>0) y1=y-1;
        if(y<dimy-1) y2=y+1;
        if(dim==3) {
          if(z>0) z1=z-1;
          if(z<dimz-1) z2=z+1;
        }
        for(int nz=z1; nz<=z2; nz++)
          for(int ny=y1; ny<=y2; ny++)
            for(int nx=x1; nx<=x2; nx++)
              if(z!=nz || y!=ny || x!=nx)
                if(img->m[nz][ny][nx][0]<0.5)
                  img->m[nz][ny][nx][0]=id;
      }
    }
  }

  return(0);
}
#endif
/*****************************************************************************/

/*****************************************************************************/
/**
 *  Main
 */
int main(int argc, char **argv)
{
  int     ai, help=0, version=0, verbose=1;
  int     ret;
  char    maskfile[FILENAME_MAX], outfile[FILENAME_MAX];
  int     dim=3; // 3 (xyz) or 2 (xy)
  char   *cptr=NULL;


  /*
   *  Get arguments
   */
  if(argc==1) {tpcPrintUsage(argv[0], info, stderr); return(1);}
  maskfile[0]=outfile[0]=(char)0;
  /* Options */
  for(ai=1; ai<argc; ai++) if(*argv[ai]=='-') { /* options */
    cptr=argv[ai]+1; if(*cptr=='-') cptr++; if(cptr==NULL) continue;
    if(tpcProcessStdOptions(argv[ai], &help, &version, &verbose)==0) continue;
    if(strcasecmp(cptr, "XYZ")==0) {
      dim=3; continue;
    } else if(strcasecmp(cptr, "XY")==0) {
      dim=2; continue;
    }
    fprintf(stderr, "Error: invalid option '%s'.\n", argv[ai]); 
    return(1);
  } else break;
  
  /* Print help or version? */
  if(help==2) {tpcHtmlUsage(argv[0], info, ""); return(0);}
  if(help) {tpcPrintUsage(argv[0], info, stdout); return(0);}
  if(version) {tpcPrintBuild(argv[0], stdout); return(0);}

  /* Process other arguments, starting from the first non-option */
  if(ai<argc) {strlcpy(maskfile, argv[ai], FILENAME_MAX); ai++;}
  if(ai<argc) {strlcpy(outfile, argv[ai], FILENAME_MAX); ai++;}
  if(ai<argc) {fprintf(stderr, "Error: too many arguments.\n"); return(1);}

  /* Did we get all the information that we need? */
  if(!maskfile[0]) {
    fprintf(stderr, "Error: missing command-line argument; use option --help\n");
    return(1);
  }


  /* In verbose mode print options */
  if(verbose>1) {
    printf("maskfile := %s\n", maskfile);
    if(outfile[0]) printf("outfile := %s\n", outfile);
    printf("dim := %d\n", dim);
    fflush(stdout);
  }

  /* If file name for output was not given, the overwrite input */
  if(!outfile[0]) strcpy(outfile, maskfile);


  /*
   *  Read mask
   */
  IMG mask; imgInit(&mask);

  if(verbose>0) {printf("reading %s\n", maskfile); fflush(stdout);}
  ret=imgRead(maskfile, &mask);
  if(ret) {
    fprintf(stderr, "Error: %s\n", mask.statmsg);
    if(verbose>1) printf("ret := %d\n", ret);
    return(2);
  }
  if(mask.dimt>1) {
    fprintf(stderr, "Error: mask cannot be dynamic image.\n");
    imgEmpty(&mask); return(2);
  }
  /* Check whether mask has any voxels to begin with */
  long long mn=imgMaskCount(&mask);
  if(mn==0) {
    fprintf(stderr, "Warning: initial mask contains no positive voxels.\n");
    imgEmpty(&mask);
    return(0);
  }
  long long pxlNr=(long long)mask.dimz*mask.dimy*mask.dimx;
  if(mn==pxlNr) {
    fprintf(stderr, "Warning: initial mask contains all positive voxels.\n");
    imgEmpty(&mask);
    return(0);
  }
  if(verbose>1) {
    printf("initial_nr_of_positive_voxels := %lld\n", mn);
  }


  /*
   *  Set the id number for new mask
   */
  float id=0.0;
  if(imgMax(&mask, &id)!=0 || id<0.9) {
    fprintf(stderr, "Error: invalid mask ID.\n");
    imgEmpty(&mask);
    return(3);
  }
  id=1.0+roundf(id);
  if(verbose>1) printf("new mask ID := %.0f\n", id);


  /*
   *  Cloak the mask
   */
  if(verbose>0) {printf("cloaking\n"); fflush(stdout);}
  ret=imgMaskCloak(&mask, dim, id);
  if(ret!=0) {
    fprintf(stderr, "Error: cannot cloak the mask.\n");
    if(verbose>1) printf("ret := %d\n", ret);
    imgEmpty(&mask);
    return(8);
  }
  long long mnn=imgMaskCount(&mask);
  if(verbose>0) {
    fprintf(stdout, "Cloak consists of %lld voxel.\n", mnn-mn);
    fflush(stdout);
  }

  /*
   *  Save the modified mask
   */
  if(verbose>2) printf("writing mask\n");
  ret=imgWrite(outfile, &mask);
  if(ret) {
    fprintf(stderr, "Error: %s\n", mask.statmsg);
    imgEmpty(&mask); return(11);
  }
  imgEmpty(&mask);
  if(verbose>0) printf("dilated mask written in %s.\n\n", outfile);

  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/// @endcond
