/** @file masklbl.c
 *  @brief Mask image region labelling.
 *  @copyright (c) Turku PET Centre
 *  @author Vesa Oikonen
 */
/// @cond
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <string.h>
#include <unistd.h>
#include <time.h>
/*****************************************************************************/
#include "libtpcmisc.h"
#include "libtpcimgio.h"
#include "libtpcimgp.h"
/*****************************************************************************/

/*****************************************************************************/
static char *info[] = {
  "Region labelling (1) for mask image in ECAT 6.3 or 7.x, NIfTI-1, or",
  "Analyze 7.5 format.",
  "All pixels with value < 0.1 are considered as background pixels; other",
  "pixels are considered as foreground pixels, and tagged with values 2, 3, ...",
  "Notice that mask image file will be overwritten by labelled mask file.",
  " ",
  "Usage: @P [Options] maskfile",
  " ",
  "Options:",
  " -stdoptions", // List standard options like --help, -v, etc
  " ",
  "References:",
  "1. Burger W and Burge MJ: Principles of Digital Image Processing - Core",
  "   Algorithms, Springer, 2009, DOI 10.1007/978-1-84800-195-4.",
  " ",
  "See also: masksize, imgthrs, maskinv, maskconj, imgmask",
  " ",
  "Keywords: image, mask, ROI",
  0};
/*****************************************************************************/

/*****************************************************************************/
/* Turn on the globbing of the command line, since it is disabled by default in
   mingw-w64 (_dowildcard=0); in MinGW32 define _CRT_glob instead, if necessary;
   In Unix&Linux wildcard command line processing is enabled by default. */
/*
#undef _CRT_glob
#define _CRT_glob -1
*/
int _dowildcard = -1;
/*****************************************************************************/

/*****************************************************************************/
/**
 *  Main
 */
int main(int argc, char **argv)
{
  int     ai, help=0, version=0, verbose=1;
  int     ret, n;
  char    maskfile[FILENAME_MAX];
//char   *cptr=NULL;


  /*
   *  Get arguments
   */
  if(argc==1) {tpcPrintUsage(argv[0], info, stderr); return(1);}
  maskfile[0]=(char)0;
  /* Options */
  for(ai=1; ai<argc; ai++) if(*argv[ai]=='-') { /* options */
    if(tpcProcessStdOptions(argv[ai], &help, &version, &verbose)==0) continue;
    //cptr=argv[ai]+1; if(*cptr=='-') cptr++; if(cptr==NULL) continue;
    fprintf(stderr, "Error: invalid option '%s'.\n", argv[ai]); 
    return(1);
  } else break;
  
  /* Print help or version? */
  if(help==2) {tpcHtmlUsage(argv[0], info, ""); return(0);}
  if(help) {tpcPrintUsage(argv[0], info, stdout); return(0);}
  if(version) {tpcPrintBuild(argv[0], stdout); return(0);}

  /* Process other arguments, starting from the first non-option */
  if(ai<argc) {strlcpy(maskfile, argv[ai], FILENAME_MAX); ai++;}
  if(ai<argc) {fprintf(stderr, "Error: too many arguments.\n"); return(1);}

  /* Did we get all the information that we need? */
  if(!maskfile[0]) {
    fprintf(stderr, "Error: missing command-line argument; use option --help\n");
    return(1);
  }


  /* In verbose mode print options */
  if(verbose>1) {
    printf("maskfile := %s\n", maskfile);
    fflush(stdout);
  }


  /*
   *  Read mask
   */
  IMG mask; imgInit(&mask);

  if(verbose>0) {printf("reading %s\n", maskfile); fflush(stdout);}
  ret=imgRead(maskfile, &mask);
  if(ret) {
    fprintf(stderr, "Error: %s\n", mask.statmsg);
    if(verbose>1) printf("ret := %d\n", ret);
    return(2);
  }
  if(mask.dimt>1) {
    fprintf(stderr, "Error: mask cannot be dynamic image.\n");
    imgEmpty(&mask); return(2);
  }


  /*
   *  Region labelling
   */
  if(verbose>0) {printf("labelling regions...\n"); fflush(stdout);}
  IMG mask2; imgInit(&mask2);
  ret=imgMaskRegionLabeling(&mask, &mask2, &n, verbose-2);
  imgEmpty(&mask);
  if(ret!=0) {
    fprintf(stderr, "Error: cannot label regions.\n");
    if(verbose>1) printf("ret := %d\n", ret);
    imgEmpty(&mask2);
    return(6);
  }
  if(verbose>0) printf("region_nr := %d\n", n);
  if(n==0) {
    fprintf(stdout, "Warning: mask file was empty.\n");
    imgEmpty(&mask2);
    return(0);
  }


  /*
   *  Save the modified mask
   */
  if(verbose>2) printf("writing labelled mask...\n");
  ret=imgWrite(maskfile, &mask2);
  if(ret) {
    fprintf(stderr, "Error: %s\n", mask2.statmsg);
    imgEmpty(&mask2); return(11);
  }
  imgEmpty(&mask2);
  if(verbose>0) printf("labelled mask written.\n\n");

  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/// @endcond
