/** @file imgfur.c
 *  @brief Calculation of FUR from dynamic or static PET image data.
 *  @copyright (c) Turku PET Centre
 *  @author Vesa Oikonen
 */
/// @cond
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <math.h>
#include <time.h>
/*****************************************************************************/
#include "libtpcmisc.h"
#include "libtpcmodel.h"
#include "libtpccurveio.h"
#include "libtpcimgio.h"
#include "libtpcimgp.h"
#include "libtpcmodext.h"
/*****************************************************************************/

/*****************************************************************************/
#ifndef DEFAULT_LC
#define DEFAULT_LC 1.00
#endif
#ifndef DEFAULT_DENSITY
#define DEFAULT_DENSITY 1.00
#endif
/*****************************************************************************/

/*****************************************************************************/
static char *info[] = {
  "Calculates of Fractional Uptake Rate (FUR) or FUR-based Metabolic Rate (MR)",
  "image from static or dynamic PET image in ECAT 6.3 or 7.2, NIfTI, or",
  "Analyze 7.5 format. Information on FUR in:",
  "http://www.turkupetcentre.net/petanalysis/model_fur.html",
  " ",
  "Usage: @P [Options] inputfile image starttime endtime furimage",
  " ",
  "FUR calculation start and stop time must be entered in minutes;",
  "set both to zero to use the whole time range from image data.",
  " ",
  "Options:",
  " -Ca=<value>",
  "     Concentration of native substrate in arterial plasma (mM),",
  "     for example plasma glucose in [18F]FDG studies.",
  "     With this option the metabolic rate (umol/(min*100 g)) is calculated.",
  " -LC=<value>",
  "     Lumped Constant in MR calculation; default is 1.0.",
  " -density=<value>",
  "     Tissue density in MR calculation; default is 1.0 g/ml.",
  " -stdoptions", // List standard options like --help, -v, etc
  " ",
  "Example 1. Calculation of FUR image from a dynamic image from 45 to 60 min:",
  "     @P ua2918ap.kbq ua2918dy1.v 45 60 ua2918fur.v",
  " ",
  "Example 2. Calculation of glucose uptake image, when tissue density is 1.04,",
  "plasma glucose concentration is 5.2 mM, lumped constant is 0.52, from",
  "a static (one frame) image:",
  "     @P -density=1.04 -Ca=5.2 -LC=0.52 a864ap.kbq a864dy1.v 0 0 a864mrglu.v",
  " ",
  "The unit of pixel values in the FUR image is",
  "(mL plasma)/(min*(mL tissue)) by default, and umol/(min*100 g) in metabolic",
  "rate image.",
  " ",
  "See also: imgki, imginteg, imgcalc, ecattime, img2tif, regfur",
  " ",
  "Keywords: image, modelling, FUR, retention index, irreversible uptake",
  0};
/*****************************************************************************/

/*****************************************************************************/
/* Turn on the globbing of the command line, since it is disabled by default in
   mingw-w64 (_dowildcard=0); in MinGW32 define _CRT_glob instead, if necessary;
   In Unix&Linux wildcard command line processing is enabled by default. */
/*
#undef _CRT_glob
#define _CRT_glob -1
*/
int _dowildcard = -1;
/*****************************************************************************/

/*****************************************************************************/
/**
 *  Main
 */
int main(int argc, char **argv)
{
  int        ai, help=0, version=0, verbose=1;
  int        ret;
  char       inpfile[FILENAME_MAX], petfile[FILENAME_MAX], outfile[FILENAME_MAX];
  char       tmp[1024], *cptr;
  DFT        input, auc;
  IMG        img, out;
  double     startTime=-1.0, endTime=-1.0;
  double     LC=-1.0, Ca=-1.0, density=-1.0;


  /*
   *  Get arguments
   */
  if(argc==1) {tpcPrintUsage(argv[0], info, stderr); return(1);}
  inpfile[0]=petfile[0]=outfile[0]=(char)0;
  imgInit(&img); imgInit(&out);
  dftInit(&input); dftInit(&auc);
  /* Get options */
  for(ai=1; ai<argc; ai++) if(*argv[ai]=='-') {
    if(tpcProcessStdOptions(argv[ai], &help, &version, &verbose)==0) continue;
    cptr=argv[ai]+1; if(*cptr=='-') cptr++; if(cptr==NULL) continue;
    if(strncasecmp(cptr, "CA=", 3)==0) {
      Ca=atof_dpi(cptr+3); if(Ca>0.0) continue;
    } else if(strncasecmp(cptr, "LC=", 3)==0) {
      LC=atof_dpi(cptr+3); if(LC>0.0) continue;
    } else if(strncasecmp(cptr, "D=", 2)==0) {
      density=atof_dpi(cptr+2); if(density>0.0) continue;
    } else if(strncasecmp(cptr, "DENSITY=", 8)==0) {
      density=atof_dpi(cptr+8); if(density>0.0) continue;
    }
    fprintf(stderr, "Error: invalid option '%s'.\n", argv[ai]);
    return(1);
  } else break;
  
  /* Print help or version? */
  if(help==2) {tpcHtmlUsage(argv[0], info, ""); return(0);}
  if(help) {tpcPrintUsage(argv[0], info, stdout); return(0);}
  if(version) {tpcPrintBuild(argv[0], stdout); return(0);}

  /* Process other arguments, starting from the first non-option */
  for(; ai<argc; ai++) {
    if(!inpfile[0]) {
      strcpy(inpfile, argv[ai]); continue;
    } else if(!petfile[0]) {
      strcpy(petfile, argv[ai]); continue;
    } else if(startTime<0.0) {
      startTime=atof_dpi(argv[ai]); if(startTime<0.0) startTime=0.0;
      continue;
    } else if(endTime<0.0) {
      endTime=atof_dpi(argv[ai]); if(endTime<0.0) endTime=0.0;
      continue;
    } else if(!outfile[0]) {
      strcpy(outfile, argv[ai]); continue;
    }
    fprintf(stderr, "Error: invalid argument '%s'.\n", argv[ai]);
    return(1);
  }

  /* Is something missing? */
  if(!outfile[0]) {
    fprintf(stderr, "Error: missing result file name.\n");
    return(1);
  }
  /* If MR will be calculated, set defaults and give warnings as necessary */
  if(Ca>0.0) {
    if(LC<0.0) {
      LC=DEFAULT_LC;
      fprintf(stderr, "Warning: LC not set, using default %g\n", LC);
    }
    if(density<0.0) {
      density=DEFAULT_DENSITY;
      fprintf(stderr, "Warning: tissue density not set, using default %g\n",
        density);
    } 
  } else { /* Warnings if density or LC set when MR will not be calculated */
    if(LC>0.0) fprintf(stderr, "Warning: LC was set but is not used.\n");
    if(density>0.0) 
      fprintf(stderr, "Warning: tissue density was set but is not used.\n");
  }
  /* Check the time range */
  if(endTime<startTime) {
    fprintf(stderr, "Error: invalid time range.\n");
    return(1);
  }
  if(startTime==endTime && startTime>0.5) {
    startTime-=0.5; endTime+=0.5;
  }


  /* In verbose mode print arguments and options */
  if(verbose>1) {
    printf("inpfile := %s\n", inpfile);
    printf("petfile := %s\n", petfile);
    printf("outfile := %s\n", outfile);
    if(endTime>0.0) {
      printf("startTime := %g min\n", startTime);
      printf("endTime := %g min\n", endTime);
    }
    if(Ca>0.0) {
      printf("Ca := %g\n", Ca);
      printf("LC := %g\n", LC);
      printf("density := %g\n", density);
    }
  }
  if(verbose>9) IMG_TEST=verbose-9; else IMG_TEST=0;


  /*
   *  Read image
   */
  if(verbose>0) fprintf(stdout, "reading image %s\n", petfile);
  ret=imgRead(petfile, &img); 
  if(ret) {
    fprintf(stderr, "Error: %s\n", img.statmsg); if(verbose>1) imgInfo(&img);
    return(2);
  }
  if(imgNaNs(&img, 1)>0)
    if(verbose>0) fprintf(stderr, "Warning: missing pixel values.\n");
  /* Check if PET data is raw or image */
  if(img.type!=IMG_TYPE_IMAGE) {
    fprintf(stderr, "Error: %s is not an image.\n", petfile);
    imgEmpty(&img); return(2);
  }
  /* Check that frame times are available */
  if(imgExistentTimes(&img)==0) {
    fprintf(stderr, "Error: %s does not contain frame times.\n", petfile);
    imgEmpty(&img); return(2);
  }
  //imgInfo(&img);
  /* Make sure that there is no overlap in image frames */ 
  if(img.dimt>1) {
    if(verbose>1) fprintf(stdout, "checking frame overlap in %s\n", petfile);
    ret=imgDeleteFrameOverlap(&img); 
    if(ret) {
      fprintf(stderr, "Error: image %s has overlapping frame times.\n", petfile);
      imgEmpty(&img); return(2);
    }
  }

  /* If image is in NIfTI or Analyze format, it does not contain calibration
     unit; then assume that it is kBq/mL, and warn user about that */
  if(img._fileFormat==IMG_NIFTI_1D || img._fileFormat==IMG_NIFTI_1S ||
     img._fileFormat==IMG_ANA || img._fileFormat==IMG_ANA_L)
  {
    if(img.unit==IMGUNIT_UNKNOWN) {
      img.unit=IMGUNIT_KBQ_PER_ML;
      fprintf(stderr, "Warning: image calibration unit assumed to be %s.\n",
              imgUnit(img.unit));
    }
  }

  /* If user did not specify start and end times, then get those from data */
  if(endTime<=0.0) {
    startTime=img.start[0]/60.0;
    endTime=img.end[img.dimt-1]/60.0;
    if(verbose>0) {
      printf("startTime := %g min\n", startTime);
      printf("endTime := %g min\n", endTime);
    }
  }

  /*
   *  Calculate the average over specified time range
   */
  if(verbose>1) fprintf(stdout, "calculating average image\n");
  ret=imgTimeIntegral(&img, startTime*60.0, endTime*60.0, &out, 1, tmp, 
                      verbose-4);
  if(ret!=0) {
    fprintf(stderr, "Error: %s.\n", tmp);
    imgEmpty(&img); imgEmpty(&out); return(3);
  }
  if(verbose>1) fprintf(stdout, "%s.\n", tmp);

  /* Original dynamic image is not needed anymore */
  imgEmpty(&img);


  /*
   *  Read input data
   */
  if(verbose>1) printf("Reading input file %s\n", inpfile);
  if(dftRead(inpfile, &input)) {
    fprintf(stderr, "Error in reading '%s': %s\n", inpfile, dfterrmsg);
    imgEmpty(&out); return(4);
  }
  if(input.voiNr>1) {
    fprintf(stderr, "Warning: only first TAC is used as input.\n");
    input.voiNr=1;
  }
  if(dft_nr_of_NA(&input)>0) {  // check if file contains NAs (missing values)
    fprintf(stderr, "Error: missing values in %s\n", inpfile);
    imgEmpty(&out); dftEmpty(&input); return(4);
  }
  /* Set time unit to min */
  ret=dftTimeunitConversion(&input, TUNIT_MIN);
  if(ret) fprintf(stderr, "Warning: check that input times are in minutes.\n");

  /*
   *  Check the image and plasma concentration units
   */
  ret=cunit_check_dft_vs_img(&input, &out, tmp, verbose-4);
  if(ret==0) {if(verbose>0) fprintf(stdout, "%s\n", tmp);
  } else if(ret<0) {fprintf(stderr, "Warning: %s\n", tmp);
  } else {
    fprintf(stderr, "Error: %s.\n", tmp);
    imgEmpty(&out); dftEmpty(&input); return(5);
  }

  /*
   *  Calculate input integral from 0 to PET middle time point
   */
  if(verbose>1) printf("calculating input AUC\n");
  ret=dftTimeIntegral(&input, 0.0, 0.5*(startTime+endTime), &auc, 0, tmp,
                      verbose-4);
  if(ret!=0) {
    fprintf(stderr, "Error: %s.\n", tmp);
    if(verbose>1) printf("error_code := %d\n", ret);
    imgEmpty(&out); dftEmpty(&input); dftEmpty(&auc); return(6);
  }
  if(verbose>0)
    printf("AUC[%g-%g] := %g\n", auc.x1[0], auc.x2[0], auc.voi[0].y[0]);

  /* Original input curve is not needed anymore */
  dftEmpty(&input);


  /*
   *  Divide average PET data by input AUC
   */
  ret=imgArithmConst(&out, auc.voi[0].y[0], ':', 1.0E+12, verbose-5);
  if(ret!=0) {
    fprintf(stderr, "Error: cannot calculate of FUR.\n"); 
    imgEmpty(&out); dftEmpty(&auc);
    return(7);
  }
  out.unit=IMGUNIT_PER_MIN;

  /* Input AUC is not needed anymore */
  dftEmpty(&auc);


  /*
   *  Calculate metabolic rate, if necessary
   */
  if(Ca>0.0) {
    double MRf;
    MRf=100.*Ca/(density*LC);
    if(verbose>0)
      fprintf(stdout, "converting FUR to metabolic rate with factor %g\n", MRf);
    ret=imgArithmConst(&out, MRf, '*', 1.0E+06, verbose-5);
    if(ret!=0) {
      fprintf(stderr, "Error: cannot calculate metabolic rate.\n"); 
      imgEmpty(&out); return(8);
    }
    out.unit=IMGUNIT_UMOL_PER_MIN_PER_100G;
  }


  /*
   *  Save parametric Ki image
   */
  if(backupExistingFile(outfile, NULL, tmp)!=0) {
    fprintf(stderr, "Error: %s\n", tmp); 
    imgEmpty(&out); return(11);
  }
  if(verbose>0) printf("%s\n", tmp);
  ret=imgWrite(outfile, &out); 
  if(ret) {
    fprintf(stderr, "Error: %s\n", out.statmsg); 
    imgEmpty(&out); return(12);
  }
  if(verbose>0) {
    if(Ca<=0.0) fprintf(stdout, "FUR image %s saved.\n", outfile);
    else fprintf(stdout, "MR image %s saved.\n", outfile);
  }

  /* Free memory */
  imgEmpty(&out);

  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/// @endcond
