#!/bin/bash
#: Title      : test_patlak
#: Date       : 2020-05-26
#: Author     : "Vesa Oikonen" <vesa.oikonen@utu.fi>
#: Options    : None

# Set the name of executable to test
if [ -n "${OS+1}" ] && [ "$OS" = "Windows_NT" ]; then
EXT=.exe;
else
EXT=;
fi
PROGRAM=../../patlak$EXT;
if [ ! -f $PROGRAM ]; then
  printf "Failed: executable does not exist.\n"
  exit 1
fi

printf "=====================================================================\n"
printf "testing %s\n" $PROGRAM
printf "=====================================================================\n"

echo ""
if [ -f dynamic.dft ]; then
echo "Test data seems to exist, thus not created now."
else 
echo ===============================================================================
echo "creating test data"
echo ===============================================================================
echo ""
extrapol -mintime=20 plasma.dat 240 plasma_extrapolated.dat
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
tacunit -yset=kBq/ml -xset=min plasma_extrapolated.dat
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
PLANE=1
p2t_3c -nosub --silent plasma_extrapolated.dat 0.05 0.03 0.001 0 0 0 tis1.dft
p2t_3c -nosub --silent plasma_extrapolated.dat 0.05 0.10 0.001 0 0 0 tis2.dft
p2t_3c -nosub --silent plasma_extrapolated.dat 0.05 0.25 0.001 0 0 0 tis3.dft
p2t_3c -nosub --silent plasma_extrapolated.dat 0.05 0.03 0.005 0 0 0 tis4.dft
p2t_3c -nosub --silent plasma_extrapolated.dat 0.05 0.10 0.005 0 0 0 tis5.dft
p2t_3c -nosub --silent plasma_extrapolated.dat 0.05 0.25 0.005 0 0 0 tis6.dft
p2t_3c -nosub --silent plasma_extrapolated.dat 0.05 0.03 0.01 0 0 0 tis7.dft
p2t_3c -nosub --silent plasma_extrapolated.dat 0.05 0.10 0.01 0 0 0 tis8.dft
p2t_3c -nosub --silent plasma_extrapolated.dat 0.05 0.25 0.01 0 0 0 tis9.dft
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
tacadd --silent -ovr plane$PLANE.dft tis1.dft 1
tacadd --silent      plane$PLANE.dft tis2.dft 1
tacadd --silent      plane$PLANE.dft tis3.dft 1
tacadd --silent      plane$PLANE.dft tis4.dft 1
tacadd --silent      plane$PLANE.dft tis5.dft 1
tacadd --silent      plane$PLANE.dft tis6.dft 1
tacadd --silent      plane$PLANE.dft tis7.dft 1
tacadd --silent      plane$PLANE.dft tis8.dft 1
tacadd --silent      plane$PLANE.dft tis9.dft 1
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
simframe --silent plane$PLANE.dft frames.dat plane$PLANE'f.dft' F-18
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
tacren --silent plane$PLANE'f.dft' 1 ambf_dx_Pl00$PLANE
tacren --silent plane$PLANE'f.dft' 2 ambf_sin_Pl00$PLANE
tacren --silent plane$PLANE'f.dft' 3 ambf_avg_Pl00$PLANE
tacren --silent plane$PLANE'f.dft' 4 bedr_dx_Pl00$PLANE
tacren --silent plane$PLANE'f.dft' 5 bedr_sin_Pl00$PLANE
tacren --silent plane$PLANE'f.dft' 6 bedr_avg_Pl00$PLANE
tacren --silent plane$PLANE'f.dft' 7 ciztup__Pl00$PLANE
tacren --silent plane$PLANE'f.dft' 8 biztup__Pl00$PLANE
tacren --silent plane$PLANE'f.dft' 9 ferm_avg_Pl00$PLANE
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi

PLANE=2
p2t_3c -nosub --silent plasma_extrapolated.dat 0.20 0.05 0 0 0 0 tis1.dft
p2t_3c -nosub --silent plasma_extrapolated.dat 0.20 0.20 0 0 0 0 tis2.dft
p2t_3c -nosub --silent plasma_extrapolated.dat 0.20 0.80 0 0 0 0 tis3.dft
p2t_3c -nosub --silent plasma_extrapolated.dat 0.20 0.05 0.002 0 0 0 tis4.dft
p2t_3c -nosub --silent plasma_extrapolated.dat 0.20 0.20 0.002 0 0 0 tis5.dft
p2t_3c -nosub --silent plasma_extrapolated.dat 0.20 0.80 0.002 0 0 0 tis6.dft
p2t_3c -nosub --silent plasma_extrapolated.dat 0.20 0.05 0.01 0 0 0 tis7.dft
p2t_3c -nosub --silent plasma_extrapolated.dat 0.20 0.20 0.01 0 0 0 tis8.dft
p2t_3c -nosub --silent plasma_extrapolated.dat 0.20 0.80 0.01 0 0 0 tis9.dft
tacadd --silent -ovr plane$PLANE.dft tis1.dft 1
tacadd --silent      plane$PLANE.dft tis2.dft 1
tacadd --silent      plane$PLANE.dft tis3.dft 1
tacadd --silent      plane$PLANE.dft tis4.dft 1
tacadd --silent      plane$PLANE.dft tis5.dft 1
tacadd --silent      plane$PLANE.dft tis6.dft 1
tacadd --silent      plane$PLANE.dft tis7.dft 1
tacadd --silent      plane$PLANE.dft tis8.dft 1
tacadd --silent      plane$PLANE.dft tis9.dft 1
simframe --silent plane$PLANE.dft frames.dat plane$PLANE'f.dft' F-18
tacren --silent plane$PLANE'f.dft' 1 revers_dx_Pl00$PLANE
tacren --silent plane$PLANE'f.dft' 2 revers_sin_Pl00$PLANE
tacren --silent plane$PLANE'f.dft' 3 revers_avg_Pl00$PLANE
tacren --silent plane$PLANE'f.dft' 4 bedr_dx_Pl00$PLANE
tacren --silent plane$PLANE'f.dft' 5 bedr_sin_Pl00$PLANE
tacren --silent plane$PLANE'f.dft' 6 bedr_avg_Pl00$PLANE
tacren --silent plane$PLANE'f.dft' 7 ciztup__Pl00$PLANE
tacren --silent plane$PLANE'f.dft' 8 biztup__Pl00$PLANE
tacren --silent plane$PLANE'f.dft' 9 ferm_avg_Pl00$PLANE

PLANE=3
p2t_3c -nosub --silent plasma_extrapolated.dat 0.60 0.10 0.005 0 0 0 tis1.dft
p2t_3c -nosub --silent plasma_extrapolated.dat 0.60 0.40 0.005 0 0 0 tis2.dft
p2t_3c -nosub --silent plasma_extrapolated.dat 0.60 1.60 0.005 0 0 0 tis3.dft
p2t_3c -nosub --silent plasma_extrapolated.dat 0.60 0.10 0.02 0 0 0 tis4.dft
p2t_3c -nosub --silent plasma_extrapolated.dat 0.60 0.40 0.02 0 0 0 tis5.dft
p2t_3c -nosub --silent plasma_extrapolated.dat 0.60 1.60 0.02 0 0 0 tis6.dft
p2t_3c -nosub --silent plasma_extrapolated.dat 0.60 0.10 0.1 0 0 0 tis7.dft
p2t_3c -nosub --silent plasma_extrapolated.dat 0.60 0.40 0.1 0 0 0 tis8.dft
p2t_3c -nosub --silent plasma_extrapolated.dat 0.60 1.60 0.1 0 0 0 tis9.dft
tacadd --silent -ovr plane$PLANE.dft tis1.dft 1
tacadd --silent      plane$PLANE.dft tis2.dft 1
tacadd --silent      plane$PLANE.dft tis3.dft 1
tacadd --silent      plane$PLANE.dft tis4.dft 1
tacadd --silent      plane$PLANE.dft tis5.dft 1
tacadd --silent      plane$PLANE.dft tis6.dft 1
tacadd --silent      plane$PLANE.dft tis7.dft 1
tacadd --silent      plane$PLANE.dft tis8.dft 1
tacadd --silent      plane$PLANE.dft tis9.dft 1
simframe --silent plane$PLANE.dft frames.dat plane$PLANE'f.dft' F-18
tacren --silent plane$PLANE'f.dft' 1 ambf_dx_Pl00$PLANE
tacren --silent plane$PLANE'f.dft' 2 ambf_sin_Pl00$PLANE
tacren --silent plane$PLANE'f.dft' 3 ambf_avg_Pl00$PLANE
tacren --silent plane$PLANE'f.dft' 4 bedr_dx_Pl00$PLANE
tacren --silent plane$PLANE'f.dft' 5 bedr_sin_Pl00$PLANE
tacren --silent plane$PLANE'f.dft' 6 bedr_avg_Pl00$PLANE
tacren --silent plane$PLANE'f.dft' 7 ciztup__Pl00$PLANE
tacren --silent plane$PLANE'f.dft' 8 biztup__Pl00$PLANE
tacren --silent plane$PLANE'f.dft' 9 ferm_avg_Pl00$PLANE

tacadd --silent -ovr dynamic_long.dft plane1f.dft
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
tacadd --silent dynamic_long.dft plane2f.dft
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
tacadd --silent dynamic_long.dft plane3f.dft
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
tacunit -yset=kBq/ml -xset=min dynamic_long.dft

echo ""
taccut dynamic_long.dft 0 120 dynamic.dft
tacstudy dynamic_long.dft long1
tacstudy dynamic.dft test1

echo ""
tacformat -f=simple -hdr=no dynamic.dft dynamic.dat
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi

tac2svg dynamic.svg -l dynamic.dft plasma.dat
echo ""

fi


echo ""
echo "Create PMOD format files for testing"
tacformat -f=pmod dynamic.dft dynamic.tac
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
tacformat -f=pmod plasma.dat plasma.bld
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
echo ""

echo ===============================================================================
echo ""
echo ===============================================================================
echo "1.1.2: Unknown option."
if [ -f stderr.txt ] ; then rm stderr.txt ; fi
if [ -f output.res ] ; then rm output.res ; fi
$PROGRAM -stupidoption dynamic.dft plasma.dat 30 999 output.res 2> stderr.txt
if [ $? -eq 0 ] ; then echo Failed! ; exit 1 ; fi
echo passed.
iftisval stderr.txt Error "invalid option '-stupidoption'."
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
echo passed. ; echo ""
echo ===============================================================================
echo "1.1.3: Extra argument."
if [ -f stderr.txt ] ; then rm stderr.txt ; fi
$PROGRAM dynamic.dft plasma.dat 30 999 output.res extra 2> stderr.txt
if [ $? -eq 0 ] ; then echo Failed! ; exit 1 ; fi
echo passed.
iftisval stderr.txt Error "invalid argument 'extra'."
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
echo passed. ; echo ""
echo ===============================================================================
echo ""
echo ===============================================================================
rm -f stdout.txt
echo "1.1.4: Missing argument."
$PROGRAM  plasma.dat 30 999 output.res > stdout.txt
if [ $? -eq 0 ] ; then echo Failed! ; exit 1 ; fi
echo passed.
$PROGRAM dynamic.dft  30 999 output.res > stdout.txt
if [ $? -eq 0 ] ; then echo Failed! ; exit 1 ; fi
echo passed.
$PROGRAM dynamic.dft plasma.dat  999 output.res > stdout.txt
if [ $? -eq 0 ] ; then echo Failed! ; exit 1 ; fi
echo passed.
$PROGRAM dynamic.dft plasma.dat 30  output.res > stdout.txt
if [ $? -eq 0 ] ; then echo Failed! ; exit 1 ; fi
echo passed.
$PROGRAM dynamic.dft plasma.dat 30 999 > stdout.txt
if [ $? -eq 0 ] ; then echo Failed! ; exit 1 ; fi
echo passed.
echo passed. ; echo ""
echo ===============================================================================
echo ""
echo ===============================================================================
rm -f stdout.txt
echo "1.5.1: option -ca"
echo "$PROGRAM --debug=3 -ca=5.2 dynamic.dft plasma.dat 30 999 output.res"
$PROGRAM --debug=3 -ca=5.2 dynamic.dft plasma.dat 30 999 output.res &> stdout.txt
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
echo passed. ; echo ""
iftisval stdout.txt Ca "5.2"
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
echo passed. ; echo ""
iftisval stdout.txt Warning "LC not set, using default 1"
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
echo passed. ; echo ""
iftisval stdout.txt Warning "tissue density not set, using default 1"
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
echo passed. ; echo ""

rm -f stdout.txt
$PROGRAM --debug=3 -ca=5,2 dynamic.dft plasma.dat 30 999 output.res &> stdout.txt
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
echo passed.
iftisval stdout.txt Ca "5.2"
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
echo passed. ; echo ""

rm -f stdout.txt
$PROGRAM --debug=3 dynamic.dft plasma.dat 30 999 output.res &> stdout.txt
iftisval stdout.txt Ca
if [ $? -ne 10 ] ; then echo Failed! ; exit 1 ; fi
echo passed. ; echo ""
echo ===============================================================================
echo "1.5.2: option -ca erroneously"
if [ -f stdout.txt ] ; then rm stdout.txt ; fi
$PROGRAM --debug=3 -ca dynamic.dft plasma.dat 30 999 output.res > stdout.txt
if [ $? -eq 0 ] ; then echo Failed! ; exit 1 ; fi
echo passed.

if [ -f stdout.txt ] ; then rm stdout.txt ; fi
$PROGRAM --debug=3 -ca=0 dynamic.dft plasma.dat 30 999 output.res > stdout.txt
if [ $? -eq 0 ] ; then echo Failed! ; exit 1 ; fi
echo passed.

echo passed. ; echo ""
echo ===============================================================================
echo ""
echo ===============================================================================
echo "1.5.3: option -lc"
if [ -f stdout.txt ] ; then rm stdout.txt ; fi
$PROGRAM --debug=3 -ca=5.2 -lc=0.82 dynamic.dft plasma.dat 30 999 output.res > stdout.txt
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
echo passed.
iftisval stdout.txt LC "0.82"
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
echo passed. ; echo ""

if [ -f stdout.txt ] ; then rm stdout.txt ; fi
$PROGRAM --debug=3 -ca=5.2 -lc=0,82 dynamic.dft plasma.dat 30 999 output.res > stdout.txt
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
echo passed.
iftisval stdout.txt LC "0.82"
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
echo passed. ; echo ""

if [ -f stdout.txt ] ; then rm stdout.txt ; fi
if [ -f stderr.txt ] ; then rm stderr.txt ; fi
$PROGRAM --debug=3 -lc=0.82 dynamic.dft plasma.dat 30 999 output.res > stdout.txt 2> stderr.txt
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
echo passed.
iftisval stdout.txt Ca
if [ $? -ne 10 ] ; then echo Failed! ; exit 1 ; fi
echo passed. ; echo ""
iftisval stdout.txt LC
if [ $? -ne 10 ] ; then echo Failed! ; exit 1 ; fi
echo passed. ; echo ""
iftisval stderr.txt Warning "LC was set but is not used."
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
echo passed. ; echo ""
echo ===============================================================================
echo "1.5.4: option -lc erroneously"
if [ -f stdout.txt ] ; then rm stdout.txt ; fi
$PROGRAM --debug=3 -ca=5.2 -LC dynamic.dft plasma.dat 30 999 output.res > stdout.txt
if [ $? -eq 0 ] ; then echo Failed! ; exit 1 ; fi
echo passed.

if [ -f stdout.txt ] ; then rm stdout.txt ; fi
$PROGRAM --debug=3 -ca=5.2 -LC= dynamic.dft plasma.dat 30 999 output.res > stdout.txt
if [ $? -eq 0 ] ; then echo Failed! ; exit 1 ; fi
echo passed.

echo passed. ; echo ""
echo ===============================================================================
echo ""
echo ===============================================================================
echo "1.5.5: option -density"
if [ -f stdout.txt ] ; then rm stdout.txt ; fi
$PROGRAM --debug=3 -ca=5.2 -density=1.02 dynamic.dft plasma.dat 30 999 output.res > stdout.txt
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
echo passed.
iftisval stdout.txt density "1.02"
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
echo passed. ; echo ""

if [ -f stdout.txt ] ; then rm stdout.txt ; fi
$PROGRAM --debug=3 -ca=5.2 -d=1,02 dynamic.dft plasma.dat 30 999 output.res > stdout.txt
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
echo passed.
iftisval stdout.txt density "1.02"
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
echo passed. ; echo ""

if [ -f stdout.txt ] ; then rm stdout.txt ; fi
if [ -f stderr.txt ] ; then rm stderr.txt ; fi
$PROGRAM --debug=3 -density=1.02 dynamic.dft plasma.dat 30 999 output.res > stdout.txt 2> stderr.txt
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
echo passed.
iftisval stdout.txt Ca
if [ $? -ne 10 ] ; then echo Failed! ; exit 1 ; fi
echo passed. ; echo ""
iftisval stdout.txt density
if [ $? -ne 10 ] ; then echo Failed! ; exit 1 ; fi
echo passed. ; echo ""
iftisval stderr.txt Warning "tissue density was set but is not used."
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
echo passed. ; echo ""
echo ===============================================================================
echo "1.5.6: option -density erroneously"
if [ -f stdout.txt ] ; then rm stdout.txt ; fi
$PROGRAM --debug=3 -ca=5.2 -density dynamic.dft plasma.dat 30 999 output.res > stdout.txt
if [ $? -eq 0 ] ; then echo Failed! ; exit 1 ; fi
echo passed.

if [ -f stdout.txt ] ; then rm stdout.txt ; fi
$PROGRAM --debug=3 -ca=5.2 -density= dynamic.dft plasma.dat 30 999 output.res > stdout.txt
if [ $? -eq 0 ] ; then echo Failed! ; exit 1 ; fi
echo passed.

echo passed. ; echo ""
echo ===============================================================================
echo ""
echo ===============================================================================
echo "1.6.1: option -ic"

if [ -f stderr.txt ] ; then rm stderr.txt ; fi
$PROGRAM -ic dynamic.dft plasma.dat 30 999 output.res 2> stderr.txt
if [ $? -eq 0 ] ; then echo Failed! ; exit 1 ; fi
echo passed.
iftisval stderr.txt Error "invalid option '-ic'."
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
echo passed. ; echo ""

echo "positive ic with decimal point"
if [ -f stdout.txt ] ; then rm stdout.txt ; fi
$PROGRAM --debug=3 -ic=0.1 dynamic.dft plasma.dat 30 999 output.res > stdout.txt
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
echo passed.
iftisval stdout.txt fixed_Ic "0.1"
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
echo passed. ; echo ""

echo "negative ic with decimal comma"
if [ -f stdout.txt ] ; then rm stdout.txt ; fi
$PROGRAM --debug=3 -ic=-0,1 dynamic.dft plasma.dat 30 999 output.res > stdout.txt
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
echo passed.
iftisval stdout.txt fixed_Ic "-0.1"
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
echo passed. ; echo ""

echo "zero ic: FUR proposed"
if [ -f stderr.txt ] ; then rm stderr.txt ; fi
if [ -f stdout.txt ] ; then rm stdout.txt ; fi
$PROGRAM --debug=3 -ic=0 dynamic.dft plasma.dat 30 999 output.res > stdout.txt 2>stderr.txt
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
echo passed.
iftisval stdout.txt Suggestion "for FUR calculation use regfur."
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
echo passed. ; echo ""

echo ===============================================================================
echo ""
echo ===============================================================================
echo "1.7.1: option -sd=y"
if [ -f stdout.txt ] ; then rm stdout.txt ; fi
if [ -f stderr.txt ] ; then rm stderr.txt ; fi
$PROGRAM --debug=3 -sd=y dynamic.dft plasma.dat 30 999 output.res > stdout.txt 2> stderr.txt
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
echo passed.
iftisval stdout.txt save_stat "1"
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
echo passed. ; echo ""

echo "option -sd"
if [ -f stdout.txt ] ; then rm stdout.txt ; fi
if [ -f stderr.txt ] ; then rm stderr.txt ; fi
$PROGRAM --debug=3 -sd dynamic.dft plasma.dat 30 999 output.res > stdout.txt 2> stderr.txt
if [ $? -eq 0 ] ; then echo Failed! ; exit 1 ; fi
echo passed.
iftisval stderr.txt Error "invalid option '-sd'."
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
echo passed. ; echo ""

echo "without option -sd"
if [ -f stdout.txt ] ; then rm stdout.txt ; fi
if [ -f stderr.txt ] ; then rm stderr.txt ; fi
$PROGRAM --debug=3 dynamic.dft plasma.dat 30 999 output.res > stdout.txt 2> stderr.txt
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
echo passed.
iftisval stdout.txt save_stat "1"
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
echo passed. ; echo ""
echo ===============================================================================
echo "1.7.2: option -sd=n"
if [ -f stdout.txt ] ; then rm stdout.txt ; fi
if [ -f stderr.txt ] ; then rm stderr.txt ; fi
$PROGRAM --debug=3 -sd=n dynamic.dft plasma.dat 30 999 output.res > stdout.txt 2> stderr.txt
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
echo passed.
iftisval stdout.txt save_stat "0"
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
echo passed. ; echo ""

echo ===============================================================================
echo ""
echo ===============================================================================
echo "1.8.1: Option -svg"

if [ -f stdout.txt ] ; then rm stdout.txt ; fi
if [ -f stderr.txt ] ; then rm stderr.txt ; fi
$PROGRAM --debug=3 -svg dynamic.dft plasma.dat 30 999 output.res > stdout.txt 2> stderr.txt
if [ $? -eq 0 ] ; then echo Failed! ; exit 1 ; fi
echo passed.
iftisval stderr.txt Error "invalid option '-svg'."
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
echo passed. ; echo ""

echo "-svg=filename"
if [ -f stdout.txt ] ; then rm stdout.txt ; fi
if [ -f plot.svg ] ; then rm plot.svg ; fi
$PROGRAM --debug=3 -svg=plot.svg dynamic.dft plasma.dat 30 999 output.res > stdout.txt
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
echo passed.
iftisval stdout.txt sfile "plot.svg"
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
echo passed. ; echo ""
if [ -f plot.svg ] ; then echo passed. ; echo "" ;
else echo Failed! ; exit 1 ; fi
echo ===============================================================================
echo ""
echo "1.8.2: Option -svg and -bw"
echo ""
$PROGRAM -bw -svg=plot_black-and-white.svg dynamic.dft plasma.dat 30 999 output.res
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
echo passed. ; echo ""
if [ -f plot_black-and-white.svg ] ; then echo passed. ; echo "" ;
else echo Failed! ; exit 1 ; fi

echo ===============================================================================
echo ""
echo ===============================================================================
echo "1.9.1: Option -plotdata"

if [ -f stdout.txt ] ; then rm stdout.txt ; fi
if [ -f stderr.txt ] ; then rm stderr.txt ; fi
$PROGRAM --debug=3 -plotdata dynamic.dft plasma.dat 30 999 output.res > stdout.txt 2> stderr.txt
if [ $? -eq 0 ] ; then echo Failed! ; exit 1 ; fi
echo passed.
iftisval stderr.txt Error "invalid option '-plotdata'."
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
echo passed. ; echo ""

echo "-plotdata=filename"
if [ -f stdout.txt ] ; then rm stdout.txt ; fi
if [ -f plot.htm ] ; then rm plot.htm ; fi
$PROGRAM --debug=3 -plotdata=plot.htm dynamic.dft plasma.dat 30 999 output.res > stdout.txt
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
echo passed.
iftisval stdout.txt pfile "plot.htm"
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
echo passed. ; echo ""
if [ -f plot.htm ] ; then echo passed. ; echo "" ;
else echo Failed! ; exit 1 ; fi
echo ===============================================================================
echo ""
echo ===============================================================================
echo "1.10.1: option -mid=n"
if [ -f stdout.txt ] ; then rm stdout.txt ; fi
if [ -f stderr.txt ] ; then rm stderr.txt ; fi
$PROGRAM --debug=3 -mid=n dynamic.dft plasma.dat 30 999 output.res > stdout.txt 2> stderr.txt
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
echo passed.
iftisval stdout.txt always_mid "0"
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
echo passed. ; echo ""

echo "without option -mid"
if [ -f stdout.txt ] ; then rm stdout.txt ; fi
if [ -f stderr.txt ] ; then rm stderr.txt ; fi
$PROGRAM --debug=3 dynamic.dft plasma.dat 30 999 output.res > stdout.txt 2> stderr.txt
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
echo passed.
iftisval stdout.txt always_mid "0"
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
echo passed. ; echo ""
echo ===============================================================================
echo "1.10.2: option -mid=y"
if [ -f stdout.txt ] ; then rm stdout.txt ; fi
if [ -f stderr.txt ] ; then rm stderr.txt ; fi
$PROGRAM --debug=3 -mid=y dynamic.dft plasma.dat 30 999 output.res > stdout.txt 2> stderr.txt
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
echo passed.
iftisval stdout.txt always_mid "1"
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
echo passed. ; echo ""

echo "option -mid"
if [ -f stdout.txt ] ; then rm stdout.txt ; fi
if [ -f stderr.txt ] ; then rm stderr.txt ; fi
$PROGRAM --debug=3 -mid dynamic.dft plasma.dat 30 999 output.res > stdout.txt 2> stderr.txt
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
echo passed.
iftisval stdout.txt always_mid "1"
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
echo passed. ; echo ""
echo ===============================================================================
echo ""
echo ===============================================================================
echo "1.11.1: Data formats"
echo ""
echo "       Plasma data with some header information"
echo "       Tissue data with frame start and end times"
if [ -f stdout.txt ] ; then rm stdout.txt ; fi
if [ -f output.res ] ; then rm output.res ; fi
$PROGRAM --silent -sd=n dynamic_long.dft plasma_extrapolated.dat 120 240 output.res > stdout.txt
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
echo passed.
resmatch -res=1 -abs=0.001 correct_ki.res output.res
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
echo passed. ; echo ""
echo ""
echo "       Plasma data without any header information"
if [ -f stdout.txt ] ; then rm stdout.txt ; fi
if [ -f output.res ] ; then rm output.res ; fi
if [ -f temp.dat ] ; then rm temp.dat ; fi
tacformat -hdr=no -f=simple plasma.dat temp.dat
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
$PROGRAM --silent -sd=no dynamic.dft temp.dat 30 90 output.res > stdout.txt
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
echo passed.
resmatch -res=1 -abs=0.01 correct_ki.res output.res
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
echo passed. ; echo ""
echo ""

echo "       Tissue data with middle frame times"
if [ -f stdout.txt ] ; then rm stdout.txt ; fi
if [ -f output.res ] ; then rm output.res ; fi
if [ -f temp.dat ] ; then rm temp.dat ; fi
tacformat -hdr=no -f=pmod dynamic.dft temp.dat
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
$PROGRAM --silent -sd=no temp.dat plasma.dat 30 90 output.res > stdout.txt
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
echo passed.
resmatch -res=1 -abs=0.01 correct_ki.res output.res
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
echo passed. ; echo ""

echo "       Data in PMOD format"
if [ -f stdout.txt ] ; then rm stdout.txt ; fi
if [ -f output.res ] ; then rm output.res ; fi
if [ -f temp.dat ] ; then rm temp.dat ; fi
$PROGRAM --silent -sd=no dynamic.tac plasma.bld 30 90 output_pmod.res > stdout.txt
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
echo passed.
resmatch -res=1 -abs=0.01 correct_ki.res output_pmod.res
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
echo passed.
$PROGRAM --silent -sd=no dynamic.dft plasma.dat 30 90 output_dft.res > stdout.txt
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
echo passed.
resmatch -abs=0.001 -header=n output_dft.res output_pmod.res
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
echo passed. ; echo ""

echo ===============================================================================
echo "1.11.2: Input is region name"
if [ -f stdout.txt ] ; then rm stdout.txt ; fi
if [ -f output.res ] ; then rm output.res ; fi
$PROGRAM --silent -sd=n dynamic_long.dft revers 180 240 output.res > stdout.txt
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
echo passed.
resmatch -res=1 -abs=0.002 -regions=y -param=y correct_kiref.res output.res
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
echo passed. ; echo ""
echo ""
echo "nonexisting region"
if [ -f stderr.txt ] ; then rm stderr.txt ; fi
$PROGRAM --silent -sd=n dynamic.dft noroi 45 90 output.res 2> stderr.txt
if [ $? -eq 0 ] ; then echo Failed! ; exit 1 ; fi
echo passed.
iftisval stderr.txt "Error in reading 'noroi'" "cannot find region"
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
echo passed. ; echo ""
echo ===============================================================================
echo "1.11.3: Input is region number"
if [ -f stdout.txt ] ; then rm stdout.txt ; fi
if [ -f output.res ] ; then rm output.res ; fi
$PROGRAM --silent -sd=n dynamic_long.dft 12 180 240 output.res > stdout.txt
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
echo passed.
resmatch -res=1 -abs=0.01 -regions=y -param=y correct_kiref_nr.res output.res
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
echo passed. ; echo ""
echo ""
echo "nonexisting region"
if [ -f stderr.txt ] ; then rm stderr.txt ; fi
$PROGRAM --silent -sd=n dynamic.dft 108 45 90 output.res 2> stderr.txt
if [ $? -eq 0 ] ; then echo Failed! ; exit 1 ; fi
echo passed.
iftisval stderr.txt "Error in reading '108'" "cannot find region"
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
echo passed. ; echo ""
echo ===============================================================================
echo "General validity of results"
echo ===============================================================================
echo "2.1.1: Valid data and fit range"
if [ -f stdout.txt ] ; then rm stdout.txt ; fi
if [ -f output.res ] ; then rm output.res ; fi
$PROGRAM --silent -sd=n dynamic_long.dft plasma_extrapolated.dat 120 240 output.res > stdout.txt
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
echo passed.
resmatch -res=1 -abs=0.001 correct_ki.res output.res
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
echo passed. ; echo ""
echo ""
iftisval output.res "Data file" dynamic_long.dft
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
echo passed.
iftisval output.res "Plasma file" plasma_extrapolated.dat
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
echo passed.
iftisval output.res "Data nr" 6
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
echo passed.
iftisval output.res "Data range" "120 - 240 min"
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
echo passed.
echo ""
echo ===============================================================================
echo "2.1.2: Same as before but MR is calculated"
if [ -f output.res ] ; then rm output.res ; fi
if [ -f output2.res ] ; then rm output2.res ; fi
$PROGRAM --silent -sd=n -Ca=6.3 -LC=0.8 -density=1.05 dynamic.dft plasma.dat 30 60 output.res
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
echo passed.

# Same MR values are achieved by multilying data before calculation 
if [ -f temp.dft ] ; then rm temp.dft.res ; fi
taccalc --silent dynamic.dft x 750 temp.dft
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
$PROGRAM --silent -sd=n temp.dft plasma.dat 30 60 output2.res
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
resmatch -res=1 -abs=0.1 output.res output2.res
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
echo passed. ; echo ""

# Check also headers
iftisval output.res "Data file" dynamic.dft
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
echo passed.
iftisval output.res "Plasma file" plasma.dat
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
echo passed.
iftisval output.res "Data nr" 6
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
echo passed.
iftisval output.res "Data range" "30 - 60 min"
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
echo passed.
iftisval output.res "Tissue density" "1.05"
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
echo passed.
iftisval output.res "Lumped constant" "0.8"
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
echo passed.
iftisval output.res "Concentration" "6.3"
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
echo passed. ; echo ""

# Check parameter names and units
resmatch -param=y -res=0 output.res correct_mr.res
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
echo passed. ; echo ""

echo ===============================================================================
echo ""
echo ===============================================================================
echo "2.2.1: Program can internally convert between kBq/ml and Bq/cc"
if [ -f output.res ] ; then rm output.res ; fi
if [ -f output2.res ] ; then rm output2.res ; fi

$PROGRAM --silent -sd=n dynamic.dft plasma.dat 30 60 output.res > stdout.txt
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
echo passed.

echo ""
echo "       PET data in Bq/cc, plasma in kBq/ml"
if [ -f temp.dft ] ; then rm temp.dft ; fi
cp -f dynamic.dft temp.dft
tacunit -yconv=Bq/cc temp.dft
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
$PROGRAM --silent -sd=n temp.dft plasma.dat 30 60 output2.res > stdout.txt
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
echo passed.
resmatch -roughly output.res output2.res
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
echo passed. ; echo ""

echo ""
echo "       PET data in kBq/ml, plasma in Bq/ml"
if [ -f temp.dat ] ; then rm temp.dat ; fi
cp -f plasma.dat temp.dat
tacunit -yconv=Bq/ml temp.dat
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
if [ -f output2.res ] ; then rm output2.res ; fi
$PROGRAM --silent -sd=n dynamic.dft temp.dat 30 60 output2.res > stdout.txt
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
echo passed.
resmatch -roughly output.res output2.res
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
echo passed. ; echo ""
echo ===============================================================================
echo ""
echo ===============================================================================
echo "2.2.2: Program can internally convert plasma sample times from sec to min"
if [ -f temp.dat ] ; then rm temp.dat ; fi
cp -f plasma.dat temp.dat
tacunit -xconv=sec temp.dat
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
if [ -f output2.res ] ; then rm output2.res ; fi
$PROGRAM --silent -sd=n dynamic.dft temp.dat 30 60 output2.res > stdout.txt
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
echo passed.
resmatch -roughly output.res output2.res
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
echo passed. ; echo ""
echo ===============================================================================
echo "2.2.3: Program can internally convert tissue sample times from sec to min"
if [ -f temp.dft ] ; then rm temp.dft ; fi
cp -f dynamic.dft temp.dft
tacunit -xconv=sec temp.dft
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
if [ -f output2.res ] ; then rm output2.res ; fi
$PROGRAM --silent -sd=n temp.dft plasma.dat 30 60 output2.res > stdout.txt
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
echo passed.
resmatch -roughly output.res output2.res
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
echo passed. ; echo ""
echo ===============================================================================
echo ""
echo ===============================================================================
echo "2.3.1: Data that contains frame lengths produces with option -mid=y"
echo "       the same results as if only frame mid times were present,"
echo "       and the same results as an early Patlak version"
if [ -f output.res ] ; then rm output.res ; fi
if [ -f output2.res ] ; then rm output2.res ; fi

$PROGRAM --silent -sd=n s06883dy1_mrp_mid.dft s06883vp.kbq 3 20 output.res
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
echo passed.
resmatch -res=1-2 -roughly output.res s06883_oldpatlak.res
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
echo passed. ; echo ""

$PROGRAM --silent -sd=n -mid=y s06883dy1_mrp.dft s06883vp.kbq 3 20 output2.res
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
echo passed.
resmatch -res=1-2 -roughly output.res output2.res
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
echo passed. ; echo ""
resmatch -res=1-2 -roughly output.res s06883_oldpatlak.res
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
echo passed. ; echo ""
echo ===============================================================================
echo ""
echo ===============================================================================
echo "3.1.1: Ki from time range that exists in the data"
echo "       Tested already in 2.1.1."
echo ""
echo ===============================================================================
echo "3.1.2: User gives a large value as the stop time"
if [ -f output.res ] ; then rm output.res ; fi
if [ -f output2.res ] ; then rm output2.res ; fi
echo previous results removed.

$PROGRAM --silent -sd=n dynamic.dft plasma.dat 45 120 output.res > stdout.txt
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
echo passed.
$PROGRAM --silent -sd=n dynamic.dft plasma.dat 45 999 output2.res > stdout.txt
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
echo passed.
resmatch output.res output2.res
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
echo passed. ; echo ""
iftisval output2.res "Data range" "45 - 120 min"
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
echo passed. ; echo ""
echo ===============================================================================
echo "3.1.3: User gives too short time range"
if [ -f stderr.txt ] ; then rm stderr.txt ; fi
$PROGRAM --silent -sd=n dynamic.dft plasma.dat 70 75 output.res 2> stderr.txt
if [ $? -eq 0 ] ; then echo Failed! ; exit 1 ; fi
echo passed.
iftisval stderr.txt "Error" "cannot make plot from less than 2 points."
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
echo passed. ; echo ""
echo ""
echo ===============================================================================
echo ""
echo ===============================================================================
echo "3.2.1: Data does not extend to the calculation range"
echo ""
echo "       PET data stops too early"
if [ -f stderr.txt ] ; then rm stderr.txt ; fi
if [ -f short.dft ] ; then rm short.dft ; fi
taccut dynamic.dft 0 30 short.dft
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
if [ -f output.res ] ; then rm output.res ; fi
$PROGRAM --silent -sd=n short.dft plasma.dat 45 120 output.res 2> stderr.txt
if [ $? -eq 0 ] ; then echo Failed! ; exit 1 ; fi
echo passed.
iftisval stderr.txt "Error" "data does not contain the specified time range."
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
echo passed. ; echo ""
echo ""

echo "       Plasma data stops too early"
if [ -f stderr.txt ] ; then rm stderr.txt ; fi
if [ -f short.dat ] ; then rm short.dat ; fi
taccut plasma.dat 0 15 short.dat
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
if [ -f output.res ] ; then rm output.res ; fi
$PROGRAM --silent -sd=n dynamic.dft short.dat 45 120 output.res 2> stderr.txt
if [ $? -eq 0 ] ; then echo Failed! ; exit 1 ; fi
iftisval stderr.txt "Error in reading 'short.dat'" "too short data for interpolation"
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
echo passed. ; echo ""
echo ""

echo "       Plasma AND PET data stop too early"
if [ -f stderr.txt ] ; then rm stderr.txt ; fi
if [ -f output.res ] ; then rm output.res ; fi
$PROGRAM --silent -sd=n short.dat short.dft 45 120 output.res 2> stderr.txt
if [ $? -eq 0 ] ; then echo Failed! ; exit 1 ; fi
echo passed. ; echo ""
echo ===============================================================================
echo "3.2.2: Data does not extend to the calculation range"
if [ -f short.dft ] ; then rm short.dft ; fi
taccut dynamic.dft 90 120 short.dft
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
if [ -f short.dat ] ; then rm short.dat ; fi
taccut plasma.dat 90 120 short.dat
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi

echo ""
echo "       PET data starts too late"
if [ -f stderr.txt ] ; then rm stderr.txt ; fi
if [ -f output.res ] ; then rm output.res ; fi
$PROGRAM --silent -sd=n short.dft plasma.dat 25 80 output.res 2> stderr.txt
if [ $? -eq 0 ] ; then echo Failed! ; exit 1 ; fi
echo passed.
iftisval stderr.txt "Error" "data does not contain the specified time range."
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
echo passed. ; echo ""
echo ""
$PROGRAM --silent -sd=n short.dft plasma.dat 25 100 output.res 2> stderr.txt
if [ $? -eq 0 ] ; then echo Failed! ; exit 1 ; fi
echo passed.
iftisval stderr.txt "Error" "cannot make plot from less than 2 points."
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
echo passed. ; echo ""
echo ""

echo "       Plasma data starts too late"
if [ -f stderr.txt ] ; then rm stderr.txt ; fi
if [ -f output.res ] ; then rm output.res ; fi
$PROGRAM --silent -sd=n dynamic.dft short.dat 25 80 output.res 2> stderr.txt
if [ $? -eq 0 ] ; then echo Failed! ; exit 1 ; fi
iftisval stderr.txt "Error in reading 'short.dat'" "input TAC should start at time zero"
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
echo passed. ; echo ""
echo ""

echo ===============================================================================
echo ""
echo ===============================================================================
echo "3.3.1: Only few data points inside fit range"
echo ""
echo "       PET data stops too early"
if [ -f stderr.txt ] ; then rm stderr.txt ; fi
if [ -f short.dft ] ; then rm short.dft ; fi
taccut dynamic.dft 0 30 short.dft
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
if [ -f output.res ] ; then rm output.res ; fi
$PROGRAM --silent -sd=n short.dft plasma.dat 25 90 output.res 2> stderr.txt
if [ $? -eq 0 ] ; then echo Failed! ; exit 1 ; fi
echo passed.
iftisval stderr.txt "Error" "cannot make plot from less than 2 points."
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
echo passed. ; echo ""
echo ""
$PROGRAM --silent -sd=n short.dft plasma.dat 20 90 output.res 2> stderr.txt
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
echo passed.
iftisval stderr.txt "Warning" "only two samples in the time range."
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
echo passed. ; echo ""
echo ""

echo ""
echo "       PET data starts too late"
if [ -f stderr.txt ] ; then rm stderr.txt ; fi
if [ -f short.dft ] ; then rm short.dft ; fi
taccut dynamic.dft 90 120 short.dft
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
if [ -f output.res ] ; then rm output.res ; fi
$PROGRAM --silent -sd=n short.dft plasma.dat 25 90 output.res 2> stderr.txt
if [ $? -eq 0 ] ; then echo Failed! ; exit 1 ; fi
echo passed.
iftisval stderr.txt "Error" "data does not contain the specified time range."
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
echo passed. ; echo ""
echo ""
$PROGRAM --silent -sd=n short.dft plasma.dat 25 100 output.res 2> stderr.txt
if [ $? -eq 0 ] ; then echo Failed! ; exit 1 ; fi
echo passed.
iftisval stderr.txt "Error" "cannot make plot from less than 2 points."
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
echo passed. ; echo ""
echo ""
$PROGRAM --silent -sd=n short.dft plasma.dat 25 110 output.res 2> stderr.txt
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
echo passed.
iftisval stderr.txt "Warning" "only two samples in the time range."
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
echo passed. ; echo ""
echo ""
echo ===============================================================================
echo ""
echo ===============================================================================
echo "4.1.1: Ki from static data, user lets program find the times"
if [ -f stderr.txt ] ; then rm stderr.txt ; fi
if [ -f stdout.txt ] ; then rm stdout.txt ; fi
if [ -f static.dft ] ; then rm static.dft ; fi
dftinteg -avg dynamic.dft 100 20 static.dft
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
tacstudy static.dft sta1
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi

if [ -f output.res ] ; then rm output.res ; fi
$PROGRAM -sd=n static.dft plasma.dat 0 999 output.res > stdout.txt 2> stderr.txt
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
echo passed.
iftisval stdout.txt "Suggestion" "for FUR calculation use regfur."
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
echo passed. ; echo ""
resmatch -res=1 -abs=0.05 output.res correct_ki.res
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
echo passed. ; echo ""
echo ===============================================================================
echo "4.1.2: Ki from static data, user sets times to frame middle time"
if [ -f stderr.txt ] ; then rm stderr.txt ; fi
if [ -f stdout.txt ] ; then rm stdout.txt ; fi
if [ -f output.res ] ; then rm output.res ; fi
$PROGRAM -sd=n -ic=0 static.dft plasma.dat 110 110 output.res > stdout.txt 2> stderr.txt
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
echo passed.
resmatch -res=1 -abs=0.05 output.res correct_ki.res
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
echo passed. ; echo ""

$PROGRAM -sd=n static.dft plasma.dat 110 110 output.res > stdout.txt 2> stderr.txt
if [ $? -eq 0 ] ; then echo Failed! ; exit 1 ; fi
echo passed.
iftisval stderr.txt "Error" "invalid time range 110-110."
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
echo passed. ; echo ""
echo ===============================================================================
echo ""
echo ===============================================================================
echo "5.1.1: FUR from static data"
if [ -f static.dft ] ; then rm static.dft ; fi
dftinteg -avg dynamic.dft 100 20 static.dft
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
tacstudy static.dft sta1
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi

if [ -f output.res ] ; then rm output.res ; fi
regfur --silent plasma.dat static.dft 100 120 output.res
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
echo passed.

if [ -f output2.res ] ; then rm output2.res ; fi
$PROGRAM -sd=n -ic=0 static.dft plasma.dat 0 999 output2.res
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
echo passed.

resmatch -abs=0.001 -res=1 output.res output2.res
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
echo passed. ; echo ""
echo ===============================================================================
echo "5.1.2: FUR from dynamic data"

if [ -f output.res ] ; then rm output.res ; fi
regfur --silent plasma.dat dynamic.dft 30 60 output.res
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
echo passed.

if [ -f output2.res ] ; then rm output2.res ; fi
$PROGRAM -sd=n -ic=0 dynamic.dft plasma.dat 30 60 output2.res
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
echo passed.

resmatch -abs=0.010 -res=1 output.res output2.res
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
echo passed. ; echo ""
echo ===============================================================================
echo ""
echo ===============================================================================
echo "5.2.1: Intercept fixed with static data"
if [ -f static.dft ] ; then rm static.dft ; fi
dftinteg -avg dynamic.dft 100 20 static.dft
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
tacstudy static.dft sta1
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi

if [ -f k.dat ] ; then rm k.dat ; fi
if [ -f k.res ] ; then rm k.res ; fi
IC=0.1
interpol -f=static.dft plasma.dat plasmai.dat
interpol -i -f=static.dft plasma.dat plasmaii.dat
taccalc static.dft : plasmai.dat y.dat
taccalc plasmaii.dat : plasmai.dat xtac.dft
dftinteg -avg xtac.dft 100 20 x.dat
taccalc y.dat - $IC y2.dat
taccalc y2.dat : x.dat k.dat
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
dft2res k.dat
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi

if [ -f output.res ] ; then rm output.res ; fi
$PROGRAM -sd=n -ic=$IC static.dft plasma.dat 100 120 output.res
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
echo passed.
resmatch -abs=0.001 -res=1 output.res k.res
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
echo passed. ; echo ""

echo ===============================================================================
echo "5.2.2: Intercept fixed with dynamic data"

if [ -f k.dat ] ; then rm k.dat ; fi
if [ -f k.res ] ; then rm k.res ; fi
IC=0.1
interpol -f=dynamic.dft plasma.dat plasmai.dat
interpol -i -f=dynamic.dft plasma.dat plasmaii.dat
taccalc dynamic.dft : plasmai.dat ytac.dft
taccalc plasmaii.dat : plasmai.dat xtac.dft
dftinteg -avg ytac.dft 30 30 y.dat
dftinteg -avg xtac.dft 30 30 x.dat
taccalc y.dat - $IC y2.dat
taccalc y2.dat : x.dat k.dat
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
dft2res k.dat
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi

if [ -f output.res ] ; then rm output.res ; fi
$PROGRAM -sd=n -ic=$IC dynamic.dft plasma.dat 30 60 output.res
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
echo passed.
resmatch -abs=0.01 -res=1 output.res k.res
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
echo passed. ; echo ""
echo ===============================================================================
echo ""
echo ===============================================================================
echo "6.1.1: Reference tissue input; reference tissue in the same file"

if [ -f output.res ] ; then rm output.res ; fi
$PROGRAM -sd=n dynamic_long.dft revers 120 240 output.res
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
echo passed.
resmatch -abs=0.0015 -res=1 output.res correct_kiref.res
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
echo passed. ; echo ""
resmatch -abs=3 -res=2 output.res correct_kiref.res
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
echo passed. ; echo ""
echo ===============================================================================
echo "6.1.2: Reference tissue input; reference tissue in separate file"
if [ -f output.res ] ; then rm output.res ; fi
if [ -f output2.res ] ; then rm output2.res ; fi

$PROGRAM -sd=n dynamic_long.dft revers 120 240 output.res
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
echo passed.

if [ -f temp.dft ] ; then rm temp.dft ; fi
if [ -f temp2.dft ] ; then rm temp2.dft ; fi
tacadd -ovr --silent temp2.dft dynamic_long.dft "revers avg Pl002"
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
cp dynamic_long.dft temp.dft
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
tacdel --silent temp.dft revers
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi

$PROGRAM -sd=n temp.dft temp2.dft 120 240 output2.res
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
echo passed.
resmatch -abs=0.0015 -res=1,2 output.res output2.res
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
echo passed. ; echo ""
resmatch -abs=1 -res=2 output.res output2.res
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
echo passed. ; echo ""
echo ===============================================================================
echo ""
echo ===============================================================================
echo "6.2.1: Reference tissue input not applicable to late-scan studies"
if [ -f static.dft ] ; then rm static.dft ; fi
dftinteg -avg dynamic.dft 100 20 static.dft
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
tacstudy static.dft sta1
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi

if [ -f stderr.txt ] ; then rm stderr.txt ; fi
$PROGRAM -sd=n static.dft "revers avg" 100 120 output.res 2> stderr.txt
if [ $? -eq 0 ] ; then echo Failed! ; exit 1 ; fi
echo passed.
iftisval stderr.txt "Error" "too few samples."
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
echo passed. ; echo ""
echo ===============================================================================
echo "6.2.2: Small delay in scan start leads to warning with ref input"
if [ -f temp.dft ] ; then rm temp.dft ; fi
# taccut dynamic.dft 1.0 999 temp.dft
tactime dynamic.dft +1 temp.dft
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
tacstudy temp.dft sho1
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi

if [ -f stderr.txt ] ; then rm stderr.txt ; fi
$PROGRAM -sd=n temp.dft "revers avg" 100 120 output.res 2> stderr.txt
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
echo passed.
iftisval stderr.txt "Warning" "input TAC should start at time zero."
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
echo passed. ; echo ""

echo ===============================================================================
echo ""
echo ===============================================================================
echo "6.3.1: Reference tissue input; reference tissue in the same file"

if [ -f output.res ] ; then rm output.res ; fi
$PROGRAM -sd=n dynamic_long.dft revers 120 240 output.res
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
echo passed.

echo "       First frame can be very long"
dftinteg -avg dynamic_long.dft 0 60 temp.dft
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
taccut dynamic_long.dft 60 999 temp2.dft
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
dftcat temp.dft temp2.dft
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi

if [ -f output2.res ] ; then rm output2.res ; fi
$PROGRAM -sd=n temp.dft revers 120 240 output2.res
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
echo passed.
resmatch -abs=0.0015 -res=1 output.res output2.res
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
echo passed. ; echo ""
echo ===============================================================================
echo ""
echo ===============================================================================
echo ""
#exit 0
echo ""
rm -f stdout.* stderr.*
echo ===============================================================================
echo ""
echo "All passed!"
echo ""
echo ===============================================================================
exit 0

