/** @file nii_lhdr.c
 *  @brief List the information in NIfTI-1 header.
 *  @copyright (c) Turku PET Centre
 *  @author Vesa Oikonen
 */
/// @cond
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <string.h>
#include <time.h>
/*****************************************************************************/
#include "libtpcmisc.h"
#include "libtpcimgio.h"
/*****************************************************************************/
#include "niftiift.h"
/*****************************************************************************/

/*****************************************************************************/
static char *info[] = {
  "Lists the information in NIfTI-1 image header.",
  "Optionally, only the specified header field is printed.",
  " ",
  "Usage: @P niftifile [fieldname]",
  " ",
  "Options:",
  " -stdoptions", // List standard options like --help, -v, etc
  " ",
  "See also: nii_ehdr, ana_lhdr, iftlist, iftadd, iftdel, iftmatch",
  " ",
  "Keywords: image, NIfTI, header, IFT",
  0};
/*****************************************************************************/

/*****************************************************************************/
/* Turn on the globbing of the command line, since it is disabled by default in
   mingw-w64 (_dowildcard=0); in MinGW32 define _CRT_glob instead, if necessary;
   In Unix&Linux wildcard command line processing is enabled by default. */
/*
#undef _CRT_glob
#define _CRT_glob -1
*/
int _dowildcard = -1;
/*****************************************************************************/

/*****************************************************************************/
/**
 *  Main
 */
int main(int argc, char **argv)
{
  int       ai, help=0, version=0, verbose=1;
  int       i, ret;
  char      hdrfile[FILENAME_MAX], tmp[256], keyname[256];
  NIFTI_DSR dsr;
  IFT       ift;


  /*
   *  Get arguments
   */
  if(argc==1) {tpcPrintUsage(argv[0], info, stderr); return(1);}
  hdrfile[0]=keyname[0]=(char)0;
  iftInit(&ift);
  /* Options */
  for(ai=1; ai<argc; ai++) if(*argv[ai]=='-') { /* options */
    if(tpcProcessStdOptions(argv[ai], &help, &version, &verbose)==0) continue;
    fprintf(stderr, "Error: invalid option '%s'.\n", argv[ai]);
    return(1);
  } else break;
  
  /* Print help or version? */
  if(help==2) {tpcHtmlUsage(argv[0], info, ""); return(0);}
  if(help) {tpcPrintUsage(argv[0], info, stdout); return(0);}
  if(version) {tpcPrintBuild(argv[0], stdout); return(0);}

  /* Process other arguments, starting from the first non-option */
  for(; ai<argc; ai++) {
    if(!hdrfile[0]) {strlcpy(hdrfile, argv[ai], FILENAME_MAX); continue;}
    else if(!keyname[0]) {strlcpy(keyname, argv[ai], 256); continue;}
    fprintf(stderr, "Error: invalid argument '%s'.\n", argv[ai]);
    return(1);
  }

  /* Is something missing? */
  if(!hdrfile[0]) {
    fprintf(stderr, "Error: missing command-line argument.\n"); 
    return(1);
  }


  /* In verbose mode print arguments and options */
  if(verbose>1) {
    printf("hdrfile := %s\n", hdrfile);
    if(keyname[0]) printf("keyname := %s\n", keyname);
  }


  /*
   *  Read header file
   */
  if(verbose>1) printf("reading %s\n", hdrfile);
  ret=niftiReadHeader(hdrfile, &dsr, verbose-2, tmp);
  if(ret) {
    fprintf(stderr, "Error: cannot read %s: %s.\n", hdrfile, tmp);
    return(2);
  }

  /*
   *  Print header
   */
  /* Copy header contents to IFT struct */
  ret=niftiHeaderToIFT(&dsr, &ift, verbose-4);
  if(ret) {
    fprintf(stderr, "Error: invalid header structure (%d).\n", ret);
    return(5);
  }
  /* If key name was given, then print only that field, otherwise all */
  if(keyname[0]) {
    i=iftGet(&ift, keyname, 0);
    if(i<0) {
      fprintf(stderr, "Error: '%s' not found in %s\n", keyname, hdrfile); 
      iftEmpty(&ift);
      return(3);
    }
    iftWriteItem(&ift, i, stdout, 0);
  } else {
    iftWrite(&ift, "stdout", 0);
  }
  iftEmpty(&ift);

  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/// @endcond
