/** @file rescoll.c
 *  @brief Collects results from separate PET studies.
 *  @copyright (c) Turku PET Centre
 *  @author Vesa Oikonen
 *  @todo Add option to save as CSV table.
 */
/// @cond
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <math.h>
#include <time.h>
/*****************************************************************************/
#include "libtpcmisc.h"
#include "libtpccurveio.h"
/*****************************************************************************/

/*****************************************************************************/
/* Local functions */
void print_usage();
void print_build();
ResVOI *resGetNextResVOI(
    RES *reslist, int resNr, ResVOI *lastResVOI, int *resIndex);
int rescoll_tabulate(FILE *fp, RES *coll, int collNr);
/*****************************************************************************/

/*****************************************************************************/
static char *info[] = {
  "Collects the model analysis results from a set of PET studies, and",
  "by default tabulates regional results and means with standard deviations;",
  "then results of only one model analysis program can be used at a time.",
  " ",
  "Usage: @P [Options] collection_file result_files",
  " ",
  "Options:",
  " -sort",
  "     Results will be sorted by study number.",
  " -strict",
  "     Quits with error when normally a warning would be displayed.",
  " -list",
  "     Regional results are not collected but result files are just",
  "     tabulated into one HTML file.",
  " -par[ameter]",
  "     Regional and individual results are collected for each parameter.",
  " -stdoptions", // List standard options like --help, -v, etc
  " ",
  "Example:",
  "     rescoll patlak_means.html ua4???ki.res",
  " ",
  "The collected data is saved in tables in XHTML format, therefore the",
  "collection filename extension has to be .htm(l).",
  "File can then be viewed and printed with a web browser or Excel.",
  "In Excel the data can be also be processed further and saved in Excel",
  "format.",
  " ",
  "See also: reslist, pardiff, dft2res, fit2res, resmatch, resdel, paradd",
  " ",
  "Keywords: results, tabulation, reporting, modelling, simulation, tool",
  0};
/*****************************************************************************/

/*****************************************************************************/
/* Turn on the globbing of the command line, since it is disabled by default in
   mingw-w64 (_dowildcard=0); in MinGW32 define _CRT_glob instead, if necessary;
   In Unix&Linux wildcard command line processing is enabled by default. */
/*
#undef _CRT_glob
#define _CRT_glob -1
*/
int _dowildcard = -1;
/*****************************************************************************/

/*****************************************************************************/
/**
 *  Main
 */
int main(int argc, char **argv)
{
  int    ai, help=0, version=0, verbose=1;
  int    ri, rj, vi, pi, ret, len;
  int    resNr=0;
  int    doSort=0, beStrict=0;
  int    doMode=0; /* 0=regional results collected, 1=studies simply listed,
                      2=parametric results collected */
  int    first_res=-1;
  int    saveSD=0; /* 0=SD and CLs are not included; 1=included, if available */
  RES   *res, coll, tres;
  char   colfile[FILENAME_MAX], resfile[FILENAME_MAX], *cptr, tmp[1024];
  struct tm *st;
  FILE   *fp;
  ResVOI *resvoi1, *resvoi2;


  /*
   *  Get arguments
   */
  if(argc==1) {tpcPrintUsage(argv[0], info, stderr); return(1);}
  colfile[0]=resfile[0]=(char)0;
  resInit(&coll); resInit(&tres);
  /* Options */
  for(ai=1; ai<argc; ai++) if(*argv[ai]=='-') {
    if(tpcProcessStdOptions(argv[ai], &help, &version, &verbose)==0) continue;
    cptr=argv[ai]+1; if(*cptr=='-') cptr++; if(cptr==NULL) continue;
    if(strcasecmp(cptr, "STRICT")==0) {
      beStrict=1; continue;
    } else if(strncasecmp(cptr, "SORT", 2)==0) {
      doSort=1; continue;
    } else if(strncasecmp(cptr, "LIST", 2)==0) {
      doMode=1; continue;
    } else if(strncasecmp(cptr, "PARAMETER", 3)==0) {
      doMode=2; continue;
    }
    fprintf(stderr, "Error: invalid option '%s'.\n", argv[ai]); 
    return(1);
  } else break;
  
  /* Print help or version? */
  if(help==2) {tpcHtmlUsage(argv[0], info, ""); return(0);}
  if(help) {tpcPrintUsage(argv[0], info, stdout); return(0);}
  if(version) {tpcPrintBuild(argv[0], stdout); return(0);}

  /* Process other arguments, starting from the first non-option */
  for(; ai<argc; ai++) {
    if(!colfile[0]) {
      strlcpy(colfile, argv[ai], FILENAME_MAX);
      /* Check the output file extension */
      cptr=strrchr(colfile, '.');
      if(strcasecmp(cptr, ".HTM")!=0 && strcasecmp(cptr, ".HTML")!=0) {
        fprintf(stderr, "Error: collection file extension must be .htm or .html\n");
        return(1);
      }
      continue;
    }
    /* this must be a input result file. Check that it exists */
    if(access(argv[ai], 0) == -1) {
      fprintf(stderr, "Error: result file %s does not exist.\n", argv[ai]);
      return(1);
    }
    if(first_res<0) first_res=ai;
    resNr++; /* calculate how many result files there will be */
  }

  /* Is something missing? */
  if(resNr<1) {
    fprintf(stderr, "Error: no result files to process.\n");
    return(1);
  } else if(resNr<2 && doMode!=1) {
    fprintf(stderr, "Error: at least two result files must be given.\n");
    return(1);
  }

  /* In verbose mode print arguments and options */
  if(verbose>0) {
    printf("colfile := %s\n", colfile);
    printf("resNr := %d\n", resNr);
    printf("doSort := %d\n", doSort);
    printf("doMode := %d\n", doMode);
    printf("saveSD := %d\n", saveSD);
    printf("first_res := %d\n", first_res);
  }
  if(verbose>25) RESULT_TEST=verbose-25; else RESULT_TEST=0;


  /*
   *  Read result files
   */
  if(verbose>1) printf("reading result files\n");
  /* Allocate memory for separate study results */
  res=(RES*)calloc(resNr, sizeof(RES));
  if(res==NULL) {fprintf(stderr, "Error: out of memory.\n"); return(2);}
  /* Initiate study results */
  for(ri=0; ri<resNr; ri++) resInit(res+ri);
  /* Read result files */
  for(ai=first_res, vi=0; ai<argc; ai++) {
    strcpy(resfile, argv[ai]);
    if(verbose>0) fprintf(stdout, "file[%d] := %s\n", vi+1, resfile);
    /* read this result file */
    ret=resRead(resfile, res+vi, verbose-3);
    if(ret) {
      if(beStrict==0) {
        fprintf(stderr, "Warning: '%s' not read: %s\n", resfile, reserrmsg);
        resEmpty(res+vi); continue;
      } else {
        fprintf(stderr, "Error in reading '%s': %s\n", resfile, reserrmsg);
        for(ri=0; ri<vi; ri++) resEmpty(res+ri);
        return(3);
      }
    }
    if(verbose>2) printf("res.voiNr := %d\n", res[vi].voiNr);
    /* Make sure that TAC names do not need encoding in XML */
    for(int i=0; i<res[vi].voiNr; i++) {
      strCleanForXML(res[vi].voi[i].name);
      strCleanForXML(res[vi].voi[i].voiname);
      strCleanForXML(res[vi].voi[i].hemisphere);
      strCleanForXML(res[vi].voi[i].place);
    }
    /* sort rois */
    if(doMode!=1) resSortByName(res+vi);
    /* make sure that there are no duplicate rois */
    if(doMode!=1 && resIsDuplicateNames(res+vi)>0) {
      fprintf(stderr, "Error: %s contains duplicate region names.\n", resfile);
      for(ri=0; ri<vi; ri++) resEmpty(res+ri);
      return(3);
    }
    /* make sure that result contains some sort of study number */
    if(strlen(res[vi].studynr)<1 || strcmp(res[vi].studynr, ".")==0) {
      cptr=strrchr(resfile, '/'); if(cptr==NULL) cptr=strrchr(resfile, '\\');
      if(cptr==NULL) cptr=resfile; else cptr++;
      strlcpy(res[vi].studynr, cptr, MAX_STUDYNR_LEN);
      cptr=strrchr(res[vi].studynr, '.'); if(cptr!=NULL) *cptr=(char)0;
    }
    /* prepare for the next result file */
    vi++;
  }
  if(verbose>0) fflush(stdout);
  resNr=vi;
  if(resNr<1) {
    fprintf(stderr, "Error: no result file(s) could be read.\n");
    for(ri=0; ri<resNr; ri++) resEmpty(res+ri);
    return(3);
  } else if(resNr<2 && doMode!=1) {
    fprintf(stderr, "Error: only one result file could be read.\n");
    for(ri=0; ri<resNr; ri++) resEmpty(res+ri);
    return(3);
  }
  if(verbose>0) printf("%d result files read.\n", resNr);
  if(verbose>20) for(ri=0; ri<resNr; ri++) resPrint(res+ri);
  /* Sort the files by study number, if required */
  if(doSort) {
    if(verbose>1) printf("sorting results by study number\n");
    for(ri=0; ri<resNr-1; ri++) for(rj=ri+1; rj<resNr; rj++)
      if(strcmp(res[ri].studynr, res[rj].studynr)>0) {
        memcpy(&tres, res+ri, sizeof(RES));
        memcpy(res+ri, res+rj, sizeof(RES));
        memcpy(res+rj, &tres, sizeof(RES));
      }
  }
  if(verbose>19) for(ri=0; ri<resNr; ri++) resPrint(res+ri);
  /* Print the study numbers */
  if(verbose>0) {
    for(ri=0; ri<resNr; ri++)
      printf("study_number[%d] := %s\n", ri+1, res[ri].studynr);
    fflush(stdout);
  }

  /*
   *  Check that results are of one kind only
   */
  if(doMode!=1) {
    if(verbose>1) printf("checking that results can be combined\n");
    strcpy(tmp, res[0].program); cptr=strtok(tmp, " \t\n\r(");
    if(cptr==NULL) len=strlen(res[0].program); else len=strlen(cptr);
    for(ri=1; ri<resNr; ri++) {
      if(strncasecmp(res[0].program, res[ri].program, len)==0) continue;
      fprintf(stderr, "Error: different software used in result files.\n");
      for(ri=0; ri<resNr; ri++) resEmpty(res+ri);
      return(4);
    }
    for(ri=1, ret=0; ri<resNr; ri++) {
      /* parNr */
      if(res[ri].parNr!=res[0].parNr) {
        fprintf(stderr, "Error: different parameter nr in result files.\n");
        ret=1; break;
      }
      /* Parameter names and units */
      if(resMatchParameternames(&res[ri], &res[0])!=0) {
        fprintf(stderr, "Error: different parameters in result files.\n");
        ret=2; break;
      }
    }
    if(ret>0) {
      for(ri=0; ri<resNr; ri++) resEmpty(res+ri);
      return(4);
    }
  }

  /*
   *  Check that result file nr is not too big in parameter table mode
   *  (including median, mean and sd)
   */
  if(doMode==2) {
    if(resNr>=MAX_RESPARAMS-3) {
      fprintf(stderr, "Error: too many result files; %d allowed.\n", MAX_RESPARAMS);
      for(ri=0; ri<resNr; ri++) resEmpty(res+ri);
      return(1);
    }
  }

  /*
   *  Check that the same regions exist in all result files in parameter table
   *  mode
   */
  if(doMode==2) {
    ret=0;
    for(ri=1; ri<resNr; ri++) {
      if(res[0].voiNr!=res[ri].voiNr) {ret=1; break;}
      if(resMatchRegions(res+0, res+ri)) {ret=2; break;}
    }
    if(ret>0) {
      if(ret==1) fprintf(stderr, "Error: in this mode region nr must match.\n");
      else fprintf(stderr, "Error: in this mode regions must match.\n");
      for(ri=0; ri<resNr; ri++) resEmpty(res+ri);
      return(1);
    }
  }

  if(verbose>0) printf("collecting results...\n");

  /*
   *  Open the collection file
   */
  if(verbose>1) printf("writing %s\n", colfile);

  /* Check if the collection file exists; backup, if necessary */
  (void)backupExistingFile(colfile, NULL, NULL);

  /* Open the collection file for writing */
  if((fp=fopen(colfile, "w"))==NULL) {
    fprintf(stderr, "Error: cannot open file '%s' for write.\n", colfile);
    for(ri=0; ri<resNr; ri++) resEmpty(res+ri);
    return(11);
  }

  /* Write XHTML 1.1 doctype and head */
  tpcProgramName(argv[0], 1, 1, tmp, 128);
  if(resWriteXHTML11_doctype(fp) || resWriteXHTML11_head(fp, tmp)) {
    fprintf(stderr, "Error: cannot write in file '%s'.\n", colfile);
    for(ri=0; ri<resNr; ri++) resEmpty(res+ri);
    fclose(fp); remove(colfile); return(11);
  }

  /* Start writing the body of the HTML file */
  fprintf(fp, "\n<body>\n");

  /* Write a title */
  fprintf(fp, "\n<div id=\"title\">\n");
  strcpy(tmp, res[0].program); cptr=strtok(tmp, " \t\n\r(");
  if(cptr==NULL) cptr=res[0].program;
  if(doMode!=1 && strlen(cptr)>1)
    fprintf(fp, "  <h1>Results of %s</h1>\n", cptr);
  else
    fprintf(fp, "  <h1>Results</h1>\n");
  fprintf(fp, "</div>\n");

  /*
   *  Start with a list of links
   */
  if(doMode==0) {
    /* Make a list of rois with links */
    fprintf(fp, "\n<div id=\"regcontainer\">\n");
    fprintf(fp, "<ul id=\"reglist\">\n");
    resvoi2=NULL; resvoi1=resGetNextResVOI(res, resNr, NULL, NULL);
    if(resvoi1==NULL) {
      fprintf(stderr, "Error: cannot get result file contents.\n");
      for(ri=0; ri<resNr; ri++) resEmpty(res+ri);
      fclose(fp); remove(colfile); return(9);
    }
    while(resvoi1!=NULL) {
      if(resvoi2==NULL || strcasecmp(resvoi1->name, resvoi2->name)) {
        strcpy(tmp, resvoi1->name);
        if(verbose>10) printf("NEW name := %s\n", resvoi1->name);
        /* replace dots and spaces with underscores */
        while((cptr=strpbrk(tmp, " ."))!=NULL) *cptr='_';
        fprintf(fp, "  <li><a href=\"#%s\">%s</a></li>\n", tmp, resvoi1->name);
      }
      resvoi2=resvoi1; if(verbose>11) printf("name := %s\n", resvoi2->name);
      resvoi1=resGetNextResVOI(res, resNr, resvoi2, NULL);
    }
    fprintf(fp, "</ul>\n");
    fprintf(fp, "</div>\n");
  } else if(doMode==2) {
    /* Make a list of parameters with links */
    fprintf(fp, "\n<div id=\"parcontainer\">\n");
    fprintf(fp, "<ol id=\"parlist\">\n");
    for(pi=0; pi<res[0].parNr; pi++) {
      sprintf(tmp, "par%d", pi+1);
      fprintf(fp, "  <li><a href=\"#%s\">%s</a></li>\n", tmp, res[0].parname[pi]);
    }
    fprintf(fp, "</ol>\n");
    fprintf(fp, "</div>\n");
  }

  /* Start the div for tables */
  fprintf(fp, "\n<div id=\"tables\">\n");

  ret=0;
  if(doMode==0) {
    /*
     *  Collect and process each region, one at a time
     */
    if(verbose>1) printf("collecting results of one region at a time\n");
    /* Allocate memory for results of one region */
    if(resSetmem(&coll, resNr)) {
      fprintf(stderr, "Error: out of memory.\n");
      for(ri=0; ri<resNr; ri++) resEmpty(res+ri);
      return(5);
    }
    /* Search through results for each region */
    resvoi2=NULL; resvoi1=resGetNextResVOI(res, resNr, NULL, &ri);
    if(resvoi1==NULL) {
      fprintf(stderr, "Error: cannot get result file contents.\n");
      for(ri=0; ri<resNr; ri++) resEmpty(res+ri);
      fclose(fp); remove(colfile); return(9);
    }
    coll.voiNr=0; vi=0; ret=0;
    while(resvoi1!=NULL && ret==0) {
      if(resvoi2==NULL || strcasecmp(resvoi1->name, resvoi2->name)) {
        vi++;
        /* This is a new (or the first) region */
        if(coll.voiNr>0) { // tabulate the previous region, if this is not 1st
          ret=rescoll_tabulate(fp, &coll, vi);
        }
        /* Copy region information into collection data */
        coll.voiNr=1;
        if(verbose>0) printf("region_name[%d] := %s\n", vi, resvoi1->name);
        strcpy(coll.datafile, resvoi1->name);
        char *cptr=strstr(coll.datafile, " . ."); if(cptr!=NULL) *cptr=(char)0;
        coll.parNr=res[ri].parNr;
        for(pi=0; pi<coll.parNr; pi++) {
          strcpy(coll.parname[pi], res[ri].parname[pi]);
          strcpy(coll.parunit[pi], res[ri].parunit[pi]);
        }
      } else {
        /* the same region, just a new study */
        coll.voiNr++;
      }
      if(verbose>2) printf("studynr := %s\n", res[ri].studynr);
      /* Copy study information into collection data */
      /* Set the study name */
      strcpy(coll.voi[coll.voiNr-1].name, res[ri].studynr);
      /* set the calculation date */
      st=localtime(&res[ri].time);
      if(st!=NULL) strftime(tmp, 256, "%Y-%m-%d", st); else strcpy(tmp, "          ");
      strlcpy(coll.voi[coll.voiNr-1].voiname, tmp, 7);
      strlcpy(coll.voi[coll.voiNr-1].hemisphere, tmp+6, 7);
      /* parameter values */
      for(pi=0; pi<res[ri].parNr; pi++)
        coll.voi[coll.voiNr-1].parameter[pi]=resvoi1->parameter[pi];
      /* Prepare for the next one */
      resvoi2=resvoi1; resvoi1=resGetNextResVOI(res, resNr, resvoi2, &ri);
    } // while
    /* tabulate the last region */
    if(coll.voiNr>0) {
      ret=rescoll_tabulate(fp, &coll, vi);
    }

  } else if (doMode==1) {
    /*
     *  Write one result file as one HTML table
     */
    if(verbose>1) printf("writing result files as HTML tables\n");
    for(ri=0, ret=0; ri<resNr; ri++) {
      /* Remove SDs and CLs if required */
      if(saveSD==0) {
        for(vi=0; vi<res[ri].voiNr; vi++) for(pi=0; pi<res[ri].parNr; pi++)
          res[ri].voi[vi].cl1[pi]=res[ri].voi[vi].cl2[pi]=
            res[ri].voi[vi].sd[pi]=nan("");
      }
      /* Add empty line to HTML */
      if(ri>0) fprintf(fp, "\n<br />\n");
      /* Write table */
      ret=resWriteHTML_table(res+ri, fp);
      if(ret!=0) break;
    }
  } else if(doMode==2) {
    /*
     *  Collect and process each parameter, one at a time
     */
    if(verbose>1) printf("collecting results of one parameter at a time\n");
    /* SDs and CLs are not used */
    for(ri=0, ret=0; ri<resNr; ri++) {
      for(vi=0; vi<res[ri].voiNr; vi++) for(pi=0; pi<res[ri].parNr; pi++)
        res[ri].voi[vi].cl1[pi]=res[ri].voi[vi].cl2[pi]=
        res[ri].voi[vi].sd[pi]=nan("");
    }
    /* Allocate memory for results of one parameter */
    if(resSetmem(&coll, res[0].voiNr)) {
      fprintf(stderr, "Error: out of memory.\n");
      for(ri=0; ri<resNr; ri++) resEmpty(res+ri);
      return(5);
    }
    /* Set common result header information */
    strcpy(coll.program, "");
    coll.voiNr=res[0].voiNr;
    coll.parNr=resNr+3;
    coll.isweight=res[0].isweight;
    coll.time=time(NULL);
    for(pi=0; pi<resNr; pi++) {
      strcpy(coll.parname[pi], res[pi].studynr);
    }
    strcpy(coll.parname[pi++], "Median");
    strcpy(coll.parname[pi++], "Mean");
    strcpy(coll.parname[pi++], "SD");
    for(ri=0; ri<res[0].voiNr; ri++) {
      strcpy(coll.voi[ri].name, res[0].voi[ri].name);
      strcpy(coll.voi[ri].voiname, res[0].voi[ri].voiname);
      strcpy(coll.voi[ri].hemisphere, res[0].voi[ri].hemisphere);
      strcpy(coll.voi[ri].place, res[0].voi[ri].place);
    }
    /* One parameter at a time */
    double tdata[MAX_RESPARAMS];
    for(pi=0; pi<res[0].parNr; pi++) {
      strcpy(coll.program, res[0].parname[pi]);
      /* Fill the table for this parameter */
      for(ri=0; ri<res[0].voiNr; ri++) {
        for(vi=0; vi<resNr; vi++) {
          coll.voi[ri].parameter[vi]=tdata[vi]=res[vi].voi[ri].parameter[pi];
        }
        /* Calculate Median, Mean and SD */
        resMedian(tdata, resNr, &coll.voi[ri].parameter[resNr], NULL, NULL);
        resMean(tdata, resNr, &coll.voi[ri].parameter[resNr+1],
                              &coll.voi[ri].parameter[resNr+2]);
      }
      /* Add empty line to HTML */
      if(pi>0) fprintf(fp, "\n<br />\n");
      /* Write the anchor */
      fprintf(fp, "\n<a id=\"par%d\"></a>\n", pi+1);
      /* Write table */
      ret=resWriteHTML_table(&coll, fp);
      if(ret!=0) break;
    } // next res parameter
  }
  if(ret!=0) {
    fprintf(stderr, "Error: cannot tabulate result files.\n");
    resEmpty(&coll); for(ri=0; ri<resNr; ri++) resEmpty(res+ri);
    fclose(fp); remove(colfile); return(9);
  }


  if(verbose>0) fprintf(fp, "\n");
  /* Close collection file, writing HTLM end code */
  if(verbose>1) printf("closing collection file.\n");
  fprintf(fp, "</div>\n");
  fprintf(fp, "</body></html>\n");
  fclose(fp);

  if(verbose>0) printf("%s written.\n", colfile);

  /* Free up memory */
  resEmpty(&coll);
  for(ri=0; ri<resNr; ri++) resEmpty(res+ri);

  return(0);
}
/*****************************************************************************/
/// @endcond
/*****************************************************************************/
/** Get the next study region from a list of study results.
    This function depends on sw2 in result structures.
    @return Returns NULL in case of error, or if there are no more regions.
 */
ResVOI *resGetNextResVOI(
  /** Input: pointer to the list of RES structures */
  RES *reslist,
  /** Input: Nr of RES structures */
  int resNr,
  /** Input: pointer to the last VOI */
  ResVOI *lastResVOI,
  /** Output: index [0..resNr-1] of results containing the returned VOI */
  int *resIndex
) {
  int ri, vi;

  /* Check the input */
  if(resIndex!=NULL) *resIndex=0;
  if(reslist==NULL || resNr<1) return(NULL);
  if(reslist[0].voiNr==0) return(NULL);
  /* If lastResVOI is NULL, that means that first one is required */
  if(lastResVOI==NULL) {
    for(ri=0; ri<resNr; ri++) for(vi=0; vi<reslist[ri].voiNr; vi++)
      reslist[ri].voi[vi].sw2=0;
    reslist[0].voi[0].sw2=1;
    return(reslist[0].voi);
  }
  /* Search the next study with the same region name */
  for(ri=0; ri<resNr; ri++)
    for(vi=0; vi<reslist[ri].voiNr; vi++) if(reslist[ri].voi[vi].sw2==0)
      if(strcasecmp(reslist[ri].voi[vi].name, lastResVOI->name)==0) {
        reslist[ri].voi[vi].sw2=1; if(resIndex!=NULL) *resIndex=ri;
        return(reslist[ri].voi+vi);
      }
  /* If not found, then proceed to the next region name */
  for(ri=0; ri<resNr; ri++)
    for(vi=0; vi<reslist[ri].voiNr; vi++) if(reslist[ri].voi[vi].sw2==0) {
      reslist[ri].voi[vi].sw2=1; if(resIndex!=NULL) *resIndex=ri;
      return(reslist[ri].voi+vi);
    }
  /* If not found, then the previous one must have been the last one */
  return(NULL);
}
/*****************************************************************************/

/*****************************************************************************/
/** Tabulate the collected results of a single region.
    Calculates and tabulates also the statistics.
    @return Returns zero if successful, otherwise non-zero.
 */
int rescoll_tabulate(
  /** Pointer to an opened file, where XHTML headers etc have already been written */
  FILE *fp,
  /** Pointer to the structure that contains the collected data */
  RES *coll,
  /** Number of regions collected this far, including this one */
  int collNr
) {
  char *cptr, tmp[1024];
  int si, pi, partype[MAX_RESPARAMS], n;
  double resstat[5][MAX_RESPARAMS], *data;


  /* Check the input */
  if(fp==NULL || coll==NULL || coll->voiNr<0 || collNr<1) return(1);
  if(coll->voiNr<1) return(0);

  /* Calculate the statistics if more than one study */
  if(coll->voiNr>1) {
    data=(double*)malloc(coll->voiNr*sizeof(double));
    for(pi=0; pi<coll->parNr; pi++) {
      /* Copy data to a temp array */
      for(si=0; si<coll->voiNr; si++) data[si]=coll->voi[si].parameter[pi];
      /* Calculate median, min and max */
      resMedian(data, coll->voiNr, &resstat[2][pi], &resstat[0][pi],
                &resstat[1][pi]);
      /* Calculate avg and sd */
      resMean(data, coll->voiNr, &resstat[3][pi], &resstat[4][pi]);
    }
    free(data);
  }

  /* Put some space in between tables, if this is not the first table */
  if(collNr>1) {
    fprintf(fp, "\n<br />\n");
  }

  /* Write the anchor */
  strcpy(tmp, coll->datafile);
  /* replace dots and spaces with underscores */
  while((cptr=strpbrk(tmp, " ."))!=NULL) *cptr='_';
  fprintf(fp, "\n<a id=\"%s\"></a>\n", tmp);

  /* Start the table */
  fprintf(fp, "<table>\n");
  fprintf(fp, "  <thead>\n");

  /* Write the region name */
  fprintf(fp, "  <tr><th colspan=\"%d\">", 3+coll->parNr);
  fprintf(fp, "%s", coll->datafile);
  fprintf(fp, "</th></tr>\n");

  /* Write the titles */
  fprintf(fp, "  <tr>\n    <td>Nr</td><td>Study</td><td>Date</td>\n");
  /* continue with the parameter names */
  for(pi=0; pi<coll->parNr; pi++)
    fprintf(fp, "    <td>%s</td>\n", coll->parname[pi]);
  fprintf(fp, "  </tr>\n");
  fprintf(fp, "  </thead>\n");

  /* Determine the best print format for each of the parameters */
  for(pi=0; pi<coll->parNr; pi++)
    partype[pi]=resParameterPrintType(coll, pi);

  /* Save all study results from this region */
  fprintf(fp, "  <tbody>\n");
  for(si=0; si<coll->voiNr; si++) {
    if(si%2) strcpy(tmp, "evenstudy"); else strcpy(tmp, "oddstudy");
    fprintf(fp, "  <tr class=\"%s\">\n", tmp);
    fprintf(fp, "    <td>%-3d</td>\n", si+1);
    fprintf(fp, "    <td>%s</td>\n", coll->voi[si].name);
    fprintf(fp, "    <td>%s%s</td>\n", coll->voi[si].voiname, coll->voi[si].hemisphere);
    for(pi=0; pi<coll->parNr; pi++) {
      if(coll->voi[si].parameter[pi]>=0) n=4; else n=3;
      switch(partype[pi]) {
        case 0: sprintf(tmp, "%.0f", coll->voi[si].parameter[pi]); break;
        case 1: sprintf(tmp, "%.*f", n, coll->voi[si].parameter[pi]); break;
        case 2:
        default: sprintf(tmp, "%.*e", n, coll->voi[si].parameter[pi]); break;
      }
      fprintf(fp, "    <td>%s</td>\n", tmp);
    }
    fprintf(fp, "  </tr>\n");
  }
  fprintf(fp, "  </tbody>\n");

  /* Continue with the next region, if only one study was found */
  if(coll->voiNr==1) {
    fprintf(fp, "</table>\n");
    return(0);
  }

  /* Write the statistics */
  fprintf(fp, "  <tbody>\n");
  /* min */
  fprintf(fp, "  <tr class=\"oddsum\">\n");
  fprintf(fp, "    <td colspan=\"3\">Min</td>\n");
  for(pi=0; pi<coll->parNr; pi++) {
    if(resstat[0][pi]>=0) n=4; else n=3;
    switch(partype[pi]) {
      case 0: sprintf(tmp, "%.0f", resstat[0][pi]); break;
      case 1: sprintf(tmp, "%.*f", n, resstat[0][pi]); break;
      case 2:
      default: sprintf(tmp, "%.*e", n, resstat[0][pi]); break;
    }
    fprintf(fp, "    <td>%s</td>\n", tmp);
  }
  fprintf(fp, "  </tr>\n");
  /* max */
  fprintf(fp, "  <tr class=\"evensum\">\n");
  fprintf(fp, "    <td colspan=\"3\">Max</td>\n");
  for(pi=0; pi<coll->parNr; pi++) {
    if(resstat[1][pi]>=0) n=4; else n=3;
    switch(partype[pi]) {
      case 0: sprintf(tmp, "%.0f", resstat[1][pi]); break;
      case 1: sprintf(tmp, "%.*f", n, resstat[1][pi]); break;
      case 2:
      default: sprintf(tmp, "%.*e", n, resstat[1][pi]); break;
    }
    fprintf(fp, "    <td>%s</td>\n", tmp);
  }
  fprintf(fp, "  </tr>\n");
  /* median */
  fprintf(fp, "  <tr class=\"oddsum\">\n");
  fprintf(fp, "    <td colspan=\"3\">Median</td>\n");
  for(pi=0; pi<coll->parNr; pi++) {
    if(resstat[2][pi]>=0) n=6; else n=5;
    switch(partype[pi]) {
      case 0: sprintf(tmp, "%.2f", resstat[2][pi]); break;
      case 1: sprintf(tmp, "%.*f", n, resstat[2][pi]); break;
      case 2:
      default: sprintf(tmp, "%.*e", n, resstat[2][pi]); break;
    }
    fprintf(fp, "    <td>%s</td>\n", tmp);
  }
  fprintf(fp, "  </tr>\n");
  /* mean */
  fprintf(fp, "  <tr class=\"evensum\">\n");
  fprintf(fp, "    <td colspan=\"2\">Mean</td>\n");
  fprintf(fp, "    <td>n=%d</td>\n", coll->voiNr);
  for(pi=0; pi<coll->parNr; pi++) {
    if(resstat[3][pi]>=0) n=6; else n=5;
    switch(partype[pi]) {
      case 0: sprintf(tmp, "%.2f", resstat[3][pi]); break;
      case 1: sprintf(tmp, "%.*f", n, resstat[3][pi]); break;
      case 2:
      default: sprintf(tmp, "%.*e", n, resstat[3][pi]); break;
    }
    fprintf(fp, "    <td>%s</td>\n", tmp);
  }
  fprintf(fp, "  </tr>\n");
  /* sd */
  fprintf(fp, "  <tr class=\"oddsum\">\n");
  fprintf(fp, "    <td colspan=\"3\">SD</td>\n");
  for(pi=0; pi<coll->parNr; pi++) {
    if(resstat[4][pi]>=0) n=6; else n=5;
    switch(partype[pi]) {
      case 0: sprintf(tmp, "%.2f", resstat[4][pi]); break;
      case 1: sprintf(tmp, "%.*f", n, resstat[4][pi]); break;
      case 2:
      default: sprintf(tmp, "%.*e", n, resstat[4][pi]); break;
    }
    fprintf(fp, "    <td>%s</td>\n", tmp);
  }
  fprintf(fp, "  </tr>\n");
  fprintf(fp, "  </tbody>\n");

  /* End the table */
  fprintf(fp, "</table>\n");

  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
