/** @file reslist.c
 *  @brief Listing part of result file contents.
 *  @copyright (c) Turku PET Centre
 *  @author Vesa Oikonen
 */
/// @cond
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <time.h>
/*****************************************************************************/
#include "libtpcmisc.h"
#include "libtpccurveio.h"
/*****************************************************************************/

/*****************************************************************************/
static char *info[] = {
  "Lists the user-specified regional results on screen.",
  "Note: result file must not be in HTML format.",
  " ",
  "Usage: @P [Options] filename [tacid]",
  " ",
  "Options:",
  " -n",
  "     CL or SD values are not listed.",
  " -ift",
  "     Results are listed in IFT format.",
  " -stdoptions", // List standard options like --help, -v, etc
  " ",
  "Example:",
  "     @P ut6789ki.res put",
  " ",
  "See also: resdel, rescoll, resmatch, fit2res, resrmdpl, parformat, parget",
  " ",
  "Keywords: results, IFT, software testing",
  0};
/*****************************************************************************/

/*****************************************************************************/
/* Turn on the globbing of the command line, since it is disabled by default in
   mingw-w64 (_dowildcard=0); in MinGW32 define _CRT_glob instead, if necessary;
   In Unix&Linux wildcard command line processing is enabled by default. */
/*
#undef _CRT_glob
#define _CRT_glob -1
*/
int _dowildcard = -1;
/*****************************************************************************/

/*****************************************************************************/
/**
 *  Main
 */
int main(int argc, char **argv)
{
  int   ai, help=0, version=0, verbose=1;
  int   pi, ri, ret, m, n, listStat=1, toIFT=0;
  RES   res;
  char  resfile[FILENAME_MAX], *cptr;


  /*
   *  Get arguments
   */
  if(argc==1) {tpcPrintUsage(argv[0], info, stderr); return(1);}
  resfile[0]=(char)0;
  resInit(&res);
  for(ai=1; ai<argc; ai++) if(*argv[ai]=='-') {
    if(tpcProcessStdOptions(argv[ai], &help, &version, &verbose)==0) continue;
    cptr=argv[ai]+1; if(*cptr=='-') cptr++; if(!cptr[0]) continue;
    if(*cptr=='N' || *cptr=='n') {
      listStat=0; continue;
    } else if(strcasecmp(cptr, "IFT")==0) {
      toIFT=1; continue;
    }
    fprintf(stderr, "Error: invalid option '%s'.\n", argv[ai]);
    return(1);
  } else break;
  
  /* Print help or version? */
  if(help==2) {tpcHtmlUsage(argv[0], info, ""); return(0);}
  if(help) {tpcPrintUsage(argv[0], info, stdout); return(0);}
  if(version) {tpcPrintBuild(argv[0], stdout); return(0);}

  /* Get the name of result file */
  if(ai<argc) {
    strlcpy(resfile, argv[ai], FILENAME_MAX); ai++;
  } else {
    fprintf(stderr, "Error: missing command-line argument; use option --help\n");
    return(1);
  }

  /* In verbose mode print arguments and options */
  if(verbose>1) {
    printf("resfile := %s\n", resfile);
    printf("listStat := %d\n", listStat);
    printf("toIFT := %d\n", toIFT);
  }


  /*
   *  Read result file
   */
  if(verbose>1) printf("reading %s\n", resfile);
  ret=resRead(resfile, &res, verbose-2);
  if(ret) {
    fprintf(stderr, "Error in reading '%s': %s\n", resfile, reserrmsg);
    resEmpty(&res);
    return(2);
  }


  /*
   *  Select the regions with user-defined IDs, or, if not defined,
   *  then select all for listing.
   */
  if(ai>=argc) {
    for(ri=0; ri<res.voiNr; ri++) res.voi[ri].sw=1;
  } else {
    m=0;
    for(; ai<argc; ai++) {
      if(verbose>2) printf("searching for region %s\n", argv[ai]);
      n=0; ri=atoi(argv[ai])-1; /* maybe its the region number? */
      if(ri<res.voiNr && ri>=0) { /* select that region if it exists */
        res.voi[ri].sw=1; n=1;
      } else { /* if not then it is the region name */
        n=resSelectRegions(&res, argv[ai], 0);
      }
      m+=n;
    }
    if(m==0) {
      fprintf(stderr, "Error: TAC IDs did not match with any of the regions.\n");
      resEmpty(&res); return(1);
    }
  }




  /*
   *  Delete those regions that were not selected
   */
  ri=res.voiNr-1;
  while(ri>=0) {
    if(res.voi[ri].sw!=1) ret=resDelete(&res, ri);
    ri--;
  }


  /*
   *  Set SDs and CVs to NaN, if they are not to be printed
   */
  if(listStat==0) for(ri=0; ri<res.voiNr; ri++) for(pi=0; pi<res.parNr; pi++) {
    res.voi[ri].sd[pi]=nan("");
    res.voi[ri].cl1[pi]=nan("");
    res.voi[ri].cl2[pi]=nan("");
  }


  /*
   *  Print the ones that are left
   */
  if(toIFT==0) {
    resPrint(&res);
  } else {
    IFT ift;
    iftInit(&ift);
    ret=res2ift(&res, &ift, verbose-1); 
    if(ret==0) iftWrite(&ift, "stdout", 0);
    iftEmpty(&ift);
  }
  fprintf(stdout, "\n");

  resEmpty(&res);

  return(0);
}
/*****************************************************************************/
/// @endcond
/*****************************************************************************/
