#!/bin/bash
#: Title      : test_b2t_mo2
#: Date       : 2016-02-28
#: Author     : "Vesa Oikonen" <vesa.oikonen@utu.fi>
#: Options    : None

# Set the name of executable to test
if [ -n "${OS+1}" ] && [ "$OS" = "Windows_NT" ]; then
EXT=.exe;
else
EXT=;
fi
PROGRAM=../../b2t_mo2$EXT;

printf "=====================================================================\n"
printf "creating test data for %s\n" $PROGRAM
printf "=====================================================================\n"


printf "\n verifying that required data exists \n"

if [ ! -f $PROGRAM ]; then
  printf "Failed: executable does not exist.\n"
  exit 1
fi

if [ ! -f blood_o.kbq ] || [ ! -f blood_w.kbq ] || [ ! -f blood_o_ss.kbq ]; then
  printf "Failed: required data does not exist.\n"
  exit 1
fi


printf "\n making test data \n"

if [ ! -f blood.kbq ]; then
  taccalc --force blood_o.kbq + blood_w.kbq blood.kbq
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi

if [ ! -f oxygen.dat ]; then
  printf " 0 0 \n"   > oxygen.dat
  printf "10 10 \n" >> oxygen.dat
  printf "20 50 \n" >> oxygen.dat
  printf "30 55 \n" >> oxygen.dat
  printf "40 35 \n" >> oxygen.dat
  printf "50 15 \n" >> oxygen.dat
  printf "60 5  \n" >> oxygen.dat
fi
if [ ! -f water.dat ]; then
  printf " 0 0 \n"   > water.dat
  printf "10  0 \n" >> water.dat
  printf "20  1 \n" >> water.dat
  printf "30  5 \n" >> water.dat
  printf "40 12 \n" >> water.dat
  printf "50 13 \n" >> water.dat
  printf "60 12 \n" >> water.dat
fi


printf "\n\n"
printf "=====================================================================\n"
printf "testing %s\n" $PROGRAM
printf "=====================================================================\n"



printf "\n 1.0.0 \n"
printf " Test case: Normal usage with default settings. \n"
printf " Expected result: Reasonable curves simulated without errors. \n\n"

rm -f lowmro2.dat
$PROGRAM blood_o.kbq blood_w.kbq 3 0.25 lowmro2.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tac2svg -legend=y lowmro2.svg -l lowmro2.dat 
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

rm -f highmro2.dat
$PROGRAM blood_o.kbq blood_w.kbq 50 0.75 highmro2.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tac2svg -legend=y highmro2.svg -l highmro2.dat 
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 1.0.1 \n"
printf " Test case: Nonphysiological values. \n"
printf " Expected result: Error. \n\n"

rm -f output.dat stdout.txt
$PROGRAM blood_o.kbq blood_w.kbq 3 99999 output.dat &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error "Invalid OER."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
if [ -f output.dat ]; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 1.0.2 \n"
printf " Test case 1: Simple data files are accepted. \n"
printf " Test case 2: Flow and OER can be given with decimal comma. \n"
printf " Expected result: Reasonable curves simulated without errors. \n\n"

rm -f simple.dat
$PROGRAM -sub oxygen.dat water.dat 300,0 0,75 simple.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tac2svg -legend=y simple.svg -l simple.dat 
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n 1.1.0 \n"
printf " Test case: Default VOI names. \n"
printf " Expected result: Correct VOI names. \n\n"


rm -f output.tac output.ift
$PROGRAM -sub blood_o.kbq blood_w.kbq 5 0.20 output.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
taclist -na -nv -ift output.tac > output.ift
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

iftisval output.ift tacname[1] "Muscle"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval output.ift tacname[2] "Muscle_O2"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval output.ift tacname[3] "Muscle_H2O"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval output.ift tacname[4] "Muscle_artery"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval output.ift tacname[5] "Muscle_vena_O2"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval output.ift tacname[6] "Muscle_vena_H2O"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 1.1.1 \n"
printf " Test case: Optional setting of VOI name. \n"
printf " Expected result: Correct VOI names. \n\n"


rm -f output.tac output.ift
$PROGRAM -voiname=lihas -sub blood_o.kbq blood_w.kbq 5 0.20 output.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
taclist -na -nv -ift output.tac > output.ift
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

iftisval output.ift tacname[1] "lihas"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval output.ift tacname[2] "lihas_O2"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval output.ift tacname[3] "lihas_H2O"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval output.ift tacname[4] "lihas_artery"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval output.ift tacname[5] "lihas_vena_O2"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval output.ift tacname[6] "lihas_vena_H2O"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"




printf "\n===================================================================\n"
printf "Quantitation  \n"
printf "===================================================================\n\n"


printf "\n 2.0.0 \n"
printf " Test case 1: Flow is set to zero. \n"
printf " Test case 2: Option -Vb is used. \n"
printf " Test case 3: Venous BTACs are saved with options. \n"
printf " Expected result 1: Output contains only vascular contribution. \n"
printf " Expected result 2: Venous BTACs are the same as input BTACs. \n\n"

rm -f output.dat venao.dat venaw.dat
taccalc --force blood.kbq x 0.1 temp.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

$PROGRAM -Vb=10 -venao=venao.dat -venaw=venaw.dat blood_o.kbq blood_w.kbq 0 0.25 output.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

tacmatch -rel=0.1 -abs=0.1 temp.dat output.dat 
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

tacmatch -rel=0.1 -abs=0.1 blood_o.kbq venao.dat 
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -rel=0.1 -abs=0.1 blood_w.kbq venaw.dat 
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"




printf "\n 2.1.0 \n"
printf " Test case 1: Steady-state situation. \n"
printf " Test case 2: Vb is set to zero. \n"
printf " Test case 3: Radiowater input is zero (fictional). \n"
printf " Test case 4: K1/k2 is set to 1. \n"
printf " Test case 5: Subcompartments are saved. \n"
printf " Expected result: Correct steady state sub TTAC concentration. \n\n"

rm -f output.dat
taccalc --force blood_o_ss.kbq x 0 zeroinput.kbq
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

$PROGRAM -Vb=0 -K1k2=1.0 -sub blood_o_ss.kbq zeroinput.kbq 50 0.50 output.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacadd -ovr temp.dat output.dat 2
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
interpol -y -x=1800 temp.dat vrt1.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
# K1/k2=1; OER=k3/(k2+k3)=0.5; K1=k2=k3; K1/(k2+k3)=0.5;
# SSo2blood=50; SSo2tis=25
interpol -y -x=1800 blood_o_ss.kbq vrt2.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
taccalc --force vrt2.dat x 0.5 vrt2.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -rel=0.1 -abs=0.1 vrt2.dat vrt1.dat 
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 2.2.0 \n"
printf " Test case 1: Bolus situation. \n"
printf " Test case 2: Vb is set to zero. \n"
printf " Test case 3: Radiowater input is zero (fictional). \n"
printf " Test case 4: K1/k2 is set to 0.5. \n"
printf " Test case 5: Subcompartments are saved. \n"
printf " Expected result: Correct sub TTAC concentrations for oxygen and water. \n\n"

rm -f output.dat output1.dat output2.dat output3.dat

# Tissue oxygen TAC should be similar than from 1-tissue CM with K1 and k2+k3
# as model parameters:
p2t_3c -nosub blood_o.kbq 0.01 0.0333333333 0 0 0 0 output2.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

taccalc --force blood_w.kbq x 0 zeroinput.kbq
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"


$PROGRAM -Vb=0 -K1k2=0.5 -pH2O=0.8 -sub blood_o.kbq zeroinput.kbq 60 0.40 output.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacadd -ovr output1.dat output.dat 2
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -rel=0.1 -abs=0.1 output2.dat output1.dat 
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

# Tissue oxygen TAC should be similar than from 1-tissue CM with K1 and k2+k3
# as model parameters:
p2t_3c -sub blood_o.kbq 0.01 0.02 0.01333333333 0 0.0125 0 output2.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacadd -ovr output3.dat output2.dat 3
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacadd -ovr output1.dat output.dat 3
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -rel=0.1 -abs=0.1 output3.dat output1.dat 
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"




printf "\n 2.3.0 \n"
printf " Test case 1: Bolus situation. \n"
printf " Test case 2: Vb is set to zero. \n"
printf " Test case 3: oxygen input is zero (fictional). \n"
printf " Test case 4: pH2O is set to 0.8. \n"
printf " Test case 5: Subcompartments are saved. \n"
printf " Expected result: Radiowater TTAC is the same as when simulated using b2t_h2o. \n\n"

rm -f output.dat output2.dat

taccalc --force blood_o.kbq x 0 zeroinput.kbq
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

b2t_h2o -sub blood.kbq 50 0.8 1 0 100 output2.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

$PROGRAM -Vb=0 -pH2O=0.8 -sub zeroinput.kbq blood.kbq 50 0.50 output.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacadd -ovr vrt1.dat output.dat 3
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacadd -ovr vrt2.dat output2.dat 2
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -rel=0.1 -abs=0.1 vrt2.dat vrt1.dat 
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n\n"
printf "=====================================================================\n"
printf "error testing for %s\n" $PROGRAM
printf "=====================================================================\n"


printf "\n 3.0.0 \n"
printf " Test case: Unknown option. \n"
printf " Expected result: Error. \n"

rm -f stdout.txt
$PROGRAM -stupidoption blood_o.kbq blood_w.kbq 30 0.4 output.dat &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error "invalid option '-stupidoption'."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 3.0.1 \n"
printf " Test case: Extra argument entered. \n"
printf " Expected result: Error. \n"

rm -f stdout.txt
$PROGRAM blood_o.kbq blood_w.kbq 30 0.4 output.dat extra &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error "invalid argument 'extra'."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 3.0.2 \n"
printf " Test case: Missing argument(s). \n"
printf " Expected result: Error. \n"

rm -f stdout.txt
$PROGRAM blood_o.kbq blood_w.kbq 30 0.4  &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error "missing command-line argument."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

rm -f stdout.txt
$PROGRAM blood_o.kbq blood_w.kbq output.dat  &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 3.1.0 \n"
printf " Test case: Input file does not exist. \n"
printf " Expected result: Error. \n"

rm -f stdout.txt
$PROGRAM nonexistingfile.dat blood_w.kbq 30 0.4 output.dat &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Error in reading 'nonexistingfile.dat'" "cannot open file"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


rm -f stdout.txt
$PROGRAM blood_o.kbq nonexistingfile.dat 30 0.4 output.dat &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Error in reading 'nonexistingfile.dat'" "cannot open file"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 3.1.1 \n"
printf " Test case: Wrong BTAC format. \n"
printf " Expected result: Error. \n"

printf "ommmom moo moon\nommon 1 2 3\nnamnam 2 3\n" > stupid.dat

rm -f stdout.txt
$PROGRAM stupid.dat blood_w.kbq 30 0.4 output.dat &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Error in reading 'stupid.dat'" "wrong format"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"







rm -f *.bak
printf "\n===================================================================\n"
printf "   All passed! \n"
printf "===================================================================\n"
exit 0
