/** @file tac2svg.c
 *  @brief Plotting TAC data in SVG format XY plots.
 *  @copyright (c) Turku PET Centre
 *  @author Vesa Oikonen
 */
/// @cond
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <unistd.h>
#include <string.h>
#include <time.h>
/*****************************************************************************/
#include "libtpcmisc.h"
#include "libtpcmodel.h"
#include "libtpccurveio.h"
#include "libtpcsvg.h"
/*****************************************************************************/

/*****************************************************************************/
static char *info[] = {
  "Make XY plots of PET time-activity curves (TACs) in Scalable Vector Graphics",
  "(SVG) 1.1. format; specification in http://www.w3.org/TR/SVG/",
  "SVG files can be viewed with web browsers and processed further using for",
  "example Inkscape, Batik, and Gimp.",
  " ",
  "Usage: @P [Options] svg_file [-L|-S|-B] tac_file1 [-L|-S|-B] tac_file2 ...",
  " ",
  "Options:",
  " -L  TACs that are given after this option are plotted with lines only,",
  " -S  TACs that are given after this option are plotted with symbols only,",
  " -B  TACs that are given after this option are plotted with lines and",
  "     symbols (default).",
  " -x1=<start of x axis>",
  " -x2=<end of x axis>",
  " -y1=<start of y axis>",
  " -y2=<end of y axis>",
  " -nmt, -nxt, -nyt",
  "     Do not show main title or x or y axis titles.",
  " -mt=\"<main title>\"",
  "     User-specified string to override any default main title,",
  " -xt=\"<x axis title>\"",
  "     User-specified string to override any default x axis title,",
  " -yt=\"<y axis title>\"",
  "     User-specified string to override any default y axis title.",
  " -legend=<yes|No|auto>",
  "     Show TAC legends (yes), do not show (no, default), or automatically",
  "     determine whether to show or not (auto).",
  " -color, -bw", // grayscale not supported yet
  "     Create plots with colors (default), or in black-and-white.",
  /* note that -s means silent only if given as option; but if given after
     svg filename, it is interpreted as 'plot this with symbols' */
  " -stdoptions", // List standard options like --help, -v, etc
  " ",
  "Normally, SVG graphics file should have extension .svg. Alternatively,",
  "SVG graphics can be written inline in XHTML file by setting the extension",
  "to .xhtml.",
  " ",
  "Example 1.",
  "  @P gj247.svg -s gj247.tac -l gj247_fitted.tac",
  " ",
  "See also: fit2dat, dftscale, dftmax, tac4frpl, tacformat, tacjoin, tac2xml",
  " ",
  "Keywords: simulation, modelling, plotting, tool, DFT, TAC, SVG, XML",
  0};
/*****************************************************************************/

/*****************************************************************************/
/* Turn on the globbing of the command line, since it is disabled by default in
   mingw-w64 (_dowildcard=0); in MinGW32 define _CRT_glob instead, if necessary;
   In Unix&Linux wildcard command line processing is enabled by default. */
/*
#undef _CRT_glob
#define _CRT_glob -1
*/
int _dowildcard = -1;
/*****************************************************************************/

/*****************************************************************************/
typedef struct {
    DFT dft;
    int type; // 1=line, 2=symbol, 0=both
} DFTLIST;
/*****************************************************************************/

/*****************************************************************************/
/**
 *  Main
 */
int main(int argc, char **argv)
{
  int ai, help=0, version=0, verbose=1;
  int ti, ri, ret, fileNr=0, n, ns, nl, datarg=0;
  char *cptr, errmsg[128], temp[128];
  char svgfile[FILENAME_MAX], tacfile[FILENAME_MAX];
  int do_XHTML=0, plot_fit=0;
  int show_main_title=1;
  int show_x_title=1;
  int show_y_title=1;
  char main_title[64], x_title[64], y_title[64], tac_id[32], tac_title[64];
  int color_scale=0; // 0=color, 1=grayscale, 2=monochrome
  int symbol_fill=1;
  FILE *fp_svg;
  struct svg_viewports viewports; svg_init_viewports(&viewports);
  DFTLIST *dftl;
  double minx, maxx, miny, maxy, tx1, tx2, ty1, ty2;
  double preset_minx, preset_maxx, preset_miny, preset_maxy;
  int symbol_nr=0, color_nr=0, color_nr2=0, is_label=0;
  int max_color_nr, max_symbol_nr;
  SVG_LEGENDS legends; svg_init_legends(&legends);
  int use_studynr=0;
  int use_tacname=0;

#ifdef MINGW
  // Use Unix/Linux default of two-digit exponents in MinGW on Windows
  _set_output_format(_TWO_DIGIT_EXPONENT);
#endif


  /*
   *  Get arguments
   */
  if(argc==1) {tpcPrintUsage(argv[0], info, stderr); return(1);}
  svgfile[0]=tacfile[0]=(char)0;
  main_title[0]=x_title[0]=y_title[0]=(char)0;
  tac_id[0]=tac_title[0]=(char)0;
  preset_minx=preset_maxx=preset_miny=preset_maxy=nan("");
  /* Options */
  for(ai=1; ai<argc; ai++) if(*argv[ai]=='-') {
    cptr=argv[ai]+1; if(*cptr=='-') cptr++; if(cptr==NULL) continue;
    if(tpcProcessStdOptions(argv[ai], &help, &version, &verbose)==0) continue;
    cptr=argv[ai]+1;
    if(strncasecmp(cptr, "X1=", 3)==0) {
      preset_minx=atof_dpi(cptr+3); continue;
    } else if(strncasecmp(cptr, "X2=", 3)==0) {
      preset_maxx=atof_dpi(cptr+3); continue;
    } else if(strncasecmp(cptr, "Y1=", 3)==0) {
      preset_miny=atof_dpi(cptr+3); continue;
    } else if(strncasecmp(cptr, "Y2=", 3)==0) {
      preset_maxy=atof_dpi(cptr+3); continue;
    } else if(strcasecmp(cptr, "L")==0) {
      continue; // for now, see below
    } else if(strcasecmp(cptr, "S")==0) {
      continue; // for now, see below
    } else if(strcasecmp(cptr, "B")==0) {
      continue; // for now, see below
    } else if(strcasecmp(cptr, "XHTML")==0) {
      do_XHTML=1; continue;
    } else if(strcasecmp(cptr, "NMT")==0) {
      show_main_title=0; continue;
    } else if(strcasecmp(cptr, "NXT")==0) {
      show_x_title=0; continue;
    } else if(strcasecmp(cptr, "NYT")==0) {
      show_y_title=0; continue;
    } else if(strncasecmp(cptr, "MT=", 3)==0) {
      ret=strnCopyClean(main_title, cptr+3, 64);
      if(ret==0) show_main_title=0; else show_main_title=1;
      continue;
    } else if(strncasecmp(cptr, "XT=", 3)==0) {
      ret=strnCopyClean(x_title, cptr+3, 64);
      if(ret==0) show_x_title=0; else show_x_title=1;
      continue;
    } else if(strncasecmp(cptr, "YT=", 3)==0) {
      ret=strnCopyClean(y_title, cptr+3, 64);
      if(ret==0) show_y_title=0; else show_y_title=1;
      continue;
    } else if(strncasecmp(cptr, "LEGEND=", 7)==0) {
      cptr+=7;
      if(strncasecmp(cptr, "YES", 1)==0) {is_label=1; continue;}
      if(strncasecmp(cptr, "NO", 1)==0) {is_label=0; continue;}
      if(strncasecmp(cptr, "AUTO", 1)==0) {is_label=2; continue;}
    } else if(strcasecmp(cptr, "COLOR")==0) {
      color_scale=0; continue;
    } else if(strncasecmp(cptr, "GRAY", 2)==0) {
      color_scale=1; continue;
    } else if(strncasecmp(cptr, "MONOCHROME", 2)==0 || strcasecmp(cptr, "BW")==0) {
      color_scale=2; continue;
    }
    fprintf(stderr, "Error: invalid option '%s'.\n", argv[ai]); 
    return(1);
  } else break;

  /* Print help or version? */
  if(help==2) {tpcHtmlUsage(argv[0], info, ""); return(0);}
  if(help) {tpcPrintUsage(argv[0], info, stdout); return(0);}
  if(version) {tpcPrintBuild(argv[0], stdout); return(0);}

  /* Process other arguments, starting from the first non-option */
  for(; ai<argc; ai++) {
    /* Get SVG filename, and set the argument index for data files */
    if(!svgfile[0]) {strcpy(svgfile, argv[ai]); datarg=ai+1; continue;}
    else if(strcasecmp(argv[ai], "-S")==0) continue;
    else if(strcasecmp(argv[ai], "-B")==0) continue;
    else if(strcasecmp(argv[ai], "-L")==0) continue;
    else {
      /* Check that file(s) do exist */
      if(access(argv[ai], 0) == -1) {
        fprintf(stderr, "Error: %s does not exist.\n", argv[ai]);
        return(2);
      }
      /* Count them */
      fileNr++;
    }
  }

  /* Is something missing? */
  if(!svgfile[0] || fileNr<1) {
    fprintf(stderr, "Error: missing command-line argument; try %s --help\n", argv[0]);
    return(1);
  }

  /* Get the output file format by interpreting the extension */
  cptr=strrchr(svgfile, '.'); if(cptr!=NULL) {
    cptr++;
    if(strcasecmp(cptr, "XHTML")==0) {
      do_XHTML=1;
    } else if(strcasecmp(cptr, "HTM")==0 || strcasecmp(cptr, "HTML")==0) {
      fprintf(stderr, "Warning: file with HTM(L) extension will probably not open\n");
      fprintf(stderr, "         correctly in web browsers; use .xhtml instead.\n");
      do_XHTML=1;
    } else if(strcasecmp(cptr, "SVG")!=0) {
      /* If output file name extension is not .svg, then user probably just forgot to give 
         file name for it. Do not allow overwrite! */
      if(access(svgfile, 0) != -1) {
        fprintf(stderr, "Error: output file extension is not .svg\n");
        return(2);
      }
      /* If output file extension is not .svg, but file does not exist, then just give a warning */
      fprintf(stderr, "Warning: filename with other than .svg extension may not\n");
      fprintf(stderr, "         open correctly in other programs.\n");
    }
  }


  /* In verbose mode print arguments and options */
  if(verbose>1) {
    printf("svgfile := %s\n", svgfile);
    printf("fileNr := %d\n", fileNr);
    printf("do_XHTML := %d\n", do_XHTML);
    if(!isnan(preset_minx)) printf("preset_minx := %g\n", preset_minx);
    if(!isnan(preset_maxx)) printf("preset_maxx := %g\n", preset_maxx);
    if(!isnan(preset_miny)) printf("preset_miny := %g\n", preset_miny);
    if(!isnan(preset_maxy)) printf("preset_maxy := %g\n", preset_maxy);
    printf("show_main_title := %d\n", show_main_title);
    printf("show_x_title := %d\n", show_x_title);
    if(show_x_title) printf("requested_x_title := '%s'\n", x_title);
    printf("show_y_title := %d\n", show_y_title);
    if(show_y_title) printf("requested_y_title := '%s'\n", y_title);
    printf("is_label := %d\n", is_label);
    printf("color_scale := %d\n", color_scale);
    fflush(stdout);
  }


  /*
   *  Read the TAC file(s), file name(s) from command-line
   */
  dftl=(DFTLIST*)calloc(fileNr, sizeof(DFTLIST));
  if(dftl==NULL) {
    fprintf(stderr, "Error: out of memory.\n");
    return(2);
  }
  for(ti=0; ti<fileNr; ti++) {
    dftInit(&dftl[ti].dft);
    dftl[ti].type=-1;
  }
  ti=0;
  for(ai=datarg; ai<argc; ai++) {
    if(*argv[ai]=='-') {
      if(strcasecmp(argv[ai], "-L")==0) dftl[ti].type=1;
      else if(strcasecmp(argv[ai], "-S")==0) dftl[ti].type=2;
      else if(strcasecmp(argv[ai], "-B")==0) dftl[ti].type=0;
    } else {
      strcpy(tacfile, argv[ai]);
      if(ti>0 && dftl[ti].type<0) dftl[ti].type=dftl[ti-1].type;
      if(dftl[ti].type<0) dftl[ti].type=0;
      if(verbose>0) printf("  reading %s\n", tacfile);
      if(dftRead(tacfile, &dftl[ti].dft)) {
        fprintf(stderr, "Error in reading '%s': %s\n", tacfile, dfterrmsg);
        for(ti=0; ti<fileNr; ti++) dftEmpty(&dftl[ti].dft);
        free(dftl);
        return(2);
      }
      /* Set study number, if not yet available */
      if(strlen(dftl[ti].dft.studynr)<1) studynr_from_fname(tacfile, dftl[ti].dft.studynr);
      ti++;
    }
  }
  fileNr=ti;
  if(verbose>1) {
    for(ti=0; ti<fileNr; ti++) {
      printf("voiNr[%d] := %d\n", ti+1, dftl[ti].dft.voiNr);
      printf("frameNr[%d] := %d\n", ti+1, dftl[ti].dft.frameNr);
      printf("type[%d] := %d\n", ti+1, dftl[ti].type);
      printf("studynr[%d] := %s\n", ti+1, dftl[ti].dft.studynr);
    }
  }

  /*
   *  Determine the plot titles
   */
  /* Search for common study number for use as main title */
  if(show_main_title!=0 && !main_title[0]) {
    for(ti=0; ti<fileNr; ti++) {
      if(strlen(dftl[ti].dft.studynr)<1) continue;
      if(strcmp(dftl[ti].dft.studynr, ".")==0) continue;
      if(strlen(main_title)==0) { // take any string specified as study number 
        strcpy(main_title, dftl[ti].dft.studynr); continue;
      } else {
        // if study number was previously set, check that it is the same
        if(strcasecmp(main_title, dftl[ti].dft.studynr)==0) continue;
        // if not the same, then we can not set the main_title
        strcpy(main_title, ""); break;
      }
    }
  }
  /* Search for common activity unit */
  ret=n=CUNIT_UNKNOWN;
  for(ti=0; ti<fileNr; ti++) {
    n=dftUnitId(dftl[ti].dft.unit); if(n!=CUNIT_UNKNOWN) {ret=n; break;}
  }
  if(verbose>1) printf("common_cunit := %d\n", ret);
  if(ret!=CUNIT_UNKNOWN && ret!=CUNIT_UNITLESS)
    if(show_y_title!=0 && !y_title[0]) strcpy(y_title, dftUnit(ret));
  /* If activity unit is known, try to convert all known units to that */
  if(ret!=CUNIT_UNKNOWN)
    for(ti=0; ti<fileNr; ti++) dftUnitConversion(&dftl[ti].dft, ret);
  /* Search for common time unit */
  ret=TUNIT_UNKNOWN;
  for(ti=0; ti<fileNr; ti++) {
    if(dftl[ti].dft.timeunit!=TUNIT_UNKNOWN) {ret=dftl[ti].dft.timeunit; break;}
  }
  if(verbose>1) printf("common_tunit := %d\n", ret);
  if(show_x_title!=0 && !x_title[0]) {
    if(ret==TUNIT_SEC || ret==TUNIT_MIN)
      sprintf(x_title, "Time (%s)", petTunit(ret));
    else if(ret==TUNIT_MM || ret==TUNIT_CM || ret==TUNIT_UM)
      sprintf(x_title, "Distance (%s)", petTunit(ret));
    else if(ret!=TUNIT_UNKNOWN && ret!=DFTUNIT_UNITLESS)
      strcpy(x_title, petTunit(ret));
  }
  /* If time unit is known, try to convert all time units to that */
  if(ret!=TUNIT_UNKNOWN) {
    for(ti=0; ti<fileNr; ti++) dftTimeunitConversion(&dftl[ti].dft, ret);
  }
  if(verbose>1) {
    printf("main_title := '%s'\n", main_title);
    printf("x_title := '%s'\n", x_title);
    printf("y_title := '%s'\n", y_title);
  }


  /* Determine the plot min and max values */
  minx=maxx=miny=maxy=0;
  for(ti=0; ti<fileNr; ti++) {
    ret=dftMinMax(&dftl[ti].dft, &tx1, &tx2, &ty1, &ty2);
    if(ret) {
      fprintf(stderr, "Error: insufficient data.\n");
      if(verbose>1) dftPrint(&dftl[ti].dft);
      for(ti=0; ti<fileNr; ti++) dftEmpty(&dftl[ti].dft);
      free(dftl);
      return(5);
    }
    if(ti==0) {
      minx=tx1; maxx=tx2; miny=ty1; maxy=ty2;
    } else {
      if(minx>tx1) minx=tx1;
      if(maxx<tx2) maxx=tx2;
      if(miny>ty1) miny=ty1;
      if(maxy<ty2) maxy=ty2;
    }
  }
  if(verbose>3) printf("minx:=%g\nmaxx:=%g\nminy:=%g\nmaxy:=%g\n", minx, maxx, miny, maxy);
  /* Replace x range by preset range if acceptable */
  if(!isnan(preset_minx)) tx1=preset_minx; else tx1=minx;
  if(!isnan(preset_maxx)) tx2=preset_maxx; else tx2=maxx;
  if(tx1<tx2) {minx=tx1; maxx=tx2;}
  /* If x range was changed, then check the y data range inside x range */
  if(!isnan(preset_minx) || !isnan(preset_maxx)) {
    miny=maxy=nan("");
    for(ti=0; ti<fileNr; ti++) {
      ret=dftMaxY(&dftl[ti].dft, minx, maxx, &ty1, &ty2);
      if(ret) {
        fprintf(stderr, "Warning: insufficient data in the range.\n");
        continue;
      }
      if(isnan(miny)) miny=ty1; else if(miny>ty1) miny=ty1; 
      if(isnan(maxy)) maxy=ty2; else if(maxy<ty2) maxy=ty2; 
    }
    if(verbose>3) printf("in_range_miny:=%g\nin_range_maxy:=%g\n", miny, maxy);
    if(isnan(miny) || isnan(maxy)) {
      fprintf(stderr, "Error: insufficient data in the range.\n");
      for(ti=0; ti<fileNr; ti++) dftEmpty(&dftl[ti].dft);
      free(dftl);
      return(5);
    }
  }

  /* Replace y range by pre-set range if acceptable */
  if(!isnan(preset_miny)) ty1=preset_miny; else ty1=miny;
  if(!isnan(preset_maxy)) ty2=preset_maxy; else ty2=maxy;
  if(ty1<ty2) {miny=ty1; maxy=ty2;}
  if(verbose>2) printf("minx:=%g\nmaxx:=%g\nminy:=%g\nmaxy:=%g\n", minx, maxx, miny, maxy);

  /* Calculate the axis ticks */
  viewports.x.min=minx; viewports.x.max=maxx;
  if(preset_minx==minx) viewports.x.fixed_min=1; else viewports.x.fixed_min=0;
  if(preset_maxx==maxx) viewports.x.fixed_max=1; else viewports.x.fixed_max=0;
  viewports.y.min=miny; viewports.y.max=maxy;
  if(preset_miny==miny) viewports.y.fixed_min=1; else viewports.y.fixed_min=0;
  if(preset_maxy==maxy) viewports.y.fixed_max=1; else viewports.y.fixed_max=0;
  ret=svg_calculate_axes(&viewports, verbose-8);
  if(ret) {
    fprintf(stderr, "Error (%d) in setting axis scales.\n", ret);
    for(ti=0; ti<fileNr; ti++) dftEmpty(&dftl[ti].dft);
    free(dftl);
    return(2);
  }
  if(verbose>3) {
    printf("viewports.x.min := %g\n", viewports.x.min);
    printf("viewports.x.max := %g\n", viewports.x.max);
    printf("viewports.y.min := %g\n", viewports.y.min);
    printf("viewports.y.max := %g\n", viewports.y.max);
  }

  /* Check if TAC titles should contain studynr and/or TAC name */
  // search the first existing studynr
  for(ti=n=0; ti<fileNr; ti++, n++)
    if(strlen(dftl[ti].dft.studynr)>0 && strcmp(dftl[ti].dft.studynr, ".")!=0) break;
  // search if there exists a different studynr
  for(; ti<fileNr; ti++)
    if(strlen(dftl[ti].dft.studynr)>0 && strcmp(dftl[ti].dft.studynr, ".")!=0)
      if(strcasecmp(dftl[ti].dft.studynr, dftl[n].dft.studynr)!=0) {
        if(verbose>2)
          printf("  different studynr: '%s' vs '%s'\n", dftl[n].dft.studynr, dftl[ti].dft.studynr);
        use_studynr=1; break;
      }
  // search if there exists a different region name
  for(ti=0; ti<fileNr; ti++) {
    if(ti==0) ri=1; else ri=0;
    for(; ri<dftl[ti].dft.voiNr; ri++)
      if(strcasecmp(dftl[ti].dft.voi[ri].name, dftl[0].dft.voi[0].name)!=0) {use_tacname=1; break;}
  }
  /* If no different stydynrs or tacnames were found, then use tacname anyway
     but turn legends off if auto legends were requested */
  if(use_studynr==0 && use_tacname==0) {
    use_tacname=1; if(is_label==2) is_label=0;
  } else {
    if(is_label==2) is_label=1;
  }

  /* Set the plot window and window area sizes */
  ret=svg_define_viewports(0, 0, strlen(main_title), strlen(y_title),
    strlen(x_title), is_label, &viewports, verbose-9);
  if(ret) {
    fprintf(stderr, "Error (%d) in setting viewports.\n", ret);
    for(ti=0; ti<fileNr; ti++) dftEmpty(&dftl[ti].dft);
    free(dftl);
    return(2);
  }

  /*
   *  If file exists, then backup it first
   */
  if((ret=backupExistingFile(svgfile, NULL, errmsg))!=0) {
    fprintf(stderr, "Error: %s.\n", errmsg);
    for(ti=0; ti<fileNr; ti++) dftEmpty(&dftl[ti].dft);
    free(dftl);
    return(10);
  }

  /*
   *  Initiate graphics file
   */
  if(verbose>1) printf("  initiating graphics file %s\n", svgfile);
  if(do_XHTML) {
    /* Initiate XHTML file for one or more inline SVG plots */
    fp_svg=svg_xhtml_initiate(svgfile, main_title, errmsg, verbose-8);
    if(fp_svg==NULL) {
      fprintf(stderr, "Error in creating %s: %s\n", svgfile, errmsg);
      for(ti=0; ti<fileNr; ti++) dftEmpty(&dftl[ti].dft);
      free(dftl);
      return(11);
    }
    /* Initiate inline SVG */
    ret=svg_xhtml_svg_open(fp_svg, 0, 0, &viewports, errmsg, verbose-8);
    if(fp_svg==NULL) {
      fprintf(stderr, "Error in writing SVG in %s: %s\n", svgfile, errmsg);
      for(ti=0; ti<fileNr; ti++) dftEmpty(&dftl[ti].dft);
      free(dftl);
      return(11);
    }
  } else {
    /* Initiate SVG file */
    fp_svg=svg_initiate(svgfile, 0, 0, &viewports, errmsg, verbose-8);
    if(fp_svg==NULL) {
      fprintf(stderr, "Error in creating %s: %s\n", svgfile, errmsg);
      for(ti=0; ti<fileNr; ti++) dftEmpty(&dftl[ti].dft);
      free(dftl);
      return(11);
    }
  }

  /*
   *  Put the graph title into its own viewport
   */
  if(verbose>1) printf("creating main title\n");
  ret=svg_create_main_title(fp_svg, main_title, "", &viewports, errmsg, verbose-5);
  if(ret) {
    fprintf(stderr, "Error in creating %s: %s\n", svgfile, errmsg);
    fclose(fp_svg); remove(svgfile);
    for(ti=0; ti<fileNr; ti++) dftEmpty(&dftl[ti].dft);
    free(dftl);
    return(13);
  }

  /*
   *  Put the axis titles into their own viewports
   */
  if(verbose>1) printf("creating axis titles\n");
  /* y axis */
  ret=svg_create_yaxis_title(fp_svg, y_title, &viewports, errmsg, verbose-4);
  if(ret) {
    fprintf(stderr, "Error in creating %s: %s\n", svgfile, errmsg);
    fclose(fp_svg); remove(svgfile);
    for(ti=0; ti<fileNr; ti++) dftEmpty(&dftl[ti].dft);
    free(dftl);
    return(14);
  }
  /* x axis */
  ret=svg_create_xaxis_title(fp_svg, x_title, &viewports, errmsg, verbose-6);
  if(ret) {
    fprintf(stderr, "Error in creating %s: %s\n", svgfile, errmsg);
    fclose(fp_svg); remove(svgfile);
    for(ti=0; ti<fileNr; ti++) dftEmpty(&dftl[ti].dft);
    free(dftl);
    return(13);
  }

  /*
   *  Put the plot into its own viewport
   */
  if(verbose>1) printf("starting plot viewport\n");
  ret=svg_start_plot_viewport(fp_svg, &viewports, errmsg, verbose-9);
  if(ret) {
    fprintf(stderr, "Error in creating %s: %s\n", svgfile, errmsg);
    fclose(fp_svg); remove(svgfile);
    for(ti=0; ti<fileNr; ti++) dftEmpty(&dftl[ti].dft);
    free(dftl);
    return(21);
  }

  /*
   *  Start coordinate area viewport
   */
  if(verbose>1) printf("starting coordinate area viewport\n");
  ret=svg_start_coordinate_viewport(fp_svg, &viewports, errmsg, verbose-9);
  if(ret) {
    fprintf(stderr, "Error in creating %s: %s\n", svgfile, errmsg);
    fclose(fp_svg); remove(svgfile);
    for(ti=0; ti<fileNr; ti++) dftEmpty(&dftl[ti].dft);
    free(dftl);
    return(22);
  }

  /* Write plot axes */
  if(verbose>1) printf("writing axes\n");
  ret=svg_write_axes(fp_svg, &viewports, errmsg, verbose-9);
  if(ret) {
    fprintf(stderr, "Error in creating %s: %s\n", svgfile, errmsg);
    fclose(fp_svg); remove(svgfile);
    for(ti=0; ti<fileNr; ti++) dftEmpty(&dftl[ti].dft); 
    free(dftl);
    return(23);
  }

  /*
   *  Plot the TACs
   */
  if(verbose>1) printf("plotting the TACs\n");
  if(color_scale==2) { // monochrome
    max_color_nr=1;
  } else { // color and grayscale
    max_color_nr=0; while(svgColorName(max_color_nr)!=NULL) max_color_nr++;
  }
  if(verbose>1) printf("max_color_nr := %d\n", max_color_nr);
  max_symbol_nr=0; while(svgSymbolName(max_symbol_nr)!=NULL) max_symbol_nr++;
  if(verbose>1) printf("max_symbol_nr := %d\n", max_symbol_nr);
  /* Start with filled symbols if colors are used, and with open symbols in case of monochrome */
  if(color_scale==0) symbol_fill=1; else symbol_fill=0;
  /* Check if equal nr of curves with symbols and lines, and nothing else,
     then we will use same color for symbols and lines, and only make one common legend */
  n=nl=ns=0;
  for(ti=0, n=0; ti<fileNr; ti++) for(ri=0; ri<dftl[ti].dft.voiNr; ri++) {
    n++; if(dftl[ti].type==1) nl++; else if(dftl[ti].type==2) ns++;}
  if(nl==ns && n==(nl+ns)) plot_fit=1;
  if(verbose>1) printf("plot_fit := %d\n", plot_fit);
  /* plot */
  for(ti=0, n=0, color_nr=color_nr2=0, symbol_nr=0; ti<fileNr; ti++) { // each TAC file
    if(verbose>8) printf("ti := %d\n", ti);
    if(dftl[ti].dft.voiNr>1) color_nr=color_nr2=1; //else color_nr=0;
    if(verbose>5) printf("type := %d\n", dftl[ti].type);
    for(ri=0; ri<dftl[ti].dft.voiNr; ri++) { // each TAC in this TAC file
      if(verbose>8) {
        printf("  ri := %d\n", ri);
        printf("  color1 := %d\n", color_nr%max_color_nr);
        printf("  color2 := %d\n", color_nr2%max_color_nr);
      }
      sprintf(tac_id, "plot_%d", n);
      if(use_studynr==0) strcpy(tac_title, "");
      else strcpy(tac_title, dftl[ti].dft.studynr);
      if(use_studynr!=0 && use_tacname!=0) strcat(tac_title, ": ");
      if(use_tacname!=0) {
        rnameRmDots(dftl[ti].dft.voi[ri].name, temp);
        strcat(tac_title, temp);
      }

      if(plot_fit==0 || dftl[ti].type!=1) { // using color1
        ret=svg_write_tac(fp_svg, &viewports, dftl[ti].type, tac_id, tac_title,
            dftl[ti].dft.x, dftl[ti].dft.voi[ri].y, dftl[ti].dft.frameNr,
            svgColorName(color_nr%max_color_nr), symbol_nr%max_symbol_nr, symbol_fill,
            errmsg, verbose-9);
      } else { // using color2
        ret=svg_write_tac(fp_svg, &viewports, dftl[ti].type, tac_id, tac_title,
            dftl[ti].dft.x, dftl[ti].dft.voi[ri].y, dftl[ti].dft.frameNr,
            svgColorName(color_nr2%max_color_nr), symbol_nr%max_symbol_nr, symbol_fill,
            errmsg, verbose-9);
      }
      if(ret) {
        fprintf(stderr, "Error in creating %s: %s\n", svgfile, errmsg);
        fclose(fp_svg); remove(svgfile);
        for(ti=0; ti<fileNr; ti++) dftEmpty(&dftl[ti].dft); 
        free(dftl);
        svg_legend_empty(&legends); return(22);
      }

      /* Set legend too, if requested */
      if(is_label!=0) {
        if(plot_fit==0) // not fit plotting mode, thus plot legends as requested
          svg_legend_add(&legends, dftl[ti].type, symbol_nr%max_symbol_nr, symbol_fill,
                         color_nr%max_color_nr, tac_title);
        else if(dftl[ti].type!=1) // fit plotting mode, plot legends with both symbols with lines
          svg_legend_add(&legends, 0, symbol_nr%max_symbol_nr, symbol_fill,
                         color_nr%max_color_nr, tac_title);
      }

      /* Prepare for the next TAC */
      if(plot_fit==0) {
        color_nr++;
        symbol_nr++;
      } else {
        if(dftl[ti].type!=1) {
          color_nr++;
          symbol_nr++;
        } else {
          color_nr2++;
        }
      }

      /* If black-and-white and all symbols consumed, then switch between open and filled symbols */
      if(color_scale==2 && symbol_nr==max_symbol_nr) {
        if(symbol_fill==0) symbol_fill=1; else symbol_fill=0;
        symbol_nr=0;
      }
      /* If all colors and symbols consumed, then switch between open and filled symbols */
      if(color_scale==0 && color_nr==max_color_nr && symbol_nr==max_symbol_nr) {
        if(symbol_fill==0) symbol_fill=1; else symbol_fill=0;
        color_nr=symbol_nr=0; 
      }
      n++;
    } /* next region TAC */
  } /* next DFT */


  /* Close the coordinate viewport */
  if(verbose>1) printf("closing coordinate area viewport\n");
  ret=svg_end_coordinate_viewport(fp_svg, errmsg, verbose-8);
  if(ret) {
    fprintf(stderr, "Error in creating %s: %s\n", svgfile, errmsg);
    fclose(fp_svg); remove(svgfile);
    for(ti=0; ti<fileNr; ti++) dftEmpty(&dftl[ti].dft); 
    free(dftl);
    svg_legend_empty(&legends); return(28);
  }

  /* Write the axis ticks */
  if(verbose>1) printf("writing axis ticks\n");
  if(svg_write_xticks(fp_svg, &viewports, errmsg, verbose-8)!=0) {
    fprintf(stderr, "Error in creating %s: %s\n", svgfile, errmsg);
    fclose(fp_svg); remove(svgfile);
    for(ti=0; ti<fileNr; ti++) dftEmpty(&dftl[ti].dft); 
    free(dftl);
    svg_legend_empty(&legends); return(29);
  }
  if(svg_write_yticks(fp_svg, &viewports, errmsg, verbose-8)!=0) {
    fprintf(stderr, "Error in creating %s: %s\n", svgfile, errmsg);
    fclose(fp_svg); remove(svgfile);
    for(ti=0; ti<fileNr; ti++) dftEmpty(&dftl[ti].dft); 
    free(dftl);
    svg_legend_empty(&legends); return(30);
  }

  /* Close the plot viewport */
  if(verbose>1) printf("closing plot viewport\n");
  ret=svg_end_plot_viewport(fp_svg, errmsg, verbose-7);
  if(ret) {
    fprintf(stderr, "Error in creating %s: %s\n", svgfile, errmsg);
    fclose(fp_svg); remove(svgfile);
    for(ti=0; ti<fileNr; ti++) dftEmpty(&dftl[ti].dft); 
    free(dftl);
    svg_legend_empty(&legends); return(41);
  }

  /*
   *  Make the plot legends into their own viewport
   */
  if(viewports.label_area_viewport.is!=0) {
    if(verbose>1) printf("creating plot legends\n");
    ret=svg_create_legends(fp_svg, &viewports, &legends, errmsg, verbose-6);
    if(ret) {
      fprintf(stderr, "Error in creating %s: %s\n", svgfile, errmsg);
      fclose(fp_svg); remove(svgfile);
      for(ti=0; ti<fileNr; ti++) dftEmpty(&dftl[ti].dft); 
      free(dftl);
      svg_legend_empty(&legends); return(51);
    }
  }
  svg_legend_empty(&legends);


  /* TAC data is not needed anymore */
  for(ti=0; ti<fileNr; ti++) dftEmpty(&dftl[ti].dft); 
  free(dftl);


  /* Close the SVG file */
  if(verbose>1) printf("closing the file\n");
  if(do_XHTML) {
    ret=svg_xhtml_svg_close(fp_svg, errmsg, verbose-8);
    if(ret) {
      fprintf(stderr, "Error in creating %s: %s\n", svgfile, errmsg);
      return(61);
    }
    ret=svg_xhtml_close(fp_svg, errmsg, verbose-8);
    if(ret) {
      fprintf(stderr, "Error in creating %s: %s\n", svgfile, errmsg);
      return(62);
    }
  } else {
    ret=svg_close(fp_svg, errmsg, verbose-6);
    if(ret) {
      fprintf(stderr, "Error in creating %s: %s\n", svgfile, errmsg);
      return(61);
    }
  }
  if(verbose>0) printf("  %s written.\n", svgfile);

  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/// @endcond
