/** @file halflife.c
 *  @brief Listing of PET isotope half-lives.
 *  @copyright (c) Turku PET Centre
 *  @author Vesa Oikonen
 */
/// @cond
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <time.h>
#include <string.h>
/*****************************************************************************/
#include "tpcextensions.h"
#include "tpcisotope.h"
/*****************************************************************************/

/*****************************************************************************/
/* Local functions */

/*****************************************************************************/

/*****************************************************************************/
static char *info[] = {
  "List the PET isotopes and their half-lives (min) that are identified by",
  "most TPC software.",
  " ",
  "Usage: @P [options] [> outputfile]",
  " ",
  "Options:",
  " -i=<Isotope code>",
  "     Show only the given isotope.",
  " -lambda",
  "     Lambda value (1/min) is listed.",
  " -pf",
  "     Positron fraction (branching ratio) is listed.",
  " -sec",
  "     Half-lives are listed in seconds and lambdas 1/sec.",
  " -stdoptions", // List standard options like --help, -v, etc
  " ",
  "See also: tacdecay, imgdecay, tactime, ecattime, dcftime",
  " ",
  "Keywords: physical decay, decay correction, halflife, isotope",
  0};
/*****************************************************************************/

/*****************************************************************************/
/* Turn on the globbing of the command line, since it is disabled by default in
   mingw-w64 (_dowildcard=0); in MinGW32 define _CRT_glob instead, if necessary;
   In Unix&Linux wildcard command line processing is enabled by default. */
/*
#undef _CRT_glob
#define _CRT_glob -1
*/
int _dowildcard = -1;
/*****************************************************************************/

/*****************************************************************************/
/**
 *  Main
 */
int main(int argc, char **argv)
{
  int ai, help=0, version=0, verbose=1;
  int do_lambda=0;
  int do_br=0;
  int tunit=UNIT_MIN, isotope=ISOTOPE_UNKNOWN;
  double f;


  /*
   *  Get arguments
   */
  /* Options */
  for(ai=1; ai<argc; ai++) if(*argv[ai]=='-') {
    if(tpcProcessStdOptions(argv[ai], &help, &version, &verbose)==0) continue;
    char *cptr=argv[ai]+1;
    if(strncasecmp(cptr, "I=", 2)==0) {
      cptr+=2; isotope=isotopeIdentify(cptr);
      if(isotope!=ISOTOPE_UNKNOWN) continue;
      fprintf(stderr, "Error: invalid isotope '%s'\n", cptr); return(1);
    } else if(strncasecmp(cptr, "LAMBDA", 3)==0) {
      do_lambda=1; continue;
    } else if(strcasecmp(cptr, "PF")==0 || strcasecmp(cptr, "BR")==0) {
      do_br=1; continue;
    } else if(strncasecmp(cptr, "SECONDS", 3)==0) {
      tunit=UNIT_SEC; continue;
    }
    fprintf(stderr, "Error: invalid option '%s'\n", argv[ai]);
    return(1);
  }
  /* Non-options */
  for(ai=1; ai<argc; ai++) if(*argv[ai]!='-') {
    fprintf(stderr, "Error: invalid argument '%s'\n", argv[ai]);
    return(1);
  }
  /* Print help or version? */
  if(help==2) {tpcHtmlUsage(argv[0], info, ""); return(0);}
  if(help) {tpcPrintUsage(argv[0], info, stdout); return(0);}
  if(version) {tpcPrintBuild(argv[0], stdout); return(0);}
  /* In verbose mode print arguments and options */
  if(verbose>1) {
    for(ai=0; ai<argc; ai++) printf("%s ", argv[ai]); 
    printf("\n");
    printf("do_lambda := %d\n", do_lambda);
    printf("do_br := %d\n", do_br);
    printf("isotope := %s\n", isotopeName(isotope));
    printf("tunit := %d\n", tunit);
  }

  /* Print specified isotope, or all isotopes */
  f=1.0; if(tunit==UNIT_SEC) f=60.0;
  if(isotope!=ISOTOPE_UNKNOWN) {
    fprintf(stdout, "isotope := %s\n", isotopeName(isotope));
    fprintf(stdout, "halflife := %g\n", f*isotopeHalflife(isotope));
    if(do_lambda!=0) fprintf(stdout, "lambda := %.3e\n", lambdaFromIsotope(isotope)/f);
    if(do_br!=0) fprintf(stdout, "positron_fraction := %g\n", isotopeBranching(isotope));
  } else {
    int i=1;
    while(!isnan(isotopeHalflife(i))) {
      fprintf(stdout, "isotope := %s\n", isotopeName(i));
      fprintf(stdout, "halflife := %g\n", f*isotopeHalflife(i));
      if(do_lambda!=0) fprintf(stdout, "lambda := %.3e\n", lambdaFromIsotope(i)/f);
      if(do_br!=0) fprintf(stdout, "positron_fraction := %g\n", isotopeBranching(i));
      i++;
    }
  }
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/// @endcond
