/** @file iftadd.c
 *  @brief Deletes the items with specified key name from an
           Interfile-type file.
 *  @copyright (c) Turku PET Centre
 *  @author Vesa Oikonen
 */
/// @cond
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
/*****************************************************************************/
#include "tpcextensions.h"
#include "tpcift.h"
/*****************************************************************************/

/*****************************************************************************/
static char *info[] = {
  "Deletes the items with specified key names from an Interfile-type file.",
  "Notice that also all lines containing no key are deleted.",
  " ",
  "Usage: @P [Options] filename key(s)",
  " ",
  "Options:",
  " -stdoptions", // List standard options like --help, -v, etc
  " ",
  "Example:",
  "     @P iea345.hdr patient_name patient_id",
  " ",
  "See also: iftlist, iftadd, iftren, iftedit, iftisval, taclist, nii_lhdr",
  " ",
  "Keywords: header, IFT, tool",
  0};
/*****************************************************************************/

/*****************************************************************************/
/* Turn on the globbing of the command line, since it is disabled by default in
   mingw-w64 (_dowildcard=0); in MinGW32 define _CRT_glob instead, if necessary;
   In Unix&Linux wildcard command line processing is enabled by default. */
/*
#undef _CRT_glob
#define _CRT_glob -1
*/
int _dowildcard = -1;
/*****************************************************************************/

/*****************************************************************************/
/**
 *  Main
 */
int main(int argc, char **argv)
{
  int   ai, help=0, version=0, verbose=1;
  int   ret, keyNr=0, fki=0;
  char  iftfile[FILENAME_MAX];
  IFT   ift;
  FILE *fp;


  /*
   *  Get arguments
   */
  if(argc==1) {tpcPrintUsage(argv[0], info, stderr); return(1);}
  iftInit(&ift);
  iftfile[0]=(char)0;
  /* Options */
  for(ai=1; ai<argc; ai++) if(*argv[ai]=='-') {
    if(tpcProcessStdOptions(argv[ai], &help, &version, &verbose)==0) continue;
    fprintf(stderr, "Error: invalid option '%s'.\n", argv[ai]);
    return(1);
  } else break; // later arguments may start with '-'

  /* Print help or version? */
  if(help==2) {tpcHtmlUsage(argv[0], info, ""); return(0);}
  if(help) {tpcPrintUsage(argv[0], info, stdout); return(0);}
  if(version) {tpcPrintBuild(argv[0], stdout); return(0);}

  TPCSTATUS status; statusInit(&status);
  statusSet(&status, __func__, __FILE__, __LINE__, TPCERROR_OK);
  status.verbose=verbose-1;

  /* Process other arguments, starting from the first non-option */
  keyNr=0;
  for(; ai<argc; ai++) {
    if(!iftfile[0]) {strcpy(iftfile, argv[ai]); continue;}
    if(keyNr==0) fki=ai; 
    keyNr++;
  }

  /* Check that we got the filename and at least one key */
  if(!iftfile[0] || keyNr<1) {tpcPrintUsage(argv[0], info, stderr); return(1);}

  /* In verbose mode print arguments and options */
  if(verbose>1) {
    printf("iftfile := %s\n", iftfile);
    printf("keyNr := %d\n", keyNr);
  }


  /*
   *  Read the IFT file
   */
  if(verbose>1) printf("reading %s\n", iftfile);
  fp=fopen(iftfile, "r"); if(fp==NULL) {
    fprintf(stderr, "Error: cannot open file %s\n", iftfile);
    return(2);
  }
  ret=iftRead(&ift, fp, 1, 1, &status); fclose(fp);
  if(ret) {
    fprintf(stderr, "Error (%d): %s\n", ret, errorMsg(status.error));
    iftFree(&ift); return(2);
  }
  if(verbose>2) printf("list_size: %d item(s)\n", ift.keyNr);


  /*
   *  Delete the items with the given key(s)
   */
  int n=0;
  for(ai=fki; ai<argc; ai++) {
    if(verbose>3) printf("key := '%s'\n", argv[ai]);
    if(iftFindNrOfKeys(&ift, argv[ai])<1) {
      if(keyNr>1) fprintf(stderr, "Warning: key '%s' was not found.\n", argv[ai]);
      continue;
    }
    iftDeleteKey(&ift, argv[ai]);
    n++;
  } // next key
  if(n==0) {
    if(keyNr==1) fprintf(stderr, "Error: key not found.\n");
    else fprintf(stderr, "Error: none of keys found.\n");
    iftFree(&ift); return(3);
  }
  if(ift.keyNr==0) {
    fprintf(stderr, "Error: all items would be removed.\n");
    iftFree(&ift); return(4);
  }


  /*
   *  Write the modified IFT contents.
   */
  if(verbose>1) printf("writing modified IFT in %s\n", iftfile);
  fp=fopen(iftfile, "w"); if(fp==NULL) {
    fprintf(stderr, "Error: cannot open file %s\n", iftfile);
    iftFree(&ift); return(11);
  }
  ret=iftWrite(&ift, fp, &status); fclose(fp);
  if(ret!=TPCERROR_OK) {
    fprintf(stderr, "Error: %s\n", errorMsg(status.error));
    iftFree(&ift); return(12);
  }
  iftFree(&ift);

  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/// @endcond
