/** @file iftisval.c
 *  @brief Check if specified key exists in an Interfile-type file, and if
           it has the required value.
 *  @copyright (c) Turku PET Centre
 *  @author Vesa Oikonen
 */
/// @cond
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
/*****************************************************************************/
#include "tpcextensions.h"
#include "tpcift.h"
/*****************************************************************************/

/*****************************************************************************/
static char *info[] = {
  "Check whether specified Interfile-type (IFT) key and value exist in",
  "the header file, for example:",
  "     CALIBRATED := Yes",
  "Key and value are case-insensitive.",
  "Program returns 0, if key, and optional value, are found,",
  "code 10, if key is not found, and 11, if key was found but without",
  "matching value.",
  " ",
  "Usage: @P [options] filename [key [value]]",
  " ",
  "Options:",
  "-lt | -gt | -abs=<limit>",
  "     Values are tested numerically (not as text), to be less (-lt) or",
  "     greater (-gt) than the given value, or the absolute difference is not",
  "     allowed to exceed the specified limit (-abs=<limit>)",
  " -stdoptions", // List standard options like --help, -v, etc
  " ",
  "Example 1:",
  "     @P iea345ab.kbq calibrated yes",
  "Example 2:",
  "     @P -abs=0.02 header.dat halflife 2.05",
  " ",
  "See also: iftadd, iftdel, iftlist, iftmatch, iftvalc",
  " ",
  "Keywords: header, IFT, tool",
  0};
/*****************************************************************************/

/*****************************************************************************/
/* Turn on the globbing of the command line, since it is disabled by default in
   mingw-w64 (_dowildcard=0); in MinGW32 define _CRT_glob instead, if necessary;
   In Unix&Linux wildcard command line processing is enabled by default. */
/*
#undef _CRT_glob
#define _CRT_glob -1
*/
int _dowildcard = -1;
/*****************************************************************************/

/*****************************************************************************/
/**
 *  Main
 */
int main(int argc, char **argv)
{
  int ai, help=0, version=0, verbose=1;
  int ret, m, li;
  double test_abs=-1.0, v1, v2;
  int test_lt=0, test_gt=0;
  char *cptr, iftfile[FILENAME_MAX];
  IFT ift;
  char *key, *value;

  /*
   *  Get arguments
   */
  if(argc==1) {tpcPrintUsage(argv[0], info, stderr); return(1);}
  iftInit(&ift); iftfile[0]=(char)0; key=value=NULL;
  /* Options */
  for(ai=1; ai<argc; ai++) if(*argv[ai]=='-') {
    if(tpcProcessStdOptions(argv[ai], &help, &version, &verbose)==0) continue;
    cptr=argv[ai]+1;
    if(strncasecmp(cptr, "ABS=", 4)==0) {
      test_abs=atofVerified(cptr+4); if(!isnan(test_abs)) continue;
    } else if(strcasecmp(cptr, "LT")==0) {
      test_lt=1; continue;
    } else if(strcasecmp(cptr, "GT")==0) {
      test_gt=1; continue;
    }
    fprintf(stderr, "Error: invalid option '%s'\n", argv[ai]);
    free(key); free(value);
    return(1);
  } else break; // later arguments may start with '-'

  /* Print help or version? */
  if(help==2) {tpcHtmlUsage(argv[0], info, ""); return(0);}
  if(help) {tpcPrintUsage(argv[0], info, stdout); return(0);}
  if(version) {tpcPrintBuild(argv[0], stdout); return(0);}

  TPCSTATUS status; statusInit(&status);
  statusSet(&status, __func__, __FILE__, __LINE__, TPCERROR_OK);
  status.verbose=verbose;

  /* Process other arguments, starting from the first non-option */
  for(; ai<argc; ai++) {
    if(!iftfile[0]) {
      strcpy(iftfile, argv[ai]); continue;
    } else if(key==NULL) {
      key=strdup(argv[ai]); continue;
    } else if(value==NULL) {
      value=strdup(argv[ai]); continue;
    }
    fprintf(stderr, "Error: invalid argument '%s'.\n", argv[ai]);
    free(key); free(value);
    return(1);
  }
  /* Is something missing? */
  if(key==NULL) {tpcPrintUsage(argv[0], info, stdout); return(1);}

  /* In verbose mode print arguments and options */
  if(verbose>1) {
    for(ai=0; ai<argc; ai++)
      printf("%s ", argv[ai]); 
    printf("\n");
    printf("iftfile := %s\n", iftfile);
    printf("key := %s\n", key);
    printf("value := %s\n", value);
    if(test_abs>=0.0) printf("test_abs := %g\n", test_abs);
    printf("test_lt := %d\n", test_lt);
    printf("test_gt := %d\n", test_gt);
  }

  /* If testing for numerical values, then get the test value */
  ret=0; if(test_abs>=0.0) ret++; if(test_lt) ret++; if(test_gt) ret++;
  if(ret>1) {
    fprintf(stderr, "Error: invalid combination of options.\n");
    free(key); free(value); return(1);
  }
  if(ret==1) {
    if(value==NULL) {
      fprintf(stderr,
        "Error: value must be given with options -lt, -gt, and -abs.\n");
      free(key); return(1);
    }
    ret=atofCheck(value, &v1);
    if(ret!=0) {
      fprintf(stderr, "Error: invalid numerical value for testing.\n");
      free(key); free(value); return(1);
    }
    if(verbose>1) printf("numerical_value := %g\n", v1);
  } 

  /*
   *  Read IFT file
   */
  if(verbose>1) printf("reading %s\n", iftfile);
  FILE *fp=fopen(iftfile, "r"); if(fp==NULL) {
    fprintf(stderr, "Error: cannot open file %s\n", iftfile);
    free(key); free(value); return(2);
  }
  ret=iftRead(&ift, fp, 1, 1, &status); fclose(fp);
  if(ret) {
    fprintf(stderr, "Error (%d): %s\n", ret, errorMsg(status.error));
    free(key); free(value); iftFree(&ift); return(2);
  }
  if(verbose>2) printf("list size: %d item(s)\n", ift.keyNr);


  /*
   *  Try to find the key
   */
  if(verbose>1) printf("searching for '%s'\n", key);
  m=0;
  do {
    li=iftFindKey(&ift, key, m);
    if(li<0) {
      if(m==0 && verbose>0)
        fprintf(stderr, "  key \"%s\" was not found.\n", key);
      else break;
      iftFree(&ift); free(key); free(value); return(10);
    }
    m=li+1;

    if(value==NULL) {
      /* Key was found, value not specified, then everything is fine */
      if(verbose>0)
        fprintf(stdout, "  key \"%s\" is found in %s\n", key, iftfile);
      iftFree(&ift); free(key); free(value); return(0);
    }

    /* If value was specified, then check if they match */

    /* If numerical values are to be tested, then get the number */
    if(test_abs>=0 || test_lt>0 || test_gt>0) {
      ret=doubleGetWithUnit(ift.item[li].value, &v2, NULL);
      /* If key value was not numerical, try to find other matching key */
      if(ret!=0) continue;
      if(verbose>1) printf("  keys_numerical_value := %g\n", v2);
    }

    /* Test for abs difference, if requested */
    if(test_abs>=0) {
      if(verbose>2) printf("testing abs difference between %g and %g\n", v1, v2);
      if(fabs(v1-v2)<=test_abs) {
        /* Key was found, and value is matching, then everything is fine */
        if(verbose>0)
          fprintf(stdout, "  key \"%s\" with matching value %g is found in %s\n",
                  key, v2, iftfile);
        iftFree(&ift); free(key); free(value); return(0);
      }
      /* Else, try to find other matching key */
      continue;
    }
    /* Test for less than value, if requested */
    if(test_lt) {
      if(verbose>2) printf("testing whether %g < %g\n", v2, v1);
      if(v2<v1) {
        /* Key was found, and value is matching, then everything is fine */
        if(verbose>0) fprintf(stdout, 
              "  key \"%s\" with value %g less than %g is found in %s\n",
              key, v2, v1, iftfile);
        iftFree(&ift); free(key); free(value); return(0);
      }
      /* Else, try to find other matching key */
      continue;
    }
    /* Test for larger than value, if requested */
    if(test_gt) {
      if(verbose>2) printf("testing whether %g > %g\n", v2, v1);
      if(v2>v1) {
        /* Key was found, and value is matching, then everything is fine */
        if(verbose>0) fprintf(stdout, 
              "  key \"%s\" with value %g greater than %g is found in %s\n",
              key, v2, v1, iftfile);
        iftFree(&ift); free(key); free(value); return(0);
      }
      /* Else, try to find other matching key */
      continue;
    }

    /* Check if strings do match */
    if(strcasecmp(value, ift.item[li].value)==0) {
      /* Key was found, and value is matching, then everything is fine */
      if(verbose>0)
        fprintf(stdout, "  key \"%s\" with matching value \"%s\" is found in %s\n",
                key, value, iftfile);
      iftFree(&ift); free(key); free(value); return(0);
    }

  } while(li>=0); // try if next key would have matching value

  /* We are here only if key was found but value did not match */
  if(verbose>0) {
    if(test_lt>0)
      fprintf(stdout, "  key \"%s\" had not value less than \"%s\" in %s\n",
              key, value, iftfile);
    else if(test_gt>0)
      fprintf(stdout, "  key \"%s\" had not value larger than \"%s\" in %s\n",
              key, value, iftfile);
    else
      fprintf(stdout, "  key \"%s\" with value \"%s\" was not found in %s\n",
            key, value, iftfile);
  }

  iftFree(&ift); free(key); free(value);
  return(11);
}
/*****************************************************************************/

/*****************************************************************************/
/// @endcond
