/** @file iftmatch.c
 *  @brief Check if the information in two IFT files is the same.
 *  @copyright (c) Turku PET Centre
 *  @author Vesa Oikonen
 */
/// @cond
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <strings.h>
#include <math.h>
/*****************************************************************************/
#include "tpcextensions.h"
#include "tpcift.h"
/*****************************************************************************/

/*****************************************************************************/
/* Local functions */
int iftCheckKeyValue(
  IFT *ift1, IFT *ift2, char *key, int test_lt, int test_gt,
  double test_abs, int verbose
);
int iftCheckKeyValues(
  IFT *ift1, IFT *ift2, char *key1, char *key2, int test_lt, int test_gt,
  double test_abs, int verbose
);
/*****************************************************************************/

/*****************************************************************************/
static char *info[] = {
  "Verify that the contents in two IFT files is similar.",
  "By default, all contents (keys and values) are tested, but by specifying",
  "a key, only values of that that key are tested. If the second IFT file",
  "has a different key name, then it can be specified as the second key name.",
  " ",
  "IFT files are text files with key and (optional) value at each line,",
  "for example:",
  "   calibrated := yes",
  "   calibration_coefficient := 6.78901E-005",
  " ",
  "Key and value strings are case-insensitive. Lines may be in any order,",
  "except that if more than one key instances are found, the order of the",
  "values must be the same.",
  " ",
  "Program returns 0, if match is found, 1-9 in case of an error, or",
  "10, if matching key or value is not found.", 
  " ",
  "Usage: @P [options] filename1 filename2 [key [key2]]",
  " ",
  "Options:",
  "-lt | -gt | -abs=<limit>",
  "     Values in file 1 and 2 are tested numerically (not as text), to be less",
  "     (-lt) or greater (-gt) in file 1 than in file 2, or the absolute",
  "     difference is not allowed to exceed the specified limit (-abs=<limit>)",
  " -stdoptions", // List standard options like --help, -v, etc
  " ",
  "Example 1:",
  "     @P data1.ift data2.ift",
  "Example 2:",
  "     @P iea345header.txt validheader.txt calibrated",
  "Example 3:",
  "     @P -abs=0.02 iea345hdr.txt validhdr.txt halflife",
  "Example 4:",
  "     @P -abs=0.02 iea345hdr.txt validhdr.txt half-life halflife",
  " ",
  "See also: iftisval, iftlist, csv2ift",
  " ",
  "Keywords: IFT, tool, software testing",
  0};
/*****************************************************************************/

/*****************************************************************************/
/* Turn on the globbing of the command line, since it is disabled by default in
   mingw-w64 (_dowildcard=0); in MinGW32 define _CRT_glob instead, if necessary;
   In Unix&Linux wildcard command line processing is enabled by default. */
/*
#undef _CRT_glob
#define _CRT_glob -1
*/
int _dowildcard = -1;
/*****************************************************************************/

/*****************************************************************************/
/**
 *  Main
 */
int main(int argc, char **argv)
{
  int ai, help=0, version=0, verbose=1;
  int ret, i;
  char *cptr, iftfile1[FILENAME_MAX], iftfile2[FILENAME_MAX];
  IFT ift1, ift2;
  char *key, *key2;
  double test_abs=-1.0;
  int test_lt=0, test_gt=0;

  /*
   *  Get arguments
   */
  if(argc==1) {tpcPrintUsage(argv[0], info, stderr); return(1);}
  iftInit(&ift1); iftInit(&ift2);
  iftfile1[0]=iftfile2[0]=(char)0; key=NULL; key2=NULL;
  /* Options */
  for(ai=1; ai<argc; ai++) if(*argv[ai]=='-') {
    if(tpcProcessStdOptions(argv[ai], &help, &version, &verbose)==0) continue;
    cptr=argv[ai]+1;
    if(strncasecmp(cptr, "ABS=", 4)==0) {
      test_abs=atofVerified(cptr+4); if(!isnan(test_abs)) continue;
    } else if(strcasecmp(cptr, "LT")==0) {
      test_lt=1; continue;
    } else if(strcasecmp(cptr, "GT")==0) {
      test_gt=1; continue;
    }
    fprintf(stderr, "Error: invalid option '%s'\n", argv[ai]);
    free(key); free(key2);
    return(1);
  } else break; // later arguments may start with '-'

  /* Print help or version? */
  if(help==2) {tpcHtmlUsage(argv[0], info, ""); return(0);}
  if(help) {tpcPrintUsage(argv[0], info, stdout); return(0);}
  if(version) {tpcPrintBuild(argv[0], stdout); return(0);}

  TPCSTATUS status; statusInit(&status);
  statusSet(&status, __func__, __FILE__, __LINE__, TPCERROR_OK);
  status.verbose=verbose;

  /* Process other arguments, starting from the first non-option */
  for(; ai<argc; ai++) {
    if(!iftfile1[0]) {
      strcpy(iftfile1, argv[ai]); continue;
    } else if(!iftfile2[0]) {
      strcpy(iftfile2, argv[ai]); continue;
    } else if(key==NULL) {
      key=strdup(argv[ai]); continue;
    } else if(key2==NULL) {
      key2=strdup(argv[ai]); continue;
    }
    fprintf(stderr, "Error: invalid argument '%s'.\n", argv[ai]);
    free(key); free(key2);
    return(1);
  }

  /* Is something missing? */
  if(!iftfile2[0]) {tpcPrintUsage(argv[0], info, stdout); return(1);}

  /* In verbose mode print arguments and options */
  if(verbose>1) {
    for(ai=0; ai<argc; ai++)
      printf("%s ", argv[ai]); 
    printf("\n");
    printf("iftfile1 := %s\n", iftfile1);
    printf("iftfile2 := %s\n", iftfile2);
    if(key!=NULL) printf("key := %s\n", key);
    if(key2!=NULL) printf("key2 := %s\n", key2);
    if(test_abs>=0.0) printf("test_abs := %g\n", test_abs);
    printf("test_lt := %d\n", test_lt);
    printf("test_gt := %d\n", test_gt);
  }


  /*
   *  Read IFT files
   */
  FILE *fp;
  if(verbose>1) printf("reading %s\n", iftfile1);
  fp=fopen(iftfile1, "r"); if(fp==NULL) {
    fprintf(stderr, "Error: cannot open file %s\n", iftfile1);
    free(key); free(key2); return(2);
  }
  ret=iftRead(&ift1, fp, 1, 1, &status); fclose(fp);
  if(ret) {
    fprintf(stderr, "Error (%d): %s\n", ret, errorMsg(status.error));
    free(key); free(key2); iftFree(&ift1); return(2);
  }
  if(verbose>2) printf("list1 size: %d item(s)\n", ift1.keyNr);

  if(verbose>1) printf("reading %s\n", iftfile2);
  fp=fopen(iftfile2, "r"); if(fp==NULL) {
    fprintf(stderr, "Error: cannot open file %s\n", iftfile2);
    free(key); free(key2); iftFree(&ift1); return(2);
  }
  ret=iftRead(&ift2, fp, 1, 1, &status); fclose(fp);
  if(ret) {
    fprintf(stderr, "Error (%d): %s\n", ret, errorMsg(status.error));
    free(key); free(key2); iftFree(&ift1); iftFree(&ift2); return(2);
  }
  if(verbose>2) printf("list2 size: %d item(s)\n", ift2.keyNr);

  /* If key was specified, then... */
  if(key!=NULL) {
    if(key2==NULL) {
      if(verbose>1) printf("testing key '%s'\n", key);
      ret=iftCheckKeyValue(&ift1, &ift2, key, test_lt, test_gt, test_abs,
                           verbose-1);
    } else {
      if(verbose>1) printf("testing keys '%s' and '%s'\n", key, key2);
      ret=iftCheckKeyValues(&ift1, &ift2, key, key2, test_lt, test_gt, test_abs,
                           verbose-1);
    }
    if(ret==0) {
      if(verbose>=0) fprintf(stdout, "Match was found.\n");
    } else {
      if(verbose>=0) fprintf(stderr, "No match: values are not the same.\n");
      ret=10;
    }
    iftFree(&ift1); iftFree(&ift2); free(key); free(key2);
    return(ret);
  }

  /* We are here to test all contents */
  if(verbose>1) printf("full content testing\n");

  /* First check that nr of keys is the same */
  if(ift1.keyNr!=ift2.keyNr) {
    if(verbose>=0) fprintf(stderr, "No match: key number is not the same.\n");
    iftFree(&ift1); iftFree(&ift2); free(key); free(key2);
    return(10);
  }
  
  /* Go through the list items */
  ret=0;
  for(i=0; i<ift1.keyNr; i++) {
    // in case of multiple instances of the same key, it will be tested
    // many times, but that does not matter so much

    if(verbose>1) printf("testing key '%s'\n", ift1.item[i].key);
    ret=iftCheckKeyValue(&ift1, &ift2, ift1.item[i].key, test_lt, test_gt,
                           test_abs, verbose-1);
    if(ret==0) {
      if(verbose>1) fprintf(stdout, "Match was found.\n");
    } else {
      if(verbose>=0)
        fprintf(stderr, "No match: values are not the same for the key %s.\n",
	        ift1.item[i].key);
      iftFree(&ift1); iftFree(&ift2); free(key); free(key2);
      return(10);
    }
  
  } // next item
  
  
  iftFree(&ift1); iftFree(&ift2); free(key); free(key2);
  if(verbose>=0) fprintf(stdout, "Match was found.\n");
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/// @endcond
/** Verify that specified key string is found in both IFT structs with the same
 *  values. If more than one instance of key is found, then all are tested.
\return Returns 0 if match is found and >0 in case of no match.
 */
int iftCheckKeyValue(
  /** Pointer to filled IFT struct */
  IFT *ift1,
  /** Pointer to filled IFT struct */
  IFT *ift2,
  /** Pointer to case-insensitive key name */
  char *key,
  /** If set to non-zero, value in ift1 must be smaller than corresponding
   *  value in ift2 */
  int test_lt,
  /** If set to non-zero, value in ift1 must be greater than corresponding
   *  value in ift2 */
  int test_gt,
  /** Allowed absolute difference in values; set to <=0 if no difference
   *  is allowed. */  
  double test_abs,
  /** Verbose level; if zero, nothing is printed into stdout or stderr */
  int verbose
) {
  if(verbose>2)
    printf("iftCheckKeyValue(*ift1, *ift2, \"%s\", %d, %d, %g, ...)\n",
           key, test_lt, test_gt, test_abs);
  if(ift1==NULL || ift2==NULL) return(1);
  if(key==NULL || strlen(key)<1) return(1);

  /* Check first that key can be found in both files */
  int n1, n2;
  n1=iftFindNrOfKeys(ift1, key);  
  n2=iftFindNrOfKeys(ift2, key);  
  if(verbose>0) {
    printf("%d instance(s) of key in file1.\n", n1);
    printf("%d instance(s) of key in file2.\n", n2);
  }
  if(n1<1 || n2<1) {
    if(verbose>0) fprintf(stderr, "No match: key not found.\n");
    return(10);
  }
  /* Verify that the same key(s) and value(s) are found in both */
  if(n1!=n2) {
    if(verbose>0) fprintf(stderr, "No match: different nr of keys.\n");
    return(10);
  }
  char *s1, *s2;
  int i, li, lj;
  double v1, v2;
  s1=(char*)NULL; s2=(char*)NULL;
  for(i=0, li=lj=-1; i<n1; i++) {
    li=iftFindKey(ift1, key, li+1); if(verbose>5) printf("li=%d\n", li);
    lj=iftFindKey(ift2, key, lj+1); if(verbose>5) printf("lj=%d\n", lj);
    if(li<0 || lj<0) break;
    /* Get pointers to value strings */
    s1=(char*)ift1->item[li].value; s2=(char*)ift2->item[lj].value;
    if(verbose>2) printf("testing strings '%s' == '%s'\n", s1, s2);
    /* Check if both values are NULL or empty; considered as match */
    if(s1==NULL && s2==NULL) continue;
    if(strlen(s1)==0 && strlen(s2)==0) continue;
    /* Check if values are the same as strings */
    if(strcasecmp(s1, s2)==0) continue;
    /* Try to convert them to doubles, ignoring unit */
    if(doubleGetWithUnit(s1, &v1, NULL)!=0) break;
    if(doubleGetWithUnit(s2, &v2, NULL)!=0) break;
    /* Compare the doubles */
    if(test_lt) {
      if(verbose>2) printf("testing whether %g < %g\n", v1, v2);
      if(v1<v2) continue;
    } else if(test_gt) {
      if(verbose>2) printf("testing whether %g > %g\n", v1, v2);
      if(v1>v2) continue;
    } else if(test_abs>=0) {
      if(verbose>2) printf("testing whether |%g-%g| < %g\n", v1, v2, test_abs);
      if(fabs(v1-v2)<test_abs) continue;
    } else {
      if(verbose>2) printf("testing whether %g == %g\n", v1, v2);
      if(v1==v2) continue;
    }
    break;
  }
  if(i!=n1) {
    if(verbose>0) fprintf(stderr, "No match: values are not the same.\n");
    if(verbose>1) printf("value1 := '%s'\nvalue2 := '%s'\n", s1, s2);
    return(10);
  }
  if(verbose>0) fprintf(stdout, "Match was found: values are the same.\n");
  return(0);
}
/******************************************************************************/

/******************************************************************************/
/** Verify that items with specified key strings are found in both IFT structs,
 *  with the same item values.
 *  If more than one instance of key is found, then all of them are tested.
\return Returns 0 if match is found and >0 in case of no match.
 */
int iftCheckKeyValues(
  /** Pointer to filled IFT struct */
  IFT *ift1,
  /** Pointer to filled IFT struct */
  IFT *ift2,
  /** Pointer to case-insensitive key name for ift1 */
  char *key1,
  /** Pointer to case-insensitive key name for ift2 */
  char *key2,
  /** If set to non-zero, value in ift1 must be smaller than corresponding
   *  value in ift2 */
  int test_lt,
  /** If set to non-zero, value in ift1 must be greater than corresponding
   *  value in ift2 */
  int test_gt,
  /** Allowed absolute difference in values; set to <=0 if no difference
   *  is allowed. */  
  double test_abs,
  /** Verbose level; if zero, nothing is printed into stdout or stderr */
  int verbose
) {
  if(verbose>2)
    printf("iftCheckKeyValues(*ift1, *ift2, \"%s\", \"%s\", %d, %d, %g, ...)\n",
           key1, key2, test_lt, test_gt, test_abs);
  if(ift1==NULL || ift2==NULL) return(1);
  if(key1==NULL || strlen(key1)<1) return(1);
  if(key2==NULL) key2=key1;

  /* Check first that keys can be found in both files */
  int n1, n2;
  n1=iftFindNrOfKeys(ift1, key1);  
  n2=iftFindNrOfKeys(ift2, key2);  
  if(verbose>0) {
    printf("%d instance(s) of key in file1.\n", n1);
    printf("%d instance(s) of key in file2.\n", n2);
  }
  if(n1<1 || n2<1) {
    if(verbose>0) fprintf(stderr, "No match: key not found.\n");
    return(10);
  }
  /* Verify that the same key(s) and value(s) are found in both */
  if(n1!=n2) {
    if(verbose>0) fprintf(stderr, "No match: different nr of keys.\n");
    return(10);
  }
  char *s1, *s2;
  int i, li, lj;
  double v1, v2;
  s1=(char*)NULL; s2=(char*)NULL;
  for(i=0, li=lj=-1; i<n1; i++) {
    li=iftFindKey(ift1, key1, li+1); if(verbose>5) printf("li=%d\n", li);
    lj=iftFindKey(ift2, key2, lj+1); if(verbose>5) printf("lj=%d\n", lj);
    if(li<0 || lj<0) break;
    /* Get pointers to value strings */
    s1=(char*)ift1->item[li].value; s2=(char*)ift2->item[lj].value;
    if(verbose>2) printf("testing strings '%s' == '%s'\n", s1, s2);
    /* Check if both values are NULL or empty; considered as match */
    if(s1==NULL && s2==NULL) continue;
    if(strlen(s1)==0 && strlen(s2)==0) continue;
    /* Check if values are the same as strings */
    if(strcasecmp(s1, s2)==0) continue;
    /* Try to convert them to doubles */
    if(doubleGetWithUnit(s1, &v1, NULL)!=0) break;
    if(doubleGetWithUnit(s2, &v2, NULL)!=0) break;
    /* Compare the doubles */
    if(test_lt) {
      if(verbose>2) printf("testing whether %g < %g\n", v1, v2);
      if(v1<v2) continue;
    } else if(test_gt) {
      if(verbose>2) printf("testing whether %g > %g\n", v1, v2);
      if(v1>v2) continue;
    } else if(test_abs>=0) {
      if(verbose>2) printf("testing whether |%g-%g| < %g\n", v1, v2, test_abs);
      if(fabs(v1-v2)<test_abs) continue;
    } else {
      if(verbose>2) printf("testing whether %g == %g\n", v1, v2);
      if(v1==v2) continue;
    }
    break;
  }
  if(i!=n1) {
    if(verbose>0) fprintf(stderr, "No match: values are not the same.\n");
    if(verbose>1) printf("value1 := '%s'\nvalue2 := '%s'\n", s1, s2);
    return(10);
  }
  if(verbose>0) fprintf(stdout, "Match was found: values are the same.\n");
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
