/** @file injdifft.c
 *  @brief Calculates the time difference between injection times.
 *  @copyright (c) Turku PET Centre
 *  @author Vesa Oikonen
 */
/// @cond
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
/*****************************************************************************/
#include "tpcextensions.h"
#include "tpcift.h"
/*****************************************************************************/

/*****************************************************************************/
static char *info[] = {
  "Calculates time difference between zero times of PET scan and blood data.",
  "Zero times are given in two text files in format",
  "scan_start_time := YYYY-MM-DD hh:mm:ss, or",
  "injection_time := YYYY-MM-DD hh:mm:ss.",
  " ",
  "The result, time difference (time2-time1 in seconds) will be written in",
  "stdout or specified file in format",
  "time_difference := <time2-time1>.",
  " ",
  "Usage: @P [Options] timefile1 timefile2 [outputfile]",
  " ",
  "Options:",
  " -min | -sec",
  "     Time difference is written in minutes or in seconds (default).",
  " -small",
  "     Software ignores time differences of multiples of one hour, that may",
  "     have been caused by wrong time zone or daylight saving setting in",
  "     either of the zero times.",
  " -stdoptions", // List standard options like --help, -v, etc
  " ",
  "Example:",
  "  @P -small s95432dy1.v.ift s95432blo.kbq s95432.dift",
  " ",
  "See also: egetstrt, absscal, iftlist, tactime, blotimes, fitdelay",
  " ",
  "Keywords: input, header, IFT, tool",
  0};
/*****************************************************************************/

/*****************************************************************************/
/* Turn on the globbing of the command line, since it is disabled by default in
   mingw-w64 (_dowildcard=0); in MinGW32 define _CRT_glob instead, if necessary;
   In Unix&Linux wildcard command line processing is enabled by default. */
/*
#undef _CRT_glob
#define _CRT_glob -1
*/
int _dowildcard = -1;
/*****************************************************************************/

/*****************************************************************************/
/**
 *  Main
 */
int main(int argc, char **argv)
{
  int       ai, help=0, version=0, verbose=1;
  int       ret, li, time_unit=UNIT_SEC, hastobe_small=0;
  char      iftfile1[FILENAME_MAX], iftfile2[FILENAME_MAX], 
            outfile[FILENAME_MAX];
  IFT       ift1, ift2;
  char     *cptr, value1[256], value2[256];
  struct tm st1, st2;
  double    tdiff;
  FILE     *fp;


  /*
   *  Get arguments
   */
  if(argc==1) {tpcPrintUsage(argv[0], info, stderr); return(1);}
  iftfile1[0]=iftfile2[0]=outfile[0]=(char)0;
  iftInit(&ift1); iftInit(&ift2);
  /* Options */
  for(ai=1; ai<argc; ai++) if(*argv[ai]=='-') {
    if(tpcProcessStdOptions(argv[ai], &help, &version, &verbose)==0) continue;
    cptr=argv[ai]+1; if(*cptr=='-') cptr++; if(cptr==NULL) continue;
    if(strncasecmp(cptr, "MINUTES", 3)==0) {
      time_unit=UNIT_MIN; continue;
    } else if(strncasecmp(cptr, "SECONDS", 3)==0) {
      time_unit=UNIT_SEC; continue;
    } else if(strncasecmp(cptr, "SMALL", 3)==0) {
      hastobe_small=1; continue;
    }
    fprintf(stderr, "Error: invalid option '%s'.\n", argv[ai]);
    return(1);
  } else break; // tac name argument may start with '-'

  TPCSTATUS status; statusInit(&status);
  statusSet(&status, __func__, __FILE__, __LINE__, TPCERROR_OK);
  status.verbose=verbose-1;
  
  /* Print help or version? */
  if(help==2) {tpcHtmlUsage(argv[0], info, ""); return(0);}
  if(help) {tpcPrintUsage(argv[0], info, stdout); return(0);}
  if(version) {tpcPrintBuild(argv[0], stdout); return(0);}

  /* Arguments */
  for(; ai<argc; ai++) {
    if(!iftfile1[0]) {
      strlcpy(iftfile1, argv[ai], FILENAME_MAX); continue;
    } else if(!iftfile2[0]) {
      strlcpy(iftfile2, argv[ai], FILENAME_MAX); continue;
    } else if(!outfile[0]) {
      strlcpy(outfile, argv[ai], FILENAME_MAX); continue;
    }
    fprintf(stderr, "Error: invalid argument '%s'.\n", argv[ai]);
    return(1);
  }

  /* Is something missing? */
  if(!iftfile2[0]) {
    fprintf(stderr, "Error: missing command-line argument; use option --help\n");
    return(1);
  }

  /* In verbose mode print arguments and options */
  if(verbose>1) {
    for(ai=0; ai<argc; ai++)
      printf("%s ", argv[ai]); 
    printf("\n");
    printf("iftfile1 := %s\n", iftfile1);
    printf("iftfile2 := %s\n", iftfile2);
    printf("outfile := %s\n", outfile);
    printf("time_unit := %d\n", time_unit);
    printf("hastobe_small := %d\n", hastobe_small);
  }


  /*
   *  Read IFT files
   */
  if(verbose>1) printf("reading %s\n", iftfile1);
  fp=fopen(iftfile1, "r"); if(fp==NULL) {
    fprintf(stderr, "Error: cannot open file %s\n", iftfile1);
    iftFree(&ift1); return(2);
  }
  ret=iftRead(&ift1, fp, 1, 1, &status); fclose(fp);
  if(ret) {
    fprintf(stderr, "Error (%d): %s\n", ret, errorMsg(status.error));
    iftFree(&ift1); return(2);
  }
  if(verbose>2) printf("list1 size: %d item(s)\n", ift1.keyNr);

  if(verbose>1) printf("reading %s\n", iftfile2);
  fp=fopen(iftfile2, "r"); if(fp==NULL) {
    fprintf(stderr, "Error: cannot open file %s\n", iftfile2);
    iftFree(&ift1); iftFree(&ift2); return(2);
  }
  ret=iftRead(&ift2, fp, 1, 1, &status); fclose(fp);
  if(ret) {
    fprintf(stderr, "Error (%d): %s\n", ret, errorMsg(status.error));
    iftFree(&ift1); iftFree(&ift2); return(2);
  }
  if(verbose>2) printf("list2 size: %d item(s)\n", ift2.keyNr);


  /*
   *  Read the two times
   */
  if(verbose>1) printf("reading times\n");
  /* Time #1 */
  li=iftFindKey(&ift1, "scan_start_time", 0);
  if(li<0) li=iftFindKey(&ift1, "injection_time", 0);
  if(li<0) {
    fprintf(stderr, "Error: %s does not contain required key.\n", iftfile1);
    iftFree(&ift1); iftFree(&ift2); return(4);
  }
  strlcpy(value1, ift1.item[li].value, 256); 
  if(verbose>2) printf("File #1:\n%s := %s\n", ift1.item[li].key, value1);
  if(strDateTimeRead(value1, &st1)) {
    fprintf(stderr, "Error: wrong date/time format in %s.\n", iftfile1);
    iftFree(&ift1); iftFree(&ift2); return(5);
  }
  /* Time #2 */
  li=iftFindKey(&ift2, "scan_start_time", 0);
  if(li<0) li=iftFindKey(&ift2, "injection_time", 0);
  if(li<0) {
    fprintf(stderr, "Error: %s does not contain required key.\n", iftfile2);
    iftFree(&ift1); iftFree(&ift2); return(4);
  }
  strlcpy(value2, ift2.item[li].value, 256); 
  if(verbose>2) printf("File #2:\n%s := %s\n", ift2.item[li].key, value2);
  if(strDateTimeRead(value2, &st2)) {
    fprintf(stderr, "Error: wrong date/time format in %s.\n", iftfile2);
    iftFree(&ift1); iftFree(&ift2); return(5);
  }

  iftFree(&ift1); iftFree(&ift2);


  /*
   *  Calculate the time difference in seconds
   */
  tdiff=tmDifference(&st2, &st1);
  if(verbose>1) printf("tdif=%g s\n", tdiff);
  /* Ignore differences of multiples of hour that maybe caused by wrong
     time zone etc (option -small) */
  if(hastobe_small) {
    double f;
    f=tdiff/3600.0; f=(f<0.0)?(int)(f-0.5):(int)(f+0.5);
    tdiff-=3600.0*f;
    if(verbose>1) 
      printf("tdif=%g s , when ignoring %g hour(s)\n", tdiff, f);
  }


  /*
   *  Write the result into file or stdout
   */
  if(outfile[0]) {
    fp=fopen(outfile, "w"); if(fp==NULL) {
      fprintf(stderr, "Error: cannot open %s for writing.\n", outfile);
      return(11);
    }
  } else
    fp=stdout;
  /* write */
  fprintf(fp, "filename1 := %s\n", iftfile1);
  fprintf(fp, "filename2 := %s\n", iftfile2);
  if(time_unit==UNIT_SEC) {
    ret=fprintf(fp, "time_difference := %g [s]\n", tdiff);
  } else {
    ret=fprintf(fp, "time_difference := %g [min]\n", tdiff/60.0);
  }
  if(outfile[0]) fclose(fp);
  if(ret<24) {
    fprintf(stderr, "Error: cannot write into %s\n", outfile);
    return(12);
  }

  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/// @endcond
