#!/bin/bash
#: Title      : test_b2ptrap
#: Date       : 2023-03-12
#: Author     : "Vesa Oikonen" <vesa.oikonen@utu.fi>
#: Options    : None

# Set the name of executable to test
if [ -n "${OS+1}" ] && [ "$OS" = "Windows_NT" ]; then
EXT=.exe;
else
EXT=;
fi

PROGRAM=../../b2ptrap$EXT;
if [ ! -f $PROGRAM ]; then
  printf "Failed: executable does not exist.\n"
  exit 1
fi


printf "=====================================================================\n"
printf "creating test data for %s\n" $PROGRAM
printf "=====================================================================\n"

if [ ! -f plasma.fit ] || [ ! -f p2rbc.par ] || [ ! -f frames.sif ]; then
  printf "Failed: test data does not exist.\n"
  exit 1
fi

if [ ! -f plasma.bld ]; then
  rm -f blood.bld bloodf.tac plasmaf.tac
  fit2dat -c=0,10,0.02 plasma.fit plasma.bld
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi

if [ ! -f plasmac.bld ] || [ ! -f rbcc.bld ] || [ ! -f blood.bld ]; then
  rm -f bloodf.tac plasmaf.tac
# Computing PTAC as component of blood as (1-HCT)*Cp where HCT=0.39
  taccalc plasma.bld x 0.61 plasmac.bld
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
# Simulating RBC concentration curve dCrbc=kc*(1-HCT)*Cp
# where kc=0.60 and thus k2=kc*HCT=0.234
  sim_3tcm p2rbc.par plasmac.bld plasmac.bld rbc.bld
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
# Calculating the blood radioactivity residing in RBCs
  taccalc rbc.bld x 0.39 rbcc.bld
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
# Calculating the blood curve as Cb=(1-HCT)*Cp+HCT*Crbc
  taccalc plasmac.bld + rbcc.bld blood.bld
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
# Setting file header information for plotting
  tacren blood.bld 1 Blood
  tacren plasma.bld 1 Plasma
  tacren plasmac.bld 1 Plasma
  tacren rbc.bld 1 RBC
  tacren rbcc.bld 1 RBC
  tacstudy --force blood.bld sim1
  tacstudy --force plasma.bld sim1
  tacstudy --force plasmac.bld sim1
  tacstudy --force rbc.bld sim1
  tacstudy --force rbcc.bld sim1
# Plotting blood components
  tac2svg -legend=y -yt="SUV" -mt="" -x2=10.2 blood.svg -l blood.bld plasmac.bld rbcc.bld
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi

if [ ! -f plasmaf.tac ] || [ ! -f bloodf.tac ]; then
  # Simulating PET frames
  simframe plasmac.bld frames.sif plasmaf.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  simframe blood.bld frames.sif bloodf.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi




printf "\n\n"
printf "=====================================================================\n"
printf "testing %s\n" $PROGRAM
printf "=====================================================================\n"

printf "\n 1.0.0 \n"
printf " Test case 1: Compute PTAC with known k. \n"
printf " Test case 2: Data with high sampling frequency. \n"
printf " Expected result: PTAC is close to correct. \n\n"

rm -f p100.tac
$PROGRAM blood.bld 0.234 p100.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tac2svg -legend=y p100.svg -l blood.bld plasmac.bld p100.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.1 -rel=0.1 plasmac.bld p100.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 1.0.1 \n"
printf " Test case 1: Compute PTAC with known k. \n"
printf " Test case 2: Framed data. \n"
printf " Expected result: PTAC is close to correct. \n\n"

rm -f p101.tac
$PROGRAM bloodf.tac 0.234 p101.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tac2svg -legend=y p101.svg -s plasmaf.tac -l p101.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.2 -rel=0.1 plasmaf.tac p101.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 2.0.0 \n"
printf " Test case 1: Compute PTAC and fit k. \n"
printf " Test case 2: Data with high sampling frequency. \n"
printf " Expected result: PTAC is close to correct. \n\n"

rm -f p200.tac
$PROGRAM -ztime=5 blood.bld FIT p200.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tac2svg -legend=y p200.svg -l blood.bld plasmac.bld p200.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.5 -rel=0.1 plasmac.bld p200.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 2.0.1 \n"
printf " Test case 1: Compute PTAC and fit k. \n"
printf " Test case 2: Framed data. \n"
printf " Expected result: PTAC is close to correct. \n\n"

rm -f p201.tac
$PROGRAM -ztime=5 bloodf.tac FIT p201.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tac2svg -legend=y p201.svg -s plasmaf.tac -l p201.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.5 -rel=0.1 plasmaf.tac p201.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n===================================================================\n"
printf "   All passed! \n"
printf "===================================================================\n"
exit 0

