#!/bin/bash
#: Title      : test_tacmove
#: Date       : 2023-05-10
#: Author     : "Vesa Oikonen" <vesa.oikonen@utu.fi>
#: Options    : None

# Set the name of executable to test
if [ -n "${OS+1}" ] && [ "$OS" = "Windows_NT" ]; then
EXT=.exe;
else
EXT=;
fi

PROGRAM=../../tacmove$EXT;
if [ ! -f $PROGRAM ]; then
  printf "Failed: executable does not exist.\n"
  exit 1
fi


printf "=====================================================================\n"
printf "creating test data for %s\n" $PROGRAM
printf "=====================================================================\n"


if [ ! -f bolus1.fit ] || [ ! -f bolus2.fit ]; then
  rm -f bolus1.bld bolus2.bld bolus.svg
  printf "FIT1        generic\n" > bolus1.fit
  printf "Date:	2023-03-01 20:42:59\n" >> bolus1.fit
  printf "Data file:	bolus.dat\n" >> bolus1.fit
  printf "Data unit:	g/mL\n" >> bolus1.fit
  printf "Time unit:	min\n" >> bolus1.fit
  printf "Nr of VOIs:	1\n" >> bolus1.fit
  printf "Region Plane	Start	End	dataNr	WSS	parNr	Type	Parameters\n" >> bolus1.fit
  printf "TAC1	0.000	10.000	23	0	7	1313	4010	-11.7	469	-20.2	9.43	-0.895	0.5\n" >> bolus1.fit

  printf "FIT1        generic\n" > bolus2.fit
  printf "Date:	2023-03-01 20:42:59\n" >> bolus2.fit
  printf "Data file:	bolus.dat\n" >> bolus2.fit
  printf "Data unit:	g/mL\n" >> bolus2.fit
  printf "Time unit:	min\n" >> bolus2.fit
  printf "Nr of VOIs:	1\n" >> bolus2.fit
  printf "Region Plane	Start	End	dataNr	WSS	parNr	Type	Parameters\n" >> bolus2.fit
  printf "TAC2	0.000	10.000	23	0	7	1313	4010	-11.7	469	-20.2	9.43	-0.895	1.5\n" >> bolus2.fit
fi


if [ ! -f bolus.svg ]; then
  fit2dat -c=0,10,0.02 bolus1.fit bolus1.bld
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  fit2dat -c=0,10,0.02 bolus2.fit bolus2.bld
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tac2svg bolus.svg -l bolus1.bld bolus2.bld
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi

if [ ! -f frames.sif ]; then
  printf "12/4/2023 11:03:43  8 2 1 SIM1 O-15\n" > frames.sif
  printf "      0      30\n" >> frames.sif
  printf "     30      60\n" >> frames.sif
  printf "     60     120\n" >> frames.sif
  printf "    120     180\n" >> frames.sif
  printf "    180     240\n" >> frames.sif
  printf "    240     300\n" >> frames.sif
  printf "    300     420\n" >> frames.sif
  printf "    420     540\n" >> frames.sif
fi


if [ ! -f tacs1.fit ] || [ ! -f tacs2.fit ] || [ ! -f delays.csv ]; then
  printf "FIT1        generic\n" > tacs1.fit
  printf "Date:	2023-03-01 20:42:59\n" >> tacs1.fit
  printf "Data file:	bolus.dat\n" >> tacs1.fit
  printf "Data unit:	g/mL\n" >> tacs1.fit
  printf "Time unit:	min\n" >> tacs1.fit
  printf "Nr of VOIs:	4\n" >> tacs1.fit
  printf "Region Plane	Start	End	dataNr	WSS	parNr	Type	Parameters\n" >> tacs1.fit
  printf "TAC1	0.000	10.000	23	0	7	1313	4010	-11.7	469	-20.2	9.43	-0.895	0.5\n" >> tacs1.fit
  printf "TAC2	0.000	10.000	23	0	7	1313	4010	-11.7	469	-20.2	9.43	-0.895	0.6\n" >> tacs1.fit
  printf "TAC3	0.000	10.000	23	0	7	1313	4010	-11.7	469	-20.2	9.43	-0.895	0.7\n" >> tacs1.fit
  printf "TAC4	0.000	10.000	23	0	7	1313	4010	-11.7	469	-20.2	9.43	-0.895	0.8\n" >> tacs1.fit

  printf "FIT1        generic\n" > tacs2.fit
  printf "Date:	2023-03-01 20:42:59\n" >> tacs2.fit
  printf "Data file:	bolus.dat\n" >> tacs2.fit
  printf "Data unit:	g/mL\n" >> tacs2.fit
  printf "Time unit:	min\n" >> tacs2.fit
  printf "Nr of VOIs:	4\n" >> tacs2.fit
  printf "Region Plane	Start	End	dataNr	WSS	parNr	Type	Parameters\n" >> tacs2.fit
  printf "TAC1	0.000	10.000	23	0	7	1313	4010	-11.7	469	-20.2	9.43	-0.895	0.6\n" >> tacs2.fit
  printf "TAC2	0.000	10.000	23	0	7	1313	4010	-11.7	469	-20.2	9.43	-0.895	0.6\n" >> tacs2.fit
  printf "TAC3	0.000	10.000	23	0	7	1313	4010	-11.7	469	-20.2	9.43	-0.895	0.6\n" >> tacs2.fit
  printf "TAC4	0.000	10.000	23	0	7	1313	4010	-11.7	469	-20.2	9.43	-0.895	0.6\n" >> tacs2.fit

  printf "Parameters,K1,delayT\n" > delays.csv
  printf "TAC2,0.3,0.0\n" >> delays.csv
  printf "TAC1,0.3,-0.1\n" >> delays.csv
  printf "TAC3,0.3,0.1\n" >> delays.csv
  printf "TAC4,0.3,0.2\n" >> delays.csv
fi


printf "\n\n"
printf "=====================================================================\n"
printf "testing %s\n" $PROGRAM
printf "=====================================================================\n"

printf "\n 1.0.0 \n"
printf " Test case 1: Single TAC with sample times. \n"
printf " Test case 2: Time given. \n"
printf " Expected result: Correct output. \n\n"


fit2dat -c=0,5,0.2 bolus1.fit temp1.bld
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fit2dat -c=0,5,0.2 bolus2.fit temp2.bld
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi

$PROGRAM temp1.bld 1
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.01 -rel=0.01 temp2.bld temp1.bld
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 1.1.0 \n"
printf " Test case 1: Single TAC with sample times. \n"
printf " Test case 2: Time in parameter file. \n"
printf " Expected result: Correct output. \n\n"

printf "Parameters,K1,dT[min]\n" > temp.csv
printf "Input,0.3,1.0\n" >> temp.csv

fit2dat -c=0,5,0.2 bolus1.fit temp1.bld
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi

$PROGRAM temp1.bld temp.csv
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.01 -rel=0.01 temp2.bld temp1.bld
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 1.1.1 \n"
printf " Test case 1: Time in parameter file in seconds. \n"
printf " Expected result: Correct output. \n\n"

printf "Parameters,K1,dT[sec]\n" > temp.csv
printf "Input,0.3,60.0\n" >> temp.csv

fit2dat -c=0,5,0.2 bolus1.fit temp1.bld
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi

$PROGRAM temp1.bld temp.csv
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.01 -rel=0.01 temp2.bld temp1.bld
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 1.1.2 \n"
printf " Test case 1: Option -opposite. \n"
printf " Expected result: Correct output. \n\n"

printf "Parameters,K1,dT[sec]\n" > temp.csv
printf "Input,0.3,-60.0\n" >> temp.csv

fit2dat -c=0,5,0.2 bolus1.fit temp1.bld
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi

$PROGRAM -opposite temp1.bld temp.csv
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.01 -rel=0.01 temp2.bld temp1.bld
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 1.2.0 \n"
printf " Test case 1: Several TACs with sample times. \n"
printf " Test case 2: Times for each TAC given in parameter file. \n"
printf " Test case 3: TACs in parameter file are in different order. \n"
printf " Expected result: Correct output. \n\n"


fit2dat -c=0,5,0.1 tacs1.fit temp1.bld
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fit2dat -c=0,5,0.1 tacs2.fit temp2.bld
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi

$PROGRAM -opp temp1.bld delays.csv
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.1 -rel=0.1 temp2.bld temp1.bld
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"





printf "\n 2.0.0 \n"
printf " Test case 1: Single TAC with time frames. \n"
printf " Test case 2: Positive time given. \n"
printf " Expected result: Correct output. \n\n"


fit2dat -c=0,10,0.02 bolus1.fit temp1.bld
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fit2dat -c=0,10,0.02 bolus2.fit temp2.bld
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
simframe temp1.bld frames.sif temp1.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
simframe temp2.bld frames.sif temp2.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi

$PROGRAM temp1.tac 1
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.05 -rel=0.1 temp2.tac temp1.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 2.0.1 \n"
printf " Test case 1: Single TAC with time frames. \n"
printf " Test case 2: Negative time given. \n"
printf " Expected result: Correct output. \n\n"


fit2dat -c=0,10,0.02 bolus1.fit temp1.bld
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
simframe temp1.bld frames.sif temp1.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi

$PROGRAM temp2.tac -1
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
taccut temp1.tac 2 10 temp1b.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
taccut temp2.tac 2 10 temp2b.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
tacmatch -abs=0.5 -rel=0.1 temp1b.tac temp2b.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

dftinteg temp1.tac 0 9 i1.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
dftinteg temp2.tac 0 9 i2.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
tacmatch -rel=0.5 i1.tac i2.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"





printf "\n 2.1.0 \n"
printf " Test case 1: Several TACs with time frames. \n"
printf " Test case 2: Times for each TAC given in parameter file. \n"
printf " Test case 3: TACs in parameter file are in different order. \n"
printf " Expected result: Correct output. \n\n"


fit2dat -c=0,10,0.02 tacs1.fit temp1.bld
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fit2dat -c=0,10,0.02 tacs2.fit temp2.bld
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
simframe temp1.bld frames.sif temp1.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
simframe temp2.bld frames.sif temp2.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi

$PROGRAM -opp temp1.tac delays.csv
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
dftinteg temp1.tac 0 9 i1.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
dftinteg temp2.tac 0 9 i2.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
tacmatch -rel=0.5 i1.tac i2.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"




printf "\n===================================================================\n"
printf "   All passed! \n"
printf "===================================================================\n"
exit 0

