/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include "tpcextensions.h"
#include "test_tpccm.h"
/*****************************************************************************/

/*****************************************************************************/
int test_simDispersion(
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  if(verbose>0) {
    printf("\n=====================================\n");
    printf("\n%s\n", __func__);
    printf("\n=====================================\n");
  }

  int ret;

  if(verbose>1) printf("\n testing with empty data \n");
  ret=simDispersion(NULL, NULL, 0, 0.0, 0.0, NULL);
  if(verbose>2) printf("did not segfault, that is good.\n");
  if(ret==0) return(1);
  if(verbose>2) printf("\n more testing with empty data \n");
  ret=simDispersion(NULL, NULL, 1000, 2.0, 5.0, NULL);
  if(verbose>2) printf("did not segfault, that is good.\n");
  if(ret==0) return(2);

  TAC input, output;
  tacInit(&input); tacInit(&output);

  if(verbose>2) printf("\n yet more testing with empty data \n");
  ret=simDispersion(input.x, input.x, 1000, 2.0, 5.0, NULL);
  if(verbose>2) printf("did not segfault, that is good.\n");
  if(ret==0) return(3);

  statusSet(status, __func__, __FILE__, __LINE__, 0);
  if(verbose>1) printf("\n testing with perfect bolus (nonrealistic for radiowater)\n");
  {
    PAR par; parInit(&par); parExamplePerfectBolus(&par, NULL);
    ret=tacAllocateWithPAR(&input, &par, 6001, status);
    if(ret!=TPCERROR_OK) {parFree(&par); tacFree(&input); return(100);}
    /* set sample times */
    double fdur=1.0;
    for(int i=0; i<input.sampleNr; i++) {
      input.x1[i]=(double)i*fdur;
      input.x2[i]=input.x1[i]+fdur; input.x[i]=0.5*(input.x1[i]+input.x2[i]);
    }
    /* compute input values */
    ret=mfEvalFrameY(modelCode(par.r[0].model), modelParNr(par.r[0].model),
                     par.r[0].p, input.sampleNr, input.x1, input.x2, input.c[0].y, 
                     verbose-100);
    if(ret) {tacFree(&input); parFree(&par); return(101);}
    parFree(&par);
  }

  statusSet(status, __func__, __FILE__, __LINE__, 0);
  if(verbose>1) printf("\n testing with dispersion constants set to zero \n");
  {
    double y2[input.sampleNr], buf[input.sampleNr];
    for(int i=0; i<input.sampleNr; i++) y2[i]=input.c[0].y[i];
    ret=simDispersion(input.x, y2, input.sampleNr, 0.0, 0.0, buf);
    if(ret) {tacFree(&input); return(1001);}
    ret=doubleArrayMatch(y2, input.c[0].y, input.sampleNr, 1.0E-100);
    if(!ret) {tacFree(&input); return(1002);}

    for(int i=0; i<input.sampleNr; i++) y2[i]=input.c[0].y[i];
    ret=simDispersion(input.x, y2, input.sampleNr, 0.0, 0.0, NULL);
    if(ret) {tacFree(&input); return(1011);}
    ret=doubleArrayMatch(y2, input.c[0].y, input.sampleNr, 1.0E-100);
    if(!ret) {tacFree(&input); return(1012);}
  }

  statusSet(status, __func__, __FILE__, __LINE__, 0);
  if(verbose>1) printf("\n testing with the first dispersion constant \n");
  {
    double y2[input.sampleNr], buf[input.sampleNr];
    for(int i=0; i<input.sampleNr; i++) y2[i]=input.c[0].y[i];
    ret=simDispersion(input.x, y2, input.sampleNr, 5.0, 0.0, buf);
    if(ret) {tacFree(&input); return(1101);}
    ret=doubleArrayMatch(y2, input.c[0].y, input.sampleNr, 0.1);
    if(ret) {tacFree(&input); return(1102);}
    ret=simC1(input.x, input.c[0].y, input.sampleNr, 1.0/5.0, 1.0/5.0, buf);
    if(ret) {tacFree(&input); return(1103);}
    ret=doubleArrayMatch(y2, buf, input.sampleNr, 0.1);
    if(!ret) {tacFree(&input); return(1104);}

    for(int i=0; i<input.sampleNr; i++) y2[i]=input.c[0].y[i];
    ret=simDispersion(input.x, y2, input.sampleNr, 5.0, 0.0, NULL);
    if(ret) {tacFree(&input); return(1111);}
    ret=doubleArrayMatch(y2, input.c[0].y, input.sampleNr, 0.1);
    if(ret) {tacFree(&input); return(1112);}
    ret=doubleArrayMatch(y2, buf, input.sampleNr, 0.1);
    if(!ret) {tacFree(&input); return(1114);}
  }

  statusSet(status, __func__, __FILE__, __LINE__, 0);
  if(verbose>1) printf("\n testing with the second dispersion constant \n");
  {
    double y2[input.sampleNr], buf[input.sampleNr];
    for(int i=0; i<input.sampleNr; i++) y2[i]=input.c[0].y[i];
    ret=simDispersion(input.x, y2, input.sampleNr, 0.0, 5.0, buf);
    if(ret) {tacFree(&input); return(1201);}
    ret=doubleArrayMatch(y2, input.c[0].y, input.sampleNr, 0.1);
    if(ret) {tacFree(&input); return(1202);}
    ret=simC1(input.x, input.c[0].y, input.sampleNr, 1.0/5.0, 1.0/5.0, buf);
    if(ret) {tacFree(&input); return(1203);}
    ret=doubleArrayMatch(y2, buf, input.sampleNr, 0.1);
    if(!ret) {tacFree(&input); return(1204);}

    for(int i=0; i<input.sampleNr; i++) y2[i]=input.c[0].y[i];
    ret=simDispersion(input.x, y2, input.sampleNr, 0.0, 5.0, NULL);
    if(ret) {tacFree(&input); return(1211);}
    ret=doubleArrayMatch(y2, input.c[0].y, input.sampleNr, 0.1);
    if(ret) {tacFree(&input); return(1212);}
    ret=doubleArrayMatch(y2, buf, input.sampleNr, 0.1);
    if(!ret) {tacFree(&input); return(1214);}
  }

  statusSet(status, __func__, __FILE__, __LINE__, 0);
  if(verbose>1) printf("\n testing with both dispersion constants \n");
  {
    double y2[input.sampleNr], buf[input.sampleNr], buf2[input.sampleNr];
    for(int i=0; i<input.sampleNr; i++) y2[i]=input.c[0].y[i];
    ret=simDispersion(input.x, y2, input.sampleNr, 2.0, 3.0, buf);
    if(ret) {tacFree(&input); return(1301);}
    ret=doubleArrayMatch(y2, input.c[0].y, input.sampleNr, 0.1);
    if(ret) {tacFree(&input); return(1302);}
    ret=simC1(input.x, input.c[0].y, input.sampleNr, 1.0/2.0, 1.0/2.0, buf);
    if(ret) {tacFree(&input); return(1303);}
    ret=simC1(input.x, buf, input.sampleNr, 1.0/3.0, 1.0/3.0, buf2);
    if(ret) {tacFree(&input); return(1304);}
    ret=doubleArrayMatch(y2, buf2, input.sampleNr, 0.1);
    if(!ret) {tacFree(&input); return(1305);}

    for(int i=0; i<input.sampleNr; i++) y2[i]=input.c[0].y[i];
    ret=simDispersion(input.x, y2, input.sampleNr, 2.0, 3.0, NULL);
    if(ret) {tacFree(&input); return(1311);}
    ret=doubleArrayMatch(y2, input.c[0].y, input.sampleNr, 0.1);
    if(ret) {tacFree(&input); return(1312);}
    ret=doubleArrayMatch(y2, buf2, input.sampleNr, 0.1);
    if(!ret) {tacFree(&input); return(1315);}
  }

  tacFree(&input);

  statusSet(status, __func__, __FILE__, __LINE__, 0);
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
int test_corDispersion(
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  if(verbose>0) {
    printf("\n=====================================\n");
    printf("\n%s\n", __func__);
    printf("\n=====================================\n");
  }

  int ret;

  if(verbose>1) printf("\n testing with empty data \n");
  ret=corDispersion(NULL, NULL, 0, 0.0, NULL);
  if(verbose>2) printf("did not segfault, that is good.\n");
  if(ret==0) return(1);
  if(verbose>2) printf("\n more testing with empty data \n");
  ret=corDispersion(NULL, NULL, 1000, 2.0, NULL);
  if(verbose>2) printf("did not segfault, that is good.\n");
  if(ret==0) return(2);

  TAC input, output;
  tacInit(&input); tacInit(&output);

  if(verbose>2) printf("\n yet more testing with empty data \n");
  ret=corDispersion(input.x, input.x, 1000, 2.0, NULL);
  if(verbose>2) printf("did not segfault, that is good.\n");
  if(ret==0) return(3);


  statusSet(status, __func__, __FILE__, __LINE__, 0);
  {
    if(verbose>1) printf("\n make dispersion-free perfect bolus (non-realistic for radiowater)\n");
    PAR par; parInit(&par); parExamplePerfectBolus(&par, NULL);
    ret=tacAllocateWithPAR(&input, &par, 1001, status);
    if(ret!=TPCERROR_OK) {parFree(&par); tacFree(&input); return(100);}
    /* set sample times */
    double fdur=0.02;
    for(int i=0; i<input.sampleNr; i++) {
      input.x1[i]=(double)i*fdur;
      input.x2[i]=input.x1[i]+fdur; input.x[i]=0.5*(input.x1[i]+input.x2[i]);
    }
    /* compute input values */
    ret=mfEvalFrameY(modelCode(par.r[0].model), modelParNr(par.r[0].model),
                     par.r[0].p, input.sampleNr, input.x1, input.x2, input.c[0].y, 
                     verbose-100);
    if(ret) {tacFree(&input); parFree(&par); return(101);}
    parFree(&par);
    if(verbose>1) printf("\n add dispersion\n");
    if(tacDuplicate(&input, &output) || 
       simDispersion(output.x, output.c[0].y, output.sampleNr, 3.0, 0.0, NULL))
    {
      tacFree(&input); tacFree(&output); return(102);
    }
  }

  statusSet(status, __func__, __FILE__, __LINE__, 0);
  {
    if(verbose>1) printf("\n try to correct for the added dispersion\n");
    double cy[output.sampleNr];
    for(int i=0; i<output.sampleNr; i++) cy[i]=output.c[0].y[i];
    ret=corDispersion(output.x, cy, output.sampleNr, 3.0, NULL);
    if(ret) {tacFree(&input); tacFree(&output); return(1101);}
    if(verbose>3) {
      for(int i=0; i<input.sampleNr; i++) printf("%9.3f\t\t%12.3e\t%12.3e\t%12.3e\t\t%g\n",
          input.x[i], input.c[0].y[i], output.c[0].y[i], cy[i], cy[i]-input.c[0].y[i]);
    }
    ret=doubleArrayMatch(input.c[0].y, cy, input.sampleNr, 0.05);
    if(!ret) {tacFree(&input); tacFree(&output); return(1102);}
  }

  tacFree(&input); tacFree(&output);

  statusSet(status, __func__, __FILE__, __LINE__, 0);
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
int test_simTTM(
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  if(verbose>0) {
    printf("\n=====================================\n");
    printf("\n%s\n", __func__);
    printf("\n=====================================\n");
  }

  int ret;

  if(verbose>1) printf("\n testing with empty data \n");
  ret=simTTM(NULL, NULL, 0, 0.0, 0, NULL);
  if(verbose>2) printf("did not segfault, that is good.\n");
  if(ret==0) return(1);
  if(verbose>2) printf("\n more testing with empty data \n");
  ret=simTTM(NULL, NULL, 1000, 0.5, 5, NULL);
  if(verbose>2) printf("did not segfault, that is good.\n");
  if(ret==0) return(2);

  /* Probability density function of the Erlang distribution describes perfectly
     the temporal response function of transit-time model for unit impulse function at t=0;
     using the results of that function to validate the compartmental model code. */

  {
  if(verbose>1) {printf("\n 1CM \n"); fflush(stdout);}
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  double k=0.15;
  /* Set sample times */
  int n=simSamples(0.001, 1.0, 200.0, 1, NULL); if(verbose>10) printf("n=%d\n", n);
  double x[n];
  simSamples(0.001, 1.0, 200.0, 1, x);
  /* Calculate input function of Erlang pdf with N=1 as f(x)=k*exp(-k*x) */
  double C0[n];
  for(int i=0; i<n; i++) C0[i]=k*exp(-k*x[i]);
  /* Calculate correct result for one-way catenary 1CM using Erlang pdf with N=2 */
  double cy[n]; double p[]={1.0, 0.15, 2.0};
  if(mfEvalY("erlangpdf", 3, p, n, x, cy, 0)!=0) return(100);
  /* Simulate */
  double y[n];
  if(simTTM(x, C0, n, k, 1, y)!=0) return(101);
  if(verbose>2) {
    printf("\nX\tC0\tcY\tY\n");
    for(int i=0; i<n; i++) printf("%g\t%g\t%g\t%g\n", x[i], C0[i], cy[i], y[i]);
    fflush(stdout);
  }
  /* Check the result */
  for(int i=0; i<n; i++) {
    if(!doubleMatch(cy[i], y[i], 0.0001) && !doubleMatchRel(cy[i], y[i], 0.004)) {
      if(verbose>0) printf("x[%d]=%g C0=%g cy=%g y=%g\n", i, x[i], C0[i], cy[i], y[i]);
      return(102);
    }
  }
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  }

  {
  if(verbose>1) {printf("\n 2CM \n"); fflush(stdout);}
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  double k=0.15;
  /* Set sample times */
  int n=simSamples(0.001, 1.0, 200.0, 1, NULL); if(verbose>10) printf("n=%d\n", n);
  double x[n];
  simSamples(0.001, 1.0, 200.0, 1, x);
  /* Calculate input function of Erlang pdf with N=1 as f(x)=k*exp(-k*x) */
  double C0[n];
  for(int i=0; i<n; i++) C0[i]=k*exp(-k*x[i]);
  /* Calculate correct result for one-way catenary 2CM using Erlang pdf with N=3 */
  double cy[n]; double p[]={1.0, 0.15, 3.0};
  if(mfEvalY("erlangpdf", 3, p, n, x, cy, 0)!=0) return(110);
  /* Simulate */
  double y[n];
  if(simTTM(x, C0, n, k, 2, y)!=0) return(111);
  if(verbose>2) {
    printf("\nX\tC0\tcY\tY\n");
    for(int i=0; i<n; i++) printf("%g\t%g\t%g\t%g\n", x[i], C0[i], cy[i], y[i]);
    fflush(stdout);
  }
  /* Check the result */
  for(int i=0; i<n; i++) {
    if(!doubleMatch(cy[i], y[i], 0.0001) && !doubleMatchRel(cy[i], y[i], 0.004)) {
      if(verbose>0) printf("x[%d]=%g C0=%g cy=%g y=%g\n", i, x[i], C0[i], cy[i], y[i]);
      return(112);
    }
  }
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  }

  {
  if(verbose>1) {printf("\n 3CM \n"); fflush(stdout);}
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  double k=0.15;
  /* Set sample times */
  int n=simSamples(0.001, 1.0, 200.0, 1, NULL); if(verbose>10) printf("n=%d\n", n);
  double x[n];
  simSamples(0.001, 1.0, 200.0, 1, x);
  /* Calculate input function of Erlang pdf with N=2 */
  double C0[n]; double pi[]={1.0, 0.15, 2.0};
  if(mfEvalY("erlangpdf", 3, pi, n, x, C0, 0)!=0) return(120);
  /* Calculate correct result for one-way catenary 3CM using Erlang pdf with N=5 */
  double cy[n]; double p[]={1.0, 0.15, 5.0};
  if(mfEvalY("erlangpdf", 3, p, n, x, cy, 0)!=0) return(121);
  /* Simulate */
  double y[n];
  if(simTTM(x, C0, n, k, 3, y)!=0) return(122);
  if(verbose>2) {
    printf("\nX\tC0\tcY\tY\n");
    for(int i=0; i<n; i++) printf("%g\t%g\t%g\t%g\n", x[i], C0[i], cy[i], y[i]);
    fflush(stdout);
  }
  /* Check the result */
  for(int i=0; i<n; i++) {
    if(!doubleMatch(cy[i], y[i], 0.0001) && !doubleMatchRel(cy[i], y[i], 0.004)) {
      if(verbose>0) printf("x[%d]=%g C0=%g cy=%g y=%g\n", i, x[i], C0[i], cy[i], y[i]);
      return(123);
    }
  }
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  }


  statusSet(status, __func__, __FILE__, __LINE__, 0);
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
int test_simTTM_i(
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  if(verbose>0) {
    printf("\n=====================================\n");
    printf("\n%s\n", __func__);
    printf("\n=====================================\n");
  }

  int ret;

  if(verbose>1) printf("\n testing with empty data \n");
  ret=simTTM_i(NULL, NULL, 0, 0.0, 0, NULL);
  if(verbose>2) printf("did not segfault, that is good.\n");
  if(ret==0) return(1);
  if(verbose>2) printf("\n more testing with empty data \n");
  ret=simTTM_i(NULL, NULL, 1000, 0.5, 5, NULL);
  if(verbose>2) printf("did not segfault, that is good.\n");
  if(ret==0) return(2);

  /* Probability density function of the Erlang distribution describes perfectly
     the temporal response function of transit-time model for unit impulse function at t=0;
     using the results of that function to validate the compartmental model code. */

  {
  if(verbose>1) {printf("\n 1CM \n"); fflush(stdout);}
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  double k=0.15;
  /* Set sample times */
  int n=simSamples(0.001, 1.0, 200.0, 1, NULL); if(verbose>10) printf("n=%d\n", n);
  double x[n];
  simSamples(0.001, 1.0, 200.0, 1, x);
  /* Calculate input function integral of Erlang pdf with N=1 as f(x)=1-exp(-k*x) */
  double iC0[n];
  for(int i=0; i<n; i++) iC0[i]=1.0-exp(-k*x[i]);
  /* Calculate correct result for one-way catenary 1CM using Erlang pdf with N=2 */
  double cy[n]; double p[]={1.0, 0.15, 2.0};
  if(mfEvalY("erlangpdf", 3, p, n, x, cy, 0)!=0) return(100);
  /* Simulate */
  double y[n];
  if(simTTM_i(x, iC0, n, k, 1, y)!=0) return(101);
  if(verbose>2) {
    printf("\nX\tiC0\tcY\tY\n");
    for(int i=0; i<n; i++) printf("%g\t%g\t%g\t%g\n", x[i], iC0[i], cy[i], y[i]);
    fflush(stdout);
  }
  /* Check the result */
  for(int i=0; i<n; i++) {
    if(!doubleMatch(cy[i], y[i], 0.0001) && !doubleMatchRel(cy[i], y[i], 0.001)) {
      if(verbose>0) printf("x[%d]=%g iC0=%g cy=%g y=%g\n", i, x[i], iC0[i], cy[i], y[i]);
      return(102);
    }
  }
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  }

  {
  if(verbose>1) {printf("\n 3CM \n"); fflush(stdout);}
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  double k=0.15;
  /* Set sample times */
  int n=simSamples(0.001, 1.0, 200.0, 1, NULL); if(verbose>10) printf("n=%d\n", n);
  double x[n];
  simSamples(0.001, 1.0, 200.0, 1, x);
  /* Calculate input function integral of Erlang pdf with N=1 as f(x)=1-exp(-k*x) */
  double iC0[n];
  for(int i=0; i<n; i++) iC0[i]=1.0-exp(-k*x[i]);
  /* Calculate correct result for one-way catenary 3CM using Erlang pdf with N=4 */
  double cy[n]; double p[]={1.0, 0.15, 4.0};
  if(mfEvalY("erlangpdf", 3, p, n, x, cy, 0)!=0) return(110);
  /* Simulate */
  double y[n];
  if(simTTM_i(x, iC0, n, k, 3, y)!=0) return(111);
  if(verbose>2) {
    printf("\nX\tiC0\tcY\tY\n");
    for(int i=0; i<n; i++) printf("%g\t%g\t%g\t%g\n", x[i], iC0[i], cy[i], y[i]);
    fflush(stdout);
  }
  /* Check the result */
  for(int i=0; i<n; i++) {
    if(!doubleMatch(cy[i], y[i], 0.0001) && !doubleMatchRel(cy[i], y[i], 0.001)) {
      if(verbose>0) printf("x[%d]=%g iC0=%g cy=%g y=%g\n", i, x[i], iC0[i], cy[i], y[i]);
      return(112);
    }
  }
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  }

  {
  if(verbose>1) {printf("\n 2CM \n"); fflush(stdout);}
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  double k=0.15;
  /* Set sample times */
  int n=simSamples(0.001, 1.0, 200.0, 1, NULL); if(verbose>10) printf("n=%d\n", n);
  double x[n];
  simSamples(0.001, 1.0, 200.0, 1, x);
  /* Calculate input function integral of Erlang pdf with N=2 as f(x)=1-exp(-k*x)-k*x*exp(-k*x) */
  double iC0[n];
  for(int i=0; i<n; i++) iC0[i]=1.0-exp(-k*x[i])-k*x[i]*exp(-k*x[i]);
  /* Calculate correct result for one-way catenary 2CM using Erlang pdf with N=4 */
  double cy[n]; double p[]={1.0, 0.15, 4.0};
  if(mfEvalY("erlangpdf", 3, p, n, x, cy, 0)!=0) return(200);
  /* Simulate */
  double y[n];
  if(simTTM_i(x, iC0, n, k, 2, y)!=0) return(201);
  if(verbose>2) {
    printf("\nX\tiC0\tcY\tY\n");
    for(int i=0; i<n; i++) printf("%g\t%g\t%g\t%g\n", x[i], iC0[i], cy[i], y[i]);
    fflush(stdout);
  }
  /* Check the result */
  for(int i=0; i<n; i++) {
    if(!doubleMatch(cy[i], y[i], 0.0001) && !doubleMatchRel(cy[i], y[i], 0.001)) {
      if(verbose>0) printf("x[%d]=%g iC0=%g cy=%g y=%g\n", i, x[i], iC0[i], cy[i], y[i]);
      return(202);
    }
  }
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  }

  {
  if(verbose>1) {printf("\n 8CM \n"); fflush(stdout);}
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  double k=0.15;
  /* Set sample times */
  int n=simSamples(0.001, 1.0, 200.0, 1, NULL); if(verbose>10) printf("n=%d\n", n);
  double x[n];
  simSamples(0.001, 1.0, 200.0, 1, x);
  /* Calculate input function integral of Erlang pdf with N=10 as f(x)=1-exp(-k*x)-k*x*exp(-k*x) */
  double iC0[n];
  for(int i=0; i<n; i++) iC0[i]=1.0-exp(-k*x[i])-k*x[i]*exp(-k*x[i]);
  /* Calculate correct result for one-way catenary 2CM using Erlang pdf with N=4 */
  double cy[n]; double p[]={1.0, 0.15, 10.0};
  if(mfEvalY("erlangpdf", 3, p, n, x, cy, 0)!=0) return(210);
  /* Simulate */
  double y[n];
  if(simTTM_i(x, iC0, n, k, 8, y)!=0) return(211);
  if(verbose>2) {
    printf("\nX\tiC0\tcY\tY\n");
    for(int i=0; i<n; i++) printf("%g\t%g\t%g\t%g\n", x[i], iC0[i], cy[i], y[i]);
    fflush(stdout);
  }
  /* Check the result */
  for(int i=0; i<n; i++) {
    if(!doubleMatch(cy[i], y[i], 0.0001) && !doubleMatchRel(cy[i], y[i], 0.001)) {
      if(verbose>0) printf("x[%d]=%g iC0=%g cy=%g y=%g\n", i, x[i], iC0[i], cy[i], y[i]);
      return(212);
    }
  }
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  }



  statusSet(status, __func__, __FILE__, __LINE__, 0);
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
