/** @file csv.c
 *  @brief CSV struct processing.
 *  @test Update test function arguments. 
 */
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <time.h>
#include <string.h>
/*****************************************************************************/
#include "tpccsv.h"
/*****************************************************************************/

/*****************************************************************************/
/** Initiate the CSV struct before any use.
    @author Vesa Oikonen
    @sa csvFree, csvAllocate, csvRead
 */
void csvInit(
  /** Pointer to CSV. */
  CSV *csv
) {
  if(csv==NULL) return;
  csv->c=NULL; csv->_item_nr=csv->nr=csv->row_nr=csv->col_nr=0;
  csv->separator='\t'; //(char)0;
}
/*****************************************************************************/

/*****************************************************************************/
/** Free memory allocated for CSV data. All contents are destroyed.
    @pre Before first use initialize the CSV struct with csvInit().
    @author Vesa Oikonen
    @sa csvInit, csvAllocate, csvWrite
 */
void csvFree(
  /** Pointer to CSV. */
  CSV *csv
) {
  if(csv==NULL) return;
  for(int i=0; i<csv->_item_nr; i++) free(csv->c[i].content);
  free(csv->c);
  csvInit(csv);
}
/*****************************************************************************/

/*****************************************************************************/
/** Allocate memory for list of field items for CSV data.

    Any previous content is preserved.
    @pre Before first use initialize the CSV struct with csvInit().
    @return enum tpcerror (TPCERROR_OK when successful).
    @author Vesa Oikonen
    @sa csvInit, csvFree, csvDuplicate, csvSetDimensions, csvList
 */
int csvAllocate(
  /** Pointer to CSV. */
  CSV *csv,
  /** Nr of field items to add to the list. */
  int nr
) {
  if(csv==NULL) return TPCERROR_FAIL;
  if(nr<1) return TPCERROR_OK;

  /* If not allocated previously, then just allocate and thats it */
  if(csv->_item_nr==0) {
    csv->c=(CSV_item*)calloc(nr, sizeof(CSV_item));
    if(csv->c==NULL) return TPCERROR_OUT_OF_MEMORY;
    csv->_item_nr=nr;
    return TPCERROR_OK;
  }

  /* How many unused places there are already? */
  int unused=csv->_item_nr-csv->nr;
  /* Check if enough memory is already allocated */
  if(unused>=nr) return TPCERROR_OK;

  /* Ok, we have to reallocate more memory for the list */
  nr+=csv->nr;
  CSV_item *new_list;
  new_list=(CSV_item*)realloc(csv->c, nr*sizeof(CSV_item));
  if(new_list==NULL) return TPCERROR_OUT_OF_MEMORY;
  else csv->c=new_list;
  /* Initiate the list */
  for(int i=csv->_item_nr; i<nr; i++) {
    csv->c[i].row=0; csv->c[i].col=0; csv->c[i].content=NULL;
  }
  csv->_item_nr=nr;
  return TPCERROR_OK;
}
/*****************************************************************************/

/*****************************************************************************/
/** @brief Make a duplicate of CSV structure.
    @details Previous contents are copied.
    In the duplicate the space is allocated for row_nr times col_nr cells. 
    @sa csvInit, csvAllocate, csvFree
    @return Returns TPCERROR status (0 when successful).
 */
int csvDuplicate(
  /** Pointer to the source CSV. */
  CSV *csv1,
  /** Pointer to the target CSV; must be initiated; any old contents are deleted. */
  CSV *csv2
) {
  if(csv1==NULL || csv2==NULL) return TPCERROR_FAIL;
  if(csv1->nr<1 || csv1->_item_nr<csv1->nr || csv1->row_nr<1 || csv1->col_nr<1) return TPCERROR_NO_DATA;

  /* Empty the duplicate */
  csvFree(csv2);

  /* Allocate memory for csv2 */
  int ret=csvAllocate(csv2, csv1->row_nr*csv1->col_nr);
  if(ret!=TPCERROR_OK) return(ret);
  csv2->col_nr=csv1->col_nr;
  csv2->row_nr=csv1->row_nr;
  csv2->nr=csv2->col_nr*csv2->row_nr;

  /* Copy the contents */
  csv2->separator=csv1->separator;
  int ni=0, fulln=0;
  for(int ri=0; ri<csv2->row_nr; ri++)
    for(int ci=0; ci<csv2->col_nr; ci++) {
      csv2->c[ni].row=ri;
      csv2->c[ni].col=ci;
      char *cp=csvCell(csv1, ri, ci);
      if(cp==NULL) csv2->c[ni].content=NULL; else {csv2->c[ni].content=strdup(cp); fulln++;}
      ni++;
    }
  if(fulln<1) return TPCERROR_NO_DATA;
  return TPCERROR_OK;
}
/*****************************************************************************/

/*****************************************************************************/
/** Add specified string to CSV.
    @return tpcerror (TPCERROR_OK when successful).
    @pre Before first use initialize the CSV struct with csvInit().
    @author Vesa Oikonen
    @sa csvInit, csvPutDouble, csvPutInt, csvPutLine
 */
int csvPutString(
  /** Pointer to initiated CSV; previous contents are not changed. */
  CSV *csv,
  /** Field string to add; can be empty ("" or NULL). */
  const char *s,
  /** New line (1) or same line (0). */
  int newline
) {
  if(csv==NULL) return TPCERROR_FAIL;
  /* Allocate memory, if needed, and if needed, then a bigger chunk */
  int avail=csv->_item_nr-csv->nr;
  if(avail<1) {
    int ret=csvAllocate(csv, 10);
    if(ret!=TPCERROR_OK) return ret;
  }
  /* Write the item */
  if(s==NULL || *s=='\0') csv->c[csv->nr].content=strdup(""); 
  else csv->c[csv->nr].content=strdup(s);
  if(csv->c[csv->nr].content==NULL) return TPCERROR_OUT_OF_MEMORY;
  /* Set the row and column number */
  int r, c;
  if(csv->nr==0) {r=c=0;}
  else {
    if(newline!=0) {
      c=0;
      r=csv->c[csv->nr-1].row+1;
    } else {
      c=csv->c[csv->nr-1].col+1;
      r=csv->c[csv->nr-1].row;
    }
  }
  csv->c[csv->nr].col=c; csv->c[csv->nr].row=r;
  /* Update row_nr and max col nr, if necessary */
  c++; r++;
  if(c>csv->col_nr) csv->col_nr=c;
  if(r>csv->row_nr) csv->row_nr=r;

  csv->nr++;
  return TPCERROR_OK;
}
/*****************************************************************************/

/*****************************************************************************/
/** Add specified double value as string into CSV.
    @return tpcerror (TPCERROR_OK when successful).
    @pre Before first use initialize the CSV struct with csvInit().
    @author Vesa Oikonen
    @sa csvInit, csvPutString, csvPutInt, csvPutLine
 */
int csvPutDouble(
  /** Pointer to initiated CSV; previous contents are not changed. */
  CSV *csv,
  /** Double value to add; NaN is added as an empty field. */
  double v,
  /** New line (1) or same line (0). */
  int newline,
  /** Convert (1) or do not convert (0) commas to semicolons and dots to commas. */
  int tointl
) {
  int ret;
  if(isnan(v)) {
    ret=csvPutString(csv, "", newline);
  } else {
    char s[128];
    sprintf(s, "%g", v); if(tointl) strReplaceChar(s, '.', ',');
    ret=csvPutString(csv, s, newline);
  }
  return ret;
}
/*****************************************************************************/

/*****************************************************************************/
/** Add specified integer value as string into CSV.
    @return tpcerror (TPCERROR_OK when successful).
    @pre Before first use initialize the CSV struct with csvInit().
    @author Vesa Oikonen
    @sa csvInit, csvPutDouble, csvPutString, csvPutLine
 */
int csvPutInt(
  /** Pointer to initiated CSV; previous contents are not changed */
  CSV *csv,
  /** Integer value to add; use csvPutString() to add an empty field */
  int v,
  /** New line (1) or same line (0) */
  int newline
) {
  int ret;
  char s[128];
  sprintf(s, "%d", v);
  ret=csvPutString(csv, s, newline);
  return ret;
}
/*****************************************************************************/

/*****************************************************************************/
/** Count the nr of columns on specified CSV row.
    @author Vesa Oikonen
    @sa csvCell, csvRead, csvIsRegular, csvSetDimensions
    @return The number of columns.
 */
int csvRowLength(
  /** Pointer to CSV */
  CSV *csv,
  /** CSV row index */
  int row
) {
  if(csv==NULL) return(0);
//  int i, n=0;
//  for(i=0; i<csv->nr; i++) if(csv->c[i].row==row) n++;
//  return(n); 
  int n=-1;
  for(int i=0; i<csv->nr; i++) if(csv->c[i].row==row && csv->c[i].col>n) n=csv->c[i].col;
  return(1+n); 
}
/*****************************************************************************/

/*****************************************************************************/
/** Set the number of rows and columns in CSV table, based on the row and column
    stored with each cell. 
    CSV may not be regular, thus maximum row and column numbers are used.

    @author Vesa Oikonen
    @sa csvIsRegular, csvRowLength, csvRead, csvRemoveComments
    @return enum tpcerror (TPCERROR_OK when successful).
 */
int csvSetDimensions(
  /** Pointer to CSV structure. */
  CSV *csv
) {
  if(csv==NULL) return TPCERROR_FAIL;
  if(csv->nr<1) {csv->col_nr=0; csv->row_nr=0; return(TPCERROR_OK);}

  int maxri=0, maxci=0;
  for(int i=0; i<csv->nr; i++) {
    if(csv->c[i].row>maxri) maxri=csv->c[i].row;
    if(csv->c[i].col>maxci) maxci=csv->c[i].col;
  }
  csv->col_nr=1+maxci; csv->row_nr=1+maxri;
  return(TPCERROR_OK);
}
/*****************************************************************************/

/*****************************************************************************/
/** Check whether CSV is regular, that is, each row contain the same number of columns.
    @author Vesa Oikonen
    @sa csvTranspose, csvTrimRight, csvSetDimensions, csvRowLength, csvRead, csvDuplicate
    @return Returns 1 if CSV is regular, 0 if not.
 */
int csvIsRegular(
  /** Pointer to CSV */
  CSV *csv
) {
  if(csv==NULL) return(0);
  if(csv->nr<2) return(1);
  int i, r, n=0, m=0;
  i=0; r=csv->c[i].row; m++;
  for(i=1; i<csv->nr; i++) {
    if(r==csv->c[i].row) {
      m++; 
      continue;}
    if(n>0 && m!=n) return(0);
    r=csv->c[i].row; n=m; m=1;
  }
  if(n>0 && m!=n) return(0);
  return(1);
}
/*****************************************************************************/

/*****************************************************************************/
/** Remove empty cells from the right side of CVS table until a non-empty column is reached.

    Resulting CSV may still not be regular.

    @author Vesa Oikonen
    @sa csvIsRegular, csvRowLength, csvRead, csvTranspose
    @return enum tpcerror (TPCERROR_OK when successful).
 */
int csvTrimRight(
  /** Pointer to CSV */
  CSV *csv
) {
  if(csv==NULL) return TPCERROR_FAIL;
  if(csv->nr<1 || csv->col_nr<2 || csv->row_nr<2) return TPCERROR_NO_DATA;

  int ci=csv->col_nr-1;
  while(ci>0) {
    /* Does this column contain any data? */
    int n=0;
    for(int i=0; i<csv->nr; i++) if(csv->c[i].col==ci) {
      if(csv->c[i].content!=NULL && csv->c[i].content[0]!='\0' && csv->c[i].content[0]!='#') n++;
    }
    if(n>0) break; // yes
    /* No, so let's delete the last column */
    int i=csv->nr-1;
    while(i>=0) {
      if(csv->c[i].col==ci) {
        if(csvRemoveItem(csv, i)!=TPCERROR_OK) return TPCERROR_FAIL;
      }
      i--;
    }
    csv->col_nr--;
    /* Try the previous column */
    ci--;
  }
  return(TPCERROR_OK);
}
/*****************************************************************************/

/*****************************************************************************/
/** Get the CVS field contents in specified row and column.
    @return Returns pointer to the content string, or NULL if not found.
    @author Vesa Oikonen
    @sa csvRead, csvSearchField, csvFindField, csvCellReplace, csvSetDimensions
 */
char* csvCell(
  /** Pointer to CSV. */
  CSV *csv,
  /** CSV row index. */
  int row,
  /** CSV column index. */
  int col
) {
  if(csv==NULL) return((char*)NULL);
  for(int i=0; i<csv->nr; i++) 
    if(csv->c[i].row==row && csv->c[i].col==col)
      return(csv->c[i].content);
  return((char*)NULL); 
}
/*****************************************************************************/

/*****************************************************************************/
/** Replace the value of CVS field.
    @return tpcerror (TPCERROR_OK when successful).
    @author Vesa Oikonen
    @sa csvRead, csvSearchField, csvCell, csvRemoveItem, csvTranspose
 */
int csvCellReplace(
  /** Pointer to CSV. */
  CSV *csv,
  /** CSV row index. */
  int row,
  /** CSV column index. */
  int col,
  /** Field string to add; can be empty ("" or NULL). */
  const char *s
) {
  if(csv==NULL) return(TPCERROR_FAIL);
  int ci=-1;
  for(int i=0; i<csv->nr; i++) if(csv->c[i].row==row && csv->c[i].col==col) {ci=i; break;}
  if(ci<0) return(TPCERROR_MISSING_DATA);
  /* Delete the previous value */
  free(csv->c[ci].content);
  /* Save the new item */
  if(s==NULL || *s=='\0') csv->c[ci].content=strdup(""); 
  else csv->c[ci].content=strdup(s);
  if(csv->c[ci].content==NULL) return(TPCERROR_OUT_OF_MEMORY);
  return(TPCERROR_OK); 
}
/*****************************************************************************/

/*****************************************************************************/
/** Remove item from CSV data structure.
    @return enum tpcerror (TPCERROR_OK when successful).
    @sa csvCellReplace, csvFree, csvDuplicate, csvRemoveEmptyRows
 */
int csvRemoveItem(
  /** Pointer to CSV. */
  CSV *csv,
  /** Index of item to delete. */
  int i
) {
  if(csv==NULL || i<0) return TPCERROR_FAIL;
  if(csv->nr<1 || i>=csv->nr) return TPCERROR_NO_DATA;
  free(csv->c[i].content); csv->c[i].content=NULL;
  for(int j=i+1; j<csv->nr; j++) {
    csv->c[j-1].row=csv->c[j].row;
    csv->c[j-1].col=csv->c[j].col;
    csv->c[j-1].content=csv->c[j].content;
  }
  csv->nr--; csv->c[csv->nr].content=NULL;
  return(TPCERROR_OK); 
}
/*****************************************************************************/

/*****************************************************************************/
/** Remove empty rows from CSV.
    Rows that contain only empty cells or cells that start with '#' are considered empty.
    @return enum tpcerror (TPCERROR_OK when successful).
    @sa csvRemoveComments, csvIsRegular, csvDuplicate, csvCellReplace
 */
int csvRemoveEmptyRows(
  /** Pointer to CSV. */
  CSV *csv
) {
  if(csv==NULL) return TPCERROR_FAIL;
  if(csv->nr<1 || csv->row_nr<1 || csv->col_nr<1) return TPCERROR_NO_DATA;

  /* Check the rows */
  int ri=csv->row_nr-1;
  while(ri>=0) {
    int n=0;
    for(int ci=0; ci<csv->col_nr; ci++) {
      char *cp=csvCell(csv, ri, ci); if(cp==NULL) continue;
      if(*cp=='\0' || *cp=='#') continue;
      n++;
    }
    if(n==0) { // this line is empty
      int i=csv->nr-1;
      while(i>=0) {
        if(csv->c[i].row==ri) csvRemoveItem(csv, i);
        i--;
      }
      for(i=0; i<csv->nr; i++) if(csv->c[i].row>ri) csv->c[i].row--;
      csv->row_nr--;
    }
    ri--;
  }
  return(TPCERROR_OK);
}
/*****************************************************************************/

/*****************************************************************************/
/** Remove those cells from CSV structure whose contents start with '#'.
    @return enum tpcerror (TPCERROR_OK when successful).
    @sa csvIsRegular, csvRemoveEmptyRows
 */
int csvRemoveComments(
  /** Pointer to CSV. */
  CSV *csv
) {
  if(csv==NULL) return(TPCERROR_FAIL);
  if(csv->nr<1) return(TPCERROR_NO_DATA);

  int i=csv->nr-1;
  while(i>=0) {
    if(csv->c[i].content!=NULL && csv->c[i].content[0]=='#') csvRemoveItem(csv, i);
    i--;
  }
  return(TPCERROR_OK);
//  return(csvRemoveEmptyRows(csv));
}
/*****************************************************************************/

/*****************************************************************************/
/// @cond
/** Local function */
static int csvReorgQSort(const void *c1, const void *c2)
{
  if(((CSV_item*)c2)->row > ((CSV_item*)c1)->row) return(-1);
  if(((CSV_item*)c2)->row < ((CSV_item*)c1)->row) return(+1);
  if(((CSV_item*)c2)->col > ((CSV_item*)c1)->col) return(-1);
  if(((CSV_item*)c2)->col < ((CSV_item*)c1)->col) return(+1);
  return(0);
}
/// @endcond
/** Sort CSV data array by increasing row and column numbers.
    @return enum tpcerror (TPCERROR_OK when successful).
    @sa csvTranspose, csvList, csvAllocate, csvWrite
 */
int csvReorg(
  /** Pointer to CSV structure. */
  CSV *d,
  /** Pointer to status data; enter NULL if not needed. */
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  if(verbose>0) printf("%s()\n", __func__);
  /* Check that required data exists */
  if(d==NULL || d->nr<1) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_NO_DATA);
    return TPCERROR_NO_DATA;
  }
  statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_OK);
  if(d->nr<2) return TPCERROR_OK;
  qsort(d->c, d->nr, sizeof(CSV_item), csvReorgQSort);
  return TPCERROR_OK;
}
/*****************************************************************************/

/*****************************************************************************/
/** Transpose data in CSV structure.

    To work properly, CSV should not contain comment cells/lines.
    @return enum tpcerror (TPCERROR_OK when successful).
    @sa csvIsRegular, csvDuplicate, csvCellReplace, csvReorg, csvRmComments
 */
int csvTranspose(
  /** Pointer to CSV.*/
  CSV *csv
) {
  if(csv==NULL) return(TPCERROR_FAIL);
  if(csv->nr<1) return(TPCERROR_NO_DATA);

  /* Get dimensions */
  {
    int ret=csvSetDimensions(csv);
    if(ret!=TPCERROR_OK) return(ret);
  }
  if(csv->row_nr<1 || csv->col_nr<1) return(TPCERROR_NO_DATA);

  /* If only one row and column, then nothing to do */
  if(csv->row_nr==1 && csv->col_nr==1) return(TPCERROR_OK);

  /* Switch column and row numbers */
  for(int i=0; i<csv->nr; i++) {
    int cr=csv->c[i].col;
    csv->c[i].col=csv->c[i].row;
    csv->c[i].row=cr;
  }
  {
    int cn=csv->col_nr;
    csv->col_nr=csv->row_nr;
    csv->row_nr=cn;
  }

  /* Sort CSV data by rows and columns */
  {
    int ret=csvReorg(csv, NULL);
    if(ret!=TPCERROR_OK) return(ret);
  }

  return(TPCERROR_OK);
}
/*****************************************************************************/

/*****************************************************************************/
