/** @file csvfind.c
 *  @brief Find field contents in CSV struct.
 */
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include "tpcextensions.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <strings.h>
#include <ctype.h>
#include <math.h>
#include <time.h>
/*****************************************************************************/
#include "tpccsv.h"
/*****************************************************************************/

/*****************************************************************************/
/** Find the CSV field with specified content. Search is case-insensitive, but
    otherwise match must be exact.
    @sa csvSearchField, csvCell, csvRead, csvRemoveItem
    @return field index, or <0 if not found.
    @author Vesa Oikonen
 */
int csvFindField(
  /** Pointer to CSV. */
  CSV *csv,
  /** Field value to be searched for. */
  const char *s,
  /** CSV item index [0..nr-1] from which search is started. */
  int start_index
) {
  if(csv==NULL || s==NULL) return -2;
  if(start_index<0 || start_index>=csv->nr) return -2;
  for(int i=start_index; i<csv->nr; i++) if(csv->c[i].content!=NULL)
    if(strcasecmp(csv->c[i].content, s)==0) return i;
  return -1;
}
/*****************************************************************************/

/*****************************************************************************/
/** Find the CSV field which contains the given search string.
    Search is case-insensitive.
    @sa csvFindField, csvCell, csvRead, csvRemoveItem
    @return field index, or <0 if not found.
    @author Vesa Oikonen
 */
int csvSearchField(
  /** Pointer to CSV. */
  CSV *csv,
  /** String to be searched for inside field values. */
  const char *s,
  /** CSV field index [0..nr-1] from which search is started. */
  int start_index
) {
  if(csv==NULL || s==NULL) return -2;
  if(start_index<0 || start_index>=csv->nr) return -2;
  for(int i=start_index; i<csv->nr; i++) {
    if(csv->c[i].content==NULL || csv->c[i].content[0]=='\0') {
      if(*s=='\0') return i; else continue;
    }
    if(*s=='\0') continue;
    if(strcasestr(csv->c[i].content, s)!=NULL) return i;
  }
  return -1;
}
/*****************************************************************************/

/*****************************************************************************/
