/** @file dcmdictionary.c
    @brief DICOM dictionary for TPCCLIB.
 */
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <time.h>
#include <string.h>
/*****************************************************************************/
#include "tpcextensions.h"
/*****************************************************************************/
#include "tpcdcm.h"
/*****************************************************************************/

/*****************************************************************************/
/*! @cond PRIVATE */
/** One item for DICOM dictionary. */
typedef struct DCM_DICT {
  /** TAG group. */
  unsigned short int group;
  /** TAG element. */
  unsigned short int element;
  /** VR name; NULL terminated string. */
  char vr[8];
  /** Tag type; NULL terminated string. 
    - 1: mandatory.
    - 1C: mandatory only under certain conditions.
    - 2: required to exist, but can have a null value.
    - 2C: required to exist only under certain conditions.
    - 3: optional.
   */
  char type[8];
  /** Tag description; NULL terminated string. */
  char descr[80];
} DCM_DICT;

/** DICOM dictionary used by TPCCLIB.

    Reference: DICOM PS3.6 2017a Data Dictionary.

    This list does not contain nearly all item. More items from DICOM standard
    can be added freely, as long as whole library is recompiled. 
    In case of duplicate items, the first one is used. 
    Table can not be accessed directly outside the c file.

    Table is obligatory for reading DICOM with implicit VR.

    @note Tag types are currently not necessarily correct.

    @sa dcmDictSize, dcmDictIndexTag, dcmDictIndexVR, dcmDictIndexType,
        dcmDictIndexDescr, dcmDictFindTag
*/
static DCM_DICT dcmdict[]={
{ 0x0000, 0x0000, "UL", "3", "Command Group Length" },
{ 0x0000, 0x0001, "UL", "2", "Length to End" },
{ 0x0000, 0x0002, "UI", "3", "Affected SOP Class UID" },
{ 0x0000, 0x0003, "UI", "3", "Requested SOP Class UID" },
{ 0x0000, 0x0010, "SH", "2", "Command Recognition Code" },
{ 0x0000, 0x0100, "US", "3", "Command Field" }, //< C-STORE-RQ, C-STORE-RSP, C-GET-RQ, ...
{ 0x0000, 0x0110, "US", "3", "Message ID" },
{ 0x0000, 0x0120, "US", "3", "Message Id being Responded to" },
{ 0x0000, 0x0200, "AE", "2", "Initiator" },
{ 0x0000, 0x0300, "AE", "2", "Receiver" },
{ 0x0000, 0x0400, "AE", "2RET", "Find Location" },
{ 0x0000, 0x0600, "AE", "3", "Move Destination" },
{ 0x0000, 0x0700, "US", "3", "Priority" },
{ 0x0000, 0x0800, "US", "3", "Data Set Type" },
{ 0x0000, 0x0850, "US", "2", "Number of Matches" },
{ 0x0000, 0x0860, "US", "2", "Response Sequence Number" },
{ 0x0000, 0x0900, "US", "3", "Status" },
{ 0x0000, 0x0901, "AT", "3", "Offending Element" },
{ 0x0000, 0x0902, "LO", "3", "Error Comment" },
{ 0x0000, 0x0903, "US", "3", "Error ID" },
{ 0x0000, 0x1000, "UI", "3", "Affected SOP Instance UID" },
{ 0x0000, 0x1001, "UI", "3", "Requested SOP Instance UID" },
{ 0x0000, 0x1002, "US", "3", "Event Type ID" },
{ 0x0000, 0x1005, "AT", "3", "Attribute Identifier List" },
{ 0x0000, 0x1008, "US", "3", "Action Type ID" },
{ 0x0000, 0x1012, "UI", "3", "Requested SOP Instance UID List" },
{ 0x0000, 0x1020, "US", "3", "Number of Remaining Sub-operations" },
{ 0x0000, 0x1021, "US", "3", "Number of Completed Sub-operations" },
{ 0x0000, 0x1022, "US", "3", "Number of Failed Sub-operations" },
{ 0x0000, 0x1023, "US", "3", "Number of Warning Sub-operations" },
{ 0x0000, 0x1030, "AE", "3", "Move Originator Application Entity Title" },
{ 0x0000, 0x1031, "US", "3", "Move Originator Message ID" },
{ 0x0000, 0x4000, "LT", "2", "Dialog Receiver" },
{ 0x0000, 0x4010, "LT", "2", "Terminal Type" },
{ 0x0000, 0x5010, "LO", "3RET", "Message Set ID" },
{ 0x0000, 0x5020, "LO", "3RET", "End Message Set ID" },

{ 0x0002, 0x0000, "UL", "3", "Meta Element Group Length" },
{ 0x0002, 0x0001, "OB", "3", "File Meta Information Version" },
{ 0x0002, 0x0002, "UI", "3", "Media Stored SOP Class UID" },
{ 0x0002, 0x0003, "UI", "3", "Media Stored SOP Instance UID" },
{ 0x0002, 0x0010, "UI", "3", "Transfer Syntax UID" },
{ 0x0002, 0x0012, "UI", "3", "Implementation Class UID" },
{ 0x0002, 0x0013, "SH", "3", "Implementation Version Name" },
{ 0x0002, 0x0016, "AE", "3", "Source Application Entity Title" },
{ 0x0002, 0x0017, "AE", "3", "Sending Application Entity Title" },
{ 0x0002, 0x0018, "AE", "3", "Receiving Application Entity Title" },
{ 0x0002, 0x0100, "UI", "3", "Private Information Creator UID" },
{ 0x0002, 0x0102, "OB", "3", "Private Information" },

{ 0x0004, 0x0000, "UL", "3", "File Set Group Length" },
{ 0x0004, 0x1130, "CS", "3", "File-set ID" },
{ 0x0004, 0x1141, "CS", "3", "File-set Descriptor File File ID" },
{ 0x0004, 0x1142, "CS", "3", "Specific Character Set of File-set Descriptor File" },
{ 0x0004, 0x1200, "UL", "3", "Offset of the First Directory Record of the Root Directory Entity" },
{ 0x0004, 0x1202, "UL", "3", "Offset of the Last Directory Record of the Root Directory Entity" },
{ 0x0004, 0x1212, "US", "3", "File-set Consistency Flag" },
{ 0x0004, 0x1220, "SQ", "3", "Directory Record Sequence" },
{ 0x0004, 0x1400, "UL", "3", "Next Directory Record Offset" },
{ 0x0004, 0x1410, "US", "3", "Record In-use Flag" },
{ 0x0004, 0x1420, "UL", "3", "Offset of Referenced Lower-Level Directory Entity" },
{ 0x0004, 0x1430, "CS", "3", "Directory Record Type" }, //< PATIENT,STUDY,SERIES,IMAGE
{ 0x0004, 0x1432, "UI", "3", "Private Record UID" },
{ 0x0004, 0x1500, "CS", "3", "Referenced File ID" },
{ 0x0004, 0x1504, "UL", "3RET", "MRDR Directory Record Offset" },
{ 0x0004, 0x1510, "UI", "3", "Referenced SOP Class UID in File" },
{ 0x0004, 0x1511, "UI", "3", "Referenced SOP Instance UID in File" },
{ 0x0004, 0x1512, "UI", "3", "Referenced Transfer Syntax UID in File" },
{ 0x0004, 0x151A, "UI", "3", "Referenced Related General SOP Class UID in File" },
{ 0x0004, 0x1600, "UL", "3RET", "Number of References" },

{ 0x0008, 0x0000, "UL", "3", "Identifying Group Length" },
{ 0x0008, 0x0001, "UL", "2RET", "Length to End" }, //< Retired
{ 0x0008, 0x0005, "CS", "3", "Specific Character Set" },
{ 0x0008, 0x0006, "SQ", "3", "Language Code Sequence" },
/** ORIGINAL\PRIMARY, DERIVED\SECONDARY, ... */
{ 0x0008, 0x0008, "CS", "3", "Image Type" },
{ 0x0008, 0x000A, "US", "3", "Sequence Item Number" },
{ 0x0008, 0x0010, "CS", "2", "Recognition Code" }, //< Retired
{ 0x0008, 0x0012, "DA", "3", "Instance Creation Date" },
{ 0x0008, 0x0013, "TM", "3", "Instance Creation Time" },
{ 0x0008, 0x0014, "UI", "3", "Instance Creator UID" },
{ 0x0008, 0x0015, "DT", "3", "Instance Coercion Date Time" },
{ 0x0008, 0x0016, "UI", "3", "SOP Class UID" }, //< See dcmsop.c
{ 0x0008, 0x0018, "UI", "3", "SOP Instance UID" }, //< Globally unique in each file.
{ 0x0008, 0x001A, "UI", "3", "Related General SOP Class UID" },
{ 0x0008, 0x001B, "UI", "3", "Original Specialized SOP Class UID" },
{ 0x0008, 0x0020, "DA", "3", "Study Date" }, // YYYY.MM.DD
{ 0x0008, 0x0021, "DA", "3", "Series Date" },
{ 0x0008, 0x0022, "DA", "3", "Acquisition Date" },
{ 0x0008, 0x0023, "DA", "2C", "Content Date" },
{ 0x0008, 0x0024, "DA", "3", "Overlay Date" }, //< Retired
{ 0x0008, 0x0025, "DA", "3", "Curve Date" }, //< Retired
{ 0x0008, 0x002A, "DT", "3", "Acquisition Date Time" },
{ 0x0008, 0x0030, "TM", "3", "Study Time" }, // hh.mm.ss.frac
{ 0x0008, 0x0031, "TM", "3", "Series Time" },
{ 0x0008, 0x0032, "TM", "3", "Acquisition Time" },
{ 0x0008, 0x0033, "TM", "2C", "Content Time" },
{ 0x0008, 0x0034, "TM", "3", "Overlay Time" }, //< Retired
{ 0x0008, 0x0035, "TM", "3", "Curve Time" }, //< Retired
{ 0x0008, 0x0040, "US", "2RET", "Old Data Set Type" }, //< Retired
{ 0x0008, 0x0041, "SH", "2RET", "Old Data Set Subtype" }, //< Retired
{ 0x0008, 0x0042, "CS", "3RET", "Nuclear Medicine Series Type" }, //< Retired
{ 0x0008, 0x0050, "SH", "3", "Accession Number" },
{ 0x0008, 0x0051, "SQ", "3", "Issuer of Accession Number Sequence" },
{ 0x0008, 0x0052, "CS", "3", "Query/Retrieve Level" }, //< STUDY, SERIES, IMAGE
{ 0x0008, 0x0053, "CS", "1", "Query/Retrieve View" },
{ 0x0008, 0x0054, "AE", "3", "Retrieve AE Title" },
{ 0x0008, 0x0056, "CS", "3", "Instance Availability" },
{ 0x0008, 0x0058, "AE", "3", "Failed SOP Instance UID List" },
{ 0x0008, 0x0060, "CS", "3", "Modality" }, // PT,CT,NM,MR,ST,DS,DR,US,OT,...
{ 0x0008, 0x0061, "CS", "3CP", "Modalities in Study" },
{ 0x0008, 0x0062, "UI", "3", "SOP Classes in Study" },
{ 0x0008, 0x0063, "SQ", "3", "Anatomic Regions in Study Code Sequence" },
{ 0x0008, 0x0064, "CS", "3", "Conversion Type" },
{ 0x0008, 0x0068, "CS", "3", "Presentation Intent Type" },
{ 0x0008, 0x0070, "LO", "3", "Manufacturer" },
{ 0x0008, 0x0080, "LO", "1C", "Institution Name" }, //< Required if no (0008,0082)
{ 0x0008, 0x0081, "ST", "3", "Institution Address" },
{ 0x0008, 0x0082, "SQ", "1C", "Institution Code Sequence" }, //< Required if no (0008,0080)
{ 0x0008, 0x0090, "PN", "3", "Referring Physician's Name" },
{ 0x0008, 0x0092, "ST", "3", "Referring Physician's Address" },
{ 0x0008, 0x0094, "SH", "3", "Referring Physician's Telephone Numbers" },
{ 0x0008, 0x0096, "SQ", "3", "Referring Physician Identification Sequence" },
{ 0x0008, 0x0100, "SH", "1C", "Code Value" }, //< The identifier of the Coded Entry.
{ 0x0008, 0x0101, "LO", "1", "Extended Code Value" }, //< DICOS
{ 0x0008, 0x0102, "SH", "1COD", "Coding Scheme Designator" }, //< Mandatory if 0008,0100 is present.
{ 0x0008, 0x0103, "SH", "1COD", "Coding Scheme Version" },
{ 0x0008, 0x0104, "LO", "1", "Code Meaning" },
{ 0x0008, 0x0105, "CS", "3COD", "Mapping Resource" },
{ 0x0008, 0x0106, "DT", "1COD", "Context Group Version" },
{ 0x0008, 0x0107, "DT", "1COD", "Context Group Local Version" },
{ 0x0008, 0x010B, "CS", "3COD", "Context Group Extension Flag" },
{ 0x0008, 0x010C, "UI", "3COD", "Coding Scheme UID" },
{ 0x0008, 0x010D, "UI", "1COD", "Context Group Extension Creator UID" },
{ 0x0008, 0x010E, "SQ", "3COD", "Mapping Resource Sequence" },
{ 0x0008, 0x010F, "CS", "3COD", "Context Identifier" },
{ 0x0008, 0x0110, "SQ", "3COD", "Coding Scheme Identification Sequence" },
{ 0x0008, 0x0112, "LO", "3COD", "Coding Scheme Registry" },
{ 0x0008, 0x0114, "ST", "3COD", "Coding Scheme External ID" },
{ 0x0008, 0x0115, "ST", "3COD", "Coding Scheme Name" },
{ 0x0008, 0x0116, "ST", "3COD", "Coding Scheme Responsible Organization" },
{ 0x0008, 0x0117, "UI", "3COD", "Context UID" },
{ 0x0008, 0x0118, "UI", "3COD", "Mapping Resource UID" },
{ 0x0008, 0x0201, "SH", "3", "Time Zone Offset From UTC" },
{ 0x0008, 0x1000, "LO", "2RET", "Network ID" },
{ 0x0008, 0x1010, "SH", "3", "Station Name" },
{ 0x0008, 0x1030, "LO", "3", "Study Description" },
{ 0x0008, 0x1032, "SQ", "3", "Procedure Code Sequence" },
{ 0x0008, 0x103E, "LO", "3", "Series Description" }, //< Add for derived DICOM image.
{ 0x0008, 0x103F, "SQ", "3", "Series Description Code Sequence" }, //< Remove from derived DICOM image.
{ 0x0008, 0x1040, "LO", "3", "Institutional Department Name" },
{ 0x0008, 0x1048, "PN", "3", "Physician of Record" },
{ 0x0008, 0x1049, "SQ", "1", "Physicians​ Of Record ​Identification ​Sequence" },
{ 0x0008, 0x1050, "PN", "3", "Performing Physician's Name" },
{ 0x0008, 0x1052, "SQ", "3", "Performing Physician Identification Sequence" },
{ 0x0008, 0x1060, "PN", "3", "Name of Physician(s) Reading Study" },
{ 0x0008, 0x1062, "SQ", "3", "Physician(s) Reading Study Identification Sequence" },
{ 0x0008, 0x1070, "PN", "3", "Operator's Name" },
{ 0x0008, 0x1072, "SQ", "3", "Operator Identification Sequence" },
{ 0x0008, 0x1080, "LO", "3", "Admitting Diagnoses Description" },
{ 0x0008, 0x1084, "SQ", "3", "Admitting Diagnosis Code Sequence" },
{ 0x0008, 0x1090, "LO", "3", "Manufacturer's Model Name" },
{ 0x0008, 0x1100, "SQ", "3", "Referenced Results Sequence" },
{ 0x0008, 0x1110, "SQ", "3", "Referenced Study Sequence" },
{ 0x0008, 0x1111, "SQ", "3", "Referenced Performed Procedure Step Sequence" },
{ 0x0008, 0x1115, "SQ", "3", "Referenced Series Sequence" },
{ 0x0008, 0x1120, "SQ", "3", "Referenced Patient Sequence" },
{ 0x0008, 0x1125, "SQ", "3", "Referenced Visit Sequence" },
{ 0x0008, 0x1130, "SQ", "3", "Referenced Overlay Sequence" },
{ 0x0008, 0x1134, "SQ", "3", "Referenced Stereometric Instance Sequence" },
{ 0x0008, 0x113A, "SQ", "3", "Referenced Waveform Sequence" },
{ 0x0008, 0x1140, "SQ", "3", "Referenced Image Sequence" },
{ 0x0008, 0x1145, "SQ", "3", "Referenced Curve Sequence" },
{ 0x0008, 0x114A, "SQ", "3WAV", "Referenced Instance Sequence" },
{ 0x0008, 0x114B, "LO", "3WAV", "Referenced Real World Value Mapping Instance Sequence" },
{ 0x0008, 0x1150, "UI", "1C", "Referenced SOP Class UID" }, //< Required if 0008,1140 is sent
{ 0x0008, 0x1155, "UI", "1C", "Referenced SOP Instance UID" }, //< Required if 0008,1140 is sent
{ 0x0008, 0x115A, "UI", "3", "SOP Classes Supported" },
{ 0x0008, 0x1160, "IS", "3", "Referenced Frame Number" },
{ 0x0008, 0x1161, "UL", "3", "Simple Frame List" },
{ 0x0008, 0x1162, "UL", "3", "Calculated Frame List" },
{ 0x0008, 0x1163, "FD", "3", "TimeRange" },
{ 0x0008, 0x1164, "SQ", "3", "Frame Extraction Sequence" },
{ 0x0008, 0x1167, "UI", "3", "Multi-frame Source SOP Instance UID" },
{ 0x0008, 0x1190, "UT", "3", "Retrieve URL" },
{ 0x0008, 0x1195, "UI", "3", "Transaction UID" },
{ 0x0008, 0x1196, "US", "1", "Warning Reason" },
{ 0x0008, 0x1197, "US", "3", "Failure Reason" },
{ 0x0008, 0x1198, "SQ", "3", "Failed SOP Sequence" },
{ 0x0008, 0x1199, "SQ", "1C", "Referenced SOP Sequence" }, //< Required if 0x0040,0xA040 = COMPOSITE or IMAGE
{ 0x0008, 0x1200, "SQ", "3", "Studies Containing Other Referenced Instances Sequence" },
{ 0x0008, 0x1250, "SQ", "3", "Related Series Sequence" },
{ 0x0008, 0x2110, "CS", "3RET", "Old Lossy Image Compression" },
{ 0x0008, 0x2111, "ST", "3", "Derivation Description" },
{ 0x0008, 0x2112, "SQ", "3", "Source Image Sequence" },
{ 0x0008, 0x2120, "SH", "3", "Stage Name" },
{ 0x0008, 0x2122, "IS", "3", "Stage Number" },
{ 0x0008, 0x2124, "IS", "3", "Number of Stages" },
{ 0x0008, 0x2127, "SH", "3", "View Name" },
{ 0x0008, 0x2128, "IS", "3", "View Number" },
{ 0x0008, 0x2129, "IS", "3", "Number of Event Timers" },
{ 0x0008, 0x212A, "IS", "3", "Number of Views in Stage" },
{ 0x0008, 0x2130, "DS", "3", "Event Elapsed Time(s)" },
{ 0x0008, 0x2132, "LO", "3", "Event Timer Name(s)" },
{ 0x0008, 0x2133, "SQ", "3", "Event Timer Sequence" },
{ 0x0008, 0x2134, "FD", "3", "Event Time Offset" },
{ 0x0008, 0x2135, "SQ", "3", "Event Code Sequence" },
{ 0x0008, 0x2142, "IS", "3", "Start Trim" },
{ 0x0008, 0x2143, "IS", "3", "Stop Trim" },
{ 0x0008, 0x2144, "IS", "3", "Recommended Display Frame Rate" },
{ 0x0008, 0x2200, "CS", "3RET", "Transducer Position" },
{ 0x0008, 0x2204, "CS", "3RET", "Transducer Orientation" },
{ 0x0008, 0x2208, "CS", "3RET", "Anatomic Structure" },
{ 0x0008, 0x2218, "SQ", "3", "Anatomic Region Sequence" },
{ 0x0008, 0x2220, "SQ", "3", "Anatomic Region Modifier Sequence" },
{ 0x0008, 0x2228, "SQ", "3", "Primary Anatomic Structure Sequence" },
{ 0x0008, 0x2229, "SQ", "3", "Anatomic Structure, Space or Region Sequence" },
{ 0x0008, 0x2230, "SQ", "3", "Primary Anatomic Structure Modifier Sequence" },
{ 0x0008, 0x2240, "SQ", "3RET", "Transducer Position Sequence" },
{ 0x0008, 0x2242, "SQ", "3RET", "Transducer Position Modifier Sequence" },
{ 0x0008, 0x2244, "SQ", "3RET", "Transducer Orientation Sequence" },
{ 0x0008, 0x2246, "SQ", "3RET", "Transducer Orientation Modifier Sequence" },
{ 0x0008, 0x2251, "SQ", "1", "Anatomic Structure Space Or Region Code Sequence (Trial)" }, //< Retired
{ 0x0008, 0x2253, "SQ", "1", "Anatomic Portal Of Entrance Code Sequence (Trial)" }, //< Retired
{ 0x0008, 0x2255, "SQ", "1", "Anatomic Approach Direction Code Sequence (Trial)" }, //< Retired
{ 0x0008, 0x2256, "ST", "1", "Anatomic Perspective Description (Trial)" }, //< Retired
{ 0x0008, 0x2257, "SQ", "1", "Anatomic Perspective Code Sequence (Trial)" }, //< Retired
{ 0x0008, 0x2258, "ST", "1", "Anatomic Location Of Examining Instrument Description (Trial)" }, //< Retired
{ 0x0008, 0x2259, "SQ", "1", "Anatomic Location Of Examining Instrument Code Sequence (Trial)" }, //< Retired
{ 0x0008, 0x225A, "SQ", "1", "Anatomic Structure Space Or Region Modifier Code Sequence (Trial)" },  //< Retired
{ 0x0008, 0x225C, "SQ", "1", "On​Axis Background Anatomic Structure Code Sequence (Trial)" }, //< Retired
{ 0x0008, 0x3001, "SQ", "3", "Alternate Representation Sequence" },
{ 0x0008, 0x3010, "UI", "3", "Irradiation Event UID" },
{ 0x0008, 0x4000, "LT", "2RET", "Identifying Group Comments" },
{ 0x0008, 0x9007, "CS", "3", "Frame Type" },
{ 0x0008, 0x9092, "SQ", "3", "Referenced Image Evidence Sequence" },
{ 0x0008, 0x9121, "SQ", "3", "Referenced Raw Data Sequence" },
{ 0x0008, 0x9123, "UI", "3", "Creator-Version UID" },
{ 0x0008, 0x9124, "SQ", "3", "Derivation Image Sequence" },
{ 0x0008, 0x9154, "SQ", "3", "Source Image Evidence Sequence" },
{ 0x0008, 0x9205, "CS", "3", "Pixel Presentation" },
{ 0x0008, 0x9206, "CS", "3", "Volumetric Properties" },
{ 0x0008, 0x9207, "CS", "3", "Volume Based Calculation Technique" },
{ 0x0008, 0x9208, "CS", "3", "Complex Image Component" },
{ 0x0008, 0x9209, "CS", "3", "Acquisition Contrast" },
{ 0x0008, 0x9215, "SQ", "3", "Derivation Code Sequence" },
{ 0x0008, 0x9237, "SQ", "3", "Referenced Presentation State Sequence" },
{ 0x0008, 0x9410, "SQ", "3", "Referenced Other Plane Sequence" },
{ 0x0008, 0x9458, "SQ", "3", "Frame Display Sequence" },
{ 0x0008, 0x9459, "FL", "3", "Recommended Display Frame Rate in Float" },
{ 0x0008, 0x9460, "CS", "3", "Skip Frame Range Flag" },

{ 0x0009, 0x0000, "UL", "GEMS", "GEMS Private Group Length" },
{ 0x0009, 0x0010, "SH", "GEMS", "GEMS Private Group Creator Data Element" },
{ 0x0009, 0x1001, "LO", "GEMS", "GEMS Private Implementation Name" },
{ 0x0009, 0x1002, "SH", "GEMS", "GEMS Private Suite ID" },
{ 0x0009, 0x1004, "SH", "GEMS", "GEMS Private Product ID" },
{ 0x0009, 0x1005, "DT", "GEMS", "GEMS Private PET Patient DateTime" },
{ 0x0009, 0x1006, "SL", "GEMS", "GEMS Private PET Type" },
{ 0x0009, 0x1007, "UI", "GEMS", "GEMS Private PET Exam Id" },
{ 0x0009, 0x100A, "UI", "GEMS", "GEMS Private PET Scan Id" },
{ 0x0009, 0x100D, "DT", "GEMS", "GEMS Private PET Scan DateTime" }, //< Exam Start Time; Signa PET/MR correct decay to to this time
{ 0x0009, 0x100F, "ST", "GEMS", "GEMS Private PET Scan Description" },
{ 0x0009, 0x1011, "LO", "GEMS", "GEMS Private PET Scanner Description" },
{ 0x0009, 0x1012, "LO", "GEMS", "GEMS Private PET Scanner Manufacturer" },
{ 0x0009, 0x1015, "SH", "GEMS", "GEMS Private PET Landmark Abbreviation" },
{ 0x0009, 0x1016, "SL", "GEMS", "GEMS Private PET Patient Position" },
{ 0x0009, 0x1018, "SL", "GEMS", "GEMS Private PET Scan Type" },
{ 0x0009, 0x1019, "SL", "GEMS", "GEMS Private PET Scan Mode" },
{ 0x0009, 0x101A, "SL", "GEMS", "GEMS Private PET Start Condition" },
{ 0x0009, 0x1022, "SL", "GEMS", "GEMS Private PET Delayed Events" },
{ 0x0009, 0x1023, "SL", "GEMS", "GEMS Private PET Delayed Bias" },
{ 0x0009, 0x1024, "SL", "GEMS", "GEMS Private PET Word Size" },
{ 0x0009, 0x1025, "SL", "GEMS", "GEMS Private PET Axial Acceptance" },
{ 0x0009, 0x1026, "SL", "GEMS", "GEMS Private PET Axial Angle 3D" },
{ 0x0009, 0x1027, "SL", "GEMS", "GEMS Private Image Actual Date" },
{ 0x0009, 0x1028, "SL", "GEMS", "GEMS Private Axial Compression" },
{ 0x0009, 0x1029, "FL", "GEMS", "GEMS Private Gantry Tilt Angle" },
{ 0x0009, 0x102A, "SL", "GEMS", "GEMS Private PET Collimation" },
{ 0x0009, 0x102B, "SL", "GEMS", "GEMS Private PET Scan FOV" },
{ 0x0009, 0x102C, "SL", "GEMS", "GEMS Private PET Axial FOV" },
{ 0x0009, 0x102D, "SL", "GEMS", "GEMS Private PET Event Separation" },
{ 0x0009, 0x1036, "LO", "GEMS", "GEMS Private PET Tracer Name" },
{ 0x0009, 0x1038, "FL", "GEMS", "GEMS Private PET Tracer Activity" },
{ 0x0009, 0x1039, "DT", "GEMS", "GEMS Private PET Tracer Measurement DateTime" },
{ 0x0009, 0x103E, "SH", "GEMS", "GEMS Private PET Radionuclide Name" },
{ 0x0009, 0x103F, "FL", "GEMS", "GEMS Private PET Halflife" },
{ 0x0009, 0x1040, "FL", "GEMS", "GEMS Private PET Positron Fraction" },
{ 0x0009, 0x105A, "SL", "GEMS", "GEMS Private PET Derived" },
{ 0x0009, 0x105B, "LO", "GEMS", "GEMS Private PET Contrast Agent" },
{ 0x0009, 0x105C, "UI", "GEMS", "GEMS Private PET frame id" },
{ 0x0009, 0x105D, "UI", "GEMS", "GEMS Private PET scan id" },
{ 0x0009, 0x105E, "UI", "GEMS", "GEMS Private PET exam id" },
{ 0x0009, 0x105F, "LO", "GEMS", "GEMS Private PET patient id" },
{ 0x0009, 0x1060, "SH", "GEMS", "GEMS Private PET compatible version" },
{ 0x0009, 0x1061, "SH", "GEMS", "GEMS Private PET software version" },
{ 0x0009, 0x1062, "ST", "GEMS", "GEMS Private PET where is frame" },
{ 0x0009, 0x1063, "SL", "GEMS", "GEMS Private PET frame size" },
{ 0x0009, 0x1064, "SL", "GEMS", "GEMS Private PET file exists" },
{ 0x0009, 0x1065, "SL", "GEMS", "GEMS Private PET patient entry" },
{ 0x0009, 0x1066, "FL", "GEMS", "GEMS Private PET table height" },
{ 0x0009, 0x1067, "FL", "GEMS", "GEMS Private PET table z position" },
{ 0x0009, 0x1068, "DT", "GEMS", "GEMS Private PET landmark datetime" },
{ 0x0009, 0x1069, "SL", "GEMS", "GEMS Private PET slice count" },
{ 0x0009, 0x106A, "FL", "GEMS", "GEMS Private PET start location" },
{ 0x0009, 0x106B, "SL", "GEMS", "GEMS Private PET Acquisition Delay" },
{ 0x0009, 0x106C, "DT", "GEMS", "GEMS Private PET Acquisition Start" },
{ 0x0009, 0x106D, "SL", "GEMS", "GEMS Private PET Acquisition Duration" },
{ 0x0009, 0x106E, "SL", "GEMS", "GEMS Private PET Acquisition Bin Duration" },
{ 0x0009, 0x106F, "SL", "GEMS", "GEMS Private PET Acquisition Bin Start" },
{ 0x0009, 0x1070, "SL", "GEMS", "GEMS Private PET Actual Stop Condition" },
{ 0x0009, 0x1071, "FD", "GEMS", "GEMS Private PET Total Prompts" },
{ 0x0009, 0x1072, "FD", "GEMS", "GEMS Private PET Total Delays" },
{ 0x0009, 0x1073, "SL", "GEMS", "GEMS Private PET Frame Valid" },
{ 0x0009, 0x1074, "SL", "GEMS", "GEMS Private PET Validity Info" },
{ 0x0009, 0x1075, "SL", "GEMS", "GEMS Private PET Archived" },
{ 0x0009, 0x1076, "SL", "GEMS", "GEMS Private PET Compression" },
{ 0x0009, 0x1077, "SL", "GEMS", "GEMS Private PET Uncompressed Size" },
{ 0x0009, 0x1078, "SL", "GEMS", "GEMS Private PET Accumulated Bin Duration" },
{ 0x0009, 0x1079, "SH", "GEMS", "GEMS Private PET Compatible Version" },
{ 0x0009, 0x107A, "SH", "GEMS", "GEMS Private PET Software Version" },
{ 0x0009, 0x107B, "DT", "GEMS", "GEMS Private PET Is DateTime" },
{ 0x0009, 0x107C, "SL", "GEMS", "GEMS Private PET Is Source" },
{ 0x0009, 0x107D, "SL", "GEMS", "GEMS Private PET Is Contents" },
{ 0x0009, 0x107E, "SL", "GEMS", "GEMS Private PET Is Type" },
{ 0x0009, 0x107F, "DS", "GEMS", "GEMS Private PET Is Reference" },
{ 0x0009, 0x1080, "SL", "GEMS", "GEMS Private PET Multi Patient" },
{ 0x0009, 0x1081, "SL", "GEMS", "GEMS Private PET Number Of Normals" },
{ 0x0009, 0x1082, "UI", "GEMS", "GEMS Private PET Color Map Id" },
{ 0x0009, 0x1083, "SL", "GEMS", "GEMS Private PET Window Level Type" },
{ 0x0009, 0x1084, "FL", "GEMS", "GEMS Private PET Rotate" },
{ 0x0009, 0x1085, "SL", "GEMS", "GEMS Private PET Flip" },
{ 0x0009, 0x1086, "FL", "GEMS", "GEMS Private PET Zoom" },
{ 0x0009, 0x1087, "SL", "GEMS", "GEMS Private PET Pan X" },
{ 0x0009, 0x1088, "SL", "GEMS", "GEMS Private PET Pan Y" },
{ 0x0009, 0x1089, "FL", "GEMS", "GEMS Private PET Window Level Min" },
{ 0x0009, 0x108A, "FL", "GEMS", "GEMS Private PET Window Level Max" },
{ 0x0009, 0x108B, "SL", "GEMS", "GEMS Private PET Reconstruction Method" },
{ 0x0009, 0x108C, "SL", "GEMS", "GEMS Private PET Attenuation" },
{ 0x0009, 0x108D, "FL", "GEMS", "GEMS Private PET Attenuation Coefficient" },
{ 0x0009, 0x108E, "SL", "GEMS", "GEMS Private PET BP Filter" },
{ 0x0009, 0x108F, "FL", "GEMS", "GEMS Private PET BP Filter Cutoff" },
{ 0x0009, 0x1090, "SL", "GEMS", "GEMS Private PET BP Filter Order" },
{ 0x0009, 0x1091, "FL", "GEMS", "GEMS Private PET BP Center L" },
{ 0x0009, 0x1092, "FL", "GEMS", "GEMS Private PET BP Center P" },
{ 0x0009, 0x1093, "SL", "GEMS", "GEMS Private PET Attenuation Smooth" },
{ 0x0009, 0x1094, "SL", "GEMS", "GEMS Private PET Attenuation Smooth Parameters" },
{ 0x0009, 0x1095, "SL", "GEMS", "GEMS Private PET Angle Smooth Parameters" },
{ 0x0009, 0x1096, "UI", "GEMS", "GEMS Private PET Well Counter Calibration Id" },
{ 0x0009, 0x1097, "UI", "GEMS", "GEMS Private PET Transmission Scan Id" },
{ 0x0009, 0x1098, "UI", "GEMS", "GEMS Private PET Normalization Calibration Id" },
{ 0x0009, 0x1099, "UI", "GEMS", "GEMS Private PET Blank Calibration Id" },
{ 0x0009, 0x109A, "FL", "GEMS", "GEMS Private PET Cac Edge Threshold" },
{ 0x0009, 0x109B, "FL", "GEMS", "GEMS Private PET Cac Skull Offset" },
{ 0x0009, 0x109C, "UI", "GEMS", "GEMS Private PET Emission Subtraction Id" },
{ 0x0009, 0x109D, "SS", "GEMS", "GEMS Private PET Radial Filter 3D" },
{ 0x0009, 0x109E, "FL", "GEMS", "GEMS Private PET Radial Cutoff 3D" },
{ 0x0009, 0x109F, "SL", "GEMS", "GEMS Private PET Axial Filter 3D" },
{ 0x0009, 0x10A0, "FL", "GEMS", "GEMS Private PET Axial Cutoff 3D" },
{ 0x0009, 0x10A1, "FL", "GEMS", "GEMS Private PET Axial Start" },
{ 0x0009, 0x10A2, "FL", "GEMS", "GEMS Private PET Axial Spacing" },
{ 0x0009, 0x10A3, "SL", "GEMS", "GEMS Private PET Axial Angles Used" },
{ 0x0009, 0x10A6, "SL", "GEMS", "GEMS Private PET Slice Number" },
{ 0x0009, 0x10A7, "FL", "GEMS", "GEMS Private PET Total Counts" },
{ 0x0009, 0x10AB, "FL", "GEMS", "GEMS Private PET BP Center X" },
{ 0x0009, 0x10AC, "FL", "GEMS", "GEMS Private PET BP Center Y" },
{ 0x0009, 0x10B2, "SS", "GEMS", "GEMS Private PET IR Num Iterations" },
{ 0x0009, 0x10B3, "SS", "GEMS", "GEMS Private PET IR Num Subsets" },
{ 0x0009, 0x10B4, "FL", "GEMS", "GEMS Private PET IR Reconstruction FOV" },
{ 0x0009, 0x10B5, "SS", "GEMS", "GEMS Private PET IR Correction Model" },
{ 0x0009, 0x10B6, "SS", "GEMS", "GEMS Private PET IR Loop Filter" },
{ 0x0009, 0x10B7, "FL", "GEMS", "GEMS Private PET IR Pre Filter Parameters" },
{ 0x0009, 0x10B8, "FL", "GEMS", "GEMS Private PET IR Loop Filter Parameters" },
{ 0x0009, 0x10B9, "FL", "GEMS", "GEMS Private PET Response Filter Parameters" },
{ 0x0009, 0x10BA, "SS", "GEMS", "GEMS Private PET Post Filter" },
{ 0x0009, 0x10BB, "FL", "GEMS", "GEMS Private PET Post Filter Parameters" },
{ 0x0009, 0x10BC, "SS", "GEMS", "GEMS Private PET IR Regularize" },
{ 0x0009, 0x10BD, "FL", "GEMS", "GEMS Private PET Regularize Parameters" },
{ 0x0009, 0x10BE, "SS", "GEMS", "GEMS Private PET AC BP Filter" },
{ 0x0009, 0x10BF, "FL", "GEMS", "GEMS Private PET AC BP Filter Cuttoff" },
{ 0x0009, 0x10C0, "SL", "GEMS", "GEMS Private PET AC BP Filter Order" },
{ 0x0009, 0x10C1, "SS", "GEMS", "GEMS Private PET AC BP Image Smoothing" },
{ 0x0009, 0x10C2, "FL", "GEMS", "GEMS Private PET AC BP Image Smoothing Parameters" },
{ 0x0009, 0x10C3, "SL", "GEMS", "GEMS Private PET Scatter Method" },
{ 0x0009, 0x10C4, "SS", "GEMS", "GEMS Private PET Scatter Num Iterations" },
{ 0x0009, 0x10C5, "FL", "GEMS", "GEMS Private PET Scatter Parameters" },
{ 0x0009, 0x10C7, "SL", "GEMS", "GEMS Private PET Overlap" },
{ 0x0009, 0x10D1, "LO", "GEMS", "GEMS Private PET CTAC Conv Scale" },
{ 0x0009, 0x10D2, "UI", "GEMS", "GEMS Private PET Imageset Id" },
{ 0x0009, 0x10D4, "LO", "GEMS", "GEMS Private PET CTAC Conv Scale" },
{ 0x0009, 0x10D5, "FL", "GEMS", "GEMS Private PET Loop Filter Parameters" },
{ 0x0009, 0x10D6, "FL", "GEMS", "GEMS Private PET Image One Loc" },
{ 0x0009, 0x10D7, "FL", "GEMS", "GEMS Private PET Image Index Loc" },
{ 0x0009, 0x10D8, "SL", "GEMS", "GEMS Private PET Frame Number" }, // Gate number
{ 0x0009, 0x10D9, "SL", "GEMS", "GEMS Private PET List File Exists" },
{ 0x0009, 0x10DA, "ST", "GEMS", "GEMS Private PET Where Is List Frame" },
{ 0x0009, 0x10DD, "US", "GEMS", "GEMS Private PET Number Of RR Interval" },
{ 0x0009, 0x10DE, "US", "GEMS", "GEMS Private PET Number Of Time Slots" },
{ 0x0009, 0x10DF, "US", "GEMS", "GEMS Private PET Number Of Slices" },
{ 0x0009, 0x10E0, "US", "GEMS", "GEMS Private PET Number Of Time Slices" },
{ 0x0009, 0x10E1, "SL", "GEMS", "GEMS Private PET Unlisted Scan" },
{ 0x0009, 0x10E2, "SL", "GEMS", "GEMS Private Rest Stress" },
{ 0x0009, 0x10E3, "FL", "GEMS", "GEMS Private Phase Percentage" },
{ 0x0009, 0x10E4, "ST", "GEMS", "GEMS Private Reconstruction Protocol" },
{ 0x0009, 0x10E5, "FL", "GEMS", "GEMS Private Reconstruction Left Shift" },
{ 0x0009, 0x10E6, "FL", "GEMS", "GEMS Private Reconstruction Posterior Shift" },
{ 0x0009, 0x10E7, "FL", "GEMS", "GEMS Private Reconstruction Superior Shift" },
{ 0x0009, 0x10E8, "SL", "GEMS", "GEMS Private Acquisition Bin Number" },
{ 0x0009, 0x10E9, "FL", "GEMS", "GEMS Private Acquisition Bin Duration Percent" },
{ 0x0009, 0x10EB, "FL", "GEMS", "GEMS Private 3D Filter Cutoff" },
{ 0x0009, 0x10EC, "SL", "GEMS", "GEMS Private 3D Filter Order" },

{ 0x0010, 0x0000, "UL", "3", "Patient Group Length" },
{ 0x0010, 0x0010, "PN", "3", "Patient Name" },
{ 0x0010, 0x0020, "LO", "3", "Patient ID" },
{ 0x0010, 0x0021, "LO", "3", "Issuer of Patient ID" },
{ 0x0010, 0x0022, "CS", "3", "Type of Patient ID" },
{ 0x0010, 0x0024, "SQ", "3", "Issuer of Patient ID Qualifiers Sequence" },
{ 0x0010, 0x0030, "DA", "3", "Patient Birth Date" }, // YYYY.MM.DD
{ 0x0010, 0x0032, "TM", "3", "Patient Birth Time" },
{ 0x0010, 0x0040, "CS", "3", "Patient Sex" }, // M,F, O for other
{ 0x0010, 0x0042, "SH", "3", "Patient's Social Security Number" },
{ 0x0010, 0x0050, "SQ", "3", "Patient's Insurance Plan Code Sequence" },
{ 0x0010, 0x0101, "SQ", "3", "Patient's Primary Language Code Sequence" },
{ 0x0010, 0x0102, "SQ", "3", "Patient's Primary Language Modifier Code Sequence" },
{ 0x0010, 0x0200, "CS", "3", "Quality Control Subject" },
{ 0x0010, 0x0201, "SQ", "3", "Quality Control Subject Type Code Sequence" },
{ 0x0010, 0x1000, "LO", "3", "Other Patient IDs" },
{ 0x0010, 0x1001, "PN", "3", "Other Patient Names" },
{ 0x0010, 0x1002, "SQ", "3", "Other Patient I​Ds Sequence" },
{ 0x0010, 0x1005, "PN", "3", "Patient's Birth Name" },
{ 0x0010, 0x1010, "AS", "3", "Patient Age" }, // xxxD or W or M or Y
{ 0x0010, 0x1020, "DS", "3", "Patient Size" },
{ 0x0010, 0x1021, "SQ", "3", "Patient Size Code Sequence" },
{ 0x0010, 0x1030, "DS", "3", "Patient Weight" },
{ 0x0010, 0x1040, "LO", "3", "Patient Address" },
{ 0x0010, 0x1050, "SH", "2RET", "Insurance Plan Identification" },
{ 0x0010, 0x1060, "PN", "3", "Patient's Mother's Birth Name" },
{ 0x0010, 0x1080, "LO", "3", "Military Rank" },
{ 0x0010, 0x1081, "LO", "3", "Branch of Service" },
{ 0x0010, 0x1090, "LO", "3", "Medical Record Locator" },
{ 0x0010, 0x2000, "LO", "3", "Medical Alerts" },
{ 0x0010, 0x2110, "LO", "3", "Allergies" },
{ 0x0010, 0x2150, "LO", "3", "Country of Residence" },
{ 0x0010, 0x2152, "LO", "3", "Region of Residence" },
{ 0x0010, 0x2154, "SH", "3", "Patient's Telephone Numbers" },
{ 0x0010, 0x2160, "SH", "3", "Ethnic Group" },
{ 0x0010, 0x2180, "SH", "3", "Occupation" },
{ 0x0010, 0x21A0, "CS", "3", "Smoking Status" },
{ 0x0010, 0x21B0, "LT", "3", "Additional Patient History" },
{ 0x0010, 0x21C0, "US", "3", "Pregnancy Status" },
{ 0x0010, 0x21D0, "DA", "3", "Last Menstrual Date" },
{ 0x0010, 0x21F0, "LO", "3", "Patient's Religious Preference" },
{ 0x0010, 0x2201, "LO", "3", "Patient Species Description" },
{ 0x0010, 0x2202, "SQ", "3", "Patient Species Code Sequence" },
{ 0x0010, 0x2203, "CS", "3", "Patient's Sex Neutered" },
{ 0x0010, 0x2210, "CS", "3", "Anatomical Orientation Type" },
{ 0x0010, 0x2292, "LO", "3", "Patient Breed Description" },
{ 0x0010, 0x2293, "SQ", "3", "Patient Breed Code Sequence" },
{ 0x0010, 0x2294, "SQ", "3", "Breed Registration Sequence" },
{ 0x0010, 0x2295, "LO", "3", "Breed Registration Number" },
{ 0x0010, 0x2296, "SQ", "3", "Breed Registry Code Sequence" },
{ 0x0010, 0x2297, "PN", "3", "Responsible Person" },
{ 0x0010, 0x2298, "CS", "3", "Responsible Person Role" },
{ 0x0010, 0x2299, "LO", "3", "Responsible Organization" },
{ 0x0010, 0x4000, "LT", "3", "Patient Comments" },
{ 0x0010, 0x9431, "FL", "3", "Examined Body Thickness" },

{ 0x0012, 0x0000, "UL", "3", "Clinical Sponsor Group Length" },
{ 0x0012, 0x0010, "LO", "3", "Clinical Trial Sponsor Name" },
{ 0x0012, 0x0020, "LO", "3", "Clinical Trial Protocol ID" },
{ 0x0012, 0x0021, "LO", "3", "Clinical Trial Protocol Name" },
{ 0x0012, 0x0030, "LO", "3", "Clinical Trial Site ID" },
{ 0x0012, 0x0031, "LO", "3", "Clinical Trial Site Name" },
{ 0x0012, 0x0040, "LO", "3", "Clinical Trial Subject ID" },
{ 0x0012, 0x0042, "LO", "3", "Clinical Trial Subject Reading ID" },
{ 0x0012, 0x0050, "LO", "3", "Clinical Trial Time Point ID" },
{ 0x0012, 0x0051, "ST", "3", "Clinical Trial Time Point Description" },
{ 0x0012, 0x0060, "LO", "3", "Clinical Trial Coordinating Center Name" },
{ 0x0012, 0x0062, "CS", "3", "Patient Identity Removed" },
{ 0x0012, 0x0063, "LO", "3", "De-identification Method" },
{ 0x0012, 0x0064, "SQ", "3", "De-identification Method Code Sequence" },
{ 0x0012, 0x0071, "LO", "3", "Clinical Trial Series ID" },
{ 0x0012, 0x0072, "LO", "3", "Clinical Trial Series Description" },
{ 0x0012, 0x0081, "LO", "3", "Clinical Trial Protocol Ethics Committee Name" },
{ 0x0012, 0x0082, "LO", "3", "Clinical Trial Protocol Ethics Committee Approval Number" },
{ 0x0012, 0x0083, "SQ", "3", "Consent for Clinical Trial Use Sequence" },
{ 0x0012, 0x0084, "CS", "3", "Distribution Type" },
{ 0x0012, 0x0085, "CS", "3", "Consent for Distribution Flag" },
{ 0x0012, 0x0086, "DA", "3", "Ethics Committee Approval Effectiveness Start Date" },
{ 0x0012, 0x0087, "DA", "3", "Ethics Committee Approval Effectiveness End Date" },

{ 0x0014, 0x0000, "UL", "3", "CAD Group Length" },
{ 0x0014, 0x0023, "ST", "3RET", "CAD File Format" },
{ 0x0014, 0x0024, "ST", "3RET", "Component Reference System" },
{ 0x0014, 0x0025, "ST", "3", "Component Manufacturing Procedure" },
{ 0x0014, 0x0028, "ST", "3", "Component Manufacturer" },
{ 0x0014, 0x0030, "DS", "3", "Material Thickness" },
{ 0x0014, 0x0032, "DS", "3", "Material Pipe Diameter" },
{ 0x0014, 0x0034, "DS", "3", "Material Isolation Diameter" },
{ 0x0014, 0x0042, "ST", "3", "Material Grade" },
{ 0x0014, 0x0044, "ST", "3", "Material Properties Description" },
{ 0x0014, 0x0045, "ST", "3RET", "Material Properties File Format" },
{ 0x0014, 0x0046, "LT", "3", "Material Notes" },
{ 0x0014, 0x0050, "CS", "3", "Component Shape" },
{ 0x0014, 0x0052, "CS", "3", "Curvature Type" },
{ 0x0014, 0x0054, "DS", "3", "Outer Diameter" },
{ 0x0014, 0x0056, "DS", "3", "Inner Diameter" },
{ 0x0014, 0x0100, "LO", "3", "Component Welder IDs" },
{ 0x0014, 0x0101, "CS", "3", "Secondary Approval Status" },
{ 0x0014, 0x0102, "DA", "3", "Secondary Review Date" },
{ 0x0014, 0x0103, "TM", "3", "Secondary Review Time" },
{ 0x0014, 0x0104, "PN", "3", "Secondary Reviewer Name" },
{ 0x0014, 0x0105, "ST", "3", "Repair ID" },
{ 0x0014, 0x0106, "SQ", "3", "Multiple Component Approval Sequence" },
{ 0x0014, 0x0107, "CS", "3", "Other Approval Status" },
{ 0x0014, 0x0108, "CS", "3", "Other Secondary Approval Status" },
{ 0x0014, 0x1010, "ST", "3", "Actual Environmental Conditions" },
{ 0x0014, 0x1020, "DA", "3", "Expiry Date" },
{ 0x0014, 0x1040, "ST", "3", "Environmental Conditions" },
{ 0x0014, 0x2002, "SQ", "3", "Evaluator Sequence" },
{ 0x0014, 0x2004, "IS", "3", "Evaluator Number" },
{ 0x0014, 0x2006, "PN", "3", "Evaluator Name" },
{ 0x0014, 0x2008, "IS", "3", "Evaluation Attempt" },
{ 0x0014, 0x2012, "SQ", "3", "Indication Sequence" },
{ 0x0014, 0x2014, "IS", "3", "Indication Number" },
{ 0x0014, 0x2016, "SH", "3", "Indication Label" },
{ 0x0014, 0x2018, "ST", "3", "Indication Description" },
{ 0x0014, 0x201A, "CS", "3", "Indication Type" },
{ 0x0014, 0x201C, "CS", "3", "Indication Disposition" },
{ 0x0014, 0x201E, "SQ", "3", "Indication ROI Sequence" },
{ 0x0014, 0x2030, "SQ", "3", "Indication Physical Property Sequence" },
{ 0x0014, 0x2032, "SH", "3", "Property Label" },
{ 0x0014, 0x2202, "IS", "3", "Coordinate System Number of Axes" },
{ 0x0014, 0x2204, "SQ", "3", "Coordinate System Axes Sequence" },
{ 0x0014, 0x2206, "ST", "3", "Coordinate System Axis Description" },
{ 0x0014, 0x2208, "CS", "3", "Coordinate System Data Set Mapping" },
{ 0x0014, 0x220A, "IS", "3", "Coordinate System Axis Number" },
{ 0x0014, 0x220C, "CS", "3", "Coordinate System Axis Type" },
{ 0x0014, 0x220E, "CS", "3", "Coordinate System Axis Units" },
{ 0x0014, 0x2210, "OB", "3", "Coordinate System Axis Values" },
{ 0x0014, 0x2220, "SQ", "3", "Coordinate System Transform Sequence" },
{ 0x0014, 0x2222, "ST", "3", "Transform Description" },
{ 0x0014, 0x2224, "IS", "3", "Transform Number of Axes" },
{ 0x0014, 0x2226, "IS", "3", "Transform Order of Axes" },
{ 0x0014, 0x2228, "CS", "3", "Transformed Axis Units" },
{ 0x0014, 0x222A, "DS", "3", "Coordinate System Transform Rotation and Scale Matrix" },
{ 0x0014, 0x222C, "DS", "3", "Coordinate System Transform Translation Matrix" },
{ 0x0014, 0x3011, "DS", "3", "Internal Detector Frame Time" },
{ 0x0014, 0x3012, "DS", "3", "Number of Frames Integrated" },
{ 0x0014, 0x3020, "SQ", "3", "Detector Temperature Sequence" },
{ 0x0014, 0x3022, "ST", "3", "Sensor Name" },
{ 0x0014, 0x3024, "DS", "3", "Horizontal Offset of Sensor" },
{ 0x0014, 0x3026, "DS", "3", "Vertical Offset of Sensor" },
{ 0x0014, 0x3028, "DS", "3", "Sensor Temperature" },
{ 0x0014, 0x3040, "SQ", "3", "Dark Current Sequence" },
{ 0x0014, 0x3050, "OW", "3", "Dark Current Counts" },
{ 0x0014, 0x3060, "SQ", "3", "Gain Correction Reference Sequence" },
{ 0x0014, 0x3070, "OW", "3", "Air Counts" },
{ 0x0014, 0x3071, "DS", "3", "KV Used in Gain Calibration" },
{ 0x0014, 0x3072, "DS", "3", "MA Used in Gain Calibration" },
{ 0x0014, 0x3073, "DS", "3", "Number of Frames Used for Integration" },
{ 0x0014, 0x3074, "LO", "3", "Filter Material Used in Gain Calibration" },
{ 0x0014, 0x3075, "DS", "3", "Filter Thickness Used in Gain Calibration" },
{ 0x0014, 0x3076, "DA", "3", "Date of Gain Calibration" },
{ 0x0014, 0x3077, "TM", "3", "Time of Gain Calibration" },
{ 0x0014, 0x3080, "OB", "3", "Bad Pixel Image" },
{ 0x0014, 0x3099, "LT", "3", "Calibration Notes" },
{ 0x0014, 0x4002, "SQ", "3", "Pulser Equipment Sequence" },
{ 0x0014, 0x4004, "CS", "3", "Pulser Type" },
{ 0x0014, 0x4006, "LT", "3", "Pulser Notes" },
{ 0x0014, 0x4008, "SQ", "3", "Receiver Equipment Sequence" },
{ 0x0014, 0x400A, "CS", "3", "Amplifier Type" },
{ 0x0014, 0x400C, "LT", "3", "Receiver Notes" },
{ 0x0014, 0x400E, "SQ", "3", "Pre-Amplifier Equipment Sequence" },
{ 0x0014, 0x400F, "LT", "3", "Pre-Amplifier Notes" },
{ 0x0014, 0x4010, "SQ", "3", "Transmit Transducer Sequence" },
{ 0x0014, 0x4011, "SQ", "3", "Receive Transducer Sequence" },
{ 0x0014, 0x4012, "US", "3", "Number of Elements" },
{ 0x0014, 0x4013, "CS", "3", "Element Shape" },
{ 0x0014, 0x4014, "DS", "3", "Element Dimension A" },
{ 0x0014, 0x4015, "DS", "3", "Element Dimension B" },
{ 0x0014, 0x4016, "DS", "3", "Element Pitch A" },
{ 0x0014, 0x4017, "DS", "3", "Measured Beam Dimension A" },
{ 0x0014, 0x4018, "DS", "3", "Measured Beam Dimension B" },
{ 0x0014, 0x4019, "DS", "3", "Location of Measured Beam Diameter" },
{ 0x0014, 0x401A, "DS", "3", "Nominal Frequency" },
{ 0x0014, 0x401B, "DS", "3", "Measured Center Frequency" },
{ 0x0014, 0x401C, "DS", "3", "Measured Bandwidth" },
{ 0x0014, 0x401D, "DS", "3", "Element Pitch B" },
{ 0x0014, 0x4020, "SQ", "3", "Pulser Settings Sequence" },
{ 0x0014, 0x4022, "DS", "3", "Pulse Width" },
{ 0x0014, 0x4024, "DS", "3", "Excitation Frequency" },
{ 0x0014, 0x4026, "CS", "3", "Modulation Type" },
{ 0x0014, 0x4028, "DS", "3", "Damping" },
{ 0x0014, 0x4030, "SQ", "3", "Receiver Settings Sequence" },
{ 0x0014, 0x4031, "DS", "3", "Acquired Soundpath Length" },
{ 0x0014, 0x4032, "CS", "3", "Acquisition Compression Type" },
{ 0x0014, 0x4033, "IS", "3", "Acquisition Sample Size" },
{ 0x0014, 0x4034, "DS", "3", "Rectifier Smoothing" },
{ 0x0014, 0x4035, "SQ", "3", "DAC Sequence" },
{ 0x0014, 0x4036, "CS", "3", "DAC Type" },
{ 0x0014, 0x4038, "DS", "3", "DAC Gain Points" },
{ 0x0014, 0x403A, "DS", "3", "DAC Time Points" },
{ 0x0014, 0x403C, "DS", "3", "DAC Amplitude" },
{ 0x0014, 0x4040, "SQ", "3", "Pre-Amplifier Settings Sequence" },
{ 0x0014, 0x4050, "SQ", "3", "Transmit Transducer Settings Sequence" },
{ 0x0014, 0x4051, "SQ", "3", "Receive Transducer Settings Sequence" },
{ 0x0014, 0x4052, "DS", "3", "Incident Angle" },
{ 0x0014, 0x4054, "ST", "3", "Coupling Technique" },
{ 0x0014, 0x4056, "ST", "3", "Coupling Medium" },
{ 0x0014, 0x4057, "DS", "3", "Coupling Velocity" },
{ 0x0014, 0x4058, "DS", "3", "Probe Center Location X" },
{ 0x0014, 0x4059, "DS", "3", "Probe Center Location Z" },
{ 0x0014, 0x405A, "DS", "3", "Sound Path Length" },
{ 0x0014, 0x405C, "ST", "3", "Delay Law Identifier" },
{ 0x0014, 0x4060, "SQ", "3", "Gate Settings Sequence" },
{ 0x0014, 0x4062, "DS", "3", "Gate Threshold" },
{ 0x0014, 0x4064, "DS", "3", "Velocity of Sound" },
{ 0x0014, 0x4070, "SQ", "3", "Calibration Settings Sequence" },
{ 0x0014, 0x4072, "ST", "3", "Calibration Procedure" },
{ 0x0014, 0x4074, "SH", "3", "Procedure Version" },
{ 0x0014, 0x4076, "DA", "3", "Procedure Creation Date" },
{ 0x0014, 0x4078, "DA", "3", "Procedure Expiration Date" },
{ 0x0014, 0x407A, "DA", "3", "Procedure Last Modified Date" },
{ 0x0014, 0x407C, "TM", "3", "Calibration Time" },
{ 0x0014, 0x407E, "DA", "3", "Calibration Date" },
{ 0x0014, 0x4080, "SQ", "3", "Probe Drive Equipment Sequence" },
{ 0x0014, 0x4081, "CS", "3", "Drive Type" },
{ 0x0014, 0x4082, "LT", "3", "Probe Drive Notes" },
{ 0x0014, 0x4083, "SQ", "3", "Drive Probe Sequence" },
{ 0x0014, 0x4084, "DS", "3", "Probe Inductance" },
{ 0x0014, 0x4085, "DS", "3", "Probe Resistance" },
{ 0x0014, 0x4086, "SQ", "3", "Receive Probe Sequence" },
{ 0x0014, 0x4087, "SQ", "3", "Probe Drive Settings Sequence" },
{ 0x0014, 0x4088, "DS", "3", "Bridge Resistors" },
{ 0x0014, 0x4089, "DS", "3", "Probe Orientation Angle" },
{ 0x0014, 0x408B, "DS", "3", "User Selected Gain Y" },
{ 0x0014, 0x408C, "DS", "3", "User Selected Phase" },
{ 0x0014, 0x408D, "DS", "3", "User Selected Offset X" },
{ 0x0014, 0x408E, "DS", "3", "User Selected Offset Y" },
{ 0x0014, 0x4091, "SQ", "3", "Channel Settings Sequence" },
{ 0x0014, 0x4092, "DS", "3", "Channel Threshold" },
{ 0x0014, 0x409A, "SQ", "3", "Scanner Settings Sequence" },
{ 0x0014, 0x409B, "ST", "3", "Scan Procedure" },
{ 0x0014, 0x409C, "DS", "3", "Translation Rate X" },
{ 0x0014, 0x409D, "DS", "3", "Translation Rate Y" },
{ 0x0014, 0x409F, "DS", "3", "Channel Overlap" },
{ 0x0014, 0x40A0, "LO", "3", "Image Quality Indicator Type" },
{ 0x0014, 0x40A1, "LO", "3", "Image Quality Indicator Material" },
{ 0x0014, 0x40A2, "LO", "3", "Image Quality Indicator Size" },
{ 0x0014, 0x5002, "IS", "3", "LINAC Energy" },
{ 0x0014, 0x5004, "IS", "3", "LINAC Output" },
{ 0x0014, 0x5100, "US", "3", "Active Aperture" },
{ 0x0014, 0x5101, "DS", "3", "Total Aperture" },
{ 0x0014, 0x5102, "DS", "3", "Aperture Elevation" },
{ 0x0014, 0x5103, "DS", "3", "Main Lobe Angle" },
{ 0x0014, 0x5104, "DS", "3", "Main Roof Angle" },
{ 0x0014, 0x5105, "CS", "3", "Connector Type" },
{ 0x0014, 0x5106, "SH", "3", "Wedge Model Number" },
{ 0x0014, 0x5107, "DS", "3", "Wedge Angle Float" },
{ 0x0014, 0x5108, "DS", "3", "Wedge Roof Angle" },
{ 0x0014, 0x5109, "CS", "3", "Wedge Element 1 Position" },
{ 0x0014, 0x510A, "DS", "3", "Wedge Material Velocity" },
{ 0x0014, 0x510B, "SH", "3", "Wedge Material" },
{ 0x0014, 0x510C, "DS", "3", "Wedge Offset Z" },
{ 0x0014, 0x510D, "DS", "3", "Wedge Origin Offset X" },
{ 0x0014, 0x510E, "DS", "3", "Wedge Time Delay" },
{ 0x0014, 0x510F, "SH", "3", "Wedge Name" },
{ 0x0014, 0x5110, "SH", "3", "Wedge Manufacturer Name" },
{ 0x0014, 0x5111, "LO", "3", "Wedge Description" },
{ 0x0014, 0x5112, "DS", "3", "Nominal Beam Angle" },
{ 0x0014, 0x5113, "DS", "3", "Wedge Offset X" },
{ 0x0014, 0x5114, "DS", "3", "Wedge Offset Y" },
{ 0x0014, 0x5115, "DS", "3", "Wedge Total Length" },
{ 0x0014, 0x5116, "DS", "3", "Wedge In Contact Length" },
{ 0x0014, 0x5117, "DS", "3", "Wedge Front Gap" },
{ 0x0014, 0x5118, "DS", "3", "Wedge Total Height" },
{ 0x0014, 0x5119, "DS", "3", "Wedge Front Height" },
{ 0x0014, 0x511A, "DS", "3", "Wedge Rear Height" },
{ 0x0014, 0x511B, "DS", "3", "Wedge Total Width" },
{ 0x0014, 0x511C, "DS", "3", "Wedge In Contact Width" },
{ 0x0014, 0x511D, "DS", "3", "Wedge Chamfer Height" },
{ 0x0014, 0x511E, "CS", "3", "Wedge Curve" },
{ 0x0014, 0x511F, "DS", "3", "Radius Along the Wedge" },

{ 0x0018, 0x0000, "UL", "3", "Acquisition Group Length" },
{ 0x0018, 0x0010, "LO", "3", "Contrast/Bolus Agent" },
{ 0x0018, 0x0012, "SQ", "3", "Contrast/Bolus Agent Sequence" },
{ 0x0018, 0x0013, "FL", "3", "Contrast/Bolus T1 Relaxivity" },
{ 0x0018, 0x0014, "SQ", "3", "Contrast/Bolus Administration Route Sequence" },
{ 0x0018, 0x0015, "CS", "3", "Body Part Examined" },
{ 0x0018, 0x0020, "CS", "3", "Scanning Sequence" },
{ 0x0018, 0x0021, "CS", "3", "Sequence Variant" },
{ 0x0018, 0x0022, "CS", "3", "Scan Options" },
{ 0x0018, 0x0023, "CS", "3", "MR Acquisition Type" }, //< 2D, 3D, ...
{ 0x0018, 0x0024, "SH", "3", "Sequence Name" },
{ 0x0018, 0x0025, "CS", "3", "Angio Flag" },
{ 0x0018, 0x0026, "SQ", "3", "Intervention Drug Information Sequence" },
{ 0x0018, 0x0027, "TM", "3", "Intervention Drug Stop Time" },
{ 0x0018, 0x0028, "DS", "3", "Intervention Drug Dose" },
{ 0x0018, 0x0029, "SQ", "3", "Intervention Drug Code Sequence" },
{ 0x0018, 0x002A, "SQ", "3", "Additional Drug Sequence" },
{ 0x0018, 0x0030, "LO", "3RET", "Radionuclide" },
{ 0x0018, 0x0031, "LO", "3", "Radiopharmaceutical" },
{ 0x0018, 0x0032, "DS", "3RET", "Energy Window Centerline" },
{ 0x0018, 0x0033, "DS", "3RET", "Energy Window Total Width" },
{ 0x0018, 0x0034, "LO", "3", "Intervention Drug Name" },
{ 0x0018, 0x0035, "TM", "3", "Intervention Drug Start Time" },
{ 0x0018, 0x0036, "SQ", "3", "Intervention Therapy Sequence" },
{ 0x0018, 0x0037, "CS", "3RET", "Therapy Type" },
{ 0x0018, 0x0038, "CS", "3", "Intervention Status" },
{ 0x0018, 0x0039, "CS", "3RET", "Therapy Description" },
{ 0x0018, 0x003A, "ST", "3", "Intervention Description" },
{ 0x0018, 0x0040, "IS", "3", "Cine Rate" },
{ 0x0018, 0x0042, "CS", "1", "Initial Cine Run State" },
{ 0x0018, 0x0050, "DS", "3", "Slice Thickness (mm)" },
{ 0x0018, 0x0060, "DS", "3", "KVP" },
{ 0x0018, 0x0070, "IS", "3", "Counts Accumulated" },
{ 0x0018, 0x0071, "CS", "3", "Acquisition Termination Condition" },
{ 0x0018, 0x0072, "DS", "3", "Effective Series Duration" },
{ 0x0018, 0x0073, "CS", "3PET", "Acquisition Start Condition" },
{ 0x0018, 0x0074, "IS", "3PET", "Acquisition Start Condition Data" },
{ 0x0018, 0x0075, "IS", "3PET", "Acquisition Termination Condition Data" },
{ 0x0018, 0x0080, "DS", "3", "Repetition Time (ms)" },
{ 0x0018, 0x0081, "DS", "3", "Echo Time (ms)" },
{ 0x0018, 0x0082, "DS", "3", "Inversion Time" },
{ 0x0018, 0x0083, "DS", "3", "Number of Averages" },
{ 0x0018, 0x0084, "DS", "3", "Imaging Frequency" },
{ 0x0018, 0x0085, "SH", "3", "Imaged Nucleus" },
{ 0x0018, 0x0086, "IS", "3", "Echo Numbers(s)" },
{ 0x0018, 0x0087, "DS", "3", "Magnetic Field Strength" },
{ 0x0018, 0x0088, "DS", "3", "Spacing Between Slices" },
{ 0x0018, 0x0089, "IS", "3", "Number of Phase Encoding Steps" },
{ 0x0018, 0x0090, "DS", "3", "Data Collection Diameter" },
{ 0x0018, 0x0091, "IS", "3", "Echo Train Length" },
{ 0x0018, 0x0093, "DS", "3", "Percent Sampling" },
{ 0x0018, 0x0094, "DS", "3", "Percent Phase Field of View" },
{ 0x0018, 0x0095, "DS", "3", "Pixel Bandwidth" },
{ 0x0018, 0x1000, "LO", "3", "Device Serial Number" },
{ 0x0018, 0x1002, "UI", "3", "Device UID" },
{ 0x0018, 0x1003, "LO", "3", "Device ID" },
{ 0x0018, 0x1004, "LO", "3", "Plate ID" },
{ 0x0018, 0x1005, "LO", "3", "Generator ID" },
{ 0x0018, 0x1006, "LO", "3", "Grid ID" },
{ 0x0018, 0x1007, "LO", "3", "Cassette ID" },
{ 0x0018, 0x1008, "LO", "3", "Gantry ID" },
{ 0x0018, 0x1009, "UT", "3", "Unique Device Identifier" },
{ 0x0018, 0x100A, "SQ", "3", "Unique Device Identifier Sequence" },
{ 0x0018, 0x1010, "LO", "3", "Secondary Capture Device ID" },
{ 0x0018, 0x1011, "LO", "3RET", "Hardcopy Creation Device ID" },
{ 0x0018, 0x1012, "DA", "3", "Date of Secondary Capture" },
{ 0x0018, 0x1014, "TM", "3", "Time of Secondary Capture" },
{ 0x0018, 0x1016, "LO", "3", "Secondary Capture Device Manufacturer" },
{ 0x0018, 0x1017, "LO", "3RET", "Hardcopy Device Manufacturer" },
{ 0x0018, 0x1018, "LO", "3", "Secondary Capture Device Manufacturer's Model Name" },
{ 0x0018, 0x1019, "LO", "3", "Secondary Capture Device Software Version(s)" },
{ 0x0018, 0x101A, "LO", "1", "Hardcopy Device Software Version" }, //< Retired
{ 0x0018, 0x101B, "LO", "1", "Hardcopy Device Manufacturer's Model Name" }, //< Retired
{ 0x0018, 0x1020, "LO", "3", "Software Versions(s)" },
{ 0x0018, 0x1022, "SH", "3", "Video Image Format Acquired" },
{ 0x0018, 0x1023, "LO", "3", "Digital Image Format Acquired" },
{ 0x0018, 0x1030, "LO", "3", "Protocol Name" },
{ 0x0018, 0x1040, "LO", "3", "Contrast/Bolus Route" },
{ 0x0018, 0x1041, "DS", "3", "Contrast/Bolus Volume" },
{ 0x0018, 0x1042, "TM", "3", "Contrast/Bolus Start Time" },
{ 0x0018, 0x1043, "TM", "3", "Contrast/Bolus Stop Time" },
{ 0x0018, 0x1044, "DS", "3", "Contrast/Bolus Total Dose" },
{ 0x0018, 0x1045, "IS", "3", "Syringe Counts" },
{ 0x0018, 0x1046, "DS", "3", "Contrast Flow Rate" },
{ 0x0018, 0x1047, "DS", "3", "Contrast Flow Duration" },
{ 0x0018, 0x1048, "CS", "3", "Contrast Bolus Ingredient" },
{ 0x0018, 0x1049, "DS", "3", "Contrast Bolus Ingredient Concentration" },
{ 0x0018, 0x1050, "DS", "3", "Spatial Resolution" },
{ 0x0018, 0x1060, "DS", "3", "Trigger Time" },
{ 0x0018, 0x1061, "LO", "3", "Trigger Source or Type" }, //< Gating: EKG
{ 0x0018, 0x1062, "IS", "3", "Nominal Interval" },
{ 0x0018, 0x1063, "DS", "3", "Frame Time" },
{ 0x0018, 0x1064, "LO", "3", "Framing Type" }, //< FORW,BACK,PCNT
{ 0x0018, 0x1065, "DS", "3", "Frame Time Vector" },
{ 0x0018, 0x1066, "DS", "3", "Frame Delay" },
{ 0x0018, 0x1067, "DS", "3", "Image Trigger Delay" },
{ 0x0018, 0x1068, "DS", "3", "Multiplex Group Time Offset" },
{ 0x0018, 0x1069, "DS", "3", "Trigger Time Offset" },
{ 0x0018, 0x106A, "CS", "3", "Synchronization Trigger" },
{ 0x0018, 0x106C, "US", "3", "Synchronization Channel" },
{ 0x0018, 0x106E, "UL", "3", "Trigger Sample Position" },
{ 0x0018, 0x1070, "LO", "3", "Radiopharmaceutical Route" },
{ 0x0018, 0x1071, "DS", "3", "Radiopharmaceutical Volume (mL)" },
{ 0x0018, 0x1072, "TM", "3", "Radiopharmaceutical Start Time" },
{ 0x0018, 0x1073, "TM", "3", "Radiopharmaceutical Stop Time" },
{ 0x0018, 0x1074, "DS", "3", "Radiopharmaceutical Total Dose" },
{ 0x0018, 0x1075, "DS", "3PET", "Radionuclide Half Life" },
{ 0x0018, 0x1076, "DS", "3PET", "Radionuclide Positron Fraction" },
{ 0x0018, 0x1077, "DS", "3PET", "Radiopharmaceutical Specific Activity" },
{ 0x0018, 0x1078, "DT", "3PET", "Radiopharmaceutical Start Datetime" },
{ 0x0018, 0x1079, "DT", "3PET", "Radiopharmaceutical Stop Datetime" },
{ 0x0018, 0x1080, "CS", "3", "Beat Rejection Flag" }, //< Gating: Y,N
{ 0x0018, 0x1081, "IS", "3", "Low R-R Value" },
{ 0x0018, 0x1082, "IS", "3", "High R-R Value" },
{ 0x0018, 0x1083, "IS", "3", "Intervals Acquired" },
{ 0x0018, 0x1084, "IS", "3", "Intervals Rejected" },
{ 0x0018, 0x1085, "LO", "3", "PVC Rejection" },
{ 0x0018, 0x1086, "IS", "3", "Skip Beats" }, //< After arrhythmia
{ 0x0018, 0x1088, "IS", "3", "Heart Rate" },
{ 0x0018, 0x1090, "IS", "3", "Cardiac Number of Images" },
{ 0x0018, 0x1094, "IS", "3", "Trigger Window" },
{ 0x0018, 0x1100, "DS", "3", "Reconstruction Diameter" }, //< FOV; see 0x1149
{ 0x0018, 0x1110, "DS", "3", "Distance Source to Detector" },
{ 0x0018, 0x1111, "DS", "3", "Distance Source to Patient" },
{ 0x0018, 0x1114, "DS", "3", "Estimated Radiographic Magnification Factor" },
{ 0x0018, 0x1120, "DS", "3", "Gantry/Detector Tilt (degrees)" },
{ 0x0018, 0x1121, "DS", "3PET", "Gantry/Detector Slew" },
{ 0x0018, 0x1030, "DS", "3", "Table Height" },
{ 0x0018, 0x1131, "DS", "3", "Table Traverse" },
{ 0x0018, 0x1134, "CS", "3", "Table Motion" },
{ 0x0018, 0x1135, "DS", "3", "Table Vertical Increment" },
{ 0x0018, 0x1136, "DS", "3", "Table Lateral Increment" },
{ 0x0018, 0x1137, "DS", "3", "Table Longitudinal Increment" },
{ 0x0018, 0x1138, "DS", "3", "Table Angle" },
{ 0x0018, 0x113A, "CS", "3DX", "Table Type" },
{ 0x0018, 0x1140, "CS", "3", "Rotation Direction" },
{ 0x0018, 0x1141, "DS", "3RET", "Angular Position" },
{ 0x0018, 0x1142, "DS", "3", "Radial Position" },
{ 0x0018, 0x1143, "DS", "3", "Scan Arc" },
{ 0x0018, 0x1144, "DS", "3", "Angular Step" },
{ 0x0018, 0x1145, "DS", "3", "Center of Rotation Offset" },
{ 0x0018, 0x1146, "DS", "3RET", "Rotation Offset" },
{ 0x0018, 0x1147, "CS", "3", "Field of View Shape" }, //< CYLINDRICAL RING, HEXAGONAL, MULTIPLE PLANAR
{ 0x0018, 0x1149, "IS", "3", "Field of View Dimensions (mm)" }, //< Transverse FOV/Axial FOV; see 0x1100
{ 0x0018, 0x1150, "IS", "3", "Exposure Time" },
{ 0x0018, 0x1151, "IS", "3", "X-ray Tube Current" },
{ 0x0018, 0x1152, "IS", "3", "Exposure" },
{ 0x0018, 0x1153, "IS", "3", "Exposure in uAs" },
{ 0x0018, 0x1154, "DS", "3", "Average Pulse Width" },
{ 0x0018, 0x1155, "CS", "3", "Radiation Settings" },
{ 0x0018, 0x1156, "CS", "3DX", "Rectification Type" },
{ 0x0018, 0x115A, "CS", "3", "Radiation Mode" },
{ 0x0018, 0x115E, "DS", "3", "Image Area Dose Product" },
{ 0x0018, 0x1160, "SH", "3", "Filter Type, extremity" },
{ 0x0018, 0x1161, "LO", "3", "Type Of Filters" },
{ 0x0018, 0x1162, "DS", "3", "Intensifier Size" },
{ 0x0018, 0x1164, "DS", "3", "Imager Pixel Spacing" },
{ 0x0018, 0x1166, "CS", "3", "Grid" },
{ 0x0018, 0x1170, "IS", "3", "Generator Power" },
{ 0x0018, 0x1180, "SH", "3", "Collimator/grid Name" },
{ 0x0018, 0x1181, "CS", "3", "Collimator Type" },
{ 0x0018, 0x1182, "IS", "3", "Focal Distance" },
{ 0x0018, 0x1183, "DS", "3", "X Focus Center" },
{ 0x0018, 0x1184, "DS", "3", "Y Focus Center" },
{ 0x0018, 0x1190, "DS", "3", "Focal Spot(s)" },
{ 0x0018, 0x1191, "CS", "3", "Anode Traget Material" },
{ 0x0018, 0x11A0, "DS", "3", "Body Part Thickness" },
{ 0x0018, 0x11A2, "DS", "3", "Compression Force" },
{ 0x0018, 0x11A4, "LO", "3", "Paddle Description" },
{ 0x0018, 0x1200, "DA", "3", "Date of Last Calibration" },
{ 0x0018, 0x1201, "TM", "3", "Time of Last Calibration" },
{ 0x0018, 0x1202, "DT", "3", "DateTime of Last Calibration" },
{ 0x0018, 0x1210, "SH", "3", "Convolution Kernel" },
{ 0x0018, 0x1240, "DS", "2", "Upper/Lower Pixel Values" }, //< Retired
{ 0x0018, 0x1242, "IS", "3", "Actual Frame Duration" },
{ 0x0018, 0x1243, "IS", "3", "Count Rate" },
{ 0x0018, 0x1244, "US", "3", "Preferred Playback Sequencing" },
{ 0x0018, 0x1250, "SH", "3", "Receive Coil Name" },
{ 0x0018, 0x1251, "SH", "3", "Transmit Coil Name" },
{ 0x0018, 0x1260, "SH", "3", "Plate/Screen Type" },
{ 0x0018, 0x1261, "LO", "3", "Phosphor Type" },
{ 0x0018, 0x1271, "FD", "3", "Water Equivalent Diameter" },
{ 0x0018, 0x1272, "SQ", "3", "Water Equivalent Diameter Calculation Method Code Sequence" },
{ 0x0018, 0x1300, "IS", "3", "Scan Velocity" },
{ 0x0018, 0x1301, "CS", "3", "Whole Body Technique" },
{ 0x0018, 0x1302, "IS", "3", "Scan Length" },
{ 0x0018, 0x1310, "US", "3", "Acquisition Matrix" },
{ 0x0018, 0x1312, "CS", "3", "Phase Encoding Direction" },
{ 0x0018, 0x1314, "DS", "3", "Flip Angle" },
{ 0x0018, 0x1315, "CS", "3", "Variable Flip Angle Flag" },
{ 0x0018, 0x1316, "DS", "3", "SAR" },
{ 0x0018, 0x1318, "DS", "3", "dB/dt" },
{ 0x0018, 0x1320, "FL", "3", "B1rms" },
{ 0x0018, 0x1400, "LO", "3", "Acquisition Device Processing Description" },
{ 0x0018, 0x1401, "LO", "3", "Acquisition Device Processing Code" },
{ 0x0018, 0x1402, "CS", "3", "Cassette Orientation" },
{ 0x0018, 0x1403, "CS", "3", "Cassette Size" },
{ 0x0018, 0x1404, "US", "3", "Exposures on Plate" },
{ 0x0018, 0x1405, "IS", "3", "Relative X-ray Exposure" },
{ 0x0018, 0x1450, "DS", "3", "Column Angulation" },
{ 0x0018, 0x1460, "DS", "3", "Tomo Layer Height" },
{ 0x0018, 0x1470, "DS", "3", "Tomo Angle" },
{ 0x0018, 0x1480, "DS", "3", "Tomo Time" },
{ 0x0018, 0x1490, "CS", "3DX", "Tomo Type" },
{ 0x0018, 0x1491, "CS", "3DX", "Tomo Class" },
{ 0x0018, 0x1495, "IS", "3DX", "Number of Tomosynthesis Source Images" },
{ 0x0018, 0x1500, "CS", "1", "Positioner Motion" },
{ 0x0018, 0x1508, "CS", "1", "Positioner Type" },
{ 0x0018, 0x1510, "DS", "1", "Positioner Primary Angle" },
{ 0x0018, 0x1511, "DS", "1", "Positioner Secondary Angle" },
{ 0x0018, 0x1520, "DS", "1", "Positioner Primary Angle Increment" },
{ 0x0018, 0x1521, "DS", "1", "Positioner Secondary Angle Increment" },
{ 0x0018, 0x1530, "DS", "1", "Detector Primary Angle" },
{ 0x0018, 0x1531, "DS", "1", "Detector Secondary Angle" },
{ 0x0018, 0x1600, "CS", "1", "Shutter Shape" },
{ 0x0018, 0x1602, "IS", "1", "Shutter Left Vertical Edge" },
{ 0x0018, 0x1604, "IS", "1", "Shutter Right Vertical Edge" },
{ 0x0018, 0x1606, "IS", "1", "Shutter Upper Horizontal Edge" },
{ 0x0018, 0x1608, "IS", "1", "Shutter Lower Horizontal Edge" },
{ 0x0018, 0x1610, "IS", "2", "Center of Circular Shutter" },
{ 0x0018, 0x1612, "IS", "1", "Radius of Circular Shutter" },
{ 0x0018, 0x1620, "IS", "2", "Vertices of the Polygonal Shutter" },
{ 0x0018, 0x1622, "US", "1", "Shutter Presentation Value" },
{ 0x0018, 0x1623, "US", "1", "Shutter Overlay Group" },
{ 0x0018, 0x1624, "US", "3", "Shutter Presentation Color CIE​Lab Value" },
{ 0x0018, 0x1700, "CS", "1", "Collimator Shape" },
{ 0x0018, 0x1702, "IS", "1", "Collimator Left Vertical Edge" },
{ 0x0018, 0x1704, "IS", "1", "Collimator Right Vertical Edge" },
{ 0x0018, 0x1706, "IS", "1", "Collimator Upper Horizontal Edge" },
{ 0x0018, 0x1708, "IS", "1", "Collimator Lower Horizontal Edge" },
{ 0x0018, 0x1710, "IS", "2", "Center of Circular Collimator" },
{ 0x0018, 0x1712, "IS", "1", "Radius of Circular Collimator" },
{ 0x0018, 0x1720, "IS", "2", "Vertices of the Polygonal Collimator" },
{ 0x0018, 0x1800, "CS", "3WAV", "Acquisition Time Synchronized" },
{ 0x0018, 0x1801, "SH", "3WAV", "Time Source" },
{ 0x0018, 0x1802, "CS", "3WAV", "Time Distribution Protocol" },
{ 0x0018, 0x1803, "LO", "3", "NTP Source Address" },
{ 0x0018, 0x2001, "IS", "1", "Page Number Vector" },
{ 0x0018, 0x2002, "SH", "1", "Frame Label Vector" },
{ 0x0018, 0x2003, "DS", "1", "Frame Primary Angle Vector" },
{ 0x0018, 0x2004, "DS", "1", "Frame Secondary Angle Vector" },
{ 0x0018, 0x2005, "DS", "1", "Slice Location Vector" },
{ 0x0018, 0x2006, "SH", "1", "Display Window Label Vector" },
{ 0x0018, 0x2010, "DS", "2", "Nominal Scanned Pixel Spacing" },
{ 0x0018, 0x2020, "CS", "1", "Digitizing Device Transport Direction" },
{ 0x0018, 0x2030, "DS", "1", "Rotation of Scanned Film" },
{ 0x0018, 0x2041, "SQ", "1", "Biopsy Target Sequence" },
{ 0x0018, 0x2042, "UI", "1", "Target UID" },
{ 0x0018, 0x2043, "FL", "2", "Localizing Cursor Position" },
{ 0x0018, 0x2044, "FL", "3", "Calculated Target Position" },
{ 0x0018, 0x2045, "SH", "1", "Target Label" },
{ 0x0018, 0x2046, "FL", "1", "Displayed Z Value" },
{ 0x0018, 0x3100, "CS", "1", "IVUS Acquisition" },
{ 0x0018, 0x3101, "DS", "1", "IVUS Pullback Rate" },
{ 0x0018, 0x3102, "DS", "1", "IVUS Gated Rate" },
{ 0x0018, 0x3103, "IS", "1", "IVUS Pullback Start Frame Number" },
{ 0x0018, 0x3104, "IS", "1", "IVUS Pullback Stop Frame Number" },
{ 0x0018, 0x3105, "IS", "1", "Lesion Number" },
{ 0x0018, 0x4000, "LO", "2RET", "Acquisition Comments" },
{ 0x0018, 0x5000, "SH", "3", "Output Power" },
{ 0x0018, 0x5010, "LO", "3", "Transducer Data" },
{ 0x0018, 0x5012, "DS", "3", "Focus Depth" },
{ 0x0018, 0x5020, "LO", "3", "Preprocessing Function" },
{ 0x0018, 0x5021, "LO", "3RET", "Postprocessing Function" },
{ 0x0018, 0x5022, "DS", "3", "Mechanical Index" },
{ 0x0018, 0x5024, "DS", "3", "Thermal Index" },
{ 0x0018, 0x5026, "DS", "3", "Cranial Thermal Index" },
{ 0x0018, 0x5027, "DS", "3", "Soft Tissue Thermal Index" },
{ 0x0018, 0x5028, "DS", "3", "Soft Tissue-focus Thermal Index" },
{ 0x0018, 0x5029, "DS", "3", "Soft Tissue-surface Thermal Index" },
{ 0x0018, 0x5030, "IS", "2RET", "Dynamic Range" },
{ 0x0018, 0x5040, "IS", "2RET", "Total Gain" },
{ 0x0018, 0x5050, "IS", "3", "Depth of Scan Field" },
{ 0x0018, 0x5100, "CS", "3", "Patient Position" },
{ 0x0018, 0x5101, "CS", "3", "View Position" },
{ 0x0018, 0x5104, "SQ", "1", "Projection Eponymous Name Code Sequence" },
{ 0x0018, 0x5210, "DS", "3RET", "Image Transformation Matrix" },
{ 0x0018, 0x5212, "DS", "3RET", "Image Translation Vector" },
{ 0x0018, 0x6000, "DS", "3", "Sensitivity" },
{ 0x0018, 0x6011, "SQ", "3", "Sequence of Ultrasound Regions" },
{ 0x0018, 0x6012, "US", "3", "Region Spatial Format" },
{ 0x0018, 0x6014, "US", "3", "Region Data Type" },
{ 0x0018, 0x6016, "UL", "3", "Region Flags" },
{ 0x0018, 0x6018, "UL", "3", "Region Location Min X0" },
{ 0x0018, 0x601A, "UL", "3", "Region Location Min Y0" },
{ 0x0018, 0x601C, "UL", "3", "Region Location Max X1" },
{ 0x0018, 0x601E, "UL", "3", "Region Location Max Y1" },
{ 0x0018, 0x6020, "SL", "3", "Reference Pixel X0" },
{ 0x0018, 0x6022, "SL", "3", "Reference Pixel Y0" },
{ 0x0018, 0x6024, "US", "3", "Physical Units X Direction" },
{ 0x0018, 0x6026, "US", "3", "Physical Units Y Direction" },
{ 0x0018, 0x6028, "FD", "3", "Reference Pixel Physical Value X" },
{ 0x0018, 0x602A, "FD", "3", "Reference Pixel Physical Value Y" },
{ 0x0018, 0x602C, "FD", "3", "Physical Delta X" },
{ 0x0018, 0x602E, "FD", "3", "Physical Delta Y" },
{ 0x0018, 0x6030, "UL", "3", "Transducer Frequency" },
{ 0x0018, 0x6031, "CS", "3", "Transducer Type" },
{ 0x0018, 0x6032, "UL", "3", "Pulse Repetition Frequency" },
{ 0x0018, 0x6034, "FD", "3", "Doppler Correction Angle" },
{ 0x0018, 0x6036, "FD", "3", "Steering Angle" },
{ 0x0018, 0x6038, "UL", "3RET", "Doppler Sample Volume X Position" },
{ 0x0018, 0x6039, "SL", "3", "Doppler Sample Volume X Position" },
{ 0x0018, 0x603A, "UL", "3RET", "Doppler Sample Volume Y Position" },
{ 0x0018, 0x603B, "SL", "3", "Doppler Sample Volume Y Position" },
{ 0x0018, 0x603C, "UL", "3RET", "TM-Line Position X0" },
{ 0x0018, 0x603D, "SL", "3", "TM-Line Position X0" },
{ 0x0018, 0x603E, "UL", "3RET", "TM-Line Position Y0" },
{ 0x0018, 0x603F, "SL", "3", "TM-Line Position Y0" },
{ 0x0018, 0x6040, "UL", "3RET", "TM-Line Position X1" },
{ 0x0018, 0x6041, "SL", "3", "TM-Line Position X1" },
{ 0x0018, 0x6042, "UL", "3RET", "TM-Line Position Y1" },
{ 0x0018, 0x6043, "SL", "3", "TM-Line Position Y1" },
{ 0x0018, 0x6044, "US", "3", "Pixel Component Organization" },
{ 0x0018, 0x6046, "UL", "3", "Pixel Component Mask" },
{ 0x0018, 0x6048, "UL", "3", "Pixel Component Range Start" },
{ 0x0018, 0x604A, "UL", "3", "Pixel Component Range Stop" },
{ 0x0018, 0x604C, "US", "3", "Pixel Component Physical Units" },
{ 0x0018, 0x604E, "US", "3", "Pixel Component Data Type" },
{ 0x0018, 0x6050, "UL", "3", "Number of Table Break Points" },
{ 0x0018, 0x6052, "UL", "3", "Table of X Break Points" },
{ 0x0018, 0x6054, "FD", "3", "Table of Y Break Points" },
{ 0x0018, 0x6056, "UL", "3", "Number of Table Entries" },
{ 0x0018, 0x6058, "UL", "3", "Table of Pixel Values" },
{ 0x0018, 0x605A, "FL", "3", "Table of Parameter Values" },
{ 0x0018, 0x6060, "FL", "3", "R Wave Time Vector" },
{ 0x0018, 0x7000, "CS", "3DX", "Detector Conditions Nominal Flag" },
{ 0x0018, 0x7001, "DS", "3DX", "Detector Temperature" },
{ 0x0018, 0x7004, "CS", "3DX", "Detector Type" },
{ 0x0018, 0x7005, "CS", "3DX", "Detector Configuration" },
{ 0x0018, 0x7006, "LT", "3DX", "Detector Description" },
{ 0x0018, 0x7008, "LT", "3DX", "Detector Mode" },
{ 0x0018, 0x700A, "SH", "3DX", "Detector ID" },
{ 0x0018, 0x700C, "DA", "3DX", "Date of Last Detector Calibration " },
{ 0x0018, 0x700E, "TM", "3DX", "Time of Last Detector Calibration" },
{ 0x0018, 0x7010, "IS", "3DX", "Exposures on Detector Since Last Calibration" },
{ 0x0018, 0x7011, "IS", "3DX", "Exposures on Detector Since Manufactured" },
{ 0x0018, 0x7012, "DS", "3DX", "Detector Time Since Last Exposure" },
{ 0x0018, 0x7014, "DS", "3DX", "Detector Active Time" },
{ 0x0018, 0x7016, "DS", "3DX", "Detector Activation Offset From Exposure" },
{ 0x0018, 0x701A, "DS", "3DX", "Detector Binning" },
{ 0x0018, 0x7020, "DS", "3DX", "Detector Element Physical Size" },
{ 0x0018, 0x7022, "DS", "3DX", "Detector Element Spacing" },
{ 0x0018, 0x7024, "CS", "3DX", "Detector Active Shape" },
{ 0x0018, 0x7026, "DS", "3DX", "Detector Active Dimensions" },
{ 0x0018, 0x7028, "DS", "3DX", "Detector Active Origin" },
{ 0x0018, 0x702A, "LO", "3", "Detector Manufacturer Name" },
{ 0x0018, 0x702B, "LO", "3", "Detector Manufacturer's Model Name" },
{ 0x0018, 0x7030, "DS", "3DX", "Field of View Origin" },
{ 0x0018, 0x7032, "DS", "3DX", "Field of View Rotation" },
{ 0x0018, 0x7034, "CS", "3DX", "Field of View Horizontal Flip" },
{ 0x0018, 0x7036, "FL", "3", "Pixel Data Area Origin Relative To FOV" },
{ 0x0018, 0x7038, "FL", "3", "Pixel Data Area Rotation Angle Relative To FOV" },
{ 0x0018, 0x7040, "LT", "3DX", "Grid Absorbing Material" },
{ 0x0018, 0x7041, "LT", "3DX", "Grid Spacing Material" },
{ 0x0018, 0x7042, "DS", "3DX", "Grid Thickness" },
{ 0x0018, 0x7044, "DS", "3DX", "Grid Pitch" },
{ 0x0018, 0x7046, "IS", "3DX", "Grid Aspect Ratio" },
{ 0x0018, 0x7048, "DS", "3DX", "Grid Period" },
{ 0x0018, 0x704C, "DS", "3DX", "Grid Focal Distance" },
{ 0x0018, 0x7050, "LT", "3DX", "Filter Material" },
{ 0x0018, 0x7052, "DS", "3DX", "Filter Thickness Minimum" },
{ 0x0018, 0x7054, "DS", "3DX", "Filter Thickness Maximum" },
{ 0x0018, 0x7056, "FL", "1", "Filter Beam Path Length Minimum" },
{ 0x0018, 0x7058, "FL", "1", "Filter Beam Path Length Maximum" },
{ 0x0018, 0x7060, "CS", "3DX", "Exposure Control Mode" },
{ 0x0018, 0x7062, "LT", "3DX", "Exposure Control Mode Description" },
{ 0x0018, 0x7064, "CS", "3DX", "Exposure Status" },
{ 0x0018, 0x7065, "DS", "3DX", "Phototimer Setting" },
{ 0x0018, 0x8150, "DS", "1", "Exposure Time in uS" },
{ 0x0018, 0x8151, "DS", "1", "X-Ray Tube Current in uA" },
{ 0x0018, 0x9004, "CS", "3", "Content Qualification" },
{ 0x0018, 0x9005, "SH", "3", "Pulse Sequence Name" },
{ 0x0018, 0x9006, "SQ", "3", "MR Imaging Modifier Sequence" },
{ 0x0018, 0x9008, "CS", "3", "Echo Pulse Sequence" },
{ 0x0018, 0x9009, "CS", "3", "Inversion Recovery" },
{ 0x0018, 0x9010, "CS", "3", "Flow Compensation" },
{ 0x0018, 0x9011, "CS", "3", "Multiple Spin Echo" },
{ 0x0018, 0x9012, "CS", "3", "Multi-planar Excitation" },
{ 0x0018, 0x9014, "CS", "3", "Phase Contrast" },
{ 0x0018, 0x9015, "CS", "3", "Time of Flight Contrast" },
{ 0x0018, 0x9016, "CS", "3", "Spoiling" },
{ 0x0018, 0x9017, "CS", "3", "Steady State Pulse Sequence" },
{ 0x0018, 0x9018, "CS", "3", "Echo Planar Pulse Sequence" },
{ 0x0018, 0x9019, "FD", "3", "Tag Angle First Axis" },
{ 0x0018, 0x9020, "CS", "3", "Magnetization Transfer" },
{ 0x0018, 0x9021, "CS", "3", "T2 Preparation" },
{ 0x0018, 0x9022, "CS", "3", "Blood Signal Nulling" },
{ 0x0018, 0x9024, "CS", "3", "Saturation Recovery" },
{ 0x0018, 0x9025, "CS", "3", "Spectrally Selected Suppression" },
{ 0x0018, 0x9026, "CS", "3", "Spectrally Selected Excitation" },
{ 0x0018, 0x9027, "CS", "3", "Spatial Pre-saturation" },
{ 0x0018, 0x9028, "CS", "3", "Tagging" },
{ 0x0018, 0x9029, "CS", "3", "Oversampling Phase" },
{ 0x0018, 0x9030, "FD", "3", "Tag Spacing First Dimension" },
{ 0x0018, 0x9032, "CS", "3", "Geometry of k-Space Traversal" },
{ 0x0018, 0x9033, "CS", "3", "Segmented k-Space Traversal" },
{ 0x0018, 0x9034, "CS", "3", "Rectilinear Phase Encode Reordering" },
{ 0x0018, 0x9035, "FD", "3", "Tag Thickness" },
{ 0x0018, 0x9036, "CS", "3", "Partial Fourier Direction" },
{ 0x0018, 0x9037, "CS", "3", "Cardiac Synchronization Technique" },
{ 0x0018, 0x9041, "LO", "3", "Receive Coil Manufacturer Name" },
{ 0x0018, 0x9042, "SQ", "3", "MR Receive Coil Sequence" },
{ 0x0018, 0x9043, "CS", "3", "Receive Coil Type" },
{ 0x0018, 0x9044, "CS", "3", "Quadrature Receive Coil" },
{ 0x0018, 0x9045, "SQ", "3", "Multi-Coil Definition Sequence" },
{ 0x0018, 0x9046, "LO", "3", "Multi-Coil Configuration" },
{ 0x0018, 0x9047, "SH", "3", "Multi-Coil Element Name" },
{ 0x0018, 0x9048, "CS", "3", "Multi-Coil Element Used" },
{ 0x0018, 0x9049, "SQ", "3", "MR Transmit Coil Sequence" },
{ 0x0018, 0x9050, "LO", "3", "Transmit Coil Manufacturer Name" },
{ 0x0018, 0x9051, "CS", "3", "Transmit Coil Type" },
{ 0x0018, 0x9052, "FD", "3", "Spectral Width" },
{ 0x0018, 0x9053, "FD", "3", "Chemical Shift Reference" },
{ 0x0018, 0x9054, "CS", "3", "Volume Localization Technique" },
{ 0x0018, 0x9058, "US", "3", "MR Acquisition Frequency Encoding Steps" },
{ 0x0018, 0x9059, "CS", "3", "De-coupling" },
{ 0x0018, 0x9060, "CS", "3", "De-coupled Nucleus" },
{ 0x0018, 0x9061, "FD", "3", "De-coupling Frequency" },
{ 0x0018, 0x9062, "CS", "3", "De-coupling Method" },
{ 0x0018, 0x9063, "FD", "3", "De-coupling Chemical Shift Reference" },
{ 0x0018, 0x9064, "CS", "3", "k-space Filtering" },
{ 0x0018, 0x9065, "CS", "3", "Time Domain Filtering" },
{ 0x0018, 0x9066, "US", "3", "Number of Zero Fills" },
{ 0x0018, 0x9067, "CS", "3", "Baseline Correction" },
{ 0x0018, 0x9069, "FD", "3", "Parallel Reduction Factor In-plane" },
{ 0x0018, 0x9070, "FD", "3", "Cardiac R-R Interval Specified" },
{ 0x0018, 0x9073, "FD", "3", "Acquisition Duration" },
{ 0x0018, 0x9074, "DT", "3", "Frame Acquisition DateTime" },
{ 0x0018, 0x9075, "CS", "3", "Diffusion Directionality" },
{ 0x0018, 0x9076, "SQ", "3", "Diffusion Gradient Direction Sequence" },
{ 0x0018, 0x9077, "CS", "3", "Parallel Acquisition" },
{ 0x0018, 0x9078, "CS", "3", "Parallel Acquisition Technique" },
{ 0x0018, 0x9079, "FD", "3", "Inversion Times" },
{ 0x0018, 0x9080, "ST", "3", "Metabolite Map Description" },
{ 0x0018, 0x9081, "CS", "3", "Partial Fourier" },
{ 0x0018, 0x9082, "FD", "3", "Effective Echo Time" },
{ 0x0018, 0x9083, "SQ", "3", "Metabolite Map Code Sequence" },
{ 0x0018, 0x9084, "SQ", "3", "Chemical Shift Sequence" },
{ 0x0018, 0x9085, "CS", "3", "Cardiac Signal Source" },
{ 0x0018, 0x9087, "FD", "3", "Diffusion b-value" },
{ 0x0018, 0x9089, "FD", "3", "Diffusion Gradient Orientation" },
{ 0x0018, 0x9090, "FD", "3", "Velocity Encoding Direction" },
{ 0x0018, 0x9091, "FD", "3", "Velocity Encoding Minimum Value" },
{ 0x0018, 0x9092, "SQ", "3", "Velocity Encoding Acquisition Sequence" },
{ 0x0018, 0x9093, "US", "3", "Number of k-Space Trajectories" },
{ 0x0018, 0x9094, "CS", "3", "Coverage of k-Space" },
{ 0x0018, 0x9095, "UL", "3", "Spectroscopy Acquisition Phase Rows" },
{ 0x0018, 0x9096, "FD", "3RET", "Parallel Reduction Factor In-plane" },
{ 0x0018, 0x9098, "FD", "3", "Transmitter Frequency" },
{ 0x0018, 0x9100, "CS", "3", "Resonant Nucleus" },
{ 0x0018, 0x9101, "CS", "3", "Frequency Correction" },
{ 0x0018, 0x9103, "SQ", "3", "MR Spectroscopy FOV/Geometry Sequence" },
{ 0x0018, 0x9104, "FD", "3", "Slab Thickness" },
{ 0x0018, 0x9105, "FD", "3", "Slab Orientation" },
{ 0x0018, 0x9106, "FD", "3", "Mid Slab Position" },
{ 0x0018, 0x9107, "SQ", "3", "MR Spatial Saturation Sequence" },
{ 0x0018, 0x9112, "SQ", "3", "MR Timing and Related Parameters Sequence" },
{ 0x0018, 0x9114, "SQ", "3", "MR Echo Sequence" },
{ 0x0018, 0x9115, "SQ", "3", "MR Modifier Sequence" },
{ 0x0018, 0x9117, "SQ", "3", "MR Diffusion Sequence" },
{ 0x0018, 0x9118, "SQ", "3", "Cardiac Synchronization Sequence" },
{ 0x0018, 0x9119, "SQ", "3", "MR Averages Sequence" },
{ 0x0018, 0x9125, "SQ", "3", "MR FOV/Geometry Sequence" },
{ 0x0018, 0x9126, "SQ", "3", "Volume Localization Sequence" },
{ 0x0018, 0x9127, "UL", "3", "Spectroscopy Acquisition Data Columns" },
{ 0x0018, 0x9147, "CS", "3", "Diffusion Anisotropy Type" },
{ 0x0018, 0x9151, "DT", "3", "Frame Reference DateTime" },
{ 0x0018, 0x9152, "SQ", "3", "MR Metabolite Map Sequence" },
{ 0x0018, 0x9155, "FD", "3", "Parallel Reduction Factor out-of-plane" },
{ 0x0018, 0x9159, "UL", "3", "Spectroscopy Acquisition Out-of-plane Phase Steps" },
{ 0x0018, 0x9166, "CS", "3", "Bulk Motion Status" },
{ 0x0018, 0x9168, "FD", "3", "Parallel Reduction Factor Second In-plane" },
{ 0x0018, 0x9169, "CS", "3", "Cardiac Beat Rejection Technique" },
{ 0x0018, 0x9170, "CS", "3", "Respiratory Motion Compensation Technique" },
{ 0x0018, 0x9171, "CS", "3", "Respiratory Signal Source" },
{ 0x0018, 0x9172, "CS", "3", "Bulk Motion Compensation Technique" },
{ 0x0018, 0x9173, "CS", "3", "Bulk Motion Signal Source" },
{ 0x0018, 0x9174, "CS", "3", "Applicable Safety Standard Agency" },
{ 0x0018, 0x9175, "LO", "3", "Applicable Safety Standard Description" },
{ 0x0018, 0x9176, "SQ", "3", "Operating Mode Sequence" },
{ 0x0018, 0x9177, "CS", "3", "Operating Mode Type" },
{ 0x0018, 0x9178, "CS", "3", "Operating Mode" },
{ 0x0018, 0x9179, "CS", "3", "Specific Absorption Rate Definition" },
{ 0x0018, 0x9180, "CS", "3", "Gradient Output Type" },
{ 0x0018, 0x9181, "FD", "3", "Specific Absorption Rate Value" },
{ 0x0018, 0x9182, "FD", "3", "Gradient Output" },
{ 0x0018, 0x9183, "CS", "3", "Flow Compensation Direction" },
{ 0x0018, 0x9184, "FD", "3", "Tagging Delay" },
{ 0x0018, 0x9185, "ST", "3", "Respiratory Motion Compensation Technique Description" },
{ 0x0018, 0x9186, "SH", "3", "Respiratory Signal Source ID" },
{ 0x0018, 0x9195, "FD", "3", "Chemical Shift Minimum Integration Limit in Hz" },
{ 0x0018, 0x9196, "FD", "3", "Chemical Shift Maximum Integration Limit in Hz" },
{ 0x0018, 0x9197, "SQ", "3", "MR Velocity Encoding Sequence" },
{ 0x0018, 0x9198, "CS", "3", "First Order Phase Correction" },
{ 0x0018, 0x9199, "CS", "3", "Water Referenced Phase Correction" },
{ 0x0018, 0x9200, "CS", "3", "MR Spectroscopy Acquisition Type" },
{ 0x0018, 0x9214, "CS", "3", "Respiratory Cycle Position" },
{ 0x0018, 0x9217, "FD", "3", "Velocity Encoding Maximum Value" },
{ 0x0018, 0x9218, "FD", "3", "Tag Spacing Second Dimension" },
{ 0x0018, 0x9219, "SS", "3", "Tag Angle Second Axis" },
{ 0x0018, 0x9220, "FD", "3", "Frame Acquisition Duration" },
{ 0x0018, 0x9226, "SQ", "3", "MR Image Frame Type Sequence" },
{ 0x0018, 0x9227, "SQ", "3", "MR Spectroscopy Frame Type Sequence" },
{ 0x0018, 0x9231, "US", "3", "MR Acquisition Phase Encoding Steps in-plane" },
{ 0x0018, 0x9232, "US", "3", "MR Acquisition Phase Encoding Steps out-of-plane" },
{ 0x0018, 0x9234, "UL", "3", "Spectroscopy Acquisition Phase Columns" },
{ 0x0018, 0x9236, "CS", "3", "Cardiac Cycle Position" },
{ 0x0018, 0x9239, "SQ", "3", "Specific Absorption Rate Sequence" },
{ 0x0018, 0x9240, "US", "3", "RF Echo Train Length" },
{ 0x0018, 0x9241, "US", "3", "Gradient Echo Train Length" },
{ 0x0018, 0x9250, "CS", "3", "Arterial Spin Labeling Contrast" },
{ 0x0018, 0x9251, "SQ", "3", "MR Arterial Spin Labeling Sequence" },
{ 0x0018, 0x9252, "LO", "3", "ASL Technique Description" },
{ 0x0018, 0x9253, "US", "3", "ASL Slab Number" },
{ 0x0018, 0x9254, "FD", "3", "ASL Slab Thickness" },
{ 0x0018, 0x9255, "FD", "3", "ASL Slab Orientation" },
{ 0x0018, 0x9256, "FD", "3", "ASL Mid Slab Position" },
{ 0x0018, 0x9257, "CS", "3", "ASL Context" },
{ 0x0018, 0x9258, "UL", "3", "ASL Pulse Train Duration" },
{ 0x0018, 0x9259, "CS", "3", "ASL Crusher Flag" },
{ 0x0018, 0x925A, "FD", "3", "ASL Crusher Flow Limit" },
{ 0x0018, 0x925B, "LO", "3", "ASL Crusher Description" },
{ 0x0018, 0x925C, "CS", "3", "ASL Bolus Cut-off Flag" },
{ 0x0018, 0x925D, "SQ", "3", "ASL Bolus Cut-off Timing Sequence" },
{ 0x0018, 0x925E, "LO", "3", "ASL Bolus Cut-off Technique" },
{ 0x0018, 0x925F, "UL", "3", "ASL Bolus Cut-off Delay Time" },
{ 0x0018, 0x9260, "SQ", "3", "ASL Slab Sequence" },
{ 0x0018, 0x9295, "FD", "3", "Chemical Shift Minimum Integration Limit in ppm" },
{ 0x0018, 0x9296, "FD", "3", "Chemical Shift Maximum Integration Limit in ppm" },
{ 0x0018, 0x9297, "CS", "3", "Water Reference Acquisition" },
{ 0x0018, 0x9298, "IS", "3", "Echo Peak Position" },
{ 0x0018, 0x9301, "SQ", "3", "CT Acquisition Type Sequence" },
{ 0x0018, 0x9302, "CS", "3", "Acquisition Type" },
{ 0x0018, 0x9303, "FD", "3", "Tube Angle" },
{ 0x0018, 0x9304, "SQ", "3", "CT Acquisition Details Sequence" },
{ 0x0018, 0x9305, "FD", "3", "Revolution Time" },
{ 0x0018, 0x9306, "FD", "3", "Single Collimation Width" },
{ 0x0018, 0x9307, "FD", "3", "Total Collimation Width" },
{ 0x0018, 0x9308, "SQ", "3", "CT Table Dynamics Sequence" },
{ 0x0018, 0x9309, "FD", "3", "Table Speed" },
{ 0x0018, 0x9310, "FD", "3", "Table Feed per Rotation" },
{ 0x0018, 0x9311, "FD", "3", "Spiral Pitch Factor" },
{ 0x0018, 0x9312, "SQ", "3", "CT Geometry Sequence" },
{ 0x0018, 0x9313, "FD", "3", "Data Collection Center (Patient)" },
{ 0x0018, 0x9314, "SQ", "3", "CT Reconstruction Sequence" },
{ 0x0018, 0x9315, "CS", "3", "Reconstruction Algorithm" },
{ 0x0018, 0x9316, "CS", "3", "Convolution Kernel Group" },
{ 0x0018, 0x9317, "FD", "3", "Reconstruction Field of View" },
{ 0x0018, 0x9318, "FD", "3", "Reconstruction Target Center (Patient)" },
{ 0x0018, 0x9319, "FD", "3", "Reconstruction Angle" },
{ 0x0018, 0x9320, "SH", "3", "Image Filter" },
{ 0x0018, 0x9321, "SQ", "3", "CT Exposure Sequence" },
{ 0x0018, 0x9322, "FD", "3", "Reconstruction Pixel Spacing" },
{ 0x0018, 0x9323, "CS", "3", "Exposure Modulation Type" },
{ 0x0018, 0x9324, "FD", "3", "Estimated Dose Saving" },
{ 0x0018, 0x9325, "SQ", "3", "CT X-Ray Details Sequence" },
{ 0x0018, 0x9326, "SQ", "3", "CT Position Sequence" },
{ 0x0018, 0x9327, "FD", "3", "Table Position" },
{ 0x0018, 0x9328, "FD", "3", "Exposure Time in ms" },
{ 0x0018, 0x9329, "SQ", "3", "CT Image Frame Type Sequence" },
{ 0x0018, 0x9330, "FD", "3", "X-Ray Tube Current in mA" },
{ 0x0018, 0x9332, "FD", "3", "Exposure in mAs" },
{ 0x0018, 0x9333, "CS", "3", "Constant Volume Flag" },
{ 0x0018, 0x9334, "CS", "3", "Fluoroscopy Flag" },
{ 0x0018, 0x9335, "FD", "3", "Distance Source to Data Collection Center" },
{ 0x0018, 0x9337, "US", "3", "Contrast/Bolus Agent Number" },
{ 0x0018, 0x9338, "SQ", "3", "Contrast/Bolus Ingredient Code Sequence" },
{ 0x0018, 0x9340, "SQ", "3", "Contrast Administration Profile Sequence" },
{ 0x0018, 0x9341, "SQ", "3", "Contrast/Bolus Usage Sequence" },
{ 0x0018, 0x9342, "CS", "3", "Contrast/Bolus Agent Administered" },
{ 0x0018, 0x9343, "CS", "3", "Contrast/Bolus Agent Detected" },
{ 0x0018, 0x9344, "CS", "3", "Contrast/Bolus Agent Phase" },
{ 0x0018, 0x9345, "FD", "3", "CTDIvol" },
{ 0x0018, 0x9346, "SQ", "3", "CTDI Phantom Type Code Sequence" },
{ 0x0018, 0x9351, "FL", "3", "Calcium Scoring Mass Factor Patient" },
{ 0x0018, 0x9352, "FL", "3", "Calcium Scoring Mass Factor Device" },
{ 0x0018, 0x9353, "FL", "3", "Energy Weighting Factor" },
{ 0x0018, 0x9360, "SQ", "3", "CT Additional X-Ray Source Sequence" },
{ 0x0018, 0x9401, "SQ", "3", "Projection Pixel Calibration Sequence" },
{ 0x0018, 0x9402, "FL", "3", "Distance Source to Isocenter" },
{ 0x0018, 0x9403, "FL", "3", "Distance Object to Table Top" },
{ 0x0018, 0x9404, "FL", "3", "Object Pixel Spacing in Center of Beam" },
{ 0x0018, 0x9405, "SQ", "3", "Positioner Position Sequence" },
{ 0x0018, 0x9406, "SQ", "3", "Table Position Sequence" },
{ 0x0018, 0x9407, "SQ", "3", "Collimator Shape Sequence" },
{ 0x0018, 0x9410, "CS", "3", "Planes in Acquisition" },
{ 0x0018, 0x9412, "SQ", "3", "XA/XRF Frame Characteristics Sequence" },
{ 0x0018, 0x9417, "SQ", "3", "Frame Acquisition Sequence" },
{ 0x0018, 0x9420, "CS", "3", "X-Ray Receptor Type" },
{ 0x0018, 0x9423, "LO", "3", "Acquisition Protocol Name" },
{ 0x0018, 0x9424, "LT", "3", "Acquisition Protocol Description" },
{ 0x0018, 0x9425, "CS", "3", "Contrast/Bolus Ingredient Opaque" },
{ 0x0018, 0x9426, "FL", "3", "Distance Receptor Plane to Detector Housing" },
{ 0x0018, 0x9427, "CS", "3", "Intensifier Active Shape" },
{ 0x0018, 0x9428, "FL", "3", "Intensifier Active Dimension(s)" },
{ 0x0018, 0x9429, "FL", "3", "Physical Detector Size" },
{ 0x0018, 0x9430, "FL", "3", "Position of Isocenter Projection" },
{ 0x0018, 0x9432, "SQ", "3", "Field of View Sequence" },
{ 0x0018, 0x9433, "LO", "3", "Field of View Description" },
{ 0x0018, 0x9434, "SQ", "3", "Exposure Control Sensing Regions Sequence" },
{ 0x0018, 0x9435, "CS", "3", "Exposure Control Sensing Region Shape" },
{ 0x0018, 0x9436, "SS", "3", "Exposure Control Sensing Region Left Vertical Edge" },
{ 0x0018, 0x9437, "SS", "3", "Exposure Control Sensing Region Right Vertical Edge" },
{ 0x0018, 0x9438, "SS", "3", "Exposure Control Sensing Region Upper Horizontal Edge" },
{ 0x0018, 0x9439, "SS", "3", "Exposure Control Sensing Region Lower Horizontal Edge" },
{ 0x0018, 0x9440, "SS", "3", "Center of Circular Exposure Control Sensing Region" },
{ 0x0018, 0x9441, "US", "3", "Radius of Circular Exposure Control Sensing Region" },
{ 0x0018, 0x9442, "SS", "3", "Vertices of the Polygonal Exposure Control Sensing Region" },
{ 0x0018, 0x9447, "FL", "3", "Column Angulation (Patient)" },
{ 0x0018, 0x9449, "FL", "3", "Beam Angle" },
{ 0x0018, 0x9451, "SQ", "3", "Frame Detector Parameters Sequence" },
{ 0x0018, 0x9452, "FL", "3", "Calculated Anatomy Thickness" },
{ 0x0018, 0x9455, "SQ", "3", "Calibration Sequence" },
{ 0x0018, 0x9456, "SQ", "3", "Object Thickness Sequence" },
{ 0x0018, 0x9457, "CS", "3", "Plane Identification" },
{ 0x0018, 0x9461, "FL", "3", "Field of View Dimension(s) in Float" },
{ 0x0018, 0x9462, "SQ", "3", "Isocenter Reference System Sequence" },
{ 0x0018, 0x9463, "FL", "3", "Positioner Isocenter Primary Angle" },
{ 0x0018, 0x9464, "FL", "3", "Positioner Isocenter Secondary Angle" },
{ 0x0018, 0x9465, "FL", "3", "Positioner Isocenter Detector Rotation Angle" },
{ 0x0018, 0x9466, "FL", "3", "Table X Position to Isocenter" },
{ 0x0018, 0x9467, "FL", "3", "Table Y Position to Isocenter" },
{ 0x0018, 0x9468, "FL", "3", "Table Z Position to Isocenter" },
{ 0x0018, 0x9469, "FL", "3", "Table Horizontal Rotation Angle" },
{ 0x0018, 0x9470, "FL", "3", "Table Head Tilt Angle" },
{ 0x0018, 0x9471, "FL", "3", "Table Cradle Tilt Angle" },
{ 0x0018, 0x9472, "SQ", "3", "Frame Display Shutter Sequence" },
{ 0x0018, 0x9473, "FL", "3", "Acquired Image Area Dose Product" },
{ 0x0018, 0x9474, "CS", "3", "C-arm Positioner Tabletop Relationship" },
{ 0x0018, 0x9476, "SQ", "3", "X-Ray Geometry Sequence" },
{ 0x0018, 0x9477, "SQ", "3", "Irradiation Event Identification Sequence" },
{ 0x0018, 0x9504, "SQ", "3", "X-Ray 3D Frame Type Sequence" },
{ 0x0018, 0x9506, "SQ", "3", "Contributing Sources Sequence" },
{ 0x0018, 0x9507, "SQ", "3", "X-Ray 3D Acquisition Sequence" },
{ 0x0018, 0x9508, "FL", "3", "Primary Positioner Scan Arc" },
{ 0x0018, 0x9509, "FL", "3", "Secondary Positioner Scan Arc" },
{ 0x0018, 0x9510, "FL", "3", "Primary Positioner Scan Start Angle" },
{ 0x0018, 0x9511, "FL", "3", "Secondary Positioner Scan Start Angle" },
{ 0x0018, 0x9514, "FL", "3", "Primary Positioner Increment" },
{ 0x0018, 0x9515, "FL", "3", "Secondary Positioner Increment" },
{ 0x0018, 0x9516, "DT", "3", "Start Acquisition DateTime" },
{ 0x0018, 0x9517, "DT", "3", "End Acquisition DateTime" },
{ 0x0018, 0x9518, "SS", "3", "Primary Positioner Increment Sign" },
{ 0x0018, 0x9519, "SS", "3", "Secondary Positioner Increment Sign" },
{ 0x0018, 0x9524, "LO", "3", "Application Name" },
{ 0x0018, 0x9525, "LO", "3", "Application Version" },
{ 0x0018, 0x9526, "LO", "3", "Application Manufacturer" },
{ 0x0018, 0x9527, "CS", "3", "Algorithm Type" },
{ 0x0018, 0x9528, "LO", "3", "Algorithm Description" },
{ 0x0018, 0x9530, "SQ", "3", "X-Ray 3D Reconstruction Sequence" },
{ 0x0018, 0x9531, "LO", "3", "Reconstruction Description" },
{ 0x0018, 0x9538, "SQ", "3", "Per Projection Acquisition Sequence" },
{ 0x0018, 0x9541, "SQ", "3", "Detector Position Sequence" },
{ 0x0018, 0x9542, "SQ", "3", "X-Ray Acquisition Dose Sequence" },
{ 0x0018, 0x9543, "FD", "3", "X-Ray Source Isocenter Primary Angle" },
{ 0x0018, 0x9544, "FD", "3", "X-Ray Source Isocenter Secondary Angle" },
{ 0x0018, 0x9545, "FD", "3", "Breast Support Isocenter Primary Angle" },
{ 0x0018, 0x9546, "FD", "3", "Breast Support Isocenter Secondary Angle" },
{ 0x0018, 0x9547, "FD", "3", "Breast Support X Position to Isocenter" },
{ 0x0018, 0x9548, "FD", "3", "Breast Support Y Position to Isocenter" },
{ 0x0018, 0x9549, "FD", "3", "Breast Support Z Position to Isocenter" },
{ 0x0018, 0x9550, "FD", "3", "Detector Isocenter Primary Angle" },
{ 0x0018, 0x9551, "FD", "3", "Detector Isocenter Secondary Angle" },
{ 0x0018, 0x9552, "FD", "3", "Detector X Position to Isocenter" },
{ 0x0018, 0x9553, "FD", "3", "Detector Y Position to Isocenter" },
{ 0x0018, 0x9554, "FD", "3", "Detector Z Position to Isocenter" },
{ 0x0018, 0x9555, "SQ", "3", "X-Ray Grid Sequence" },
{ 0x0018, 0x9556, "SQ", "3", "X-Ray Filter Sequence" },
{ 0x0018, 0x9557, "FD", "3", "Detector Active Area TLHC Position" },
{ 0x0018, 0x9558, "FD", "3", "Detector Active Area Orientation" },
{ 0x0018, 0x9559, "CS", "3", "Positioner Primary Angle Direction" },
{ 0x0018, 0x9601, "SQ", "3", "Diffusion b-matrix Sequence" },
{ 0x0018, 0x9602, "FD", "3", "Diffusion b-value XX" },
{ 0x0018, 0x9603, "FD", "3", "Diffusion b-value XY" },
{ 0x0018, 0x9604, "FD", "3", "Diffusion b-value XZ" },
{ 0x0018, 0x9605, "FD", "3", "Diffusion b-value YY" },
{ 0x0018, 0x9606, "FD", "3", "Diffusion b-value YZ" },
{ 0x0018, 0x9607, "FD", "3", "Diffusion b-value ZZ" },
{ 0x0018, 0x9621, "SQ", "3", "Functional MR Sequence" },
{ 0x0018, 0x9622, "CS", "3", "Functional Settling Phase Frames Present" },
{ 0x0018, 0x9623, "DT", "3", "Functional Sync Pulse" },
{ 0x0018, 0x9624, "CS", "3", "Settling Phase Frame" },
{ 0x0018, 0x9701, "DT", "3", "Decay Correction DateTime" },
{ 0x0018, 0x9715, "FD", "3", "Start Density Threshold" },
{ 0x0018, 0x9716, "FD", "3", "Start Relative Density Difference Threshold" },
{ 0x0018, 0x9717, "FD", "3", "Start Cardiac Trigger Count Threshold" },
{ 0x0018, 0x9718, "FD", "3", "Start Respiratory Trigger Count Threshold" },
{ 0x0018, 0x9719, "FD", "3", "Termination Counts Threshold" },
{ 0x0018, 0x9720, "FD", "3", "Termination Density Threshold" },
{ 0x0018, 0x9721, "FD", "3", "Termination Relative Density Threshold" },
{ 0x0018, 0x9722, "FD", "3", "Termination Time Threshold" },
{ 0x0018, 0x9723, "FD", "3", "Termination Cardiac Trigger Count Threshold" },
{ 0x0018, 0x9724, "FD", "3", "Termination Respiratory Trigger Count Threshold" },
{ 0x0018, 0x9725, "CS", "3", "Detector Geometry" },
{ 0x0018, 0x9726, "FD", "3", "Transverse Detector Separation" },
{ 0x0018, 0x9727, "FD", "3", "Axial Detector Dimension" },
{ 0x0018, 0x9729, "US", "3", "Radiopharmaceutical Agent Number" },
{ 0x0018, 0x9732, "SQ", "3", "PET Frame Acquisition Sequence" },
{ 0x0018, 0x9733, "SQ", "3", "PET Detector Motion Details Sequence" },
{ 0x0018, 0x9734, "SQ", "3", "PET Table Dynamics Sequence" },
{ 0x0018, 0x9735, "SQ", "3", "PET Position Sequence" },
{ 0x0018, 0x9736, "SQ", "3", "PET Frame Correction Factors Sequence" },
{ 0x0018, 0x9737, "SQ", "3", "Radiopharmaceutical Usage Sequence" },
{ 0x0018, 0x9738, "CS", "3", "Attenuation Correction Source" },
{ 0x0018, 0x9739, "US", "3", "Number of Iterations" },
{ 0x0018, 0x9740, "US", "3", "Number of Subsets" },
{ 0x0018, 0x9749, "SQ", "3", "PET Reconstruction Sequence" },
{ 0x0018, 0x9751, "SQ", "3", "PET Frame Type Sequence" },
{ 0x0018, 0x9755, "CS", "3", "Time of Flight Information Used" },
{ 0x0018, 0x9756, "CS", "3", "Reconstruction Type" },
{ 0x0018, 0x9758, "CS", "3", "Decay Corrected" },
{ 0x0018, 0x9759, "CS", "3", "Attenuation Corrected" },
{ 0x0018, 0x9760, "CS", "3", "Scatter Corrected" },
{ 0x0018, 0x9761, "CS", "3", "Dead Time Corrected" },
{ 0x0018, 0x9762, "CS", "3", "Gantry Motion Corrected" },
{ 0x0018, 0x9763, "CS", "3", "Patient Motion Corrected" },
{ 0x0018, 0x9764, "CS", "3", "Count Loss Normalization Corrected" },
{ 0x0018, 0x9765, "CS", "3", "Randoms Corrected" },
{ 0x0018, 0x9766, "CS", "3", "Non-uniform Radial Sampling Corrected" },
{ 0x0018, 0x9767, "CS", "3", "Sensitivity Calibrated" },
{ 0x0018, 0x9768, "CS", "3", "Detector Normalization Correction" },
{ 0x0018, 0x9769, "CS", "3", "Iterative Reconstruction Method" },
{ 0x0018, 0x9770, "CS", "3", "Attenuation Correction Temporal Relationship" },
{ 0x0018, 0x9771, "SQ", "3", "Patient Physiological State Sequence" },
{ 0x0018, 0x9772, "SQ", "3", "Patient Physiological State Code Sequence" },
{ 0x0018, 0x9801, "FD", "3", "Depth(s) of Focus" },
{ 0x0018, 0x9803, "SQ", "3", "Excluded Intervals Sequence" },
{ 0x0018, 0x9804, "DT", "3", "Exclusion Start DateTime" },
{ 0x0018, 0x9805, "FD", "3", "Exclusion Duration" },
{ 0x0018, 0x9806, "SQ", "3", "US Image Description Sequence" },
{ 0x0018, 0x9807, "SQ", "3", "Image Data Type Sequence" },
{ 0x0018, 0x9808, "CS", "3", "Data Type" },
{ 0x0018, 0x9809, "SQ", "3", "Transducer Scan Pattern Code Sequence" },
{ 0x0018, 0x980B, "CS", "3", "Aliased Data Type" },
{ 0x0018, 0x980C, "CS", "3", "Position Measuring Device Used" },
{ 0x0018, 0x980D, "SQ", "3", "Transducer Geometry Code Sequence" },
{ 0x0018, 0x980E, "SQ", "3", "Transducer Beam Steering Code Sequence" },
{ 0x0018, 0x980F, "SQ", "3", "Transducer Application Code Sequence" },
{ 0x0018, 0x9810, "SS", "3", "Zero Velocity Pixel Value" },
{ 0x0018, 0x9900, "LO", "3", "Reference Location Label" },
{ 0x0018, 0x9901, "UT", "3", "Reference Location Description" },
{ 0x0018, 0x9902, "SQ", "3", "Reference Basis Code Sequence" },
{ 0x0018, 0x9903, "SQ", "3", "Reference Geometry Code Sequence" },
{ 0x0018, 0x9904, "DS", "3", "Offset Distance" },
{ 0x0018, 0x9905, "CS", "3", "Offset Direction" },
{ 0x0018, 0x9906, "SQ", "3", "Potential Scheduled Protocol Code Sequence" },
{ 0x0018, 0x9907, "SQ", "3", "Potential Requested Procedure Code Sequence" },
{ 0x0018, 0x9908, "UC", "3", "Potential Reasons for Procedure" },
{ 0x0018, 0x9909, "SQ", "3", "Potential Reasons for Procedure Code Sequence" },
{ 0x0018, 0x990A, "UC", "3", "Potential Diagnostic Tasks" },
{ 0x0018, 0x990B, "SQ", "3", "Contraindications Code Sequence" },
{ 0x0018, 0x990C, "SQ", "3", "Referenced Defined Protocol Sequence" },
{ 0x0018, 0x990D, "SQ", "3", "Referenced Performed Protocol Sequence" },
{ 0x0018, 0x990E, "SQ", "3", "Predecessor Protocol Sequence" },
{ 0x0018, 0x990F, "UT", "3", "Protocol Planning Information" },
{ 0x0018, 0x9910, "UT", "3", "Protocol Design Rationale" },
{ 0x0018, 0x9911, "SQ", "3", "Patient Specification Sequence" },
{ 0x0018, 0x9912, "SQ", "3", "Model Specification Sequence" },
{ 0x0018, 0x9913, "SQ", "3", "Parameters Specification Sequence" },
{ 0x0018, 0x9914, "SQ", "3", "Instruction Sequence" },
{ 0x0018, 0x9915, "US", "3", "Instruction Index" },
{ 0x0018, 0x9916, "LO", "3", "Instruction Text" },
{ 0x0018, 0x9917, "UT", "3", "Instruction Description" },
{ 0x0018, 0x9918, "CS", "3", "Instruction Performed Flag" },
{ 0x0018, 0x9919, "DT", "3", "Instruction Performed DateTime" },
{ 0x0018, 0x991A, "UT", "3", "Instruction Performance Comment" },
{ 0x0018, 0x991B, "SQ", "3", "Patient Positioning Instruction Sequence" },
{ 0x0018, 0x991C, "SQ", "3", "Positioning Method Code Sequence" },
{ 0x0018, 0x991D, "SQ", "3", "Positioning Landmark Sequence" },
{ 0x0018, 0x991E, "UI", "3", "Target Frame of Reference UID" },
{ 0x0018, 0x991F, "SQ", "3", "Acquisition Protocol Element Specification Sequence" },
{ 0x0018, 0x9920, "SQ", "3", "Acquisition Protocol Element Sequence" },
{ 0x0018, 0x9921, "US", "3", "Protocol Element Number" },
{ 0x0018, 0x9922, "LO", "3", "Protocol Element Name" },
{ 0x0018, 0x9923, "UT", "3", "Protocol Element Characteristics Summary" },
{ 0x0018, 0x9924, "UT", "3", "Protocol Element Purpose" },
{ 0x0018, 0x9930, "CS", "3", "Acquisition Motion" },
{ 0x0018, 0x9931, "SQ", "3", "Acquisition Start Location Sequence" },
{ 0x0018, 0x9932, "SQ", "3", "Acquisition End Location Sequence" },
{ 0x0018, 0x9933, "SQ", "3", "Reconstruction Protocol Element Specification Sequence" },
{ 0x0018, 0x9934, "SQ", "3", "Reconstruction Protocol Element Sequence" },
{ 0x0018, 0x9935, "SQ", "3", "Storage Protocol Element Specification Sequence" },
{ 0x0018, 0x9936, "SQ", "3", "Storage Protocol Element Sequence" },
{ 0x0018, 0x9937, "LO", "3", "Requested Series Description" },
{ 0x0018, 0x9938, "US", "3", "Source Acquisition Protocol Element Number" },
{ 0x0018, 0x9939, "US", "3", "Source Acquisition Beam Number" },
{ 0x0018, 0x993A, "US", "3", "Source Reconstruction Protocol Element Number" },
{ 0x0018, 0x993B, "SQ", "3", "Reconstruction Start Location Sequence" },
{ 0x0018, 0x993C, "SQ", "3", "Reconstruction End Location Sequence" },
{ 0x0018, 0x993D, "SQ", "3", "Reconstruction Algorithm Sequence" },
{ 0x0018, 0x993E, "SQ", "3", "Reconstruction Target Center Location Sequence" },
{ 0x0018, 0x9941, "UT", "3", "Image Filter Description" },
{ 0x0018, 0x9942, "FD", "3", "CTDIvol Notification Trigger" },
{ 0x0018, 0x9943, "FD", "3", "DLP Notification Trigger" },
{ 0x0018, 0x9944, "CS", "3", "Auto KVP Selection Type" },
{ 0x0018, 0x9945, "FD", "3", "Auto KVP Upper Bound" },
{ 0x0018, 0x9946, "FD", "3", "Auto KVP Lower Bound" },
{ 0x0018, 0x9947, "CS", "3", "Protocol Defined Patient Position" },
{ 0x0018, 0xA001, "SQ", "3", "Contributing Equipment Sequence" }, // VL=0xFFFFFFFF
{ 0x0018, 0xA002, "DT", "3", "Contribution DateTime" },
{ 0x0018, 0xA003, "ST", "3", "Contribution Description" },

// Siemens
{ 0x0019, 0x000F, "SL", "SIE", "Siemens Private" },
{ 0x0019, 0x00A5, "SS", "SIE", "Siemens Private" },
{ 0x0019, 0x00A6, "SS", "SIE", "Siemens Private" },
{ 0x0019, 0x00A7, "SL", "SIE", "Siemens Private" },
{ 0x0019, 0x00A8, "SL", "SIE", "Siemens Private" },
{ 0x0019, 0x00A9, "SL", "SIE", "Siemens Private" },
{ 0x0019, 0x00AA, "SS", "SIE", "Siemens Private" },

// GEMS
{ 0x0019, 0x1004, "DT", "GE", "GEMS Calibration DateTime" },
{ 0x0019, 0x1018, "SL", "GE", "GEMS Scan Type" },

{ 0x0020, 0x0000, "UL", "3", "Relationship Group Length" },
/** Study Instance UID is the same in images from the same scanning session, 
    for example PET and MR. 
    This must remain the same for derived DICOM images. */
{ 0x0020, 0x000D, "UI", "3", "Study Instance UID" },
/** Series Instance UID is the same in series acquired in one scan; 
    one image (for example dynamic PET or MR), but
    not in MR and PET from the same session.
    This can be used to identify files belonging to the same (dynamic) image volume.
    All series level attributes must have the same values. 
    This is globally unique, and new must be created for derived DICOM series. */
{ 0x0020, 0x000E, "UI", "3", "Series Instance UID" },
/** Study ID is the same in one image (for example PET or MR), but
    not in MR and PET from the same session. */
{ 0x0020, 0x0010, "SH", "3", "Study ID" },
/** Series Number is the same in one image (for example PET or MR), but
    not in MR and PET from the same session. 
    This should be unique, but that cannot be guaranteed. 
    New must be created for derived DICOM images. */
{ 0x0020, 0x0011, "IS", "3", "Series Number" },
{ 0x0020, 0x0012, "IS", "3", "Acquisition Number" },
/** Instance number separates each image file of an image. 
    It must be set to 1 to N for N images in the series. */
{ 0x0020, 0x0013, "IS", "2", "Instance (image) Number" },
{ 0x0020, 0x0014, "IS", "3RET", "Isotope Number" },
{ 0x0020, 0x0015, "IS", "3RET", "Phase Number" },
{ 0x0020, 0x0016, "IS", "3RET", "Interval Number" },
{ 0x0020, 0x0017, "IS", "3RET", "Time Slot Number" },
{ 0x0020, 0x0018, "IS", "3RET", "Angle Number" },
{ 0x0020, 0x0019, "IS", "1", "Item Number" },
{ 0x0020, 0x0020, "CS", "2C", "Patient Orientation" },
{ 0x0020, 0x0022, "US", "3RET", "Overlay Number" },
{ 0x0020, 0x0024, "US", "3RET", "Curve Number" },
{ 0x0020, 0x0026, "IS", "3RET", "LUT Number" },
{ 0x0020, 0x0030, "DS", "2RET", "Image Position" },
{ 0x0020, 0x0032, "DS", "3", "Image Position (Patient)" },
{ 0x0020, 0x0035, "DS", "2", "Image Orientation" },
{ 0x0020, 0x0037, "DS", "3", "Image Orientation (Patient)" },
{ 0x0020, 0x0050, "DS", "2RET", "Location" },
{ 0x0020, 0x0052, "UI", "3", "Frame of Reference UID" },
{ 0x0020, 0x0060, "CS", "3", "Laterality" },
{ 0x0020, 0x0062, "CS", "3DX", "Image Laterality" },
{ 0x0020, 0x0070, "SH", "2RET", "Image Geometry Type" },
{ 0x0020, 0x0080, "UI", "2RET", "Masking Image UID" },
{ 0x0020, 0x00AA, "IS", "1", "Report Number" }, //< Retired
{ 0x0020, 0x0100, "IS", "3", "Temporal Position Identifier" },
{ 0x0020, 0x0105, "IS", "3", "Number of Temporal Positions" },
{ 0x0020, 0x0110, "DS", "3", "Temporal Resolution" },
{ 0x0020, 0x0200, "UI", "3WAV", "Synchronization Frame Of Reference UID" },
{ 0x0020, 0x0242, "UI", "3", "SOP Instance UID of Concatenation Source" },
{ 0x0020, 0x1000, "IS", "3RET", "Series in Study" },
{ 0x0020, 0x1001, "IS", "2RET", "Acquisitions in Series" },
{ 0x0020, 0x1002, "IS", "3", "Images in Acquisition" },
{ 0x0020, 0x1003, "IS", "2RET", "Images in Series" },
{ 0x0020, 0x1004, "IS", "3RET", "Acquisitions in Study" },
{ 0x0020, 0x1005, "IS", "2RET", "Images in Study" },
{ 0x0020, 0x1020, "SH", "2RET", "Reference" },
{ 0x0020, 0x103F, "LO", "3", "Target Position Reference Indicator" },
{ 0x0020, 0x1040, "LO", "3", "Position Reference Indicator" },
{ 0x0020, 0x1041, "DS", "3", "Slice Location (mm)" },
{ 0x0020, 0x1070, "IS", "3RET", "Other Study Numbers" },
{ 0x0020, 0x1200, "IS", "3", "Number of Patient Related Studies" },
{ 0x0020, 0x1202, "IS", "3", "Number of Patient Related Series" },
{ 0x0020, 0x1204, "IS", "3", "Number of Patient Related Images" },
{ 0x0020, 0x1206, "IS", "3", "Number of Study Related Series" },
{ 0x0020, 0x1208, "IS", "3", "Number of Study Related Images" },
{ 0x0020, 0x1209, "IS", "3", "Number of Series Related Instances" },
{ 0x0020, 0x3100, "SH", "2RET", "Source Image ID" },
{ 0x0020, 0x3401, "SH", "2RET", "Modifying Device ID" },
{ 0x0020, 0x3402, "SH", "2RET", "Modified Image ID" },
{ 0x0020, 0x3403, "SH", "2RET", "Modified Image Date" },
{ 0x0020, 0x3404, "SH", "2RET", "Modifying Device Manufacturer" },
{ 0x0020, 0x3405, "SH", "2RET", "Modified Image Time" },
{ 0x0020, 0x3406, "SH", "2RET", "Modified Image Description" },
{ 0x0020, 0x4000, "LT", "3", "Image Comments" },
{ 0x0020, 0x5000, "US", "2RET", "Original Image Identification" },
{ 0x0020, 0x5002, "SH", "2RET", "Original Image Identification Nomenclature" },
{ 0x0020, 0x9056, "SH", "3", "Stack ID" },
{ 0x0020, 0x9057, "UL", "3", "In Stack Position Number" },
{ 0x0020, 0x9071, "SQ", "3", "Frame Anatomy Sequence" },
{ 0x0020, 0x9072, "CS", "3", "Frame Laterality" },
{ 0x0020, 0x9111, "SQ", "3", "Frame Content Sequence" },
{ 0x0020, 0x9113, "SQ", "3", "Plane Position Sequence" },
{ 0x0020, 0x9116, "SQ", "3", "Plane Orientation Sequence" },
{ 0x0020, 0x9128, "UL", "3", "Temporal Position Index" },
{ 0x0020, 0x9153, "FD", "3", "Nominal Cardiac Trigger Delay Time" },
{ 0x0020, 0x9154, "FL", "3", "Nominal Cardiac Trigger Time Prior To R-Peak" },
{ 0x0020, 0x9155, "FL", "3", "Actual Cardiac Trigger Time Prior To R-Peak" },
{ 0x0020, 0x9156, "US", "3", "Frame Acquisition Number" },
{ 0x0020, 0x9157, "UL", "3", "Dimension Index Values" },
{ 0x0020, 0x9158, "LT", "3", "Frame Comments" },
{ 0x0020, 0x9161, "UI", "3", "Concatenation UID" },
{ 0x0020, 0x9162, "US", "3", "In-concatenation Number" },
{ 0x0020, 0x9163, "US", "3", "In-concatenation Total Number" },
{ 0x0020, 0x9164, "UI", "3", "Dimension Organization UID" },
{ 0x0020, 0x9165, "AT", "3", "Dimension Index Pointer" },
{ 0x0020, 0x9167, "AT", "3", "Functional Group Pointer" },
{ 0x0020, 0x9170, "SQ", "3", "Unassigned Shared Converted Attributes Sequence" },
{ 0x0020, 0x9171, "SQ", "3", "Unassigned Per-Frame Converted Attributes Sequence" },
{ 0x0020, 0x9172, "SQ", "3", "Conversion Source Attributes Sequence" },
{ 0x0020, 0x9213, "LO", "3", "Dimension Index Private Creator" },
{ 0x0020, 0x9221, "SQ", "3", "Dimension Organization Sequence" },
{ 0x0020, 0x9222, "SQ", "3", "Dimension Index Sequence" },
{ 0x0020, 0x9228, "UL", "3", "Concatenation Frame Offset Number" },
{ 0x0020, 0x9238, "LO", "3", "Functional Group Private Creator" },
{ 0x0020, 0x9241, "FL", "3", "Nominal Percentage of Cardiac Phase" },
{ 0x0020, 0x9245, "FL", "3", "Nominal Percentage of Respiratory Phase" },
{ 0x0020, 0x9246, "FL", "3", "Starting Respiratory Amplitude" },
{ 0x0020, 0x9247, "CS", "3", "Starting Respiratory Phase" },
{ 0x0020, 0x9248, "FL", "3", "Ending Respiratory Amplitude" },
{ 0x0020, 0x9249, "CS", "3", "Ending Respiratory Phase" },
{ 0x0020, 0x9250, "CS", "3", "Respiratory Trigger Type" },
{ 0x0020, 0x9251, "FD", "3", "R-R Interval Time Nominal" },
{ 0x0020, 0x9252, "FD", "3", "Actual Cardiac Trigger Delay Time" },
{ 0x0020, 0x9253, "SQ", "3", "Respiratory Synchronization Sequence" },
{ 0x0020, 0x9254, "FD", "3", "Respiratory Interval Time" },
{ 0x0020, 0x9255, "FD", "3", "Nominal Respiratory Trigger Delay Time" },
{ 0x0020, 0x9256, "FD", "3", "Respiratory Trigger Delay Threshold" },
{ 0x0020, 0x9257, "FD", "3", "Actual Respiratory Trigger Delay Time" },
{ 0x0020, 0x9301, "FD", "3", "Image Position (Volume)" },
{ 0x0020, 0x9302, "FD", "3", "Image Orientation (Volume)" },
{ 0x0020, 0x9307, "CS", "3", "Ultrasound Acquisition Geometry" },
{ 0x0020, 0x9308, "FD", "3", "Apex Position" },
{ 0x0020, 0x9309, "FD", "3", "Volume to Transducer Mapping Matrix" },
{ 0x0020, 0x930A, "FD", "3", "Volume to Table Mapping Matrix" },
{ 0x0020, 0x930B, "CS", "3", "Volume to Transducer Relationship" },
{ 0x0020, 0x930C, "CS", "3", "Patient Frame of Reference Source" },
{ 0x0020, 0x930D, "FD", "3", "Temporal Position Time Offset" },
{ 0x0020, 0x930E, "SQ", "3", "Plane Position (Volume) Sequence" },
{ 0x0020, 0x930F, "SQ", "3", "Plane Orientation (Volume) Sequence" },
{ 0x0020, 0x9310, "SQ", "3", "Temporal Position Sequence" },
{ 0x0020, 0x9311, "CS", "3", "Dimension Organization Type" },
{ 0x0020, 0x9312, "UI", "3", "Volume Frame of Reference UID" },
{ 0x0020, 0x9313, "UI", "3", "Table Frame of Reference UID" },
{ 0x0020, 0x9421, "LO", "3", "Dimension Description Label" },
{ 0x0020, 0x9450, "SQ", "3", "Patient Orientation in Frame Sequence" },
{ 0x0020, 0x9453, "LO", "3", "Frame Label" },
{ 0x0020, 0x9518, "US", "3", "Acquisition Index" },
{ 0x0020, 0x9529, "SQ", "3", "Contributing SOP Instances Reference Sequence" },
{ 0x0020, 0x9536, "US", "3", "Reconstruction Index" },

{ 0x0028, 0x0000, "UL", "3", "Image Presentation Group Length" },
{ 0x0028, 0x0002, "US", "3", "Samples per Pixel" },
{ 0x0028, 0x0003, "US", "3", "Samples per Pixel Used" },
{ 0x0028, 0x0004, "CS", "3", "Photometric Interpretation" }, // MONOCHROME2: lowest=black, highest=white
{ 0x0028, 0x0005, "US", "2", "Image Dimensions (RET)" },
{ 0x0028, 0x0006, "US", "3", "Planar Configuration" },
{ 0x0028, 0x0008, "IS", "3", "Number of Frames" },
{ 0x0028, 0x0009, "AT", "3", "Frame Increment Pointer" },
{ 0x0028, 0x000A, "AT", "3", "Frame Dimension Pointer" },
{ 0x0028, 0x0010, "US", "3", "Rows" },
{ 0x0028, 0x0011, "US", "3", "Columns" },
{ 0x0028, 0x0012, "US", "3RET", "Planes" },
{ 0x0028, 0x0014, "US", "3", "Ultrasound Color Data Present" },
{ 0x0028, 0x0030, "DS", "3", "Pixel Spacing/Size (mm)" },
{ 0x0028, 0x0031, "DS", "3", "Zoom Factor" },
{ 0x0028, 0x0032, "DS", "3", "Zoom Center" },
{ 0x0028, 0x0034, "IS", "3", "Pixel Aspect Ratio" },
{ 0x0028, 0x0040, "SH", "2RET", "Image Format" },
{ 0x0028, 0x0050, "SH", "2RET", "Manipulated Image" },
{ 0x0028, 0x0051, "CS", "3", "Corrected Image" },
{ 0x0028, 0x005F, "LO", "3RET", "Compression Recognition Code" },
{ 0x0028, 0x0060, "SH", "2CRET", "Compression Code" },
{ 0x0028, 0x0100, "US", "3", "Bits Allocated" }, //< Bits allocated per image pixel
{ 0x0028, 0x0101, "US", "3", "Bits Stored" }, //< Bits used per image pixel
{ 0x0028, 0x0102, "US", "3", "High Bit" }, //< 0 when big-endian, (Bits stored)-1 when little endian
{ 0x0028, 0x0103, "US", "3", "Pixel Representation" }, //< 1=signed, 0=unsigned
{ 0x0028, 0x0104, "SS", "2RET", "Smallest Valid Pixel Value" },
{ 0x0028, 0x0105, "SS", "2RET", "Largest Valid Pixel Value" },
{ 0x0028, 0x0106, "SS", "3", "Smallest Image Pixel Value" },
{ 0x0028, 0x0107, "SS", "3", "Largest Image Pixel Value" },
{ 0x0028, 0x0108, "SS", "3", "Smallest Pixel Value in Series" },
{ 0x0028, 0x0109, "SS", "3", "Largest Pixel Value in Series" },
{ 0x0028, 0x0110, "SS", "3RET", "Smallest Pixel Value in Plane" },
{ 0x0028, 0x0111, "SS", "3RET", "Largest Pixel Value in Plane" },
{ 0x0028, 0x0120, "US", "3", "Pixel Padding Value" },
{ 0x0028, 0x0121, "US", "3", "Pixel Padding Range Limit" },
{ 0x0028, 0x0122, "FL", "3", "Float Pixel Padding Value" },
{ 0x0028, 0x0123, "FD", "3", "Double Float Pixel Padding Value" },
{ 0x0028, 0x0124, "FL", "3", "Float Pixel Padding Range Limit" },
{ 0x0028, 0x0125, "FD", "3", "Double Float Pixel Padding Range Limit" },
{ 0x0028, 0x0200, "US", "2RET", "Image Location" },
{ 0x0028, 0x0300, "CS", "3", "Quality Control Image" },
{ 0x0028, 0x0301, "CS", "3", "Burned In Annotation" },
{ 0x0028, 0x0302, "CS", "3", "Recognizable Visual Features" },
{ 0x0028, 0x0303, "CS", "3", "Longitudinal Temporal Information Modified" },
{ 0x0028, 0x0304, "UI", "3", "Referenced Color Palette Instance UID" },
{ 0x0028, 0x0700, "LO", "3RET", "DCT Label" },
{ 0x0028, 0x0701, "CS", "3RET", "Data Block Description" },
{ 0x0028, 0x0702, "AT", "3RET", "Data Block" },
{ 0x0028, 0x0710, "US", "3RET", "Normalization Factor Format" },
{ 0x0028, 0x0720, "US", "3RET", "Zonal Map Number Format" },
{ 0x0028, 0x0721, "AT", "3RET", "Zonal Map Location" },
{ 0x0028, 0x0722, "US", "3RET", "Zonal Map Format" },
{ 0x0028, 0x0730, "US", "3RET", "Adaptive Map Format" },
{ 0x0028, 0x0740, "US", "3RET", "Code Number Format" },
{ 0x0028, 0x0A02, "CS", "3", "Pixel Spacing Calibration Type" },
{ 0x0028, 0x0A04, "LO", "3", "Pixel Spacing Calibration Description" },
{ 0x0028, 0x1040, "CS", "3", "Pixel Intensity Relationship" },
{ 0x0028, 0x1041, "SS", "3DX", "Pixel Intensity Relationship Sign" },
{ 0x0028, 0x1050, "DS", "3", "Window Center" },
{ 0x0028, 0x1051, "DS", "3", "Window Width" },
{ 0x0028, 0x1052, "DS", "3", "Rescale Intercept" },
{ 0x0028, 0x1053, "DS", "3", "Rescale Slope" },
{ 0x0028, 0x1054, "LO", "3", "Rescale Type" },
{ 0x0028, 0x1055, "LO", "3", "Window Center & Width Explanation" },
{ 0x0028, 0x1056, "CS", "3", "VOI LUT Function" },
{ 0x0028, 0x1080, "SH", "2RET", "Gray Scale" },
{ 0x0028, 0x1090, "CS", "3", "Recommended Viewing Mode" },
{ 0x0028, 0x1100, "US", "2RET", "Gray Lookup Table Descriptor" },
{ 0x0028, 0x1101, "US", "3", "Red Palette Color Lookup Table Descriptor" },
{ 0x0028, 0x1102, "US", "3", "Green Palette Color Lookup Table Descriptor" },
{ 0x0028, 0x1103, "US", "3", "Blue Palette Color Lookup Table Descriptor" },
{ 0x0028, 0x1104, "US", "3", "Alpha Palette Color Lookup Table Descriptor" },
{ 0x0028, 0x1111, "US", "3RET", "Large Red Palette Color Lookup Table Descriptor" },
{ 0x0028, 0x1112, "US", "3RET", "Large Green Palette Color Lookup Table Descriptor" },
{ 0x0028, 0x1113, "US", "3RET", "Large Blue Palette Color Lookup Table Descriptor" },
{ 0x0028, 0x1199, "UI", "3", "Palette Color Lookup Table UID" },
{ 0x0028, 0x1200, "US", "2RET", "Gray Lookup Table Data" },
{ 0x0028, 0x1201, "OW", "3", "Red Palette Color Lookup Table Data" },
{ 0x0028, 0x1202, "OW", "3", "Green Palette Color Lookup Table Data" },
{ 0x0028, 0x1203, "OW", "3", "Blue Palette Color Lookup Table Data" },
{ 0x0028, 0x1204, "OW", "3", "Alpha Palette Color Lookup Table Data" },
{ 0x0028, 0x1211, "OW", "3RET", "Large Red Palette Color Lookup Table Data" },
{ 0x0028, 0x1212, "OW", "3RET", "Large Green Palette Color Lookup Table Data" },
{ 0x0028, 0x1213, "OW", "3RET", "Large Blue Palette Color Lookup Table Data" },
{ 0x0028, 0x1214, "UI", "3RET", "Large Palette Color Lookup Table UID" },
{ 0x0028, 0x1221, "OW", "3", "Segmented Red Palette Color Lookup Table Data" },
{ 0x0028, 0x1222, "OW", "3", "Segmented Green Palette Color Lookup Table Data" },
{ 0x0028, 0x1223, "OW", "3", "Segmented Blue Palette Color Lookup Table Data" },
{ 0x0028, 0x1224, "OW", "3", "Segmented Alpha Palette Color Lookup Table Data" },
{ 0x0028, 0x1230, "SQ", "3", "Stored Value Color Range Sequence" },
{ 0x0028, 0x1231, "FD", "3", "Minimum Stored Value Mapped" },
{ 0x0028, 0x1232, "FD", "3", "Maximum Stored Value Mapped" },
{ 0x0028, 0x1300, "CS", "3DFT", "Breast Implant Present" },
{ 0x0028, 0x1350, "CS", "3DFT", "Partial View" },
{ 0x0028, 0x1351, "ST", "3DFT", "Partial View Description" },
{ 0x0028, 0x1352, "SQ", "3DFT", "Partial View Code Sequence" },
{ 0x0028, 0x135A, "CS", "3DFT", "Spatial Locations Preserved" },
{ 0x0028, 0x1401, "SQ", "3", "Data Frame Assignment Sequence" },
{ 0x0028, 0x2000, "OB", "3", "ICC Profile" },
{ 0x0028, 0x2110, "CS", "3", "Lossy Image Compression" },
{ 0x0028, 0x2112, "DS", "3", "Lossy Image Compression Ratio" },
{ 0x0028, 0x2114, "CS", "3", "Lossy Image Compression Method" },
{ 0x0028, 0x3000, "SQ", "3", "Modality LUT Sequence" },
{ 0x0028, 0x3002, "US", "3", "LUT Descriptor" }, // DICOM PS3.5 2017a: SS/US; see (0x0028, 0x0103)
{ 0x0028, 0x3003, "LO", "3", "LUT Explanation" },
{ 0x0028, 0x3004, "LO", "3", "Modality LUT Type" },
{ 0x0028, 0x3006, "US", "3", "LUT Data" }, // DICOM PS3.5 2017a: US/OW
{ 0x0028, 0x3010, "SQ", "3", "VOI LUT Sequence" },
{ 0x0028, 0x3110, "SQ", "3", "Softcopy VOI LUT Sequence" },
{ 0x0028, 0x4000, "LT", "2RET", "Image Presentation Group Comments" },
{ 0x0028, 0x5000, "SQ", "3", "Biplane Acquisition Sequence" },
{ 0x0028, 0x6010, "US", "3", "Representative Frame Number" },
{ 0x0028, 0x6020, "US", "3", "Frame Numbers of Interest (FOI)" },
{ 0x0028, 0x6022, "LO", "3", "Frame of Interest Description" },
{ 0x0028, 0x6023, "CS", "3", "Frame of Interest Type" },
{ 0x0028, 0x6030, "US", "3", "Mask Pointer" },
{ 0x0028, 0x6040, "US", "3", "R Wave Pointer" },
{ 0x0028, 0x6100, "SQ", "3", "Mask Subtraction Sequence" },
{ 0x0028, 0x6101, "CS", "3", "Mask Operation" },
{ 0x0028, 0x6102, "US", "3", "Applicable Frame Range" },
{ 0x0028, 0x6110, "US", "3", "Mask Frame Numbers" },
{ 0x0028, 0x6112, "US", "3", "Contrast Frame Averaging" },
{ 0x0028, 0x6114, "FL", "3", "Mask Sub-Pixel Shift" },
{ 0x0028, 0x6120, "SS", "3", "TID Offset" },
{ 0x0028, 0x6190, "ST", "3", "Mask Operation Explanation" },
{ 0x0028, 0x7FE0, "UT", "3", "Pixel Data Provider URL" },
{ 0x0028, 0x9001, "UL", "3", "Data Point Rows" },
{ 0x0028, 0x9002, "UL", "3", "Data Point Columns" },
{ 0x0028, 0x9003, "CS", "3", "Signal Domain Columns" },
{ 0x0028, 0x9099, "US", "3RET", "Largest Monochrome Pixel Value" },
{ 0x0028, 0x9108, "CS", "3", "Data Representation" },
{ 0x0028, 0x9110, "SQ", "3", "Pixel Measures Sequence" },
{ 0x0028, 0x9132, "SQ", "3", "Frame VOI LUT Sequence" },
{ 0x0028, 0x9145, "SQ", "3", "Pixel Value Transformation Sequence" },
{ 0x0028, 0x9235, "CS", "3", "Signal Domain Rows" },
{ 0x0028, 0x9411, "FL", "3", "Display Filter Percentage" },
{ 0x0028, 0x9415, "SQ", "3", "Frame Pixel Shift Sequence" },
{ 0x0028, 0x9416, "US", "3", "Subtraction Item ID" },
{ 0x0028, 0x9422, "SQ", "3", "Pixel Intensity Relationship LUT Sequence" },
{ 0x0028, 0x9443, "SQ", "3", "Frame Pixel Data Properties Sequence" },
{ 0x0028, 0x9444, "CS", "3", "Geometrical Properties" },
{ 0x0028, 0x9445, "FL", "3", "Geometric Maximum Distortion" },
{ 0x0028, 0x9446, "CS", "3", "Image Processing Applied" },
{ 0x0028, 0x9454, "CS", "3", "Mask Selection Mode" },
{ 0x0028, 0x9474, "CS", "3", "LUT Function" },
{ 0x0028, 0x9520, "DS", "1", "Image to Equipment Mapping Matrix" },
{ 0x0028, 0x9537, "CS", "1", "Equipment Coordinate System Identification" },

// Siemens
{ 0x0029, 0x0008, "CS", "SIE", "Siemens Private" },
{ 0x0029, 0x0009, "LO", "SIE", "Siemens Private" },
{ 0x0029, 0x0010, "OB", "SIE", "Siemens Private" },
{ 0x0029, 0x0018, "CS", "SIE", "Siemens Private" },
{ 0x0029, 0x0019, "LO", "SIE", "Siemens Private" },
{ 0x0029, 0x0020, "OB", "SIE", "Siemens Private" },

{ 0x0032, 0x0000, "UL", "3", "Study Group Length" },
{ 0x0032, 0x000A, "CS", "3RET", "Study Status ID" },
{ 0x0032, 0x000C, "CS", "3RET", "Study Priority ID" },
{ 0x0032, 0x0012, "LO", "3RET", "Study ID Issuer" },
{ 0x0032, 0x0032, "DA", "3RET", "Study Verified Date" },
{ 0x0032, 0x0033, "TM", "3RET", "Study Verified Time" },
{ 0x0032, 0x0034, "DA", "3RET", "Study Read Date" },
{ 0x0032, 0x0035, "TM", "3RET", "Study Read Time" },
{ 0x0032, 0x1000, "DA", "3RET", "Scheduled Study Start Date" },
{ 0x0032, 0x1001, "TM", "3RET", "Scheduled Study Start Time" },
{ 0x0032, 0x1010, "DA", "3RET", "Scheduled Study Stop Date" },
{ 0x0032, 0x1011, "TM", "3RET", "Scheduled Study Stop Time" },
{ 0x0032, 0x1020, "LO", "3RET", "Scheduled Study Location" },
{ 0x0032, 0x1021, "AE", "3RET", "Scheduled Study Location AE Title(s)" },
{ 0x0032, 0x1030, "LO", "3RET", "Reason For Study" },
{ 0x0032, 0x1031, "SQ", "3", "Requesting Physician Identification Sequence" },
{ 0x0032, 0x1032, "PN", "3", "Requesting Physician" },
{ 0x0032, 0x1033, "LO", "3", "Requesting Service" },
{ 0x0032, 0x1034, "SQ", "3", "Requesting Service Code Sequence" },
{ 0x0032, 0x1040, "DA", "3RET", "Study Arrival Date" },
{ 0x0032, 0x1041, "TM", "3RET", "Study Arrival Time" },
{ 0x0032, 0x1050, "DA", "3RET", "Study Completion Date" },
{ 0x0032, 0x1051, "TM", "3RET", "Study Completion Time" },
{ 0x0032, 0x1055, "CS", "3RET", "Study Component Status ID" },
{ 0x0032, 0x1060, "LO", "3", "Requested Procedure Description" },
{ 0x0032, 0x1064, "SQ", "3", "Requested Procedure Code Sequence" },
{ 0x0032, 0x1070, "LO", "3", "Requested Contrast Agent" },
{ 0x0032, 0x4000, "LT", "3RET", "Study Comments" },

{ 0x0038, 0x0000, "UL", "3", "Visit Group Length" },
{ 0x0038, 0x0004, "SQ", "3", "Referenced Patient Alias Sequence" },
{ 0x0038, 0x0008, "CS", "3", "Visit Status ID" },
{ 0x0038, 0x0010, "LO", "3", "Admissin ID" },
{ 0x0038, 0x0011, "LO", "3RET", "Issuer of Admission ID" },
{ 0x0038, 0x0014, "SQ", "3", "Issuer of Admission ID Sequence" },
{ 0x0038, 0x0016, "LO", "3", "Route of Admissions" },
{ 0x0038, 0x001A, "DA", "3RET", "Scheduled Admission Date" },
{ 0x0038, 0x001B, "TM", "3RET", "Scheduled Admission Time" },
{ 0x0038, 0x001C, "DA", "3RET", "Scheduled Discharge Date" },
{ 0x0038, 0x001D, "TM", "3RET", "Scheduled Discharge Time" },
{ 0x0038, 0x001E, "LO", "3RET", "Scheduled Patient Institution Residence" },
{ 0x0038, 0x0020, "DA", "3", "Admitting Date" },
{ 0x0038, 0x0021, "TM", "3", "Admitting Time" },
{ 0x0038, 0x0030, "DA", "3RET", "Discharge Date" },
{ 0x0038, 0x0032, "TM", "3RET", "Discharge Time" },
{ 0x0038, 0x0040, "LO", "3RET", "Discharge Diagnosis Description" },
{ 0x0038, 0x0044, "SQ", "3RET", "Discharge Diagnosis Code Sequence" },
{ 0x0038, 0x0050, "LO", "3", "Special Needs" },
{ 0x0038, 0x0300, "LO", "3", "Current Patient Location" },
{ 0x0038, 0x0400, "LO", "3", "Patient's Institution Residence" },
{ 0x0038, 0x0500, "LO", "3", "Patient State" },
{ 0x0038, 0x0502, "SQ", "3", "Patient Clinical Trial Participation Sequence" },
{ 0x0038, 0x4000, "LT", "3", "Visit Comments" },

{ 0x003A, 0x0000, "UL", "3WAV", "Waveform Group Length"},
{ 0x003A, 0x0002, "SQ", "3WAV", "Waveform Sequence"},
{ 0x003A, 0x0004, "CS", "3WAV", "Waveform Originality" },
{ 0x003A, 0x0005, "US", "3WAV", "Number of Waveform Channels" },
{ 0x003A, 0x0010, "UL", "3WAV", "Number of Waveform Samples" },
{ 0x003A, 0x001A, "DS", "3WAV", "Sampling Frequency" },
{ 0x003A, 0x0020, "SH", "3WAV", "Multiplex Group Label" },
{ 0x003A, 0x0103, "CS", "3WAV", "Waveform Data Value Representation"},
{ 0x003A, 0x0200, "SQ", "3WAV", "Channel Definition Sequence" },
{ 0x003A, 0x0202, "IS", "3WAV", "Waveform Channel Number" },
{ 0x003A, 0x0203, "SH", "3WAV", "Channel Label" },
{ 0x003A, 0x0205, "CS", "3WAV", "Channel Status" },
{ 0x003A, 0x0208, "SQ", "3WAV", "Channel Source Sequence" },
{ 0x003A, 0x0209, "SQ", "3WAV", "Channel Source Modifiers Sequence" },
{ 0x003A, 0x020A, "SQ", "3WAV", "Source Waveform Sequence" },
{ 0x003A, 0x020C, "LO", "3WAV", "Channel Derivation Description" },
{ 0x003A, 0x0210, "DS", "3WAV", "Channel Sensitivity" },
{ 0x003A, 0x0211, "SQ", "3WAV", "Channel Sensitivity Units Sequence" },
{ 0x003A, 0x0212, "DS", "3WAV", "Channel Sensitivity Correction Factor" },
{ 0x003A, 0x0213, "DS", "3WAV", "Channel Baseline" },
{ 0x003A, 0x0214, "DS", "3WAV", "Channel Time Skew" },
{ 0x003A, 0x0215, "DS", "3WAV", "Channel Sample Skew" },
{ 0x003A, 0x0218, "DS", "3WAV", "Channel Offset" },
{ 0x003A, 0x021A, "US", "3WAV", "Waveform Bits Stored" },
{ 0x003A, 0x0220, "DS", "3WAV", "Filter Low Frequency" },
{ 0x003A, 0x0221, "DS", "3WAV", "Filter High Frequency" },
{ 0x003A, 0x0222, "DS", "3WAV", "Notch Filter Frequency" },
{ 0x003A, 0x0223, "DS", "3WAV", "Notch Filter Bandwidth" },

{ 0x0040, 0x0000, "UL", "3", "Schedule Group Length"},
{ 0x0040, 0x0001, "AE", "3", "Scheduled Station AE Title" },
{ 0x0040, 0x0002, "DA", "3", "Scheduled Procedure Step Start Date" },
{ 0x0040, 0x0003, "TM", "3", "Scheduled Procedure Step Start Time" },
{ 0x0040, 0x0004, "DA", "3", "Scheduled Procedure Step End Date" },
{ 0x0040, 0x0005, "TM", "3", "Scheduled Procedure Step End Time" },
{ 0x0040, 0x0006, "PN", "3", "Scheduled Performing Physician Name" },
{ 0x0040, 0x0007, "LO", "3", "Scheduled Procedure Step Description" },
{ 0x0040, 0x0008, "SQ", "3", "Scheduled Action Item Code Sequence" },
{ 0x0040, 0x0009, "SH", "3", "Scheduled Procedure Step ID" },
{ 0x0040, 0x000A, "SQ", "3", "Stage Code Sequence" },
{ 0x0040, 0x000B, "SQ", "1", "Scheduled Performing Physician Identification Sequence" },
{ 0x0040, 0x0010, "SH", "3", "Scheduled Station Name" },
{ 0x0040, 0x0011, "SH", "3", "Scheduled Procedure Step Location" },
{ 0x0040, 0x0012, "LO", "3", "Pre-Medication" },
{ 0x0040, 0x0020, "CS", "3", "Scheduled Procedure Step Status" },
{ 0x0040, 0x0026, "SQ", "3", "Order Placer Identifier Sequence" },
{ 0x0040, 0x0027, "SQ", "3", "Order Filler Identifier Sequence" },
{ 0x0040, 0x0031, "UT", "3", "Local Namespace Entity ID" },
{ 0x0040, 0x0032, "UT", "3", "Universal Entity ID" },
{ 0x0040, 0x0033, "CS", "3", "Universal Entity ID Type" },
{ 0x0040, 0x0035, "CS", "3", "Identifier Type Code" },
{ 0x0040, 0x0036, "SQ", "3", "Assigning Facility Sequence" },
{ 0x0040, 0x0039, "SQ", "3", "Assigning Jurisdiction Code Sequence" },
{ 0x0040, 0x003A, "SQ", "3", "Assigning Agency or Department Code Sequence" },
{ 0x0040, 0x0100, "SQ", "3", "Scheduled Procedure Step Sequence" },
{ 0x0040, 0x0220, "SQ", "3PPS", "Referenced Standalone SOP Instance Sequence" },
{ 0x0040, 0x0241, "AE", "3PPS", "Performed Station AE Title" },
{ 0x0040, 0x0242, "SH", "3PPS", "Performed Station Name" },
{ 0x0040, 0x0243, "SH", "3PPS", "Performed Location" },
{ 0x0040, 0x0244, "DA", "3PPS", "Performed Procedure Step Start Date" },
{ 0x0040, 0x0245, "TM", "3PPS", "Performed Procedure Step Start Time" },
{ 0x0040, 0x0250, "DA", "3PPS", "Performed Procedure Step End Date" },
{ 0x0040, 0x0251, "TM", "3PPS", "Performed Procedure Step End Time" },
{ 0x0040, 0x0252, "CS", "3PPS", "Performed Procedure Step Status" },
{ 0x0040, 0x0253, "SH", "3PPS", "Performed Procedure Step ID" },
{ 0x0040, 0x0254, "LO", "3PPS", "Performed Procedure Step Description" },
{ 0x0040, 0x0255, "LO", "3PPS", "Performed Procedure Type Description" },
{ 0x0040, 0x0260, "SQ", "3PPS", "Performed Protocol Item Sequence" },
{ 0x0040, 0x0261, "CS", "3", "Performed Protocol Type" },
{ 0x0040, 0x0270, "SQ", "3PPS", "Scheduled Step Attributes Sequence" },
{ 0x0040, 0x0275, "SQ", "3PPS", "Request Attributes Sequence" },
{ 0x0040, 0x0280, "ST", "3PPS", "Comments on the Performed Procedure Steps" },
{ 0x0040, 0x0293, "SQ", "3PPS", "Quantity Sequence" },
{ 0x0040, 0x0294, "DS", "3PPS", "Quantity" },
{ 0x0040, 0x0295, "SQ", "3PPS", "Measuring Units Sequence" },
{ 0x0040, 0x0296, "SQ", "3PPS", "Billing Item Sequence" },
{ 0x0040, 0x0300, "US", "3PPS", "Total Time of Fluoroscopy" },
{ 0x0040, 0x0301, "US", "3PPS", "Total Number of Exposures" },
{ 0x0040, 0x0302, "US", "3PPS", "Entrance Dose" },
{ 0x0040, 0x0303, "US", "3PPS", "Exposed Area" },
{ 0x0040, 0x0306, "DS", "3DX", "Distance Source to Entrance" },
{ 0x0040, 0x0307, "DS", "3RET", "Distance Source to Support" },
{ 0x0040, 0x0310, "ST", "3PPS", "Comments On Radiation Dose" },
{ 0x0040, 0x0312, "DS", "3DX", "X-Ray Output" },
{ 0x0040, 0x0314, "DS", "3DX", "Half Value Layer" },
{ 0x0040, 0x0316, "DS", "3DX", "Organ Dose" },
{ 0x0040, 0x0318, "CS", "3DX", "Organ Exposed" },
{ 0x0040, 0x0320, "SQ", "3PPS", "Billing Procedure Step Sequence" },
{ 0x0040, 0x0321, "SQ", "3PPS", "Film Consumption Sequence" },
{ 0x0040, 0x0324, "SQ", "3PPS", "Billing Supplies And Devices Sequence" },
{ 0x0040, 0x0330, "SQ", "3RET", "Referenced Procedure Step Sequence" },
{ 0x0040, 0x0340, "SQ", "3PPS", "Performed Series Sequence" },
{ 0x0040, 0x0400, "LT", "3", "Comments On Scheduled Procedure Step" },
{ 0x0040, 0x050A, "LO", "3DX", "Specimen Accession Number" },
{ 0x0040, 0x0550, "SQ", "3DX", "Specimen Sequence" },
{ 0x0040, 0x0551, "LO", "3DX", "Specimen Identifier" },
{ 0x0040, 0x0552, "SQ", "3DX", "Specimen Description Sequence" },
{ 0x0040, 0x0553, "ST", "3DX", "Specimen Description" },
{ 0x0040, 0x0555, "SQ", "3DX", "Acquisition Context Sequence" },
{ 0x0040, 0x0556, "ST", "3DX", "Acquisition Context Description" },
{ 0x0040, 0x059A, "SQ", "3DX", "Specimen Type Code Sequence" },
{ 0x0040, 0x06FA, "LO", "3DX", "Slide Identifier" },
{ 0x0040, 0x071A, "SQ", "3VLI", "Image Center Point Coordinates Sequence" },
{ 0x0040, 0x072A, "DS", "3VLI", "X Offset In Slide Coordinate System" },
{ 0x0040, 0x073A, "DS", "3VLI", "Y Offset In Slide Coordinate System" },
{ 0x0040, 0x074A, "DS", "3VLI", "Z Offset In Slide Coordinate System" },
{ 0x0040, 0x08D8, "SQ", "3VLI", "Pixel Spacing Sequence" },
{ 0x0040, 0x08DA, "SQ", "3VLI", "Coordinate System Axis Code Sequence" },
{ 0x0040, 0x08EA, "SQ", "1C", "Measurement Units Code Sequence" }, //< Required if 0xA040 = NUMERIC
{ 0x0040, 0x09F8, "SQ", "3", "Vital Stain Code Sequence" },
{ 0x0040, 0x1001, "SH", "3", "Requested Procedure ID" },
{ 0x0040, 0x1002, "LO", "3", "Reason For Requested Procedure" },
{ 0x0040, 0x1003, "SH", "3", "Requested Procedure Priority" },
{ 0x0040, 0x1004, "LO", "3", "Patient Transport Arrangements" },
{ 0x0040, 0x1005, "LO", "3", "Requested Procedure Location" },
{ 0x0040, 0x1006, "SH", "3RET", "Placer Order Number of Procedure" },
{ 0x0040, 0x1007, "SH", "3RET", "Filler Order Number of Procedure" },
{ 0x0040, 0x1008, "LO", "3", "Confidentiality Code" },
{ 0x0040, 0x1009, "SH", "3", "Reporting Priority" },
{ 0x0040, 0x1010, "PN", "3", "Names of Intended Recipients of Results" },
{ 0x0040, 0x1101, "LO", "1", "Person Identification Code Sequence" },
{ 0x0040, 0x1400, "LT", "3", "Requested Procedure Comments" },
{ 0x0040, 0x2001, "LO", "3", "Reason For Imaging Service Request" },
{ 0x0040, 0x2004, "DA", "3", "Issue Date of Imaging Service Request" },
{ 0x0040, 0x2005, "TM", "3", "Issue Time of Imaging Service Request" },
{ 0x0040, 0x2006, "SH", "3RET", "Placer Order Number of Imaging Service Request" },
{ 0x0040, 0x2007, "SH", "3RET", "Filler Order Number of Imaging Service Request" },
{ 0x0040, 0x2008, "PN", "3", "Order Entered By" },
{ 0x0040, 0x2009, "SH", "3", "Order Enterer Location" },
{ 0x0040, 0x2010, "SH", "3", "Order Callback Phone Number" },
{ 0x0040, 0x2016, "LO", "3", "Placer Order Number of Imaging Service Request" },
{ 0x0040, 0x2017, "LO", "3", "Filler Order Number of Imaging Service Request" },
{ 0x0040, 0x2400, "LT", "3", "Imaging Service Request Comments" },
{ 0x0040, 0x3001, "LO", "3", "Confidentiality Constraint On Patient Data" },
{ 0x0040, 0x9092, "SQ", "3", "Parametric Map Frame Type Sequence" },
{ 0x0040, 0x9094, "SQ", "3", "Referenced Image Real World Value Mapping Sequence" },
{ 0x0040, 0x9096, "SQ", "3", "Real World Value Mapping Sequence" },
{ 0x0040, 0x9098, "SQ", "3", "Pixel Value Mapping Code Sequence" },
{ 0x0040, 0x9210, "SH", "3", "LUT Label" },
{ 0x0040, 0x9211, "SS", "3", "Real World Value Last Value Mapped" },
{ 0x0040, 0x9212, "FD", "3", "Real World Value LUT Data" },
{ 0x0040, 0x9216, "SS", "3", "Real World Value First Value Mapped" },
{ 0x0040, 0x9220, "SQ", "3", "Quantity Definition Sequence" },
{ 0x0040, 0x9224, "FD", "3", "Real World Value Intercept" },
{ 0x0040, 0x9225, "FD", "3", "Real World Value Slope" },
{ 0x0040, 0xA010, "CS", "3STR", "Relationship Type" },
{ 0x0040, 0xA027, "LO", "3STR", "Verifying Organization" },
{ 0x0040, 0xA030, "DT", "3STR", "Verification DateTime" },
{ 0x0040, 0xA032, "DT", "3STR", "Observation DateTime" },
/** Enumerated values: DATE, TIME, DATETIME, PNAME, UIDREF, TEXT, CODE, NUMERIC,
    COMPOSITE, IMAGE. */
{ 0x0040, 0xA040, "CS", "1", "Value Type" },
{ 0x0040, 0xA043, "SQ", "1", "Concept Name Code Sequence" },
{ 0x0040, 0xA050, "CS", "3STR", "Continuity Of Content" },
{ 0x0040, 0xA073, "SQ", "3STR", "Verifying Observer Sequence" },
{ 0x0040, 0xA075, "PN", "3STR", "Verifying Observer Name" },
{ 0x0040, 0xA088, "SQ", "3STR", "Verifying Observer Identification Code Sequence" },
{ 0x0040, 0xA0B0, "US", "3WAV", "Referenced Waveform Channels" },
{ 0x0040, 0xA120, "DT", "1C", "DateTime" }, //< Required if 0xA040 = DATETIME
{ 0x0040, 0xA121, "DA", "1C", "Date" }, //< Required if 0xA040 = DATE
{ 0x0040, 0xA122, "TM", "1C", "Time" }, //< Required if 0xA040 = TIME
{ 0x0040, 0xA123, "PN", "1C", "Person Name" }, //< Required if 0xA040 = PNAME
{ 0x0040, 0xA124, "UI", "1C", "UID" }, //< Required if 0xA040 = UIDREF
{ 0x0040, 0xA130, "CS", "3WAV", "Temporal Range Type" },
{ 0x0040, 0xA132, "UL", "3WAV", "Referenced Sample Positions" },
{ 0x0040, 0xA136, "US", "3DX", "Referenced Frame Numbers" },
{ 0x0040, 0xA138, "DS", "3WAV", "Referenced Time Offsets" },
{ 0x0040, 0xA13A, "DT", "3WAV", "Referenced Datetime" },
{ 0x0040, 0xA160, "UT", "1C", "Text Value" }, //< Required if 0xA040 = TEXT
{ 0x0040, 0xA168, "SQ", "1C", "Concept Code Sequence" }, //< Required if 0xA040 = CODE
{ 0x0040, 0xA170, "SQ", "1C", "Purpose of Reference Code Sequence" },
{ 0x0040, 0xA180, "US", "3WAV", "Annotation Group Number" },
{ 0x0040, 0xA195, "SQ", "3WAV", "Modifier Code Sequence" },
{ 0x0040, 0xA300, "SQ", "3STR", "Measured Value Sequence" },
{ 0x0040, 0xA30A, "DS", "1C", "Numeric Value" }, //< Required if 0xA040 = NUMERIC
{ 0x0040, 0xA360, "SQ", "3STR", "Predecessor Documents Sequence" },
{ 0x0040, 0xA370, "SQ", "3STR", "Referenced Request Sequence" },
{ 0x0040, 0xA372, "SQ", "3STR", "Performed Procedure Code Sequence" },
{ 0x0040, 0xA375, "SQ", "3STR", "Current Requested Procedure Evidence Sequence" },
{ 0x0040, 0xA385, "SQ", "3STR", "Pertinent Other Evidence Sequence" },
{ 0x0040, 0xA390, "SQ", "3", "HL7 Structured Document Reference Sequence" },
{ 0x0040, 0xA491, "CS", "3STR", "Completion Flag" },
{ 0x0040, 0xA492, "LO", "3STR", "Completion Flag Description" },
{ 0x0040, 0xA493, "CS", "3STR", "Verification Flag" },
{ 0x0040, 0xA504, "SQ", "3STR", "Content Template Sequence" },
{ 0x0040, 0xA525, "SQ", "3STR", "Identical Documents Sequence" },
{ 0x0040, 0xA730, "SQ", "3STR", "Content Sequence" },
{ 0x0040, 0xB020, "SQ", "3WAV", "Annotation Sequence" },
{ 0x0040, 0xDB00, "CS", "3STR", "Template Identifier" },
{ 0x0040, 0xDB06, "DT", "3RET", "Template Version" },
{ 0x0040, 0xDB07, "DT", "3RET", "Template Local Version" },
{ 0x0040, 0xDB0B, "CS", "3RET", "Template Extension Flag" },
{ 0x0040, 0xDB0C, "UI", "3RET", "Template Extension Organization UID" },
{ 0x0040, 0xDB0D, "UI", "3RET", "Template Extension Creator UID" },
{ 0x0040, 0xDB73, "UL", "3STR", "Referenced Content Item Identifier" },
{ 0x0040, 0xE001, "ST", "3", "HL7 Instance Identifier" },
{ 0x0040, 0xE004, "DT", "3", "HL7 Document Effective Time" },
{ 0x0040, 0xE006, "SQ", "3", "HL7 Document Type Code Sequence" },
{ 0x0040, 0xE010, "UT", "3", "Retrieve URI" },

{ 0x0041, 0x0001, "SL", "SIE", "Siemens Private" },
{ 0x0041, 0x0002, "SL", "SIE", "Siemens Private" },
{ 0x0041, 0x0010, "LO", "PHI", "Philips Private Creator Data Element" },
{ 0x0041, 0x1012, "CS", "PHI", "Philips Private Document Status" },

{ 0x0042, 0x0010, "ST", "3", "Document Title" },
{ 0x0042, 0x0011, "OB", "3", "Encapsulated Document" },
{ 0x0042, 0x0012, "LO", "3", "MIME Type Of Encapsulated Document" },
{ 0x0042, 0x0013, "SQ", "3", "Source Instance Sequence" },

{ 0x0050, 0x0000, "UL", "3", "Calibration Group Length" },
{ 0x0050, 0x0004, "CS", "3", "Calibration Object" },
{ 0x0050, 0x0010, "SQ", "3", "Device Sequence" },
{ 0x0050, 0x0014, "DS", "3", "Device Length" },
{ 0x0050, 0x0016, "DS", "3", "Device Diameter" },
{ 0x0050, 0x0017, "CS", "3", "Device Diameter Units" },
{ 0x0050, 0x0018, "DS", "3", "Device Volume" },
{ 0x0050, 0x0019, "DS", "3", "Inter Marker Distance" },
{ 0x0050, 0x0020, "LO", "3", "Device Description" },
{ 0x0050, 0x0030, "SQ", "3", "Coded Intervention Device Sequence" },

{ 0x0054, 0x0000, "UL", "3", "Nuclear Acquisition Group Length" },
{ 0x0054, 0x0010, "US", "3", "Energy Window Vector" },
{ 0x0054, 0x0011, "US", "3", "Number of Energy Windows" },
{ 0x0054, 0x0012, "SQ", "3", "Energy Window Information Sequence" },
{ 0x0054, 0x0013, "SQ", "3", "Energy Window Range Sequence" },
{ 0x0054, 0x0014, "DS", "3", "Energy Window Lower Limit (KeV)" },
{ 0x0054, 0x0015, "DS", "3", "Energy Window Upper Limit (KeV)" },
{ 0x0054, 0x0016, "SQ", "3", "Radiopharmaceutical Information Sequence" },
{ 0x0054, 0x0017, "IS", "3", "Residual Syringe Counts" },
{ 0x0054, 0x0018, "SH", "3", "Energy Window Name" },
{ 0x0054, 0x0020, "US", "3", "Detector Vector" },
{ 0x0054, 0x0021, "US", "3", "Number of Detectors" },
{ 0x0054, 0x0022, "SQ", "3", "Detector Information Sequence" },
{ 0x0054, 0x0030, "US", "3", "Phase Vector" },
{ 0x0054, 0x0031, "US", "3", "Number of Phases" },
{ 0x0054, 0x0032, "SQ", "3", "Phase Information Sequence" },
{ 0x0054, 0x0033, "US", "3", "Number of Frames In Phase" },
{ 0x0054, 0x0036, "IS", "3", "Phase Delay" },
{ 0x0054, 0x0038, "IS", "3", "Pause Between Frames" },
{ 0x0054, 0x0039, "CS", "3", "Phase Description" },
{ 0x0054, 0x0050, "US", "3", "Rotation Vector" },
{ 0x0054, 0x0051, "US", "3", "Number of Rotations" },
{ 0x0054, 0x0052, "SQ", "3", "Rotation Information Sequence" },
{ 0x0054, 0x0053, "US", "3", "Number of Frames In Rotation" },
{ 0x0054, 0x0060, "US", "3", "R-R Interval Vector" },
{ 0x0054, 0x0061, "US", "3", "Number of R-R Intervals" },
{ 0x0054, 0x0062, "SQ", "3", "Gated Information Sequence" },
{ 0x0054, 0x0063, "SQ", "3", "Data Information Sequence" },
{ 0x0054, 0x0070, "US", "3", "Time Slot Vector" },
{ 0x0054, 0x0071, "US", "3", "Number of Time Slots" },
{ 0x0054, 0x0072, "SQ", "3", "Time Slot Information Sequence" },
{ 0x0054, 0x0073, "DS", "3", "Time Slot Time" },
{ 0x0054, 0x0080, "US", "3", "Slice Vector" },
{ 0x0054, 0x0081, "US", "3", "Number of Slices" },
{ 0x0054, 0x0090, "US", "3", "Angular View Vector" },
{ 0x0054, 0x0100, "US", "3", "Time Slice Vector" },
{ 0x0054, 0x0101, "US", "3PET", "Number Of Time Slices" },
{ 0x0054, 0x0200, "DS", "3", "Start Angle" },
{ 0x0054, 0x0202, "CS", "3", "Type of Detector Motion" },
{ 0x0054, 0x0210, "IS", "3", "Trigger Vector" },
{ 0x0054, 0x0211, "US", "3", "Number of Triggers in Phase" },
{ 0x0054, 0x0220, "SQ", "3", "View Code Sequence" },
{ 0x0054, 0x0222, "SQ", "3", "View Modifier Code Sequence" },
{ 0x0054, 0x0300, "SQ", "3", "Radionuclide Code Sequence" },
{ 0x0054, 0x0302, "SQ", "3", "Radiopharmaceutical Route Code Sequence" },
{ 0x0054, 0x0304, "SQ", "3", "Radiopharmaceutical Code Sequence" },
{ 0x0054, 0x0306, "SQ", "3", "Calibration Data Sequence" },
{ 0x0054, 0x0308, "US", "3", "Energy Window Number" },
{ 0x0054, 0x0400, "SH", "3", "Image ID" },
{ 0x0054, 0x0410, "SQ", "3", "Patient Orientation Code Sequence" },
{ 0x0054, 0x0412, "SQ", "3", "Patient Orientation Modifier Code Sequence" },
{ 0x0054, 0x0414, "SQ", "3", "Patient Gantry Relationship Code Sequence" },
{ 0x0054, 0x0500, "CS", "3", "Slice Progression Direction" },
{ 0x0054, 0x0501, "CS", "3", "Scan Progression Direction" },
{ 0x0054, 0x1000, "CS", "3PET", "Positron Emission Tomography Series Type" }, //< Value 1: STATIC,DYNAMIC,GATED,WHOLE BODY ; Value 2: IMAGE,REPROJECTION
{ 0x0054, 0x1001, "CS", "3PET", "Positron Emission Tomography Units" }, //< CNTS,NONE,CM2,PCNT,CPS,BQML,MGMINML,UMOLMINML,MLMING,MLG,1CM,UMOLML,PROPCNTS,PROPCPS,MLMINML,MLML,GML,STDDEV
{ 0x0054, 0x1002, "CS", "3PET", "Counts Source" }, //< EMISSION,TRANSMISSION
{ 0x0054, 0x1004, "CS", "3PET", "Reprojection Method" }, //< SUM,MAX,PIXEL
{ 0x0054, 0x1006, "CS", "3PET", "SUV Type" },
{ 0x0054, 0x1100, "CS", "3PET", "Randoms Correction Method" }, //< NONE,DLYD,SING
{ 0x0054, 0x1101, "LO", "3PET", "Attenuation Correction Method" },
{ 0x0054, 0x1102, "CS", "3PET", "Decay Correction" }, //< NONE, START, ADMIN
{ 0x0054, 0x1103, "LO", "3PET", "Reconstruction Method" },
{ 0x0054, 0x1104, "LO", "3PET", "Detector Lines of Response Used" },
{ 0x0054, 0x1105, "LO", "3PET", "Scatter Correction Method" },
{ 0x0054, 0x1200, "DS", "3PET", "Axial Acceptance" },
{ 0x0054, 0x1201, "IS", "3PET", "Axial Mash" },
{ 0x0054, 0x1202, "IS", "3PET", "Transverse Mash" },
{ 0x0054, 0x1203, "DS", "3PET", "Detector Element Size" },
{ 0x0054, 0x1210, "DS", "3PET", "Coincidence Window Width" },
{ 0x0054, 0x1220, "CS", "3PET", "Secondary Counts Type" },
{ 0x0054, 0x1300, "DS", "3PET", "Frame Reference Time" }, //< Time (relative to series reference time) for pixel values; may be frame start time, mid time, or a little less than mid time.
{ 0x0054, 0x1310, "IS", "3PET", "Primary Prompts Counts Accumulated" },
{ 0x0054, 0x1311, "IS", "3PET", "Secondary Counts Accumulated" },
{ 0x0054, 0x1320, "DS", "3PET", "Slice Sensitivity Factor" },
{ 0x0054, 0x1321, "DS", "3PET", "Decay Factor" },
{ 0x0054, 0x1322, "DS", "3PET", "Dose Calibration Factor" },
{ 0x0054, 0x1323, "DS", "3PET", "Scatter Fraction Factor" },
{ 0x0054, 0x1324, "DS", "3PET", "Dead Time Factor" },
{ 0x0054, 0x1330, "US", "3PET", "Image Index" },
{ 0x0054, 0x1400, "CS", "3RET", "Counts Included" }, //< TRUES,SCATTER,RANDOMS,SINGLES
{ 0x0054, 0x1401, "CS", "3RET", "Dead Time Correction Flag" }, //< Y,N

{ 0x0060, 0x3000, "SQ", "3DX", "Histogram Sequence" },
{ 0x0060, 0x3002, "US", "3DX", "Histogram Number of Bins" },
{ 0x0060, 0x3004, "US or SS", "3DX", "Histogram First Bin Value" },
{ 0x0060, 0x3006, "US or SS", "3DX", "Histogram Last Bin Value" },
{ 0x0060, 0x3008, "US", "3DX", "Histogram Bin Width" },
{ 0x0060, 0x3010, "LO", "3DX", "Histogram Explanation" },
{ 0x0060, 0x3020, "UL", "3DX", "Histogram Data" },

{ 0x0062, 0x0001, "CS", "3", "Segmentation Type" },
{ 0x0062, 0x0002, "SQ", "3", "Segment Sequence" },
{ 0x0062, 0x0003, "SQ", "3", "Segmented Property Category Code Sequence" },
{ 0x0062, 0x0004, "US", "3", "Segment Number" },
{ 0x0062, 0x0005, "LO", "3", "Segment Label" },
{ 0x0062, 0x0006, "ST", "3", "Segment Description" },
{ 0x0062, 0x0008, "CS", "3", "Segment Algorithm Type" },
{ 0x0062, 0x0009, "LO", "3", "Segment Algorithm Name" },
{ 0x0062, 0x000A, "SQ", "3", "Segment Identification Sequence" },
{ 0x0062, 0x000B, "US", "1C", "Referenced Segment Number" },
{ 0x0062, 0x000C, "US", "3", "Recommended Display Grayscale Value" },
{ 0x0062, 0x000D, "US", "3", "Recommended Display CIE Lab Value" },
{ 0x0062, 0x000E, "US", "3", "Maximum Fractional Value" },
{ 0x0062, 0x000F, "SQ", "3", "Segmented Property Type Code Sequence" },
{ 0x0062, 0x0010, "CS", "3", "Segmentation Fractional Type" },

{ 0x0064, 0x0002, "SQ", "3", "Deformable Registration Sequence" },
{ 0x0064, 0x0003, "UI", "3", "Source Frame Of Reference UID" },
{ 0x0064, 0x0005, "SQ", "3", "Deformable Registration Grid Sequence" },
{ 0x0064, 0x0007, "UL", "3", "Grid Dimensions" },
{ 0x0064, 0x0008, "FD", "3", "Grid Resolution" },
{ 0x0064, 0x0009, "OF", "3", "Vector Grid Data" },
{ 0x0064, 0x000F, "SQ", "3", "Pre Deformation Matrix Registration Sequence" },
{ 0x0064, 0x0010, "SQ", "3", "Post Deformation Matrix Registration Sequence" },

{ 0x0070, 0x0001, "SQ", "3SCP", "Graphic Annotation Sequence" },
{ 0x0070, 0x0002, "CS", "3SCP", "Graphic Layer" },
{ 0x0070, 0x0003, "CS", "3SCP", "Bounding Box Annotation Units" },
{ 0x0070, 0x0004, "CS", "3SCP", "Anchor Point Annotation Units" },
{ 0x0070, 0x0005, "CS", "3SCP", "Graphic Annotation Units" },
{ 0x0070, 0x0006, "ST", "3SCP", "Unformatted Text Value" },
{ 0x0070, 0x0008, "SQ", "3SCP", "Text Object Sequence" },
{ 0x0070, 0x0009, "SQ", "3SCP", "Graphic Object Sequence" },
{ 0x0070, 0x0010, "FL", "3SCP", "Bounding Box TLHC" },
{ 0x0070, 0x0011, "FL", "3SCP", "Bounding Box BRHC" },
{ 0x0070, 0x0012, "CS", "3SCP", "Bounding Box Text Horizontal Justification" },
{ 0x0070, 0x0014, "FL", "3SCP", "Anchor Point" },
{ 0x0070, 0x0015, "CS", "3SCP", "Anchor Point Visibility" },
{ 0x0070, 0x0020, "US", "3SCP", "Graphic Dimensions" },
{ 0x0070, 0x0021, "US", "3SCP", "Number Of Graphic Points" },
{ 0x0070, 0x0022, "FL", "3SCP", "Graphic Data" },
{ 0x0070, 0x0023, "CS", "3SCP", "Graphic Type" },
{ 0x0070, 0x0024, "CS", "3SCP", "Graphic Filled" },
{ 0x0070, 0x0040, "IS", "3RET", "Image Rotation" },
{ 0x0070, 0x0041, "CS", "3SCP", "Image Horizontal Flip" },
{ 0x0070, 0x0042, "US", "3SCP", "Image Rotation" },
{ 0x0070, 0x0050, "US", "3RET", "Displayed Area TLHC" },
{ 0x0070, 0x0051, "US", "3RET", "Displayed Area BRHC" },
{ 0x0070, 0x0052, "SL", "3SCP", "Displayed Area TLHC" }, // Top left hand corner
{ 0x0070, 0x0053, "SL", "3SCP", "Displayed Area BRHC" }, // Bottom right hand corner
{ 0x0070, 0x005A, "SQ", "3SCP", "Displayed Area Selection Sequence" },
{ 0x0070, 0x0060, "SQ", "3SCP", "Graphic Layer Sequence" },
{ 0x0070, 0x0062, "IS", "3SCP", "Graphic Layer Order" },
{ 0x0070, 0x0066, "US", "3SCP", "Graphic Layer Recommended Display Grayscale Value" },
{ 0x0070, 0x0067, "US", "3RET", "Graphic Layer Recommended Display RGB Value" },
{ 0x0070, 0x0068, "LO", "3SCP", "Graphic Layer Description" },
{ 0x0070, 0x0080, "CS", "3SCP", "Presentation Label" },
{ 0x0070, 0x0081, "LO", "3SCP", "Presentation Description" },
{ 0x0070, 0x0082, "DA", "3SCP", "Presentation Creation Date" },
{ 0x0070, 0x0083, "TM", "3SCP", "Presentation Creation Time" },
{ 0x0070, 0x0084, "PN", "3SCP", "Presentation Creator Name" },
{ 0x0070, 0x0100, "CS", "3SCP", "Presentation Size Mode" },
{ 0x0070, 0x0101, "DS", "3SCP", "Presentation Pixel Spacing" },
{ 0x0070, 0x0102, "IS", "3SCP", "Presentation Pixel Aspect Ratio" },
{ 0x0070, 0x0103, "FL", "3SCP", "Presentation Pixel Magnification Ratio" },
{ 0x0070, 0x0306, "CS", "3", "ShapeType" },
{ 0x0070, 0x0308, "SQ", "3", "Registration Sequence" },
{ 0x0070, 0x0309, "SQ", "3", "Matrix Registration Sequence" },
{ 0x0070, 0x030A, "SQ", "3", "Matrix Sequence" },
{ 0x0070, 0x030C, "CS", "3", "Frame Of Reference Transformation Matrix Type" },
{ 0x0070, 0x030D, "SQ", "3", "Registration Type Code Sequence" },
{ 0x0070, 0x030F, "ST", "3", "Fiducial Description" },
{ 0x0070, 0x0310, "SH", "3", "Fiducial Identifier" },
{ 0x0070, 0x0311, "SQ", "3", "Fiducial Identifier Code Sequence" },
{ 0x0070, 0x0312, "FD", "3", "Contour Uncertainty Radius" },
{ 0x0070, 0x0314, "SQ", "3", "Used Fiducials Sequence" },
{ 0x0070, 0x0318, "SQ", "3", "Graphic Coordinates Data Sequence" },
{ 0x0070, 0x031A, "UI", "3", "Fiducial UID" },
{ 0x0070, 0x031C, "SQ", "3", "Fiducial Set Sequence" },
{ 0x0070, 0x031E, "SQ", "3", "Fiducial Sequence" },
{ 0x0070, 0x0401, "US", "3", "Graphic Layer Recommended Display CIE Lab Value" },
{ 0x0070, 0x0402, "SQ", "3", "Blending Sequence" },
{ 0x0070, 0x0403, "FL", "3", "Relative Opacity" },
{ 0x0070, 0x0404, "SQ", "3", "Referenced Spatial Registration Sequence" },
{ 0x0070, 0x0405, "CS", "3", "Blending Position" },

// Siemens
{ 0x0071, 0x0010, "LO", "SIE", "Siemens Private PET Attributes" },
{ 0x0071, 0x1021, "UI", "SIE", "Registration Matrix UID" },
{ 0x0071, 0x1022, "DT", "SIE", "Decay Correction DateTime" },
{ 0x0071, 0x1023, "FD", "SIE", "Registration matrix" },
{ 0x0071, 0x1024, "CS", "SIE", "Table Motion" },

{ 0x0088, 0x0000, "UL", "3", "Storage Media Group Length" },
{ 0x0088, 0x0130, "SH", "3", "Storage Media Fileset ID" },
{ 0x0088, 0x0140, "UI", "3", "Storage Media Fileset UID" },
{ 0x0088, 0x0200, "SQ", "3", "Icon Image Sequence" },
{ 0x0088, 0x0904, "LO", "3RET", "Topic Title" },
{ 0x0088, 0x0906, "ST", "3RET", "Topic Subject" },
{ 0x0088, 0x0910, "LO", "3RET", "Topic Author" },
{ 0x0088, 0x0912, "LO", "3RET", "Topic Key Words" },

{ 0x00E1, 0x0000, "UL", "PHI", "Philips Private Creator Data Group Length" },
{ 0x00E1, 0x0010, "LO", "PHI", "Philips Private Creator Data Element" },
{ 0x00E1, 0x10C2, "UI", "PHI", "Philips Private Element" },

{ 0x0100, 0x0410, "CS", "3", "SOP Instance Status" },
{ 0x0100, 0x0420, "DT", "3", "SOP Authorization DateTime" },
{ 0x0100, 0x0424, "LT", "3", "SOP Authorization Comment" },
{ 0x0100, 0x0426, "LO", "3", "Authorization Equipment Certification Number" },

{ 0x01E1, 0x0010, "LO", "PHI", "Philips Private Creator Data Element" },
{ 0x01E1, 0x1021, "UI", "PHI", "Philips Private CT Gating UID" },

{ 0x01F1, 0x0010, "LO", "PHI", "Philips Private Creator Data Element" },
{ 0x01F1, 0x1039, "CS", "PHI", "Philips Private Element" },
{ 0x01F1, 0x1041, "LO", "PHI", "Philips Private Element" },

{ 0x0400, 0x0005, "US", "3", "MAC ID Number" },
{ 0x0400, 0x0010, "UI", "3", "MAC Calculation Transfer Syntax UID" },
{ 0x0400, 0x0015, "CS", "3", "MAC Algorithm" },
{ 0x0400, 0x0020, "AT", "3", "Data Elements Signed" },
{ 0x0400, 0x0100, "UI", "3", "Digital Signature UID" },
{ 0x0400, 0x0105, "DT", "3", "Digital Signature Date Time" },
{ 0x0400, 0x0110, "CS", "3", "Certificate Type" },
{ 0x0400, 0x0115, "OB", "3", "Certificate Of Signer" },
{ 0x0400, 0x0120, "OB", "3", "Signature" },
{ 0x0400, 0x0305, "CS", "3", "Certified Timestamp Type" },
{ 0x0400, 0x0310, "OB", "3", "Certified Timestamp" },
{ 0x0400, 0x0401, "SQ", "3", "Digital Signature Purpose Code Sequence" },
{ 0x0400, 0x0402, "SQ", "3", "Referenced Digital Signature Sequence" },
{ 0x0400, 0x0403, "SQ", "3", "Referenced SOP Instance MAC Sequence" },
{ 0x0400, 0x0404, "OB", "3", "MAC" },
{ 0x0400, 0x0500, "SQ", "3", "Encrypted Attributes Sequence" },
{ 0x0400, 0x0510, "UI", "3", "Encrypted Content Transfer Syntax UID" },
{ 0x0400, 0x0520, "OB", "3", "Encrypted Content" },
{ 0x0400, 0x0550, "SQ", "3", "Modified Attributes Sequence" },
{ 0x0400, 0x0561, "SQ", "3", "Original Attributes Sequence" },
{ 0x0400, 0x0562, "DT", "3", "Attribute Modification Datetime" },
{ 0x0400, 0x0563, "LO", "3", "Modifying System" },
{ 0x0400, 0x0564, "LO", "3", "Source Of Previous Values" },
{ 0x0400, 0x0565, "CS", "3", "Reason For The Attribute Modification" },

{ 0x1000, 0x0000, "UL", "2C", "Code Table Group Length" },
{ 0x1000, 0x0011, "US", "2CRET", "Run Length Triplet" },
{ 0x1000, 0x0012, "US", "2CRET", "Huffman Table Size" },
{ 0x1000, 0x0013, "US", "2CRET", "Huffman Table Triplet" },
{ 0x1000, 0x0014, "US", "2CRET", "Shift Table Size" },
{ 0x1000, 0x0015, "US", "2CRET", "Shift Table Triplet" },

{ 0x1010, 0x0000, "UL", "2C", "Zonal Map Group Length" },
{ 0x1010, 0x0004, "US", "2CRET", "Zonal Map" },

{ 0x2000, 0x0000, "UL", "3", "Film Session Group Length" },
{ 0x2000, 0x0010, "IS", "3", "Number of Copies" },
{ 0x2000, 0x0020, "CS", "3", "Print Priority" },
{ 0x2000, 0x0030, "CS", "3", "Medium Type" },
{ 0x2000, 0x0040, "CS", "3", "Film Destination" },
{ 0x2000, 0x0050, "LO", "3", "Film Session Label" },
{ 0x2000, 0x0060, "IS", "3", "Memory Allocation" },
{ 0x2000, 0x0500, "SQ", "3", "Referenced Film Box Sequence" },
{ 0x2000, 0x0510, "SQ", "3STP", "Referenced Stored Print Sequence" },

{ 0x2010, 0x0000, "UL", "3", "Film Box Group Length" },
{ 0x2010, 0x0010, "ST", "3", "Image Display Format" },
{ 0x2010, 0x0030, "CS", "3", "Annotation Display Format ID" },
{ 0x2010, 0x0040, "CS", "3", "Film Orientation" },
{ 0x2010, 0x0050, "CS", "3", "Film Size ID" },
{ 0x2010, 0x0060, "CS", "3", "Magnification Type" },
{ 0x2010, 0x0080, "CS", "3", "Smoothing Type" },
{ 0x2010, 0x0100, "CS", "3", "Border Density" },
{ 0x2010, 0x0110, "CS", "3", "Empty Image Density" },
{ 0x2010, 0x0120, "US", "3", "Min Density" },
{ 0x2010, 0x0130, "US", "3", "Max Density" },
{ 0x2010, 0x0140, "CS", "3", "Trim" },
{ 0x2010, 0x0150, "ST", "3", "Configuration Information" },
{ 0x2010, 0x0500, "SQ", "3", "Referenced Film Session Sequence" },
{ 0x2010, 0x0510, "SQ", "3", "Referenced Basic Image Box Sequence" },
{ 0x2010, 0x0520, "SQ", "3", "Referenced Basic Annotation Box Sequence" },

{ 0x2020, 0x0000, "UL", "3", "Image Box Group Length" },
{ 0x2020, 0x0010, "US", "3", "Image Box Position" },
{ 0x2020, 0x0020, "CS", "3", "Polarity" },
{ 0x2020, 0x0030, "DS", "3", "Requested Image Size" },
{ 0x2020, 0x0110, "SQ", "3", "Basic Greyscale Image Sequence" },
{ 0x2020, 0x0111, "SQ", "3", "Basic Color Image Sequence" },
{ 0x2020, 0x0130, "SQ", "3RET", "Referenced Image Overlay Box Sequence" },
{ 0x2020, 0x0140, "SQ", "3RET", "Referenced VOI LUT Box Sequence" },

{ 0x2030, 0x0000, "UL", "3", "Annotation Group Length" },
{ 0x2030, 0x0010, "US", "3", "Annotation Position" },
{ 0x2030, 0x0020, "LO", "3", "Text String" },

{ 0x2040, 0x0000, "UL", "3", "Overlay Box Group Length" },
{ 0x2040, 0x0010, "SQ", "3RET", "Referenced Overlay Plane Sequence" },
{ 0x2040, 0x0011, "US", "3RET", "Refenced Overlay Plane Groups" },
{ 0x2040, 0x0020, "SQ", "3RET", "Overlay Pixel Data Sequence" },
{ 0x2040, 0x0060, "CS", "3RET", "Overlay Magnification Type" },
{ 0x2040, 0x0070, "CS", "3RET", "Overlay Smoothing Type" },
{ 0x2040, 0x0072, "CS", "3RET", "Overlay or Image Magnification" },
{ 0x2040, 0x0074, "US", "3RET", "Magnify to Number of Columns" },
{ 0x2040, 0x0080, "CS", "3RET", "Overlay Foreground Density" },
{ 0x2040, 0x0082, "CS", "3RET", "Overlay Background Density" },
{ 0x2040, 0x0090, "CS", "3RET", "Overlay Mode" },
{ 0x2040, 0x0100, "CS", "3RET", "Threshold Density" },
{ 0x2040, 0x0500, "SQ", "3RET", "Referenced Overlay Image Box Sequence" },

{ 0x2050, 0x0010, "SQ", "3", "Presentation LUT Sequence" },
{ 0x2050, 0x0020, "CS", "3", "Presentation LUT Shape" },
{ 0x2050, 0x0500, "SQ", "3", "Referenced Presentation LUT Sequence" },

{ 0x2100, 0x0000, "UL", "3", "Print Job Group Length" },
{ 0x2100, 0x0010, "SH", "3PQ", "Print Job ID" },
{ 0x2100, 0x0020, "CS", "3", "Execution Status" },
{ 0x2100, 0x0030, "CS", "3", "Execution Status Info" },
{ 0x2100, 0x0040, "DA", "3", "Creation Date" },
{ 0x2100, 0x0050, "TM", "3", "Creation Time" },
{ 0x2100, 0x0070, "AE", "3", "Originator" },
{ 0x2100, 0x0500, "SQ", "3RET", "Referenced Print Job Sequence" },

{ 0x2110, 0x0000, "UL", "3", "Printer Group Length" },
{ 0x2110, 0x0010, "CS", "3", "Printer Status" },
{ 0x2110, 0x0020, "CS", "3", "Printer Status Info" },
{ 0x2110, 0x0030, "ST", "3", "Printer Name" },
{ 0x2110, 0x0099, "SH", "3RET", "Printer Queue ID"},

{ 0x3002, 0x0000, "UL", "3RT", "RT Image Group Length" },
{ 0x3002, 0x0002, "SH", "3RT", "RT Image Label" },
{ 0x3002, 0x0003, "LO", "3RT", "RT Image Name" },
{ 0x3002, 0x0004, "ST", "3RT", "RT Image Description" },
{ 0x3002, 0x000A, "CS", "3RT", "Reported Values Origin" },
{ 0x3002, 0x000C, "CS", "3RT", "Image Plane" },
{ 0x3002, 0x000E, "DS", "3RT", "X-Ray Image Receptor Angle" },
{ 0x3002, 0x0010, "DS", "3RT", "Image Orientation" },
{ 0x3002, 0x0011, "DS", "3RT", "Image Plane Pixel Spacing" },
{ 0x3002, 0x0012, "DS", "3RT", "Image Position" },
{ 0x3002, 0x0020, "SH", "3RT", "Radiation Machine Name" },
{ 0x3002, 0x0022, "DS", "3RT", "Radiation Machine SAD" },
{ 0x3002, 0x0024, "DS", "3RT", "Radiation Machine SSD" },
{ 0x3002, 0x0026, "DS", "3RT", "Image SID" },
{ 0x3002, 0x0028, "DS", "3RT", "Source to Reference Object Distance" },
{ 0x3002, 0x0029, "IS", "3RT", "Fraction Number" },
{ 0x3002, 0x0030, "SQ", "3RT", "Exposure Sequence" },
{ 0x3002, 0x0032, "DS", "3RT", "Meterset Exposure" },

{ 0x3004, 0x0000, "UL", "3RT", "Dose Group Length" },
{ 0x3004, 0x0001, "CS", "3RT", "DVH Type" },
{ 0x3004, 0x0002, "CS", "3RT", "Dose Units" },
{ 0x3004, 0x0004, "CS", "3RT", "Dose Type" },
{ 0x3004, 0x0006, "LO", "3RT", "Dose Comment" },
{ 0x3004, 0x0008, "DS", "3RT", "Normalization Point" },
{ 0x3004, 0x000a, "CS", "3RT", "Dose Summation Type" },
{ 0x3004, 0x000c, "DS", "3RT", "Grid Frame Offset Vector" },
{ 0x3004, 0x000e, "DS", "3RT", "Dose Grid Scaling" },
{ 0x3004, 0x0010, "SQ", "3RT", "RT Dose ROI Sequence" },
{ 0x3004, 0x0012, "DS", "3RT", "Dose Value" },
{ 0x3004, 0x0040, "DS", "3RT", "Normalization Point" },
{ 0x3004, 0x0042, "DS", "3RT", "Normalization Dose Value" },
{ 0x3004, 0x0050, "SQ", "3RT", "Sequence" },
{ 0x3004, 0x0052, "DS", "3RT", "Dose Scaling" },
{ 0x3004, 0x0054, "CS", "3RT", "Volume Units" },
{ 0x3004, 0x0056, "IS", "3RT", "Number of Bins" },
{ 0x3004, 0x0058, "DS", "3RT", "Data" },
{ 0x3004, 0x0060, "SQ", "3RT", "Referenced ROI Sequence" },
{ 0x3004, 0x0062, "CS", "3RT", "ROI Contribution Type" },
{ 0x3004, 0x0070, "DS", "3RT", "Minimum Dose" },
{ 0x3004, 0x0072, "DS", "3RT", "Maximum Dose" },
{ 0x3004, 0x0074, "DS", "3RT", "Mean Dose" },

{ 0x3006, 0x0000, "UL", "3", "ROI Group Length" },
{ 0x3006, 0x0002, "SH", "3", "Structure Set Label" },
{ 0x3006, 0x0004, "LO", "3", "Structure Set Name" },
{ 0x3006, 0x0006, "ST", "3", "Structure Set Description" },
{ 0x3006, 0x0008, "DA", "3", "Structure Set Date" },
{ 0x3006, 0x0009, "TM", "3", "Structure Set Time" },
{ 0x3006, 0x0010, "SQ", "3", "Referenced Frame of Reference Sequence" },
{ 0x3006, 0x0016, "SQ", "3", "Contour Image Sequence" },
{ 0x3006, 0x0020, "SQ", "3", "Structure Set ROI Sequence" },
{ 0x3006, 0x0022, "IS", "3", "ROI Number" },
{ 0x3006, 0x0024, "UI", "3", "Referenced Frame of Reference UID" },
{ 0x3006, 0x0026, "LO", "3", "ROI Name" },
{ 0x3006, 0x0036, "CS", "3", "ROI Generation Algorithm" },
{ 0x3006, 0x0040, "SQ", "3", "Contour Sequence" },
{ 0x3006, 0x0048, "IS", "3", "Contour Number" },
{ 0x3006, 0x0049, "IS", "3", "Attached Contours" },

{ 0x3008, 0x0000, "UL", "3RT", "Treatment Group Length" },

{ 0x300A, 0x0000, "UL", "3RT", "RT Plan Group Length" },

{ 0x300C, 0x0000, "UL", "3RT", "Referenced RT Group Length" },

{ 0x300E, 0x0000, "UL", "3RT", "RT Review Group Length" },

{ 0x4000, 0x0000, "UL", "3RET", "Text Group Length" },
{ 0x4000, 0x0010, "LT", "3RET", "Arbitrary Text" },
{ 0x4000, 0x4000, "LT", "3RET", "Text Group Comments" },

{ 0x4008, 0x0000, "UL", "3", "Results Group Length" },
{ 0x4008, 0x0040, "SH", "3RET", "Results ID" },
{ 0x4008, 0x0042, "LO", "3RET", "Results ID Issuer" },
{ 0x4008, 0x0050, "SQ", "3RET", "Referenced Interpretation Sequence" },
{ 0x4008, 0x0100, "DA", "3RET", "Interpretation Recorded Date" },
{ 0x4008, 0x0101, "TM", "3RET", "Interpretation Recorded Time" },
{ 0x4008, 0x0102, "PN", "3RET", "Interpretation Recorder" },
{ 0x4008, 0x0103, "LO", "3RET", "Reference to Recorded Sound" },
{ 0x4008, 0x0108, "DA", "3RET", "Interpretation Transcription Date" },
{ 0x4008, 0x0109, "TM", "3RET", "Interpretation Transcription Time" },
{ 0x4008, 0x010A, "PN", "3RET", "Interpretation Transcriber" },
{ 0x4008, 0x010B, "ST", "3RET", "Interpretation Text" },
{ 0x4008, 0x010C, "PN", "3RET", "Interpretation Author" },
{ 0x4008, 0x0111, "SQ", "3RET", "Interpretation Approver Sequence" },
{ 0x4008, 0x0112, "DA", "3RET", "Interpretation Approval Date" },
{ 0x4008, 0x0113, "TM", "3RET", "Interpretation Approval Time" },
{ 0x4008, 0x0114, "PN", "3RET", "Physician Approving Interpretation" },
{ 0x4008, 0x0115, "LT", "3RET", "Interpretation Diagnosis Description" },
{ 0x4008, 0x0117, "SQ", "3RET", "Interpretation Diagnosis Code Sequence" },
{ 0x4008, 0x0118, "SQ", "3RET", "Results Distribution List Sequence" },
{ 0x4008, 0x0119, "PN", "3RET", "Distribution Name" },
{ 0x4008, 0x011A, "LO", "3RET", "Distribution Address" },
{ 0x4008, 0x0200, "SH", "3RET", "Interpretation ID" },
{ 0x4008, 0x0202, "LO", "3RET", "Interpretation ID Issuer" },
{ 0x4008, 0x0210, "CS", "3RET", "Interpretation Type ID" },
{ 0x4008, 0x0212, "CS", "3RET", "Interpretation Status ID" },
{ 0x4008, 0x0300, "ST", "3RET", "Impressions" },
{ 0x4008, 0x4000, "SH", "3RET", "Results Group Comments" },

{ 0x4FFE, 0x0001, "SQ", "3", "MAC Parameters Sequence" },

{ 0x5000, 0x0000, "UL", "3", "Curve Group Length" },
{ 0x5000, 0x0005, "US", "3RET", "Curve Dimensions" },
{ 0x5000, 0x0010, "US", "3RET", "Number of Points" },
{ 0x5000, 0x0020, "CS", "3RET", "Type of Data" },
{ 0x5000, 0x0022, "LO", "3RET", "Curve Description" },
{ 0x5000, 0x0030, "SH", "3RET", "Axis Units" },
{ 0x5000, 0x0040, "SH", "3RET", "Axis Labels" },
{ 0x5000, 0x0103, "US", "3RET", "Data Value Representation" },
{ 0x5000, 0x0104, "US", "3RET", "Minimum Coordinate Value" },
{ 0x5000, 0x0105, "US", "3RET", "Maximum Coordinate Value" },
{ 0x5000, 0x0106, "SH", "3RET", "Curve Range" },
{ 0x5000, 0x0110, "US", "3RET", "Curve Data Descriptor" },
{ 0x5000, 0x0112, "US", "3RET", "Coordinate Start Value" },
{ 0x5000, 0x0114, "US", "3RET", "Coordinate Step Value" },
{ 0x5000, 0x2000, "US", "3RET", "Audio Type" },
{ 0x5000, 0x2002, "US", "3RET", "Audio Sample Format" },
{ 0x5000, 0x2004, "US", "3RET", "Number of Channels" },
{ 0x5000, 0x2006, "UL", "3RET", "Number of Samples" },
{ 0x5000, 0x2008, "UL", "3RET", "Sample Rate" },
{ 0x5000, 0x200A, "UL", "3RET", "Total Time" },
{ 0x5000, 0x200C, "OX", "3RET", "Audio Sample Data" },
{ 0x5000, 0x200E, "LT", "3RET", "Audio Comments" },
{ 0x5000, 0x2500, "LO", "3RET", "Curve Label" },
{ 0x5000, 0x2600, "SQ", "3RET", "CurveReferenced Overlay Sequence" },
{ 0x5000, 0x2610, "US", "3RET", "CurveReferenced Overlay Group" },
{ 0x5000, 0x3000, "OX", "3RET", "Curve Data" },

{ 0x5200, 0x9229, "SQ", "3", "Shared Functional Groups Sequence" },
{ 0x5200, 0x9230, "SQ", "3", "Per Frame Functional Groups Sequence" },

{ 0x5400, 0x0100, "SQ", "3WAV", "Waveform Sequence" },
{ 0x5400, 0x1010, "OW", "3WAV", "Waveform Data"}, // DICOM PS3.5 2017a

{ 0x5600, 0x0010, "OF", "3", "First Order Phase Correction Angle" },
{ 0x5600, 0x0020, "OF", "3", "Spectroscopy Data" },

{ 0x6000, 0x0000, "UL", "3", "Overlay Group Length" },
{ 0x6000, 0x0010, "US", "3", "Overlay Rows" },
{ 0x6000, 0x0011, "US", "3", "Overlay Columns" },
{ 0x6000, 0x0012, "US", "3RET", "Overlay Planes" },
{ 0x6000, 0x0015, "IS", "3", "Number of Frames in Overlay" },
{ 0x6000, 0x0022, "LO", "3", "Overlay Description" },
{ 0x6000, 0x0040, "CS", "3", "Overlay Type" },
{ 0x6000, 0x0045, "CS", "3", "Overlay Subtype" },
{ 0x6000, 0x0050, "SS", "3", "Overlay Origin" },
{ 0x6000, 0x0051, "US", "3", "Image Frame Origin" },
{ 0x6000, 0x0052, "US", "3RET", "Plane Origin" },
{ 0x6000, 0x0060, "SH", "2RET", "Overlay Compression Code" },
{ 0x6000, 0x0100, "US", "3", "Overlay Bits Allocated" },
{ 0x6000, 0x0102, "US", "3", "Overlay Bit Position" },
{ 0x6000, 0x0110, "SH", "2RET", "Overlay Format" },
{ 0x6000, 0x0200, "US", "2RET", "Overlay Location" },
{ 0x6000, 0x0800, "LO", "2CRET", "Overlay Code Label" },
{ 0x6000, 0x1100, "US", "3RET", "Overlay Descriptor - Gray" },
{ 0x6000, 0x1101, "US", "3RET", "Overlay Descriptor - Red" },
{ 0x6000, 0x1102, "US", "3RET", "Overlay Descriptor - Green" },
{ 0x6000, 0x1103, "US", "3RET", "Overlay Descriptor - Blue" },
{ 0x6000, 0x1200, "US", "3RET", "Overlays - Gray" },
{ 0x6000, 0x1201, "US", "3RET", "Overlays - Red" },
{ 0x6000, 0x1202, "US", "3RET", "Overlays - Green" },
{ 0x6000, 0x1203, "US", "3RET", "Overlays - Blue" },
{ 0x6000, 0x1301, "IS", "3", "ROI Area" },
{ 0x6000, 0x1302, "DS", "3", "ROI Mean" },
{ 0x6000, 0x1303, "DS", "3", "ROI Standard Deviation" },
{ 0x6000, 0x1500, "LO", "3", "Overlay Label" },
{ 0x6000, 0x3000, "OW", "3", "Overlay Data" }, // DICOM PS3.5 2017a
{ 0x6000, 0x4000, "SH", "2RET", "Overlay Group Comments" },

{ 0x7053, 0x0000, "UL", "PHI", "Philips Private Data Group Length" },
{ 0x7053, 0x0010, "LO", "PHI", "Philips Private Creator Data Element" },
{ 0x7053, 0x1000, "DS", "PHI", "Philips Private SUV Scale factor" },
{ 0x7053, 0x1001, "OB", "PHI", "Philips Private Element" },
{ 0x7053, 0x1002, "OB", "PHI", "Philips Private Element" },
{ 0x7053, 0x1003, "ST", "PHI", "Philips Private Original Image File Name" },
{ 0x7053, 0x1004, "OB", "PHI", "Philips Private File Data" },
{ 0x7053, 0x1005, "LO", "PHI", "Philips Private Filename" },
{ 0x7053, 0x1006, "OB", "PHI", "Philips Private Element" },
{ 0x7053, 0x1007, "SQ", "PHI", "Philips Private Acquisition File Sequence" },
{ 0x7053, 0x1009, "DS", "PHI", "Philips Private Activity Concentration Scale Factor" },
{ 0x7053, 0x100A, "LO", "PHI", "Philips Private Element" },
{ 0x7053, 0x100F, "UL", "PHI", "Philips Private Segment Size" },
{ 0x7053, 0x1010, "US", "PHI", "Philips Private Segment Number" },
{ 0x7053, 0x1011, "US", "PHI", "Philips Private Number of Segments" },
{ 0x7053, 0x1012, "SQ", "PHI", "Philips Private File Data Sequence" },
{ 0x7053, 0x1013, "SS", "PHI", "Philips Private Element" },
{ 0x7053, 0x1014, "SS", "PHI", "Philips Private Element" },
{ 0x7053, 0x1015, "SS", "PHI", "Philips Private Element" },
{ 0x7053, 0x1016, "SS", "PHI", "Philips Private Element" },
{ 0x7053, 0x1017, "SS", "PHI", "Philips Private Element" },
{ 0x7053, 0x1018, "SS", "PHI", "Philips Private Element" },
{ 0x7053, 0x1050, "SS", "PHI", "Philips Private Element" },
{ 0x7053, 0x1051, "SS", "PHI", "Philips Private Element" },
{ 0x7053, 0x1052, "SS", "PHI", "Philips Private Element" },
{ 0x7053, 0x1053, "SS", "PHI", "Philips Private Element" },
{ 0x7053, 0x1054, "SS", "PHI", "Philips Private Element" },
{ 0x7053, 0x1055, "SS", "PHI", "Philips Private Element" },
{ 0x7053, 0x1056, "SS", "PHI", "Philips Private Element" },
{ 0x7053, 0x1057, "SS", "PHI", "Philips Private Element" },
{ 0x7053, 0x1058, "SS", "PHI", "Philips Private Element" },
{ 0x7053, 0x1059, "SS", "PHI", "Philips Private Element" },
{ 0x7053, 0x105A, "SS", "PHI", "Philips Private Element" },
{ 0x7053, 0x105B, "SS", "PHI", "Philips Private Element" },
{ 0x7053, 0x105C, "SS", "PHI", "Philips Private Element" },
{ 0x7053, 0x105D, "SS", "PHI", "Philips Private Element" },
{ 0x7053, 0x105E, "SS", "PHI", "Philips Private Element" },
{ 0x7053, 0x105F, "LO", "PHI", "Philips Private Element" },
{ 0x7053, 0x1060, "SS", "PHI", "Philips Private Element" },
{ 0x7053, 0x1061, "SS", "PHI", "Philips Private Element" },
{ 0x7053, 0x1062, "SS", "PHI", "Philips Private Element" },
{ 0x7053, 0x1063, "SS", "PHI", "Philips Private Element" },
{ 0x7053, 0x1064, "SS", "PHI", "Philips Private Element" },
{ 0x7053, 0x1065, "SS", "PHI", "Philips Private Element" },
{ 0x7053, 0x1066, "SS", "PHI", "Philips Private Element" },
{ 0x7053, 0x1067, "LO", "PHI", "Philips Private Element" },
{ 0x7053, 0x1068, "LO", "PHI", "Philips Private Element" },
{ 0x7053, 0x1069, "LO", "PHI", "Philips Private Element" },
{ 0x7053, 0x106A, "LT", "PHI", "Philips Private Element" },
{ 0x7053, 0x106B, "SH", "PHI", "Philips Private Element" },
{ 0x7053, 0x106C, "LO", "PHI", "Philips Private Element" },
{ 0x7053, 0x106D, "FL", "PHI", "Philips Private Element" },
{ 0x7053, 0x106E, "FL", "PHI", "Philips Private Element" },
{ 0x7053, 0x106F, "FL", "PHI", "Philips Private Element" },
{ 0x7053, 0x1070, "SS", "PHI", "Philips Private Element" },
{ 0x7053, 0x1071, "SS", "PHI", "Philips Private Element" },
{ 0x7053, 0x1072, "SS", "PHI", "Philips Private Element" },
{ 0x7053, 0x1073, "SH", "PHI", "Philips Private Element" },
{ 0x7053, 0x1074, "SH", "PHI", "Philips Private Element" },
{ 0x7053, 0x1075, "SH", "PHI", "Philips Private Element" },
{ 0x7053, 0x1076, "SH", "PHI", "Philips Private Element" },
{ 0x7053, 0x1077, "FL", "PHI", "Philips Private Element" },
{ 0x7053, 0x1078, "FL", "PHI", "Philips Private Element" },
{ 0x7053, 0x10C2, "UI", "PHI", "Philips Private PET-CT Multi Modality Name" },
{ 0x7053, 0x10D0, "SS", "PHI", "Philips Private Element" },
{ 0x7053, 0x10D1, "FL", "PHI", "Philips Private Element" },
{ 0x7053, 0x10D2, "DT", "PHI", "Philips Private Element" },
{ 0x7053, 0x10D3, "DT", "PHI", "Philips Private Element" },
{ 0x7053, 0x10D5, "FL", "PHI", "Philips Private Element" },
{ 0x7053, 0x10D6, "FL", "PHI", "Philips Private Element" },
{ 0x7053, 0x10D7, "FL", "PHI", "Philips Private Element" },
{ 0x7053, 0x10D8, "FL", "PHI", "Philips Private Element" },
{ 0x7053, 0x10D9, "FL", "PHI", "Philips Private Element" },

{ 0x7F00, 0x0000, "UL", "2", "Variable Pixel Data Group Length" },
{ 0x7F00, 0x0010, "OW", "2RET", "Variable Pixel Data" }, //< OW or OB
{ 0x7F00, 0x0011, "AT", "2RET", "Variable Next Data Group" },
{ 0x7F00, 0x0020, "OW", "2RET", "Variable Coefficients SDVN" },
{ 0x7F00, 0x0030, "OW", "2RET", "Variable Coefficients SDHN" },
{ 0x7F00, 0x0040, "OW", "2RET", "Variable Coefficients SDDN" },

{ 0x7FE0, 0x0000, "UL", "3", "Pixel Data Group Length" },
{ 0x7FE0, 0x0001, "OV", "3", "Extended Offset Table" },
{ 0x7FE0, 0x0002, "OV", "3", "Extended Offset Table Lengths" },
{ 0x7FE0, 0x0008, "OF", "3", "Float Pixel Data" }, // DICOM PS3.5 2017a
{ 0x7FE0, 0x0009, "OD", "3", "Double Float Pixel Data" }, // DICOM PS3.5 2017a
{ 0x7FE0, 0x0010, "OW", "3", "Pixel Data" }, // DICOM PS3.5 2017a: OB or OW
{ 0x7FE0, 0x0020, "OW", "2CRET", "Coefficients SDVN" },
{ 0x7FE0, 0x0030, "OW", "2CRET", "Coefficients SDHN" },
{ 0x7FE0, 0x0040, "OW", "2CRET", "Coefficients SDDN" },

{ 0x7FE1, 0x0000, "OB", "SPI", "Binary Data" },
{ 0x7FE1, 0x0010, "OW", "SPI", "Pixel Data" },

{ 0xFFFA, 0xFFFA, "SQ", "3", "Digital Signatures Sequence" },

{ 0xFFFC, 0xFFFC, "OB", "3", "Dataset Trailing Padding" },

{ 0xFFFE, 0xE000, "DL", "3", "Item" }, // Actually VR does not exist
{ 0xFFFE, 0xE00D, "DL", "3", "Item Delimitation Item" }, // Actually VR does not exist
{ 0xFFFE, 0xE0DD, "DL", "3", "Sequence Delimitation Item" }, // Actually VR does not exist
{ 0xFFFF, 0xFFFF,    "",   "",  "Unknown"}
};
/*! @endcond */
/*****************************************************************************/

/*****************************************************************************/
/** Get the size of DICOM dictionary.
 *  @return Returns the size of the dictionary.
 */
unsigned int dcmDictSize()
{
  unsigned int n=1;
  while(dcmdict[n].group<0xFFFF && dcmdict[n].element<0xFFFF) n++;
  return(n);
}
/*****************************************************************************/

/*****************************************************************************/
/** Get the DICOM tag group and element, based on given dictionary index.
    @return Returns 0 when successful, 1 if invalid index.
    @sa dcmDictSize, dcmDictIndexVR, dcmDictIndexType,
        dcmDictIndexDescr, dcmDictFindTag
 */
int dcmDictIndexTag(
  /** Dictionary index [0..n-1]. */ 
  unsigned int i,
  /** Pointer to tag group; enter NULL if not needed. */
  unsigned short int *group,
  /** Pointer to tag element; enter NULL if not needed. */
  unsigned short int *element
) {
  if(group!=NULL) *group=0;
  if(element!=NULL) *element=0;
  if(i>=dcmDictSize()) return(1);
  if(group!=NULL) *group=dcmdict[i].group;
  if(element!=NULL) *element=dcmdict[i].element;
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/** Get the DICOM tag VR string, based on given dictionary index.
 *  @return Returns pointer to VR string, or NULL in case of an error.
 *  @sa dcmDictSize, dcmDictIndexTag, dcmDictFindTag
 */
char *dcmDictIndexVR(
  unsigned int i
) {
  if(i>=dcmDictSize()) return(char*)NULL;
  else return(dcmdict[i].vr);
}
/*****************************************************************************/

/*****************************************************************************/
/** Get the DICOM tag description, based on given dictionary index.
 *  @return Returns pointer to description string, which is 'Unknown' if
 *  index equals dictionary length, or NULL in case of an error.
 *  @sa dcmDictSize, dcmDictIndexTag, dcmDictFindTag
 */
char *dcmDictIndexDescr(
  /** Dictionary index [0..n-1]. */ 
  unsigned int i
) {
  if(i>dcmDictSize()) return(char*)NULL;
  else return(dcmdict[i].descr);
}
/*****************************************************************************/

/*****************************************************************************/
/** Get the DICOM Tag Type, based on given dictionary index.

    Tag Type is a NULL terminated string. 
    - 1: mandatory.
    - 1C: mandatory only under certain conditions.
    - 2: required to exist, but can have a null value.
    - 2C: required to exist only under certain conditions.
    - 3: optional.

    Reference: DICOM PS3.3

    @return Returns pointer to the tag type string, or NULL in case of an error.
    @sa dcmDictSize, dcmDictIndexTag, dcmDictFindTag
 */
char *dcmDictIndexType(
  /** Dictionary index [0..n-1]. */ 
  unsigned int i
) {
  if(i>=dcmDictSize()) return(char*)NULL;
  else return(dcmdict[i].type);
}
/*****************************************************************************/

/*****************************************************************************/
/** Get the dictionary index for given tag.
    @return If tag is found, returns its dictionary index; otherwise returns
     the length of the dictionary.
    @sa dcmDictSize, dcmDictIndexTag, dcmDictIndexVR, dcmDictIndexType,
        dcmDictIndexDescr
 */
unsigned int dcmDictFindTag(
  /** Pointer to DICOM tag. */
  DCMTAG *tag
) {
  unsigned int i, n=dcmDictSize();
  if(tag==NULL) return(n);
  for(i=0; i<n; i++)
    if(tag->group==dcmdict[i].group && tag->element==dcmdict[i].element) break;
  return(i);
}
/*****************************************************************************/

/*****************************************************************************/
