/** @file tpcdcm.h
    @brief Header file for libtpcdcm.
    @details Header file for library libtpcdcm.
    @author Vesa Oikonen
    @copyright (c) Turku PET Centre
 */
#ifndef _TPCDCM_H_
#define _TPCDCM_H_
/*****************************************************************************/

/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
/*****************************************************************************/
#include "tpcextensions.h"
#include "tpcift.h"
#include "tpcfileutil.h"
#include "tpcisotope.h"
/*****************************************************************************/

/*****************************************************************************/
/** DICOM tag. Tag group and element are shown in hex format. */
typedef struct DCMTAG {
  /** Group: even numbers defined in DICOM standard, odd numbers are
      vendor specific.

    Even numbers:
    - 0x0002: Data elements. Always Little Endian Explicit VR.
    - 0x0008: Identification information. 
    - 0x0010: Patient information. 
    - 0x0018: Acquisition information. 
    - 0x0020: Relationship information. 
    - 0x0028: Image presentation information. 
    - 0x4000: Text information. 
    - 0x6000-0x601E: Overlay information. 
    - 0x7FE0: Image pixel data. 
  */
  unsigned short int group;
  /** Element */
  unsigned short int element;
} DCMTAG;
/*****************************************************************************/

/*****************************************************************************/
/** @public DICOM Transfer Syntax UID. 

    In case of implicit VR, elements do NOT contain VR !
    Reference: DICOM PS3.5 2017a chapter 10.

    Items must be the same and in the same order as the dcm_truid list
    in dcmuid.c.
*/
typedef enum {
  DCM_TRUID_UNKNOWN, ///< Unknown Transfer Syntax UID  
  DCM_TRUID_LEI,     ///< Little Endian Implicit VR (DICOM default)
  DCM_TRUID_LEE,     ///< Little Endian Explicit VR
  DCM_TRUID_LEED,    ///< Little Endian Explicit Deflated VR
  DCM_TRUID_BEE,     ///< Big Endian Explicit VR (retired)
  DCM_TRUID_JPEG50,  ///< Lossy JPEG 8-bit compression
  DCM_TRUID_JPEG51,  ///< Lossy JPEG 12-bit compression
  DCM_TRUID_JPEG70,  ///< Lossless JPEG
  DCM_TRUID_JPEG80,  ///< Lossless JPEG-LS
  DCM_TRUID_JPEG81,  ///< Lossy JPEG-LS
  DCM_TRUID_JPEG90,  ///< Lossless JPEG 2000
  DCM_TRUID_JPEG91,  ///< JPEG 2000
  DCM_TRUID_JPEG92,  ///< Lossless multicomponent JPEG 2000
  DCM_TRUID_JPEG93,  ///< Multicomponent JPEG 2000
  DCM_TRUID_MPEG100, ///< MPEG-2
  DCM_TRUID_MPEG102, ///< MPEG-4
  DCM_TRUID_MPEG103, ///< MPEG-4 BD-compatible
  DCM_TRUID_RLE,     ///< Lossless RLE
  DCM_TRUID_RFC,     ///< RFC 2557
  DCM_TRUID_XML,     ///< XML encoding
  DCM_TRUID_INVALID  ///< Invalid Transfer Syntax UID
} dcmtruid;
/*****************************************************************************/

/*****************************************************************************/
/** @public DICOM value representation (VR). 

    Reference: DICOM PS3.5 2017a chapter 6.2.
 
    Items must be the same and in the same order as the dcm_vr list
    in dcmvr.c.

*/
typedef enum {
  DCM_VR_AE,        ///< DICOM application entity, max 16 bytes.
  DCM_VR_AS,        ///< DICOM age string, 4 bytes fixed.
  DCM_VR_AT,        ///< DICOM attribute tag, 4 bytes fixed.
  DCM_VR_CS,        ///< DICOM code (control) string, max 16 bytes.
  DCM_VR_DA,        ///< DICOM date in format YYYYMMDD, 8 bytes fixed. @note In old standard 10 bytes fixed, in format YYYY.MM.DD
  DCM_VR_DS,        ///< DICOM decimal string, max 16 bytes.
  DCM_VR_DT,        ///< DICOM date time, max 26 bytes.
  DCM_VR_FL,        ///< DICOM floating point single precision, 4 bytes fixed.
  DCM_VR_FD,        ///< DICOM floating point double precision, 8 bytes fixed.
  DCM_VR_IS,        ///< DICOM integer string, max 12 bytes.
  DCM_VR_LO,        ///< DICOM long string, max 64 chars.
  DCM_VR_LT,        ///< DICOM long text, max 10240 chars.
  DCM_VR_OB,        ///< DICOM other byte string, even bytes, endian insensitive.
  DCM_VR_OD,        ///< DICOM other double (64-bit) stream, endian sensitive.
  DCM_VR_OF,        ///< DICOM other float (32-bit) stream, endian sensitive.
  DCM_VR_OL,        ///< DICOM other long (32-bit) stream, endian sensitive.
  DCM_VR_OW,        ///< DICOM other word (16-bit) stream, even bytes, endian sensitive.
  DCM_VR_PN,        ///< DICOM person name, max 64 chars per component group.
  DCM_VR_SH,        ///< DICOM short string, max 16 chars.
  DCM_VR_SL,        ///< DICOM signed long (32-bit integer), 4 bytes fixed.
  DCM_VR_SQ,        ///< DICOM sequence of zero or more elements (used for nested data).
  DCM_VR_SS,        ///< DICOM signed short (16-bit integer), 2 bytes fixed.
  DCM_VR_ST,        ///< DICOM short text, max 1024 chars.
  DCM_VR_TM,        ///< DICOM time HHMMSS.FFFFFF, max 14 bytes. @note In old standard 16 bytes max.
  DCM_VR_UC,        ///< DICOM unlimited characters.
  DCM_VR_UI,        ///< DICOM unique identifier (UID), max 64 bytes.
  DCM_VR_UL,        ///< DICOM unsigned long (32-bit) integer, 4 bytes fixed.
  DCM_VR_UN,        ///< DICOM unknown, any valid length of another VR.
  DCM_VR_UR,        ///< DICOM URI or URL, string of characters.
  DCM_VR_US,        ///< DICOM unsigned short (16-bit) integer, 2 bytes fixed.
  DCM_VR_UT,        ///< DICOM unlimited text, character string.
  DCM_VR_INVALID    ///< Invalid DICOM value representation.
} dcmvr;
/*****************************************************************************/

/*****************************************************************************/
/** Data struct for one DICOM item; may be recursive. */
typedef struct DCMITEM {
  /** Nr of File pointer, NULL if not opened. */
  FILE *fp;
  /** File position of the start of this element; 0, if not set. */
  fpos_t pos;
  /** Enumerated Transfer Syntax UID. */
  dcmtruid truid;
  /** Item tag. */
  DCMTAG tag;
  /** Enumerated Value Representation. */
  dcmvr vr;
  /** Value Length. Larger data can only be stored in multiple items (fragments). */
  unsigned int vl;
  /** Pointer to linked list of child elements; NULL if none. */
  struct DCMITEM *child_item;
  /** Pointer to linked list of parent elements; NULL if none. */
  struct DCMITEM *parent_item;
  /** Pointer to next item ; NULL if none. */
  struct DCMITEM *next_item;
  /** Pointer to previous item ; NULL if none. */
  struct DCMITEM *prev_item;
  /** Pointer to raw data value (no byte conversions etc); 
      NULL if not available. */
  char *rd;
} DCMITEM;

/** Main data struct for one DICOM file */
typedef struct DCMFILE {
  /** DICOM filename. */
  char filename[FILENAME_MAX];
  /** File pointer, NULL if not opened. */
  FILE *fp;
  /** Enumerated Transfer Syntax UID. */
  dcmtruid truid;
  /** Pointer to linked list of DICOM elements; recursive; NULL if none. */
  DCMITEM *item;
} DCMFILE;
/*****************************************************************************/

/*****************************************************************************/
/** Data struct for one DICOM image matrix.
   @sa DCMML
 */
typedef struct DCMMATRIX {
  /** File name; must be allocated and freed as necessary;
      note that one file may contain many matrices. */
  char *filename;
  /** Acquisition date. */ 
  char acqDate[16];
  /** Acquisition time. */ 
  char acqTime[16];
  /** Frame [1..frameNr]; note that one frame may contain several bed positions,
      each with their own frame times, but those will be on different planes. */
  unsigned int frame;
  /** Plane [1..planeNr]. */
  unsigned int plane;
  /** Frame start time (sec). */
  double frameStart;
  /** Frame duration (sec). */
  double frameDur;
} DCMMATRIX;

/** Main data struct for all DICOM matrices. */
typedef struct DCMML {
  /** Nr of matrices. */
  unsigned int nr;
  /** Allocate size of matrix list. */
  unsigned int anr;
  /** Pointer to matrix list. */
  DCMMATRIX *m;
} DCMML;
/*****************************************************************************/

/*****************************************************************************/
/* dcmdata */
void dcmfileInit(DCMFILE *d);
void dcmitemFree(DCMITEM *d);
void dcmfileFree(DCMFILE *d);
unsigned short int dcmfileMaxDepth(DCMFILE *df);
unsigned short int dcmitemMaxDepth(DCMITEM *d);
unsigned short int dcmitemParentNr(DCMITEM *d);
char *dcmValueString(DCMITEM *d);
long int dcmitemGetInt(DCMITEM *d);
double dcmitemGetReal(DCMITEM *d);
DCMITEM *dcmFindTag(DCMITEM *d, const short int omit, DCMTAG *tag, const int verbose);
void dcmitemPrint(DCMITEM *d);
void dcmTagSet(DCMTAG *tag, unsigned short int group, unsigned short int element);
int dcmAddItem(
  DCMFILE *dcm, DCMITEM *d, short int aschild, DCMTAG tag, dcmvr vr, unsigned int vl,
  char *rd, const int verbose
);

DCMITEM *dcmFindDownTag(DCMITEM *d, const short int omit, DCMTAG *tag, const int verbose);
int dcmTagIntRange(DCMITEM *d, DCMTAG *tag, int *mi, int *ma, const int verbose);

/* dcmuid */
dcmtruid dcmTrUID(const char *s);
char *dcmTrUIDDescr(dcmtruid id);
char *dcmTrUIDString(dcmtruid id);

/* dcmsop */
unsigned int dcmSOPIdentify(const char *s);
char *dcmSOPName(unsigned int i);
char *dcmSOPUID(unsigned int i);
char *dcmSOPUIDName(const char *s);

/* dcmvr */
unsigned char dcmVRReserved(dcmvr id);
dcmvr dcmVRId(const char *s);
char *dcmVRName(dcmvr id);
size_t dcmVRVLength(dcmvr id);
char *dcmVRDescr(dcmvr id);
char *dcmDA2intl(const char *orig, char *intl);
char *dcmTM2intl(const char *orig, char *intl);
char *dcmDT2intl(const char *orig, char *intl);

/* dcmdictionary */
unsigned int dcmDictSize();
int dcmDictIndexTag(unsigned int i, unsigned short int *group, unsigned short int *element);
char *dcmDictIndexVR(unsigned int i);
char *dcmDictIndexType(unsigned int i);
char *dcmDictIndexDescr(unsigned int i);
unsigned int dcmDictFindTag(DCMTAG *tag);

/* dcmfile */
int dcmSameImage(const DCMFILE *d1, const DCMFILE *d2, const int verbose);
int dcmFileList(const char *filename, IFT *ift, TPCSTATUS *status);
int dcmFileRemove(const char *filename, TPCSTATUS *status);

/* dcmio */
int dcmVerifyMagic(const char *filename, FILE *fp);
dcmtruid dcmReadTransferSyntaxUID(FILE *fp);
int dcmReadFileTag(FILE *fp, DCMTAG *tag);
int dcmWriteFileTag(FILE *fp, DCMTAG *tag);
int dcmWriteFileSQDelimItem(FILE *fp);
int dcmWriteFileSQItemDelimTag(FILE *fp);
dcmvr dcmReadFileVR(FILE *fp, char *vrstr);
unsigned int dcmReadFileVL(FILE *fp, unsigned int n);
int dcmReadFileVRVL(FILE *fp, dcmvr *vr, unsigned int *vl, unsigned int *n);
int dcmWriteFileVRVL(FILE *fp, dcmvr vr, unsigned int vl, unsigned int *n);
int dcmFileReadNextElement(
  DCMFILE *dcm, DCMITEM *prev_item, DCMITEM *parent_item, const short int sub,
  const short int headerOnly, /*const*/ int verbose
);
int dcmFileRead(
  const char *filename, DCMFILE *dcm, const short int headerOnly,
  TPCSTATUS *status
);
int dcmFileWrite(const char *filename, DCMFILE *dcm, TPCSTATUS *status);

/* dcmimage */
int dcmImgIsotope(DCMFILE *d, isotope *isot, decaycorrection *dc, const int verbose);
int dcmImgPos(DCMFILE *d, double *imgpos, const int verbose);
int dcmImgDim(DCMFILE *d, unsigned short int *imgdim, const int verbose);
int dcmImgPxlsize(DCMFILE *d, double *pxlsize, const int verbose);
int dcmImgOrient(DCMFILE *d, double *iop, const int verbose);
int dcmImgXform(double *iop, double *xyzMM, double *imgPos, double *xform, const int verbose);
int dcmXformToQuatern(double *xform, double *quatern, double *qoffset, const int verbose);

/* dcmmatrix */
void dcmmatrixInit(DCMMATRIX *m);
void dcmmatrixFree(DCMMATRIX *m);
void dcmmlInit(DCMML *d);
void dcmmlFree(DCMML *d);
int dcmmlAllocate(DCMML *d, int mNr);
int dcmMListRead(IFT *ift, DCMML *ml, TPCSTATUS *status);
int dcmmlSortByPlane(DCMML *d, TPCSTATUS *status);
/*****************************************************************************/

/*****************************************************************************/
#endif /* TPCDCM */
