/** @file doubleutil.c
 *  @brief Working with doubles.
 *  @author Vesa Oikonen
 */
/*****************************************************************************/

/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <time.h>
#include <string.h>
#include <ctype.h>
#include <unistd.h>
#include <math.h>
#include "tpcextensions.h"
/*****************************************************************************/

/*****************************************************************************/
/** Verifies that given two doubles have the same value inside given limits.
    Values are considered to match also if both are NaNs.
    @return 1 if match is found, and 0 if not.
    @author Vesa Oikonen
    @sa floatMatch, doubleMatchRel, doubleMachEps, atofCheck, statMeanSD
 */
int doubleMatch(
  /** First value. */
  const double v1,
  /** Second value. */
  const double v2,
  /** Limit for absolute difference (if <0 then test will fail every time). */
  const double lim
) {
  if(isnan(v1) && isnan(v2)) return 1;
  if(isnan(v1) || isnan(v2)) return 0;
  if(v1==v2) return 1;
  if(isnan(lim) || lim<0.0) return 0;
  if(fabs(v1-v2)<=lim) return 1;
  return 0;
}
/*****************************************************************************/

/*****************************************************************************/
/** Verifies that given two double arrays have the same values inside given limits.
    Values are considered to match also if both are NaNs.
    @return 1 if match is found, and 0 if not.
    @author Vesa Oikonen
    @sa doubleMatch, doubleMatchRel, doubleMachEps, statMeanSD
 */
int doubleArrayMatch(
  /** First array. */
  const double *a1,
  /** Second array. */
  const double *a2,
  /** Length of arrays. */
  const unsigned int n,
  /** Limit for absolute difference (if <0 then test will fail every time). */
  const double lim
) {
  if(a1==NULL || a2==NULL || n<1) return 0;
  for(unsigned int i=0; i<n; i++) if(!doubleMatch(a1[i], a2[i], lim)) return 0;
  return 1;
}
/*****************************************************************************/

/*****************************************************************************/
/** Verifies that given two doubles have the same value inside given relative
    limit |2*(v1-v2)/(v1+v2)|.

    Values are considered to match also if both are NaNs, but not if mean
    is zero (unless both are exactly zero).
    @return 1 if match is found, and 0 if not.
    @author Vesa Oikonen
    @sa floatMatchRel, doubleMatch, doubleMachEps
 */
int doubleMatchRel(
  /** First value. */
  const double v1,
  /** Second value. */
  const double v2,
  /** Limit for relative difference (if <0 then test will fail every time). */
  const double lim
) {
  if(isnan(v1) && isnan(v2)) return 1;
  if(isnan(v1) || isnan(v2)) return 0;
  if(v1==v2) return 1;
  if(isnan(lim)) return 0;
  double mean;
  mean=0.5*(v1+v2); if(!isnormal(mean)) return 0;
  if(fabs((v1-v2)/mean)<=lim) return 1;
  return 0;
}
/*****************************************************************************/

/*****************************************************************************/
/** Estimates the machine epsilon, the upper bound on the relative error due to rounding in 
    floating point arithmetic, within one order of magnitude of the true machine epsilon.

    Standard C library should also have DBL_EPSILON in float.h.
    @return Estimate of machine epsilon.
    @author Vesa Oikonen
    @sa floatMachEps, doubleMatchRel, doubleMatch, factorial
 */
double doubleMachEps()
{
  double macheps=1.0;
  do {macheps/=2.0;} while((1.0+macheps/2.0)!=1.0);
  return(macheps);
}
/*****************************************************************************/

/*****************************************************************************/
/** Copy double values from the 2nd array to the first.
    @sa floatCopy, doubleCopyFinite
 */
void doubleCopy(
  /** Target array. */
  double *t,
  /** Source array. */
  double *s,
  /** Length of arrays. */
  const unsigned int n
) {
  unsigned int i;
  if(t==NULL || s==NULL || n<1) return;
  for(i=0; i<n; i++) t[i]=s[i];
}
/*****************************************************************************/

/*****************************************************************************/
/** Copy finite double values from the 2nd array to the first.
    @sa doubleCopy
    @return The number of finite values that were copied.
 */
unsigned int doubleCopyFinite(
  /** Target array. */
  double *t,
  /** Source array. */
  double *s,
  /** Length of arrays. */
  const unsigned int n
) {
  unsigned int i, tn=0;
  if(t==NULL || s==NULL || n<1) return(0);
  for(i=0; i<n; i++) if(isfinite(s[i])) t[tn++]=s[i];
  return(tn);
}
/*****************************************************************************/

/*****************************************************************************/
/** Check the number of NaNs in a double array.
    @sa doubleNonzeroes, doubleCopyFinite, doubleRange, doubleMean
    @return The number of NaNs.
 */
unsigned int doubleNaNs(
  /** Pointer to double array. */
  double *a,
  /** Length of array. */
  const unsigned int n
) {
  if(a==NULL || n<1) return(0);
  unsigned int i, nn=0;
  for(i=0; i<n; i++) if(isnan(a[i])) nn++;
  return(nn);
}
/*****************************************************************************/

/*****************************************************************************/
/** Get the range of finite values in a double array.
    @sa doubleMaxIndex, doubleMinIndex, doubleNonzeroes, doubleNaNs, doubleMean, statMedian
    @return The number of finite values in the array, or 0 in case of an error.
 */
unsigned int doubleRange(
  /** Pointer to double array. */
  double *a,
  /** Length of array. */
  const unsigned int n,
  /** Pointer to the double float where minimum value is stored; enter NULL, if not needed. */
  double *amin,
  /** Pointer to the double float where maximum value is stored; enter NULL, if not needed. */
  double *amax
) {
  if(amin!=NULL) *amin=nan("");
  if(amax!=NULL) *amax=nan("");
  if(a==NULL || n<1) return(0);

  double mi=nan(""), ma=nan("");
  unsigned int i, nn=0;
  for(i=0; i<n; i++) if(isfinite(a[i])) {
    nn++;
    if(!isfinite(mi) || mi>a[i]) mi=a[i];
    if(!isfinite(ma) || ma<a[i]) ma=a[i];
  }
  if(amin!=NULL) *amin=mi;
  if(amax!=NULL) *amax=ma;
  return(nn);
}
/*****************************************************************************/

/*****************************************************************************/
/** Calculate the sum of values in given double array.
    @return The sum of array values.
    @sa floatSum, doubleMean, doubleMaxIndex
 */
double doubleSum(
  /** Pointer to double array. */
  double *a,
  /** Length of array. */
  const unsigned int n
) {
  double s=0.0;
  if(a==NULL || n<1) return(s);
  for(unsigned int i=0; i<n; i++) if(!isnan(a[i])) s+=a[i];
  return(s);
}
/*****************************************************************************/

/*****************************************************************************/
/** Calculate the mean of values in given double array.
    @return The mean of array values, or NaN in case of an error.
    @sa floatMean, doubleWMean, doubleSum, doubleMaxIndex, statMeanSD, statMedian
 */
double doubleMean(
  /** Pointer to double array. */
  double *a,
  /** Length of array. */
  const unsigned int n
) {
  if(a==NULL || n<1) return(nan(""));
  double s=0.0;
  unsigned int i, ci=0;
  for(i=0; i<n; i++) if(!isnan(a[i])) {ci++; s+=a[i];}
  if(ci<1) return(nan(""));
  return(s/(double)ci);
}
/*****************************************************************************/

/*****************************************************************************/
/** Calculate the weighted mean of values in given double array.
    @return The mean of array values, or NaN in case of an error.
    @sa doubleMean, floatMean, doubleSum, doubleMaxIndex, statMeanSD
 */
double doubleWMean(
  /** Pointer to double array to calculated weighted mean from. */
  double *a,
  /** Pointer to double array containing the weights.
      Negative or NaN weight is interpreted as zero weight. */
  double *w,
  /** Length of arrays. */
  const unsigned int n
) {
  if(a==NULL || w==NULL || n<1) return(nan(""));
  double s=0.0, ws=0.0;
  unsigned int i, ci=0;
  for(i=0; i<n; i++) if(isnormal(a[i]) && isnormal(w[i]) && w[i]>0.0) {
    ci++; s+=w[i]*a[i]; ws+=w[i];}
  if(ci<1 || !(ws>0.0)) return(nan(""));
  return(s/ws);
}
/*****************************************************************************/

/*****************************************************************************/
/** Reads numerical value and its unit from argument string.

    Both decimal point and comma are accepted.
    Optional result double value is set to NaN if string was not valid value.
    Optional unit is set to UNIT_UNKNOWN, if not valid or not found.
    @return 0 if numerical value successfully read, and 1 in case of an error.
    @sa floatGetWithUnit, atofVerified, atofCheck
 */
int doubleGetWithUnit(
  /** String from where double and unit is read; string must not contain any
      extra space characters. */
  const char *s,
  /** Pointer to the double float; enter NULL, if not needed. */
  double *v,
  /** Pointer to int for unit code; enter NULL, if not needed. */
  int *u
) {
  if(v!=NULL) *v=nan(""); 
  if(u!=NULL) *u=UNIT_UNKNOWN; 
  if(s==NULL) return 1;
  int n=strTokenNr(s, " \t"); if(n==0 || n>2) return 1;
  if(n==1) return(atofCheck(s, v)); // no unit
  /* separate number and unit */
  char buf[64];
  n=strTokenNCpy(s, " \t", 1, buf, 64);
  if(atofCheck(buf, v)) return 1;
  if(u==NULL) return 0;
  n=strTokenNCpy(s, " \t", 2, buf, 64);
  *u=unitIdentify(buf);
  return 0;
}
/*****************************************************************************/

/*****************************************************************************/
/** Returns the length of array consisting of only positive (>0 and not NaN) values.
   @return Returns the index of first non-positive value in the given array.
   @sa floatSpanPositives, doubleCSpanPositives, atofList, doubleNonzeroes
 */
int doubleSpanPositives(
  /** Pointer to the array. */
  double *a,
  /** Length of the array. */
  const int n
) {
  if(a==NULL || n<1) return(0);
  int i=0;
  for(i=0; i<n; i++) if(!(a[i]>0.0)) break;
  return(i);
}
/*****************************************************************************/

/*****************************************************************************/
/** Returns the length of array consisting of non-positive (<=0 and NaN) values.
   @return Returns the index of first positive value in the given array.
   @sa floatCSpanPositives, doubleSpanPositives, atofList
 */
int doubleCSpanPositives(
  /** Pointer to the array. */
  double *a,
  /** Length of the array. */
  const int n
) {
  if(a==NULL || n<1) return(0);
  int i=0;
  for(i=0; i<n; i++) if(a[i]>0.0) break;
  return(i);
}
/*****************************************************************************/

/*****************************************************************************/
/** Find the number of non-zero values in given double array.
    Negative values are counted, but NaNs are not.
    @return The number of non-zero values in array; 0 is returned also in case of errors.
    @sa floatNonzeroes, doubleSpanPositives, doubleNaNs
 */
unsigned int doubleNonzeroes(
  /** Pointer to double array. */
  double *a,
  /** Length of array. */
  const unsigned int n
) {
  if(a==NULL || n<1) return(0);
  unsigned int i, m=0;
  for(i=0; i<n; i++) if(fabs(a[i])>0.0) m++;
  return(m);
}
/*****************************************************************************/

/*****************************************************************************/
/** Find the maximum value in given double array.
    @return The index [0..n-1] of maximum value in array; 0 is returned also in case of errors.
    @sa floatMaxIndex, doubleSpanPositives, doubleCSpanPositives, doubleGTIndex, doubleRange
 */
unsigned int doubleMaxIndex(
  /** Pointer to double array. */
  double *a,
  /** Length of array. */
  const unsigned int n
) {
  if(a==NULL || n<1) return(0);
  unsigned int i, mi=0;
  double mv=nan("");
  for(i=0; i<n; i++) if(isnan(mv) || a[i]>mv) {mv=a[i]; mi=i;}
  return(mi);
}
/*****************************************************************************/

/*****************************************************************************/
/** Find the maximum absolute value in given double array.
    @return The index [0..n-1] of maximum absolute value in array; 
     0 is returned also in case of errors.
    @sa doubleMaxIndex, doubleGTIndex, doubleGEIndex, doubleRange
 */
unsigned int doubleAbsMaxIndex(
  /** Pointer to double array. */
  double *a,
  /** Length of array. */
  const unsigned int n
) {
  if(a==NULL || n<1) return(0);
  unsigned int i, mi=0;
  double mv=nan("");
  for(i=0; i<n; i++) if(isnan(mv) || fabs(a[i])>mv) {mv=fabs(a[i]); mi=i;}
  return(mi);
}
/*****************************************************************************/

/*****************************************************************************/
/** Find the minimum value in given double array.
    @return The index [0..n-1] of minimum value in array; 0 is returned also in case of errors.
    @sa doubleMaxIndex, doubleAbsMinIndex, doubleRange
 */
unsigned int doubleMinIndex(
  /** Pointer to double array. */
  double *a,
  /** Length of array. */
  const unsigned int n
) {
  if(a==NULL || n<1) return(0);
  unsigned int i, mi=0;
  double mv=nan("");
  for(i=0; i<n; i++) if(isnan(mv) || a[i]<mv) {mv=a[i]; mi=i;}
  return(mi);
}
/*****************************************************************************/

/*****************************************************************************/
/** Find the minimum absolute value in given double array.
    @return The index [0..n-1] of minimum absolute value in array; 
     0 is returned also in case of errors.
    @sa doubleAbsMaxIndex, doubleMinIndex, doubleGTIndex, doubleGEIndex
 */
unsigned int doubleAbsMinIndex(
  /** Pointer to double array. */
  double *a,
  /** Length of array. */
  const unsigned int n
) {
  if(a==NULL || n<1) return(0);
  unsigned int i, mi=0;
  double mv=nan("");
  for(i=0; i<n; i++) if(isnan(mv) || fabs(a[i])<mv) {mv=fabs(a[i]); mi=i;}
  return(mi);
}
/*****************************************************************************/

/*****************************************************************************/
/** Get the index of first array value greater or equal than given limit.
    @return The index, or n, if not found.
    @sa doubleGTIndex, doubleMaxIndex, doubleMinIndex, fitLine, statMeanSD
 */
unsigned int doubleGEIndex(
  /** Pointer to an array of values. */
  double *a,
  /** The number of samples (length of a[]). */
  const unsigned int n,
  /** The limit value. */
  double lim
) {
  if(a==NULL || n<1) return(n);
  for(unsigned int i=0; i<n; i++) if(a[i]>=lim) return(i);
  return(n);
}
/*****************************************************************************/
/** Get the index of first array value greater than given limit.
    @return The index, or n, if not found.
    @sa doubleGEIndex, doubleMaxIndex, doubleMinIndex, fitLine, statMeanSD, doubleMatch
 */
unsigned int doubleGTIndex(
  /** Pointer to an array of values. */
  double *a,
  /** The number of samples (length of a[]). */
  const unsigned int n,
  /** The limit value. */
  double lim
) {
  if(a==NULL || n<1) return(n);
  for(unsigned int i=0; i<n; i++) if(a[i]>lim) return(i);
  return(n);
}
/*****************************************************************************/

/*****************************************************************************/
/** Inverse complementary error function erfc^{-1}(x).

    Based on the code by Acklam PJ, 2010.
    @returns Returns an approximate value of erfc^{-1}(x).
    @sa doubleMachEps, igam, igamc
 */
double inverfc(
  /** Parameter for the inverse complementary error function. */
  double x
) {
  static const double a[] = {
    -3.969683028665376E+01, 2.209460984245205E+02, -2.759285104469687E+02,
    1.383577518672690E+02, -3.066479806614716E+01,  2.506628277459239
  };
  static const double b[] = {
    -5.447609879822406E+01, 1.615858368580409E+02, -1.556989798598866E+02, 
    6.680131188771972E+01, -1.328068155288572E+01
  };
  static const double c[] = {
    -7.784894002430293E-03, -3.223964580411365E-01, -2.400758277161838, 
    -2.549732539343734, 4.374664141464968, 2.938163982698783
  };
  static const double d[] = {
    7.784695709041462E-03, 3.224671290700398E-01, 2.445134137142996, 3.754408661907416
  };
  double y, e, u;
  x/=2.0;
  if(x<0.02425) {
    double q=sqrt(-2.0 * log(x));
    y = (((((c[0]*q + c[1])*q + c[2])*q + c[3])*q + c[4])*q + c[5])
         / ((((d[0]*q + d[1])*q + d[2])*q + d[3])*q + 1.0);
  } else if(x<=0.97575) {
    double q=x-0.5;
    double r=q*q;
    y = (((((a[0]*r + a[1])*r + a[2])*r + a[3])*r + a[4])*r + a[5])*q
         / (((((b[0]*r + b[1])*r + b[2])*r + b[3])*r + b[4])*r + 1.0);
  } else {
    double q=sqrt(-2.0 * log(1.0-x));
    y = -(((((c[0]*q + c[1])*q + c[2])*q + c[3])*q + c[4])*q + c[5])
          / ((((d[0]*q + d[1])*q + d[2])*q + d[3])*q + 1.0);
  }
  e=0.5*erfc(-y/M_SQRT2)-x;
  u=(e)*M_SQRT2*M_PI*exp(0.5*y*y);
  y-=u/(1.0 + 0.5*y*u);
  return(fabs(y/M_SQRT2));
}
/*****************************************************************************/

/*****************************************************************************/
