/** @file intutil.c
 *  @brief Working with integer values.
 *  @author Vesa Oikonen
 */
/*****************************************************************************/

/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <math.h>
#include "tpcextensions.h"
/*****************************************************************************/

/*****************************************************************************/
/** Verifies that argument string at least seems like a valid integer. 
    '\0' must follow the integer part in the string.
    Optional result integer value is set to 0 if string was not valid value.
    @sa atofCheck, intlistAddFromString, strIsValidNumber
    @return 0 if successful, and 1 in case of an error.
 */
int atoiCheck(
  /** String which is converted to an integer; string must not contain any space characters. */
  const char *s,
  /** Pointer to the integer; enter NULL, if not needed. */
  int *v
) {
  if(v!=NULL) *v=0; 
  if(s==NULL || strnlen(s, 2)<1) return(1);
  errno=0; char *tail;
  *v=strtol(s, &tail, 10);
  if(errno) {*v=0; return(2);}
  if(*tail) {*v=0; return(3);}
  return 0;
}
/*****************************************************************************/

/*****************************************************************************/
/** Calculate factorial of given number.
    @sa lfactorial, igam, igamc
    @return Returns factorial n!, or zero if factorial would cause wrap-around.
 */
unsigned int factorial(
  /** Integer n, from which the factorial is calculated.
      @note Wrap-around will occur if n>12.
   */
  unsigned int n
) {
  if(n<1) return(1);
  if(n>12) return(0);
  return(n*factorial(n-1));
}
/*****************************************************************************/

/*****************************************************************************/
/** Calculate factorial of given number.
    @sa factorial, igam, igamc
    @return Returns factorial n!, or zero if factorial would cause wrap-around.
 */
unsigned long long int lfactorial(
  /** Integer n, from which the factorial is calculated.
      @note Wrap-around will occur if n>20.
   */
  unsigned long long int n
) {
  if(n<1) return(1);
  if(n>20) return(0);
  return(n*lfactorial(n-1));
}
/*****************************************************************************/

/*****************************************************************************/
/** Initiate the INTLIST structure before any use.
    @sa intlistFree, intlistAdd, intlistSort
    @author Vesa Oikonen
 */
void intlistInit(
  /** Pointer to INTLIST. */
  INTLIST *l
) {
  if(l==NULL) return;
  l->nr=0; l->_nr=0;
  l->i=NULL;
}
/** Free memory allocated for INTLIST. All data is cleared. 
    @sa intlistInit
    @author Vesa Oikonen
 */
void intlistFree(
  /** Pointer to INTLIST structure */
  INTLIST *l
) {
  if(l==NULL) return;
  free(l->i);
  intlistInit(l);
}
/*****************************************************************************/

/*****************************************************************************/
/** Add one integer to the integer array in INTLIST structure.
    @sa intlistInit, intlistFree, intlistSort, intlistAddFromString, strncatInt
    @author Vesa Oikonen
    @return Returns the nr of added integers (either 0 or 1), or <0 in case of an error.
 */
int intlistAdd(
  /** Pointer to initiated or filled INTLIST structure. */
  INTLIST *l,
  /** Integer value to add. */
  const int v,
  /** Add integer to the list only if it is new (0=no, 1=yes). */
  const int ifnew
) {
  if(l==NULL) return(-1);
  /* If only new value is to be added, then check that this is new */
  if(ifnew!=0) {for(int i=0; i<l->nr; i++) if(l->i[i]==v) return(0);}
  /* Add value to list if there is space left, and quit */
  if(l->_nr>l->nr) {l->i[l->nr++]=v; return(1);}
  /* Allocate more space */
  if(l->_nr==0) l->i=(int*)malloc(10*sizeof(int));
  else l->i=(int*)realloc(l->i, (10+l->_nr)*sizeof(int));
  if(l->i==NULL) {l->nr=l->_nr=0; return(-2);}
  l->_nr+=10;
  /* Add value to list */
  l->i[l->nr++]=v;
  return(1);
}
/*****************************************************************************/

/*****************************************************************************/
/** Sort the integer array in INTLIST structure. 
    @sa intlistInit, intlistFree, intlistAdd
    @author Vesa Oikonen
 */
void intlistSort(
  /** Pointer to INTLIST structure. */
  INTLIST *l
) {
  if(l==NULL || l->i==NULL || l->nr<2) return;
  int i, j, v;
  for(i=0; i<l->nr; i++) for(j=i+1; j<l->nr; j++) {
    if(l->i[i]>l->i[j]) {v=l->i[i]; l->i[i]=l->i[j]; l->i[j]=v;}
  }
}
/*****************************************************************************/

/*****************************************************************************/
/** Read a list of integer values from given string with given delimiters.
    @sa strTokenNr, strTokenNCpy, intlistAdd, intlistInit, intlistFree, 
        intlistSort, intlistExpandFromString.
    @return The number of added integer values, or <0 in case of an error.
    @author Vesa Oikonen
 */
int intlistAddFromString(
  /** Pointer to string from which the integers are read, for example "2,3,6,8". */
  const char *s1,
  /** String containing character delimiters, for example ", ". */
  const char *s2,
  /** Pointer to INTLIST structure; previous contents are preserved. */
  INTLIST *l,
  /** Add integer to the list only if it is new (0=no, 1=yes). */
  const int ifnew
) {
  if(l==NULL) return(-1);
  if(s1==NULL || s2==NULL) return(0);
  
  /* Get the nr of tokens */
  int i, j, m, n, v;
  n=strTokenNr((char*)s1, s2); if(n<1) return(0);
  /* Read the values */
  char tmp[128];
  for(i=j=0; i<n; i++) {
    if(strTokenNCpy(s1, s2, 1+i, tmp, 128)<1) return(-2);
    if(atoiCheck(tmp, &v)) return(-3);
    m=intlistAdd(l, v, ifnew); if(m<0) return(-4);
    j+=m;
  }
  return(j);
}
/*****************************************************************************/

/*****************************************************************************/
/** Read ranges and individual integer values from given string with given delimiters.
    @sa strTokenNCpy, intlistAddFromString, intlistInit, intlistFree, intlistSort, strIsValidNumber
    @return The number of added integer values, or <0 in case of an error.
    @author Vesa Oikonen
 */
int intlistExpandFromString(
  /** Pointer to string from which the integers are read, for example
      "0-8,12,32-28" or "0..8, 12, 28..34". */
  const char *s1,
  /** String containing character delimiters, for example ", ". */
  const char *s2,
  /** Pointer to INTLIST struct; previous contents are preserved. */
  INTLIST *l,
  /** Add integer to the list only if it is new (0=no, 1=yes). */
  const int ifnew
) {
  if(l==NULL) return(-1);
  if(s1==NULL || s2==NULL) return(0);
  
  /* Get the nr of tokens */
  int n=strTokenNr((char*)s1, s2); if(n<1) return(0);
  /* Read the values */
  char tmp[128], tmp2[128], *t, *tail;
  int i, j, m, first, last, sw;
  for(i=j=0; i<n; i++) {
    if(strTokenNCpy(s1, s2, 1+i, tmp, 128)<1) return(-2);
    t=tmp; errno=0;
    first=strtol(t, &tail, 10); if(errno) return(-3);
    if(*tail) {
      strcpy(tmp2, tail); t=tmp2;
      if(*t=='-') t++;
      else if(*t=='.') {t++; if(*t=='.') t++; else return(-4);}
      else return(-5);
      if(!*t) return(-6);
      last=strtol(t, &tail, 10); if(errno) return(-7);
      if(*tail) return(-8);
    } else {
      last=first;
    }

    if(first>last) {sw=first; first=last; last=sw;}
    for(int v=first; v<=last; v++) {
      m=intlistAdd(l, v, ifnew); if(m<0) return(-10);
      j+=m;
    }
  }
  return(j);
}
/*****************************************************************************/

/*****************************************************************************/
