/** @file studynr.c
 *  @brief Finding and verifying TPC study id number.
 *  Study number consists of a string of 1-5 letters (upper- or lower-case)
 *  followed by 1-5 digits.
 */
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "tpcextensions.h"
/*****************************************************************************/

/*****************************************************************************/
/** Check that the argument string is a valid TPC study number containing 1-5 letters (upper or 
    lower case) followed by 1-5 digits.
    @author Vesa Oikonen
    @return 0 if study number is valid, 1 if not valid, or >1 in case of failure.
    @sa studynrFromFilename, studynrStandardize
 */
int studynrVerify(
  /** Pointer to string to test. */
  const char *s,
  /** Number part of study number may start with zero (1) or may not (0). */
  int zero_ok
) {
  if(s==NULL) return(2);
  size_t len, i, j;
  len=strlen(s); if(len<2 || len>10) return(1);
  for(i=0; i<len; i++) if(!isalnum((int)s[i])) return(1);
  for(i=0; i<len; i++) if(!isalpha((int)s[i])) break;
  if(i<1 || i>5) return(1);
  if(zero_ok==0 && s[i]=='0') return(1); /* first digit must be >0 */
  for(j=0; (i+j)<len; j++) if(!isdigit((int)s[i+j])) return(1);
  if(j<1 || j>5) return(1);
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/** Remove initial zeroes in the numerical part of the PET study number,
    and convert any capital letters to lower-case.
    @return non-zero in case of failure.
    @sa studynrVerify, studynrFromFilename
 */
int studynrStandardize(
  /** Pointer to the study number string to be edited. */
  char *s
) {
  if(s==NULL) return(1);
  size_t i, j, len;
  len=strlen(s); if(len<2) return(1);
  if(isdigit((int)s[0])) return(2);
  if(!isdigit((int)s[len-1])) return(3);
  /* Convert to lower-case */
  for(i=0; i<len; i++) {
    if(isdigit((int)s[i])) break;
    s[i]=(char)tolower((int)s[i]);
  }
  /* Remove initial zeroes */
  for(i=1; i<len; i++) if(isdigit((int)s[i])) break;
  for(j=i; j<len; j++) if(s[j]!='0') break;
  if(i==j) return(0); /* no initial zeroes */
  for(;j<=len; i++, j++) s[i]=s[j];
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/** Extract study number (max MAX_STUDYNR_LEN chars) from given file name.

    If file name string includes path, the path is ignored.
    This function removes initial zeroes from the numerical part, and
    converts upper case letters to lower case, if necessary.
    @return 0 if successful.
    @sa studynrStandardize, studynrVerify, filenameRmExtensions, filenameRmPath
 */
int studynrFromFilename(
  /** File name, which may include the path. */
  const char *fname,
  /** Pointer for the resulting study number. Memory must be allocated before calling this function
     (at least MAX_STUDYNR_LEN+1 characters). */
  char *studynr,
  /** Validity of studynr is verified (0) or not verified (1). */
  int force
) {
  if(studynr==NULL) return(1); 
  strcpy(studynr, "");
  if(fname==NULL || strnlen(fname, 2)<2) return(1);
  /* Work with copy of file name */
  char *temp=strdup(fname);
  /* Remove path and extensions */
  filenameRmPath(temp); filenameRmExtensions(temp);
  /* Check if any part of base file name could be a valid study number */
  int ret=0, j;
  size_t i, k, n, len;
  len=strlen(temp); if(len<1) {free(temp); return(2);}
  for(i=1; i<len; i++) if(isalpha(temp[i-1]) && isdigit(temp[i])) {
    j=i-1; while(j>0 && isalpha(temp[j-1])) j--;
    k=i; while(k<len-1 && isdigit(temp[k+1])) k++;
    n=k-j+1; if(n>=2 && n<=10) {
      strlcpy(studynr, temp+j, n+1);
      if(force==1) {ret=1; break;}
      if(studynrStandardize(studynr)==0 && studynrVerify(studynr, 0)==0) {
        ret=2; break;}
    } 
  }
  if(ret) {free(temp); return 0;}
  if(force==0) {free(temp); return 3;}
  /* ok we have to take something */
  for(j=0; j<(int)len; j++) if(isalpha(temp[j]) || isdigit(temp[j])) break;
  for(k=j; k<len-1; k++) if(!isalpha(temp[k+1]) && !isdigit(temp[k+1])) break;
  n=k-j+1; if(n<1) {free(temp); return 4;}
  if(n>10) n=10;
  strlcpy(studynr, temp+j, n+1);
  free(temp);
  return 0;
}
/*****************************************************************************/

/*****************************************************************************/
