/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include "tpcextensions.h"
#include "test_tpcextensions.h"
/*****************************************************************************/

/*****************************************************************************/
int test_unitName(
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  if(verbose>0) {
    printf("\n=====================================\n");
    printf("\n%s\n", __func__);
    printf("\n=====================================\n");
  }
  int n;
  char unit[MAX_UNITS_LEN];

  n=0; if(verbose>1) printf("n := %d\n", n);
  strcpy(unit, unitName(n));
  if(verbose>2) printf("  -> unit := %s\n", unit);
  if(strcmp(unit, "unknown")!=0) return(1);

  n=99999; if(verbose>1) printf("n := %d\n", n);
  strcpy(unit, unitName(n));
  if(verbose>2) printf("  -> unit := %s\n", unit);
  if(strcmp(unit, "unknown")!=0) return(2);

  n=UNIT_SEC; if(verbose>1) printf("n := %d\n", n);
  strcpy(unit, unitName(n));
  if(verbose>2) printf("  -> unit := %s\n", unit);
  if(strcmp(unit, "sec")!=0) return(3);

  n=UNIT_MIN; if(verbose>1) printf("n := %d\n", n);
  strcpy(unit, unitName(n));
  if(verbose>2) printf("  -> unit := %s\n", unit);
  if(strcmp(unit, "min")!=0) return(4);

  n=UNIT_M; if(verbose>1) printf("n := %d\n", n);
  strcpy(unit, unitName(n));
  if(verbose>2) printf("  -> unit := %s\n", unit);
  if(strcmp(unit, "m")!=0) return(5);

  n=UNIT_KBQ_PER_ML; if(verbose>1) printf("n := %d\n", n);
  strcpy(unit, unitName(n));
  if(verbose>2) printf("  -> unit := %s\n", unit);
  if(strcmp(unit, "kBq/mL")!=0) return(6);

  n=UNIT_GBQ_PER_NMOL; if(verbose>1) printf("n := %d\n", n);
  strcpy(unit, unitName(n));
  if(verbose>2) printf("  -> unit := %s\n", unit);
  if(strcmp(unit, "GBq/nmol")!=0) return(7);
  
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
int test_unitIdentify(
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  if(verbose>0) {
    printf("\n=====================================\n");
    printf("\n%s\n", __func__);
    printf("\n=====================================\n");
  }
  int u;
  char *cptr, s[MAX_UNITS_LEN];

  cptr=NULL;
  u=unitIdentify(cptr);
  if(verbose>2) printf("  -> unit := %s\n", unitName(u));
  if(u!=UNIT_UNKNOWN) return(1);
  
  strcpy(s, ""); if(verbose>1) printf("s := '%s'\n", s);
  u=unitIdentify(s);
  if(verbose>2) printf("  -> unit := %s\n", unitName(u));
  if(u!=UNIT_UNKNOWN) return(2);
  
  strcpy(s, "meh"); if(verbose>1) printf("s := '%s'\n", s);
  u=unitIdentify(s);
  if(verbose>2) printf("  -> unit := %s\n", unitName(u));
  if(u!=UNIT_UNKNOWN) return(3);

  strcpy(s, unitName(UNIT_KBQ_PER_ML)); if(verbose>1) printf("s := '%s'\n", s);
  u=unitIdentify(s);
  if(verbose>2) printf("  -> unit := %s\n", unitName(u));
  if(u!=UNIT_KBQ_PER_ML) return(4);

  strcpy(s, "Bq/cc"); if(verbose>1) printf("s := '%s'\n", s);
  u=unitIdentify(s);
  if(verbose>2) printf("  -> unit := %s\n", unitName(u));
  if(u!=UNIT_BQ_PER_ML) return(5);

  strcpy(s, "s"); if(verbose>1) printf("s := '%s'\n", s);
  u=unitIdentify(s);
  if(verbose>2) printf("  -> unit := %s\n", unitName(u));
  if(u!=UNIT_SEC) return(6);

  strcpy(s, "%i.d./g"); if(verbose>1) printf("s := '%s'\n", s);
  u=unitIdentify(s);
  if(verbose>2) printf("  -> unit := %s\n", unitName(u));
  if(u!=UNIT_PID_PER_G) return(7);
  
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
int test_unitIdentifyFilename(
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  if(verbose>0) {
    printf("\n=====================================\n");
    printf("\n%s\n", __func__);
    printf("\n=====================================\n");
  }
  int u;
  char *cptr, s[FILENAME_MAX];

  cptr=NULL;
  u=unitIdentifyFilename(cptr);
  if(verbose>2) printf("  -> unit := %s\n", unitName(u));
  if(u!=UNIT_UNKNOWN) return(1);
  
  strcpy(s, ""); if(verbose>1) printf("s := '%s'\n", s);
  u=unitIdentifyFilename(s);
  if(verbose>2) printf("  -> unit := %s\n", unitName(u));
  if(u!=UNIT_UNKNOWN) return(2);
  
  strcpy(s, "C:\\filename.txt"); if(verbose>1) printf("s := '%s'\n", s);
  u=unitIdentifyFilename(s);
  if(verbose>2) printf("  -> unit := %s\n", unitName(u));
  if(u!=UNIT_UNKNOWN) return(3);
  
  strcpy(s, "C:\\filename.kbq"); if(verbose>1) printf("s := '%s'\n", s);
  u=unitIdentifyFilename(s);
  if(verbose>2) printf("  -> unit := %s\n", unitName(u));
  if(u!=UNIT_KBQ_PER_ML) return(4);
  
  strcpy(s, "../folder/file.nci.bak"); if(verbose>1) printf("s := '%s'\n", s);
  u=unitIdentifyFilename(s);
  if(verbose>2) printf("  -> unit := %s\n", unitName(u));
  if(u!=UNIT_NCI_PER_ML) return(5);

  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
int test_unitIsDistance(
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  if(verbose>0) {
    printf("\n=====================================\n");
    printf("\n%s\n", __func__);
    printf("\n=====================================\n");
  }

  if(unitIsDistance(-6)) return(1);
  if(!unitIsDistance(UNIT_CM)) return(2);
  if(unitIsDistance(UNIT_BQ_PER_ML)) return(3);
  if(unitIsDistance(UNIT_MIN)) return(4);

  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
int test_unitIsTime(
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  if(verbose>0) {
    printf("\n=====================================\n");
    printf("\n%s\n", __func__);
    printf("\n=====================================\n");
  }

  if(unitIsTime(-6)) return(1);
  if(!unitIsTime(UNIT_MIN)) return(2);
  if(unitIsTime(UNIT_BQ_PER_ML)) return(3);
  if(unitIsTime(UNIT_CM)) return(4);

  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
int test_unitIsVolume(
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  if(verbose>0) {
    printf("\n=====================================\n");
    printf("\n%s\n", __func__);
    printf("\n=====================================\n");
  }

  if(unitIsVolume(-6)) return(1);
  if(!unitIsVolume(UNIT_ML)) return(2);
  if(unitIsVolume(UNIT_BQ_PER_ML)) return(3);
  if(unitIsVolume(UNIT_CM)) return(4);

  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
int test_unitIsMass(
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  if(verbose>0) {
    printf("\n=====================================\n");
    printf("\n%s\n", __func__);
    printf("\n=====================================\n");
  }

  if(unitIsMass(-6)) return(1);
  if(!unitIsMass(UNIT_G)) return(2);
  if(unitIsMass(UNIT_BQ_PER_ML)) return(3);
  if(unitIsMass(UNIT_CM)) return(4);

  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
int test_unitIsMole(
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  if(verbose>0) {
    printf("\n=====================================\n");
    printf("\n%s\n", __func__);
    printf("\n=====================================\n");
  }

  if(unitIsMole(-6)) return(1);
  if(!unitIsMole(UNIT_MMOL)) return(2);
  if(unitIsMole(UNIT_BQ_PER_ML)) return(3);
  if(unitIsMole(UNIT_CM)) return(4);

  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
int test_unitIsRadioactivity(
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  if(verbose>0) {
    printf("\n=====================================\n");
    printf("\n%s\n", __func__);
    printf("\n=====================================\n");
  }

  if(unitIsRadioactivity(-6)) return(1);
  if(!unitIsRadioactivity(UNIT_KBQ)) return(2);
  if(!unitIsRadioactivity(UNIT_MCI)) return(3);
  if(unitIsRadioactivity(UNIT_BQ_PER_ML)) return(4);
  if(unitIsRadioactivity(UNIT_CM)) return(5);

  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
int test_unitIsCombinatorial(
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  if(verbose>0) {
    printf("\n=====================================\n");
    printf("\n%s\n", __func__);
    printf("\n=====================================\n");
  }

  if(unitIsCombinatorial(-6)) return(1);
  if(unitIsCombinatorial(9999)) return(2);
  if(unitIsCombinatorial(UNIT_KBQ)) return(3);
  if(!unitIsCombinatorial(UNIT_PER_MIN)) return(4);
  if(!unitIsCombinatorial(UNIT_BQ_PER_ML)) return(5);
  if(unitIsCombinatorial(UNIT_CM)) return(6);
  if(!unitIsCombinatorial(UNIT_ML_PER_ML)) return(7);
  if(!unitIsCombinatorial(UNIT_UMOL_PER_100G_MIN)) return(8);

  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
int test_unitConversionFactor(
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  if(verbose>0) {
    printf("\n=====================================\n");
    printf("\n%s\n", __func__);
    printf("\n=====================================\n");
  }

  double cf;
  int u1, u2;
  
  /* Everything cannot be converted */
  u1=UNIT_UNKNOWN; u2=UNIT_UNKNOWN;
  cf=unitConversionFactor(u1, u2);
  if(verbose>1) 
    printf("cf=%g for conversion %s -> %s\n", cf, unitName(u1), unitName(u2));
  if(!isnan(cf)) return 1;
    
  u1=UNIT_MIN; u2=UNIT_UNKNOWN;
  cf=unitConversionFactor(u1, u2);
  if(verbose>1) 
    printf("cf=%g for conversion %s -> %s\n", cf, unitName(u1), unitName(u2));
  if(!isnan(cf)) return 2;
    
  u1=UNIT_UNKNOWN; u2=UNIT_MIN;
  cf=unitConversionFactor(u1, u2);
  if(verbose>1) 
    printf("cf=%g for conversion %s -> %s\n", cf, unitName(u1), unitName(u2));
  if(!isnan(cf)) return 3;
    
  u1=UNIT_MM; u2=UNIT_MIN;
  cf=unitConversionFactor(u1, u2);
  if(verbose>1) 
    printf("cf=%g for conversion %s -> %s\n", cf, unitName(u1), unitName(u2));
  if(!isnan(cf)) return 4;

  u1=-6; u2=UNIT_UNITLESS;
  cf=unitConversionFactor(u1, u2);
  if(verbose>1) 
    printf("cf=%g for conversion %s -> %s\n", cf, unitName(u1), unitName(u2));
  if(!isnan(cf)) return 5;

  u1=UNIT_UNITLESS; u2=-6;
  cf=unitConversionFactor(u1, u2);
  if(verbose>1) 
    printf("cf=%g for conversion %s -> %s\n", cf, unitName(u1), unitName(u2));
  if(!isnan(cf)) return 6;

  
  /* Times can be converted */
  u1=UNIT_MIN; u2=UNIT_MIN;
  cf=unitConversionFactor(u1, u2);
  if(verbose>1) 
    printf("cf=%g for conversion %s -> %s\n", cf, unitName(u1), unitName(u2));
  if(!doubleMatch(cf, 1.0, 1.0E-30)) return 11;
  
  u1=UNIT_SEC; u2=UNIT_MIN;
  cf=unitConversionFactor(u1, u2);
  if(verbose>1) 
    printf("cf=%g for conversion %s -> %s\n", cf, unitName(u1), unitName(u2));
  if(!doubleMatch(cf, 1.0/60.0, 1.0E-30)) return 12;
  
  u1=UNIT_MIN; u2=UNIT_SEC;
  cf=unitConversionFactor(u1, u2);
  if(verbose>1) 
    printf("cf=%g for conversion %s -> %s\n", cf, unitName(u1), unitName(u2));
  if(!doubleMatch(cf, 60.0, 1.0E-30)) return 13;
  
  u1=UNIT_MIN; u2=UNIT_MSEC;
  cf=unitConversionFactor(u1, u2);
  if(verbose>1) 
    printf("cf=%g for conversion %s -> %s\n", cf, unitName(u1), unitName(u2));
  if(!doubleMatch(cf, 60.0E+3, 1.0E-30)) return 14;
    
  /* Distances can be converted */
  u1=UNIT_MM; u2=UNIT_MM;
  cf=unitConversionFactor(u1, u2);
  if(verbose>1) 
    printf("cf=%g for conversion %s -> %s\n", cf, unitName(u1), unitName(u2));
  if(!doubleMatch(cf, 1.0, 1.0E-30)) return 21;
  
  u1=UNIT_MM; u2=UNIT_CM;
  cf=unitConversionFactor(u1, u2);
  if(verbose>1) 
    printf("cf=%g for conversion %s -> %s\n", cf, unitName(u1), unitName(u2));
  if(!doubleMatch(cf, 0.1, 1.0E-30)) return 22;
  
  u1=UNIT_CM; u2=UNIT_MM;
  cf=unitConversionFactor(u1, u2);
  if(verbose>1) 
    printf("cf=%g for conversion %s -> %s\n", cf, unitName(u1), unitName(u2));
  if(!doubleMatch(cf, 10.0, 1.0E-30)) return 23;
    
  /* Volumes can be converted */
  u1=UNIT_ML; u2=UNIT_ML;
  cf=unitConversionFactor(u1, u2);
  if(verbose>1) 
    printf("cf=%g for conversion %s -> %s\n", cf, unitName(u1), unitName(u2));
  if(!doubleMatch(cf, 1.0, 1.0E-30)) return 31;

  u1=UNIT_L; u2=UNIT_ML;
  cf=unitConversionFactor(u1, u2);
  if(verbose>1) 
    printf("cf=%g for conversion %s -> %s\n", cf, unitName(u1), unitName(u2));
  if(!doubleMatch(cf, 1000.0, 1.0E-30)) return 32;

  u1=UNIT_UL; u2=UNIT_L;
  cf=unitConversionFactor(u1, u2);
  if(verbose>1) 
    printf("cf=%g for conversion %s -> %s\n", cf, unitName(u1), unitName(u2));
  if(!doubleMatch(cf, 1.0E-6, 1.0E-30)) return 33;
    
  /* Masses can be converted */
  u1=UNIT_G; u2=UNIT_G;
  cf=unitConversionFactor(u1, u2);
  if(verbose>1) 
    printf("cf=%g for conversion %s -> %s\n", cf, unitName(u1), unitName(u2));
  if(!doubleMatch(cf, 1.0, 1.0E-30)) return 41;

  u1=UNIT_G; u2=UNIT_MG;
  cf=unitConversionFactor(u1, u2);
  if(verbose>1) 
    printf("cf=%g for conversion %s -> %s\n", cf, unitName(u1), unitName(u2));
  if(!doubleMatch(cf, 1000.0, 1.0E-30)) return 42;

  u1=UNIT_100G; u2=UNIT_KG;
  cf=unitConversionFactor(u1, u2);
  if(verbose>1) 
    printf("cf=%g for conversion %s -> %s\n", cf, unitName(u1), unitName(u2));
  if(!doubleMatch(cf, 1.0E-1, 1.0E-30)) return 43;
    
  /* Moles can be converted */
  u1=UNIT_MMOL; u2=UNIT_MMOL;
  cf=unitConversionFactor(u1, u2);
  if(verbose>1) 
    printf("cf=%g for conversion %s -> %s\n", cf, unitName(u1), unitName(u2));
  if(!doubleMatch(cf, 1.0, 1.0E-30)) return 51;
  
  u1=UNIT_PMOL; u2=UNIT_MMOL;
  cf=unitConversionFactor(u1, u2);
  if(verbose>1) 
    printf("cf=%g for conversion %s -> %s\n", cf, unitName(u1), unitName(u2));
  if(!doubleMatch(cf, 1.0E-9, 1.0E-30)) return 52;
  
  u1=UNIT_MOL; u2=UNIT_MMOL;
  cf=unitConversionFactor(u1, u2);
  if(verbose>1) 
    printf("cf=%g for conversion %s -> %s\n", cf, unitName(u1), unitName(u2));
  if(!doubleMatch(cf, 1.0E+3, 1.0E-30)) return 53;

  /* Radioactivities can be converted */
  u1=UNIT_KBQ; u2=UNIT_KBQ;
  cf=unitConversionFactor(u1, u2);
  if(verbose>1) 
    printf("cf=%g for conversion %s -> %s\n", cf, unitName(u1), unitName(u2));
  if(!doubleMatch(cf, 1.0, 1.0E-30)) return 61;
  
  u1=UNIT_KBQ; u2=UNIT_MBQ;
  cf=unitConversionFactor(u1, u2);
  if(verbose>1) 
    printf("cf=%g for conversion %s -> %s\n", cf, unitName(u1), unitName(u2));
  if(!doubleMatch(cf, 1.0E-3, 1.0E-30)) return 62;
  
  u1=UNIT_MBQ; u2=UNIT_KBQ;
  cf=unitConversionFactor(u1, u2);
  if(verbose>1) 
    printf("cf=%g for conversion %s -> %s\n", cf, unitName(u1), unitName(u2));
  if(!doubleMatch(cf, 1.0E+3, 1.0E-30)) return 63;

  u1=UNIT_CI; u2=UNIT_MCI;
  cf=unitConversionFactor(u1, u2);
  if(verbose>1) 
    printf("cf=%g for conversion %s -> %s\n", cf, unitName(u1), unitName(u2));
  if(!doubleMatch(cf, 1.0E+3, 1.0E-30)) return 64;

  u1=UNIT_CI; u2=UNIT_BQ;
  cf=unitConversionFactor(u1, u2);
  if(verbose>1) 
    printf("cf=%g for conversion %s -> %s\n", cf, unitName(u1), unitName(u2));
  if(!doubleMatch(cf, 3.7E+10, 1.0E-30)) return 65;
  
  u1=UNIT_NCI; u2=UNIT_KBQ;
  cf=unitConversionFactor(u1, u2);
  if(verbose>1) 
    printf("cf=%g for conversion %s -> %s\n", cf, unitName(u1), unitName(u2));
  if(!doubleMatch(cf, 3.7E-02, 1.0E-30)) return 66;
  
  u1=UNIT_KBQ; u2=UNIT_NCI;
  cf=unitConversionFactor(u1, u2);
  if(verbose>1) 
    printf("cf=%g for conversion %s -> %s\n", cf, unitName(u1), unitName(u2));
  if(!doubleMatch(cf, 27.027, 1.0E-3)) return 67;
  
  /* Percentages and fractions */
  u1=UNIT_PERCENTAGE; u2=UNIT_PERCENTAGE;
  cf=unitConversionFactor(u1, u2);
  if(verbose>1) 
    printf("cf=%g for conversion %s -> %s\n", cf, unitName(u1), unitName(u2));
  if(!doubleMatch(cf, 1.0, 1.0E-30)) return 71;

  u1=UNIT_UNITLESS; u2=UNIT_PERCENTAGE;
  cf=unitConversionFactor(u1, u2);
  if(verbose>1) 
    printf("cf=%g for conversion %s -> %s\n", cf, unitName(u1), unitName(u2));
  if(!doubleMatch(cf, 100.0, 1.0E-30)) return 72;

  u1=UNIT_PERCENTAGE; u2=UNIT_UNITLESS;
  cf=unitConversionFactor(u1, u2);
  if(verbose>1) 
    printf("cf=%g for conversion %s -> %s\n", cf, unitName(u1), unitName(u2));
  if(!doubleMatch(cf, 0.01, 1.0E-30)) return 73;

  u1=UNIT_UNITLESS; u2=UNIT_UNITLESS;
  cf=unitConversionFactor(u1, u2);
  if(verbose>1) 
    printf("cf=%g for conversion %s -> %s\n", cf, unitName(u1), unitName(u2));
  if(!doubleMatch(cf, 1.0, 1.0E-30)) return 74;

  /* Combination units */
  u1=UNIT_KBQ_PER_ML; u2=UNIT_BQ_PER_ML;
  cf=unitConversionFactor(u1, u2);
  if(verbose>1) 
    printf("cf=%g for conversion %s -> %s\n", cf, unitName(u1), unitName(u2));
  if(!doubleMatch(cf, 1.0E+3, 1.0E-30)) return 101;
  
  u1=UNIT_PER_SEC; u2=UNIT_PER_MIN;
  cf=unitConversionFactor(u1, u2);
  if(verbose>1) 
    printf("cf=%g for conversion %s -> %s\n", cf, unitName(u1), unitName(u2));
  if(!doubleMatch(cf, 60.0, 1.0E-30)) return 102;

  u1=UNIT_UMOL_PER_ML_MIN; u2=UNIT_MMOL_PER_DL_MIN;
  cf=unitConversionFactor(u1, u2);
  if(verbose>1) 
    printf("cf=%g for conversion %s -> %s\n", cf, unitName(u1), unitName(u2));
  if(!doubleMatch(cf, 0.1, 1.0E-30)) return 103;

  u1=UNIT_NCI_PER_ML; u2=UNIT_BQ_PER_ML;
  cf=unitConversionFactor(u1, u2);
  if(verbose>1) 
    printf("cf=%g for conversion %s -> %s\n", cf, unitName(u1), unitName(u2));
  if(!doubleMatch(cf, 37.0, 1.0E-30)) return 104;

  u1=UNIT_UMOL_PER_ML_MIN; u2=UNIT_MG_PER_DL_MIN;
  cf=unitConversionFactor(u1, u2);
  if(verbose>1) 
    printf("cf=%g for conversion %s -> %s\n", cf, unitName(u1), unitName(u2));
  if(!isnan(cf)) return 105;

  u1=UNIT_CPS; u2=UNIT_KCPM;
  cf=unitConversionFactor(u1, u2);
  if(verbose>1) 
    printf("cf=%g for conversion %s -> %s\n", cf, unitName(u1), unitName(u2));
  if(!doubleMatch(cf, 0.060, 1.0E-15)) return 106;
  
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
int test_unitInverse(
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  if(verbose>0) {
    printf("\n=====================================\n");
    printf("\n%s\n", __func__);
    printf("\n=====================================\n");
  }

  if(unitInverse(-6)!=UNIT_UNKNOWN) return(1);
  if(unitInverse(UNIT_UNITLESS)!=UNIT_UNITLESS) return(2);
  if(unitInverse(UNIT_ML_PER_ML)!=UNIT_ML_PER_ML) return(3);
  if(unitInverse(UNIT_MIN)!=UNIT_PER_MIN) return(4);
  if(unitInverse(UNIT_PER_MIN)!=UNIT_MIN) return(5);

  statusSet(status, __func__, __FILE__, __LINE__, 0);
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
int test_unitDividerHasVolume(
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  if(verbose>0) {
    printf("\n=====================================\n");
    printf("\n%s\n", __func__);
    printf("\n=====================================\n");
  }

  if(unitDividerHasVolume(-6)) return(1);
  if(unitDividerHasVolume(UNIT_ML)) return(2);
  if(!unitDividerHasVolume(UNIT_BQ_PER_ML)) return(3);
  if(unitDividerHasVolume(UNIT_BQ_PER_G)) return(4);
  if(!unitDividerHasVolume(UNIT_ML_PER_ML_MIN)) return(5);

  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
int test_unitDividerHasMass(
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  if(verbose>0) {
    printf("\n=====================================\n");
    printf("\n%s\n", __func__);
    printf("\n=====================================\n");
  }

  if(unitDividerHasMass(-6)) return(1);
  if(unitDividerHasMass(UNIT_G)) return(2);
  if(unitDividerHasMass(UNIT_BQ_PER_ML)) return(3);
  if(!unitDividerHasMass(UNIT_BQ_PER_G)) return(4);
  if(unitDividerHasMass(UNIT_ML_PER_ML_MIN)) return(5);

  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
int test_unitDividendHasRadioactivity(
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  if(verbose>0) {
    printf("\n=====================================\n");
    printf("\n%s\n", __func__);
    printf("\n=====================================\n");
  }

  if(unitDividendHasRadioactivity(-6)) return(1);
  if(unitDividendHasRadioactivity(UNIT_G)) return(2);
  if(!unitDividendHasRadioactivity(UNIT_BQ)) return(3);
  if(!unitDividendHasRadioactivity(UNIT_BQ_PER_ML)) return(4);
  if(!unitDividendHasRadioactivity(UNIT_BQ_PER_G)) return(5);
  if(unitDividendHasRadioactivity(UNIT_ML_PER_ML_MIN)) return(6);
  if(!unitDividendHasRadioactivity(UNIT_SEC_KBQ_PER_ML)) return(7);

  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
int test_unitIsRAConc(
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  if(verbose>0) {
    printf("\n=====================================\n");
    printf("\n%s\n", __func__);
    printf("\n=====================================\n");
  }

  if(unitIsRAConc(-6)) return(1);
  if(unitIsRAConc(UNIT_G)) return(2);
  if(unitIsRAConc(UNIT_BQ)) return(3);
  if(!unitIsRAConc(UNIT_BQ_PER_ML)) return(4);
  if(!unitIsRAConc(UNIT_BQ_PER_G)) return(5);
  if(unitIsRAConc(UNIT_ML_PER_ML_MIN)) return(6);
  if(unitIsRAConc(UNIT_SEC_KBQ_PER_ML)) return(7);

  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
int test_unitCombination(
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  if(verbose>0) {
    printf("\n=====================================\n");
    printf("\n%s\n", __func__);
    printf("\n=====================================\n");
  }

  int u, u1, u2, v1, v2;

  u1=UNIT_UNKNOWN; u2=UNIT_UNKNOWN; v1=UNIT_UNKNOWN; v2=UNIT_UNKNOWN; 
  u=unitCombination(u1, u2, v1, v2); if(u!=UNIT_UNKNOWN) return(1);

  u1=UNIT_UNITLESS; u2=UNIT_UNITLESS; v1=UNIT_UNITLESS; v2=UNIT_UNITLESS; 
  u=unitCombination(u1, u2, v1, v2); if(u!=UNIT_UNITLESS) return(2);

  u1=UNIT_BQ; u2=UNIT_UNITLESS; v1=UNIT_ML; v2=UNIT_UNITLESS; 
  u=unitCombination(u1, u2, v1, v2); if(u!=UNIT_BQ_PER_ML) return(3);

  u1=UNIT_BQ; u2=UNIT_UNITLESS; v1=UNIT_G; v2=UNIT_UNITLESS; 
  u=unitCombination(u1, u2, v1, v2); if(u!=UNIT_BQ_PER_G) return(4);

  u1=UNIT_MMOL; u2=UNIT_UNITLESS; v1=UNIT_100G; v2=UNIT_MIN;
  u=unitCombination(u1, u2, v1, v2); if(u!=UNIT_MMOL_PER_100G_MIN) return(5);
  u1=UNIT_MMOL; u2=UNIT_UNITLESS; v1=UNIT_MIN; v2=UNIT_100G;
  u=unitCombination(u1, u2, v1, v2); if(u!=UNIT_MMOL_PER_100G_MIN) return(6);

  u1=UNIT_MIN; u2=UNIT_SEC; v1=UNIT_SEC; v2=UNIT_MBQ; 
  u=unitCombination(u1, u2, v1, v2); if(u!=UNIT_UNKNOWN) return(9);

  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
int test_unitDividerMassVolumeConversion(
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  if(verbose>0) {
    printf("\n=====================================\n");
    printf("\n%s\n", __func__);
    printf("\n=====================================\n");
  }

  int u, uo;

  uo=UNIT_UNKNOWN;
  u=unitDividerMassVolumeConversion(uo); if(u!=UNIT_UNKNOWN) return(1);

  uo=UNIT_UNITLESS;
  u=unitDividerMassVolumeConversion(uo); if(u!=UNIT_UNKNOWN) return(2);

  uo=UNIT_BQ_PER_ML;
  u=unitDividerMassVolumeConversion(uo); if(verbose>1) printf(" %s -> %s\n", unitName(uo), unitName(u));
  if(u!=UNIT_BQ_PER_G) return(3);
  uo=UNIT_BQ_PER_G;
  u=unitDividerMassVolumeConversion(uo); if(u!=UNIT_BQ_PER_ML) return(4);

  uo=UNIT_MMOL_PER_100G_MIN;
  u=unitDividerMassVolumeConversion(uo); if(u!=UNIT_MMOL_PER_DL_MIN) return(5);
  uo=UNIT_MMOL_PER_DL_MIN;
  u=unitDividerMassVolumeConversion(uo); if(u!=UNIT_MMOL_PER_100G_MIN) return(6);

  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
int test_unitMultiply(
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  if(verbose>0) {
    printf("\n=====================================\n");
    printf("\n%s\n", __func__);
    printf("\n=====================================\n");
  }

  int u, u1, u2, cu;

  u1=UNIT_UNKNOWN; u2=UNIT_UNKNOWN; cu=UNIT_UNKNOWN;
  u=unitMultiply(u1, u2);
  if(verbose>2) printf("%s x %s -> %s\n", unitName(u1), unitName(u2), unitName(u));
  if(u!=cu) return(1);

  u1=UNIT_UNITLESS; u2=UNIT_UNITLESS; cu=UNIT_UNITLESS;
  u=unitMultiply(u1, u2);
  if(verbose>2) printf("%s x %s -> %s\n", unitName(u1), unitName(u2), unitName(u));
  if(u!=cu) return(2);

  u1=UNIT_UNITLESS; u2=UNIT_UNKNOWN; cu=UNIT_UNKNOWN;
  u=unitMultiply(u1, u2);
  if(verbose>2) printf("%s x %s -> %s\n", unitName(u1), unitName(u2), unitName(u));
  if(u!=cu) return(3);

  u1=UNIT_BQ_PER_ML; u2=UNIT_SEC; cu=unitIdentify("sec*Bq/mL");
  u=unitMultiply(u1, u2);
  if(verbose>2) printf("%s x %s -> %s\n", unitName(u1), unitName(u2), unitName(u));
  if(u!=cu) return(11);

  u1=UNIT_KBQ_PER_ML; u2=UNIT_MIN; cu=unitIdentify("min*kBq/mL");
  u=unitMultiply(u1, u2);
  if(verbose>2) printf("%s x %s -> %s\n", unitName(u1), unitName(u2), unitName(u));
  if(u!=cu) return(12);

  u1=UNIT_ML_PER_G; u2=UNIT_PER_MIN; cu=UNIT_ML_PER_G_MIN;
  u=unitMultiply(u1, u2);
  if(verbose>2) printf("%s x %s -> %s\n", unitName(u1), unitName(u2), unitName(u));
  if(u!=cu) return(21);

  u1=UNIT_ML_PER_G_MIN; u2=UNIT_MIN; cu=UNIT_ML_PER_G;
  u=unitMultiply(u1, u2);
  if(verbose>2) printf("%s x %s -> %s\n", unitName(u1), unitName(u2), unitName(u));
  if(u!=cu) return(22);

  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
