/** @file backup.c
 *  @brief Functions for file backup.
 */
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include "tpcextensions.h"
/*****************************************************************************/
#include "tpcfileutil.h"
/*****************************************************************************/

/*****************************************************************************/
/** Check if specified file exists; rename existing file to a backup file.
    If also backup file exists, then remove that.
    @return Returns 0, if successful, and <>0 in case of an error.
 */
int fileBackup(
  /** Name of file which, if it exists, is renamed to a backup file. */
  const char *filename,
  /** Extension for backup file; NULL will set the default ".bak" extension. */
  const char *backup_ext,
  /** Pointer to a string (allocated for at least 64 chars) where error message
      or other execution status will be written; enter NULL, if not needed. */
  char *status
) {
  char bakfile[FILENAME_MAX];
  int ret;

  // Check the input
  if(filename==NULL || strlen(filename)<1) {
    if(status!=NULL) sprintf(status, "invalid filename");
    return 1;
  }

  // Check if file exists; if not then no need to make any backup
  if(access(filename, 0) == -1) {
    if(status!=NULL) sprintf(status, "file does not pre-exist");
    return 0;
  }
  // Create filename for the backup file
  strlcpy(bakfile, filename, FILENAME_MAX);
  if(backup_ext==NULL) strlcat(bakfile, ".bak", FILENAME_MAX);
  else strlcat(bakfile, backup_ext, FILENAME_MAX);
  // If also backup file exists, then just delete it 
  if(access(bakfile, 0) != -1) {
    ret=remove(bakfile);
    if(ret!=0) {
      if(status!=NULL) sprintf(status, "cannot delete previous backup file");
      return 3;
    }
  }
  // Rename file
  ret=rename(filename, bakfile);
  if(ret!=0) {
    if(status!=NULL) sprintf(status, "cannot rename file as backup");
    return 5;
  }
  if(status!=NULL) sprintf(status, "file renamed as backup");
  return 0;
}
/*****************************************************************************/

/*****************************************************************************/
