/** @file tpcift.h
 *  @brief Header file for library libtpcift.
 *  @details Interfile-type header information processing and file i/o
 *  @author Vesa Oikonen
 */
#ifndef _TPCIFT_H_
#define _TPCIFT_H_
/*****************************************************************************/

/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
/*****************************************************************************/
#include "tpcextensions.h"
/*****************************************************************************/

/*****************************************************************************/
/** Struct for one IFT item containing key and its value. Add new item using iftPut().
   @sa IFT, iftPut
 */
typedef struct IFT_ITEM {
  /** Comment line, starting with '#': 0=not, 1=yes. */
  char comment;
  /** Undefined short integer for the use of programmer. */
  short int sw;
  /** Pointer to the NULL terminated key string; NULL if not allocated;
      Set or change this only using dedicated functions. */
  char *key;
  /** Pointer to the NULL terminated key value string; NULL if not allocated.
      Set or change this only using dedicated functions.
      If this field is used to store decimal number or list of decimal numbers, then 
      do NOT use decimal commas. */
  char *value;
} IFT_ITEM;

/** Main struct for IFT items.
   @sa IFT_ITEM, iftInit, iftFree, CVS, TAC, PAR
 */
typedef struct IFT {
  /** Number of allocated places for keys and values */
  int _memNr;
  /** Number of stored keys and their values. */
  int keyNr;
  /** Type of the parameter file:
      0=unknown, 1=interfile ':=' , 2=setup '=', 3=result ':', 4=space ' ', 5=tab, 6=',', 7=';'
  */
  int type;
  /** Space before equal sign (0=no, otherways yes). */
  int space_before_eq;
  /** Space after equal sign (0=no, otherways yes). */
  int space_after_eq;
  /** List of key-value -pairs. */
  IFT_ITEM *item;
} IFT;
/*****************************************************************************/

/*****************************************************************************/
/* ift */
extern void iftInit(IFT *ift);
extern void iftFree(IFT *ift);
extern int iftPut(IFT *ift, const char *key, const char *value, char comment, TPCSTATUS *status);
extern int iftPutDouble(IFT *ift, const char *key, const double value, char comment, TPCSTATUS *status);
extern int iftPutInt(IFT *ift, const char *key, const int value, char comment, TPCSTATUS *status);
extern int iftPutUInt(IFT *ift, const char *key, const unsigned int value, char comment, TPCSTATUS *status);
extern int iftDelete(IFT *ift, int index);
extern int iftDuplicate(IFT *ift1, IFT *ift2);
extern int iftReplaceValue(IFT *ift, int i, const char *value, TPCSTATUS *status);
extern int iftReplaceKey(IFT *ift, int i, const char *key, TPCSTATUS *status);
extern int iftDeleteDuplicateKeys(IFT *ift, TPCSTATUS *status);
extern int iftCopyItems(
  IFT *ift1, IFT *ift2, int is_key_required, int is_value_required, int is_comment_accepted, TPCSTATUS *status
);
/*****************************************************************************/

/*****************************************************************************/
/* iftio */
extern int iftWrite(IFT *ift, FILE *fp, TPCSTATUS *status);
extern int iftWriteItem(IFT *ift, int item, FILE *fp, TPCSTATUS *status);
extern int iftRead(IFT *ift, FILE *fp, int is_key_required, int is_comment_accepted, TPCSTATUS *status);
extern int iftPutFromString(
  IFT *ift, const char *line, int is_key_required, int is_comment_accepted, TPCSTATUS *status
);
extern int iftPutFromStringWithSpaceSeparator(
  IFT *ift, const char *line, int is_comment_accepted, TPCSTATUS *status
);
/*****************************************************************************/

/*****************************************************************************/
/* iftfind */
extern int iftFindKey(IFT *ift, const char *key, int start_index);
extern int iftFindPair(IFT *ift, const char *key, const char *value, int start_index);
extern int iftSearchKey(IFT *ift, const char *s, int start_index);
extern int iftSearchValue(IFT *ift, const char *s, int start_index);
extern int iftFindNrOfKeys(IFT *ift, const char *key);
extern void iftDeleteKey(IFT *ift, const char *key);
extern int iftGetDoubleValue(IFT *ift, const char *key, int index, double *v);
/*****************************************************************************/

/*****************************************************************************/
/* iftget */
extern int iftGetFloat(IFT *ift, int index, float *v);
extern int iftGetDouble(IFT *ift, int index, double *v);
extern int iftGetInt(IFT *ift, int index, int *v);
extern int iftGetUInt(IFT *ift, int index, unsigned int *v);
extern int iftGetDoubleWithUnit(IFT *ift, int index, double *v, int *unit);
/*****************************************************************************/

/*****************************************************************************/
#endif /* TPCIFT */
