/** @file image.c
    @brief Functions to basic processing with TPCCLIB image data structure.
    @copyright (c) Turku PET Centre
    @author Vesa Oikonen
 */
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include "tpcimage.h"
/*****************************************************************************/

/*****************************************************************************/
/** Text representations of modality of image data.
    @sa imgmodality, tpcimage.h
 */
static const char *img_modality[] = {
  "Unknown",  // IMG_MODALITY_UNKNOWN
  "PET",      // IMG_MODALITY_PET
  "SPECT",    // IMG_MODALITY_SPECT
  "CT",       // IMG_MODALITY_CT
  "MRI",      // IMG_MODALITY_MRI
0};
/** Return pointer to modality of the image data.
    @return pointer to the modality string.
    @sa imgmodality
 */
char *imgModalityDescr(
  /** IMG modality code */
  imgmodality c
) {
  if(c<IMG_MODALITY_UNKNOWN || c>=IMG_MODALITY_LAST) return NULL;
  return (char*)img_modality[c];
}
/*****************************************************************************/

/*****************************************************************************/
/** @public Text representations of image data content.
    @sa imgcontent, tpcimage.h
 */
static const char *img_content[] = {
  "Unknown",     // IMG_CONTENT_UNKNOWN
  "image",       // IMG_CONTENT_IMAGE
  "raw",         // IMG_CONTENT_RAW
  "attenuation", // IMG_CONTENT_ATTN
  "polarmap",    // IMG_CONTENT_POLARMAP
0};
/** Return pointer to content type of the image data.
    @return pointer to the data content type string.
    @sa imgdata
 */
char *imgContentDescr(
  /** IMG content code */
  imgcontent c
) {
  if(c<IMG_CONTENT_UNKNOWN || c>=IMG_CONTENT_LAST) return NULL;
  return (char*)img_content[c];
}
/*****************************************************************************/

/*****************************************************************************/
/** Initiate the IMG structure before any use.
    @sa imgFree, imgAllocate, imgRead
 */
void imgInit(
  /** Pointer to IMG structure. */
  IMG *img
) {
  if(img==NULL) return;
  img->dimt=img->dimx=img->dimy=img->dimz=0;
  iftInit(&img->ih);
  iftInit(&img->oh);
  img->content=IMG_CONTENT_UNKNOWN;
  img->modality=IMG_MODALITY_UNKNOWN;
  img->format=img->oformat=IMG_FORMAT_UNKNOWN;

  for(int i=0; i<20; i++) img->scanStart[i]=(char)0;
  for(int i=0; i<MAX_STUDYNR_LEN+1; i++) img->studyNr[i]=(char)0;


  img->sizex=img->sizey=img->sizez=0.0;
  img->gapx=img->gapy=img->gapz=0.0;
  img->xform[0]=img->xform[1]=0;
  for(int i=0; i<6; i++) img->quatern[i]=0.0;
  for(int i=0; i<12; i++) img->srow[i]=0.0;
  for(int i=0; i<6; i++) img->iop[i]=0.0;
  for(int i=0; i<3; i++) img->ipp[i]=0.0;
  for(int i=0; i<12; i++) img->mt[i]=0.0;

  img->isot=ISOTOPE_UNKNOWN;
  img->decayCorrection=DECAY_UNKNOWN;

  img->weighting=WEIGHTING_UNKNOWN;
  img->weight=img->prompts=img->randoms=NULL;

  img->cunit=img->tunit=UNIT_UNKNOWN;

  img->x1=img->x2=img->x=NULL;
  img->m=(float****)0; img->p=0;
  img->_z=(float****)0; img->_y=(float***)0; img->_x=(float**)0; img->_t=(float*)0;
}
/*****************************************************************************/

/*****************************************************************************/
/** Free memory allocated for IMG. All data is cleared.
    @sa imgInit, imgAllocate, imgRead
 */
void imgFree(
  /** Pointer to IMG structure. */
  IMG *img
) {
  if(img==NULL) return;
  iftFree(&img->ih);
  iftFree(&img->oh);
  free(img->weight); free(img->prompts); free(img->randoms);
  free(img->x1); free(img->x2); free(img->x);
  free(img->_t); free(img->_z); free(img->_x); free(img->_y);
  imgInit(img);
}
/*****************************************************************************/

/*****************************************************************************/
/** Allocates memory for image data. Old contents are not saved.
    @sa imgInit, imgFree, imgWrite, imgHasData, imgCopyHeader
    @return enum tpcerror (TPCERROR_OK when successful).
 */
int imgAllocate(
  /** Pointer to initialized image structure; old contents are deleted.
      @pre Initialize structure using imgInit() before this. */
  IMG *img,
  /** Nr of image planes to allocate. */
  const unsigned int dimz,
  /** Nr of image rows to allocate. */
  const unsigned int dimy,
  /** Nr of image columns to allocate. */
  const unsigned int dimx,
  /** Nr of image time frames (samples) to allocate. */
  const unsigned int dimt,
  /** Pointer to status data; enter NULL if not needed. */
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  if(verbose>0) {
    printf("%s(img, %u, %u, %u, %u)\n", __func__, dimz, dimy, dimx, dimt); fflush(stdout);}

  if(img==NULL || dimz<1 || dimy<1 || dimx<1 || dimt<1) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL);
    return(TPCERROR_FAIL);
  }

  /* Remove any previous contents */
  imgFree(img);

  /* Memory for frame times */
  int ret=0; {img->x1=calloc(dimt, sizeof(float)); if(img->x1==NULL) ret++;}
  if(!ret) {img->x2=calloc(dimt, sizeof(float)); if(img->x2==NULL) ret++;}
  if(!ret) {img->x=calloc(dimt, sizeof(float)); if(img->x==NULL) ret++;}
  if(ret) {
    imgFree(img);
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_OUT_OF_MEMORY);
    return(TPCERROR_OUT_OF_MEMORY);
  }

  /* Memory for weights */
  ret=0;   {img->weight=calloc(dimt, sizeof(float)); if(img->weight==NULL) ret++;}
  if(!ret) {img->prompts=calloc(dimt, sizeof(float)); if(img->prompts==NULL) ret++;}
  if(!ret) {img->randoms=calloc(dimt, sizeof(float)); if(img->randoms==NULL) ret++;}
  if(ret) {
    imgFree(img);
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_OUT_OF_MEMORY);
    return(TPCERROR_OUT_OF_MEMORY);
  }

  /* Memory for pixel matrix */
  ret=0;   {img->_t=(float*)malloc((size_t)dimz*dimy*dimx*dimt*sizeof(float)); if(img->_t==NULL) ret++;}
  if(!ret) {img->_x=(float**)malloc((size_t)dimz*dimy*dimx*sizeof(float*)); if(img->_x==NULL) ret++;}
  if(!ret) {img->_y=(float***)malloc((size_t)dimz*dimy*sizeof(float**)); if(img->_y==NULL) ret++;}
  if(!ret) {img->_z=(float****)malloc((size_t)dimz*sizeof(float***)); if(img->_z==NULL) ret++;}
  if(ret) {
    imgFree(img);
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_OUT_OF_MEMORY);
    return(TPCERROR_OUT_OF_MEMORY);
  }
  /* Set matrix data pointers */
  {
    float ***yptr, **xptr, *tptr;
    yptr=img->_y; xptr=img->_x; tptr=img->_t;
    for(unsigned int zi=0; zi<dimz; zi++) {
      img->_z[zi]=yptr;
      for(unsigned int yi=0; yi<dimy; yi++) {
        *yptr++=xptr;
        for(unsigned int xi=0; xi<dimx; xi++) {
          *xptr++=tptr; tptr+=dimt;
        }
      }
    }
  }
  img->m=img->_z;
  img->p=img->_t;
  /* Set all pixel values to zero */
  for(unsigned int zi=0; zi<dimz; zi++)
    for(unsigned int yi=0; yi<dimy; yi++)
      for(unsigned int xi=0; xi<dimx; xi++)
        for(unsigned int ti=0; ti<dimt; ti++)
          img->m[zi][yi][xi][ti]=(float)0.0;
  /* Set matrix dimensions */
  img->dimz=dimz; img->dimy=dimy; img->dimx=dimx; img->dimt=dimt;

  statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_OK);
  return(TPCERROR_OK);
}
/*****************************************************************************/

/*****************************************************************************/
/** Verify that image data actually contains pixel data.
    @sa imgHasTimes, imgNaNs, imgRead, imgWrite, imgContents
    @return 0, if no data, 1 if contains data.
 */
int imgHasData(
  /** Pointer to image structure. */
  IMG *img
) {
  if(img==NULL) return(0);
  if(img->dimz<1 || img->dimy<1 || img->dimx<1 || img->dimz<1) return(0);
  if(img->m==NULL || img->p==NULL) return(0);
  if(img->x1==NULL || img->x2==NULL) return(0);
  return(1);
}
/******************************************************************************/

/******************************************************************************/
/** Verify that IMG contains frame times.
    @sa imgHasData, imgInit, imgFree, imgRead, imgWrite, imgContents
    @return 0, if no frame times, 1 if frame times exist.
 */
int imgHasTimes(
  /** Pointer to image structure. */
  IMG *img
) {
  if(imgHasData(img)==0) return(0);
  for(int i=0; i<img->dimt; i++) if(img->x2[i]>0.00000001) return(1);
  return(0);
}
/******************************************************************************/

/******************************************************************************/
/** Verify that IMG contains prompts and randoms.
    @sa imgHasData, imgHasTimes, imgHasWeights, imgInit, imgFree, imgRead, imgWrite, imgContents
    @return 0, if no prompts or randoms, 1 if prompts exist, 2 if randoms exist, and 3 if both
     prompts and randoms exist.
 */
int imgHasCounts(
  /** Pointer to image structure. */
  IMG *img
) {
  if(img==NULL || img->dimt<1) return(0);
  int p=0, r=0;
  /* If image has only one frame, then any value > 0 is fine */
  if(img->dimt==1) {
    if(img->prompts[0]>0.000000001) p=1;
    if(img->randoms[0]>0.000000001) r=2;
    return(p+r);
  }
  /* When more frames, we check that frames have different count level */
  for(unsigned short int i=1; i<img->dimt; i++) {
    if(fabsf(img->prompts[i]-img->prompts[i-1])>0.001) p=1;
    if(fabsf(img->randoms[i]-img->randoms[i-1])>0.001) r=2;
    if((p+r)>2) break;
  }
  return(p+r);
}
/******************************************************************************/

/******************************************************************************/
/** Check if image contains weights as indicated by the 'weighting' field.
    @sa imgHasCounts, imgHasData, imgHasTimes, imgInit, tacIsWeighted
    @return Returns 1 if weighted, and 0 if not or in case of an error.
 */
int imgHasWeights(
  /** Pointer to image structure. */
  IMG *img
) {
  if(img==NULL || img->dimt<1) return(0);
  if(img->weighting==WEIGHTING_UNKNOWN || img->weighting==WEIGHTING_OFF) return(0);
  if(img->weighting<WEIGHTING_LAST) return(1);
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/** Print general information on the contents of the IMG data structure.
    @sa imgInit, imgRead, imgHasData, imgHasTimes
 */
void imgContents(
  /** Pointer to IMG structure. */
  IMG *img,
  /** File pointer for output; usually stdout. */
  FILE *fp
) {
  if(fp==NULL) return;
  if(img==NULL) {fprintf(fp, "IMG pointer is NULL\n"); fflush(fp); return;}

  fprintf(fp, "modality: %s\n", imgModalityDescr(img->modality));
  fprintf(fp, "content: %s\n", imgContentDescr(img->content));
  fprintf(fp, "format: %s\n", imgFormatDescr(img->format));
  if(img->oformat!=IMG_FORMAT_UNKNOWN)
    fprintf(fp, "output_format: %s\n", imgFormatDescr(img->oformat));

  fprintf(fp, "hasData := %d\n", imgHasData(img));
  fprintf(fp, "hasTimes := %d\n", imgHasTimes(img));
  fprintf(fp, "hasCounts := %d\n", imgHasCounts(img));

  if(img->scanStart[0]) fprintf(fp, "scanStart: %s\n", img->scanStart);

  fprintf(fp, "dimensions: %u x %u x %u x %u\n", img->dimx, img->dimy, img->dimz, img->dimt);
  if(img->sizex!=0.0 || img->sizey!=0.0 || img->sizez!=0.0)
    fprintf(fp, "voxel_size: %g x %g x %g\n", img->sizex, img->sizey, img->sizez);
  if(img->gapx!=0.0 || img->gapy!=0.0 || img->gapz!=0.0)
    fprintf(fp, "voxel_gaps: %g x %g x %g\n", img->gapx, img->gapy, img->gapz);

  fprintf(fp, "qform := %d\n", img->xform[0]);
  fprintf(fp, "sform := %d\n", img->xform[1]);
  if(floatNonzeroes(img->quatern, 6)>0) {
    fprintf(fp, "quatern_b := %g\n", img->quatern[0]);
    fprintf(fp, "quatern_c := %g\n", img->quatern[1]);
    fprintf(fp, "quatern_d := %g\n", img->quatern[2]);
    fprintf(fp, "quatern_x_shift := %g\n", img->quatern[3]);
    fprintf(fp, "quatern_y_shift := %g\n", img->quatern[4]);
    fprintf(fp, "quatern_z_shift := %g\n", img->quatern[5]);
  }
  if(floatNonzeroes(img->srow, 12)>0) {
    for(int i=0; i<4; i++) fprintf(fp, "srow_x[%d] := %g\n", i, img->srow[i]);
    for(int i=0; i<4; i++) fprintf(fp, "srow_y[%d] := %g\n", i, img->srow[4+i]);
    for(int i=0; i<4; i++) fprintf(fp, "srow_z[%d] := %g\n", i, img->srow[8+i]);
  }
  if(floatNonzeroes(img->iop, 6)>0) {
    fprintf(fp, "image_orientation_patient := %g", img->iop[0]);
    for(int i=1; i<6; i++) fprintf(fp, ", %g", img->iop[i]);
    fprintf(fp, "\n");
  }
  if(floatNonzeroes(img->ipp, 3)>0) {
    fprintf(fp, "image_position_patient := %g", img->ipp[0]);
    for(int i=1; i<3; i++) fprintf(fp, ", %g", img->ipp[i]);
    fprintf(fp, "\n");
  }
  if(floatNonzeroes(img->mt, 3)>0) {
    fprintf(fp, "image_matrix_transformation_parameters := %g", img->mt[0]);
    for(int i=1; i<12; i++) fprintf(fp, ", %g", img->mt[i]);
    fprintf(fp, "\n");
  }

  if(img->isot!=ISOTOPE_UNKNOWN) fprintf(fp, "isotope: %s\n", isotopeName(img->isot));
  if(img->decayCorrection==DECAY_CORRECTED) fprintf(fp, "physical_decay: corrected\n");
  else if(img->decayCorrection==DECAY_NOTCORRECTED) fprintf(fp, "physical_decay: not corrected\n");

  if(imgHasWeights(img)) fprintf(fp, "weighting: no\n");
  else fprintf(fp, "weighting: yes\n");

  if(img->cunit!=UNIT_UNKNOWN) fprintf(fp, "pixel_unit := %s\n", unitName(img->cunit));
  if(img->tunit!=UNIT_UNKNOWN) fprintf(fp, "time_unit := %s\n", unitName(img->tunit));

  fprintf(fp, "original header: %d items\n", img->ih.keyNr);

  fflush(fp);
  return;
}
/*****************************************************************************/

/*****************************************************************************/
/** Searches the image data for missing pixel values, optionally setting those to zero.
   @sa imgHasData
   @return the number of missing pixel values.
 */
unsigned long long imgNaNs(
  /** Pointer to IMG structure. */
  IMG *img,
  /** Set (1) or do not set (0) missing pixels to zero. */
  int fix
) {
  if(img==NULL) return(0);
  unsigned long long i, n=0, pxlNr=img->dimt*img->dimz*img->dimy*img->dimx;
  for(i=0; i<pxlNr; i++) {
    if(!isfinite(img->p[i])) {
      n++;
      if(fix!=0) img->p[i]=0.0;
    }
  }
  return(n);
}
/*****************************************************************************/

/*****************************************************************************/
/** Searches the min and max pixel value in the image data.
   @sa imgXRange, imgInit, imgFree, imgHasData
   @return enum tpcerror (TPCERROR_OK when successful).
 */
int imgMinMax(
  /** Pointer to IMG structure from where min and max pixel values are searched. */
  IMG *img,
  /** Pointer to min pixel value; Enter NULL if not needed. */
  float *minvalue,
  /** Pointer to max pixel value; Enter NULL if not needed. */
  float *maxvalue
) {
  if(img==NULL || !imgHasData(img)) return(TPCERROR_FAIL);
  float f1, f2;
  f1=f2=nanf("");
  if(minvalue!=NULL) *minvalue=f1;
  if(maxvalue!=NULL) *maxvalue=f2;
  unsigned long long i, n=img->dimt*img->dimz*img->dimy*img->dimx;
  // search first valid number
  for(i=0; i<n; i++) if(isfinite(img->p[i])) break;
  if(i==n) return(TPCERROR_MISSING_DATA);
  f1=f2=img->p[i++];
  for(; i<n; i++) {
    if(img->p[i]>f2) f2=img->p[i];
    else if(img->p[i]<f1) f1=img->p[i];
  }
  if(minvalue!=NULL) *minvalue=f1;
  if(maxvalue!=NULL) *maxvalue=f2;
  return(TPCERROR_OK);
}
/*****************************************************************************/

/*****************************************************************************/
/** Get the range of x values (times) in the image data.
    @details Data is not modified. Data does not need to be sorted.
   @sa imgMinMax, imgHasData
   @return enum tpcerror (TPCERROR_OK when successful).
 */
int imgXRange(
  /** Pointer to IMG structure. */
  IMG *img,
  /** Pointer to variable for min x value (NULL if not needed). */
  double *xmin,
  /** Pointer to variable for max x value (NULL if not needed). */
  double *xmax
) {
  if(xmin!=NULL) *xmin=nan("");
  if(xmax!=NULL) *xmax=nan("");
  if(img==NULL || !imgHasData(img)) return(TPCERROR_FAIL);
  float f1=nanf(""), f2=nanf("");
  for(int i=0; i<img->dimt; i++) {
    if(isfinite(img->x1[i])) {
      if(isnan(f1) || img->x1[i]<f1) f1=img->x1[i];
    }
    if(isfinite(img->x2[i])) {
      if(isnan(f2) || img->x2[i]>f2) f2=img->x2[i];
    }
  }
  if(xmin!=NULL) *xmin=f1;
  if(xmax!=NULL) *xmax=f2;
  if(isnan(f1) || isnan(f2)) return(TPCERROR_FAIL);
  return(TPCERROR_OK);
}
/*****************************************************************************/

/*****************************************************************************/
