/** @file imageheader.c
    @brief Process header information in IMG structure.
    @copyright (c) Turku PET Centre
    @author Vesa Oikonen
 */
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include "tpcimage.h"
/*****************************************************************************/

/*****************************************************************************/
/** Copy information fields from one IMG to another.
    Image dimensions, pixel data, and frame information are not copied.
    @sa imgAllocate, imgInit, imgFree
    @return enum tpcerror (TPCERROR_OK when successful).
 */
int imgCopyHeader(
  /** Pointer to source image structure. */
  IMG *img1,
  /** Pointer to initiated target image structure. */
  IMG *img2
) {
  if(img1==NULL || img2==NULL) return(TPCERROR_FAIL);
  if(!imgHasData(img1) || !imgHasData(img2)) return(TPCERROR_FAIL);
  
  strlcpy(img2->studyNr, img1->studyNr, MAX_STUDYNR_LEN+1);
  img2->isot=img1->isot;
  img2->decayCorrection=img1->decayCorrection;
  strlcpy(img2->scanStart, img1->scanStart, 20);
  img2->content=img1->content;
  img2->modality=img1->modality;
  img2->format=img1->format;
  img2->oformat=img1->oformat;
  img2->sizex=img1->sizex;
  img2->sizey=img1->sizey;
  img2->sizez=img1->sizez;
  img2->gapx=img1->gapx;
  img2->gapy=img1->gapy;
  img2->gapz=img1->gapz;
  img2->xform[0]=img1->xform[0];
  img2->xform[1]=img1->xform[1];
  for(int i=0; i<6; i++) img2->quatern[i]=img1->quatern[i];
  for(int i=0; i<12; i++) img2->srow[i]=img1->srow[i];
  for(int i=0; i<6; i++) img2->iop[i]=img1->iop[i];
  for(int i=0; i<3; i++) img2->ipp[i]=img1->ipp[i];
  for(int i=0; i<12; i++) img2->mt[i]=img1->mt[i];
  img2->cunit=img1->cunit;
  img2->tunit=img1->tunit;

  return(TPCERROR_OK);
}
/*****************************************************************************/

/*****************************************************************************/
/** Fill/edit IMG output header.
    @remark Stub function.
    @sa imgInit, imgWrite, imgFree
    @return enum tpcerror (TPCERROR_OK when successful).
 */
int imgFillOHeader(
  /** Pointer to image structure. */
  IMG *img,
  /** Pointer to status data; enter NULL if not needed. */
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  if(verbose>0) {printf("%s(img)\n", __func__); fflush(stdout);}

  if(img==NULL || !imgHasData(img)) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_NO_DATA);
    return(TPCERROR_NO_DATA);
  }

  /* If output header is empty then copy input header;
     if file formats match, copy directly, otherwise convert contents */
  if(img->oh.keyNr==0 && img->ih.keyNr>0) {
    if(img->format==img->oformat) {
      if(verbose>1) printf("copying header\n");
      if(iftDuplicate(&img->ih, &img->oh)!=TPCERROR_OK) {
        statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_INVALID_HEADER);
        return(TPCERROR_INVALID_HEADER);
      }
    } else {
      if(verbose>1) printf("converting header\n");

    }
  }

  /* Set fields from IMG structure */

  iftDeleteKey(&img->oh, "dimx");
  if(iftPutUInt(&img->oh, "dimx", img->dimz, 0, NULL)!=0) return(TPCERROR_INVALID_HEADER);
  iftDeleteKey(&img->oh, "dimy");
  if(iftPutUInt(&img->oh, "dimy", img->dimz, 0, NULL)!=0) return(TPCERROR_INVALID_HEADER);
  iftDeleteKey(&img->oh, "dimz");
  if(iftPutUInt(&img->oh, "dimz", img->dimz, 0, NULL)!=0) return(TPCERROR_INVALID_HEADER);
  iftDeleteKey(&img->oh, "dimt");
  if(iftPutUInt(&img->oh, "dimt", img->dimz, 0, NULL)!=0) return(TPCERROR_INVALID_HEADER);

  if(img->modality!=IMG_MODALITY_UNKNOWN) {
    iftDeleteKey(&img->oh, "modality");
    if(iftPut(&img->oh, "modality", imgModalityDescr(img->modality), 0, NULL)!=0) 
      return(TPCERROR_INVALID_HEADER);
  }
  if(img->content!=IMG_CONTENT_UNKNOWN) {
    iftDeleteKey(&img->oh, "content");
    if(iftPut(&img->oh, "content", imgContentDescr(img->content), 0, NULL)!=0) 
      return(TPCERROR_INVALID_HEADER);
  }
  if(img->format!=IMG_FORMAT_UNKNOWN) {
    iftDeleteKey(&img->oh, "format");
    if(iftPut(&img->oh, "format", imgFormatDescr(img->format), 0, NULL)!=0) 
      return(TPCERROR_INVALID_HEADER);
  }
  if(img->isot!=ISOTOPE_UNKNOWN) {
    iftDeleteKey(&img->oh, "isotope");
    if(iftPut(&img->oh, "isotope", isotopeName(img->isot), 0, NULL)!=0)
      return(TPCERROR_INVALID_HEADER);
  }
  if(img->decayCorrection!=DECAY_UNKNOWN) {
    iftDeleteKey(&img->oh, "decay_correction");
    char buf[8];
    if(img->decayCorrection==DECAY_CORRECTED) strcpy(buf, "y"); else strcpy(buf, "n");
    if(iftPut(&img->oh, "decay_correction", buf, 0, NULL)!=0)
      return(TPCERROR_INVALID_HEADER);
  }
  if(img->cunit!=UNIT_UNKNOWN) {
    iftDeleteKey(&img->oh, "pixel_unit");
    if(iftPut(&img->oh, "pixel_unit", unitName(img->cunit), 0, NULL)!=0)
      return(TPCERROR_INVALID_HEADER);
  }
  if(img->tunit!=UNIT_UNKNOWN) {
    iftDeleteKey(&img->oh, "time_unit");
    if(iftPut(&img->oh, "time_unit", unitName(img->tunit), 0, NULL)!=0)
      return(TPCERROR_INVALID_HEADER);
  }

  statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_OK);
  return(TPCERROR_OK);
}
/*****************************************************************************/

/*****************************************************************************/

/*****************************************************************************/

/*****************************************************************************/
