/** @file imagesif.c
    @brief Scan information file interface with IMG structure.
    @copyright (c) Turku PET Centre
    @author Vesa Oikonen
 */
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include "tpcimage.h"
/*****************************************************************************/

/*****************************************************************************/
/** Set SIF contents based on data in IMG.
    @todo Add tests.
    @sa imgFromSIF, imgInit, tacInit, imgWrite, tacWrite, tacWriteSIF
    @return enum tpcerror (TPCERROR_OK when successful).
 */ 
int imgToSIF(
  /** Pointer to IMG struct from which content is copied to SIF. */
  IMG *img,
  /** Pointer to SIF, which actually is stored as a TAC format.
      Previous contents are optionally kept.
      @pre Must be initialized with tacInit().
   */
  TAC *sif,
  /** Select whether header contents are copied (1) or not copied (0) to SIF. */
  int copy_header,
  /** Select whether frame times are copied (1) or not copied (0) to SIF. */
  int copy_frames,
  /** Select whether counts are copied (1) or not copied (0) to SIF,
      or created if they do not exist (2). */
  int copy_counts,
  /** Verbose level; if zero, then nothing is printed to stderr or stdout. */
  int verbose
) {
  if(verbose>0) {
    printf("%s(img, sif, %d, %d, %d, ...)\n", __func__, copy_header, copy_frames, copy_counts);
    fflush(stdout);
  }

  if(img==NULL || sif==NULL) return(TPCERROR_FAIL);

  /* Verify that IMG contains frame times */
  if(!imgHasTimes(img)) {
    if(verbose>0) printf("  image does not contain frame times.\n");
    /* If not, then frame times cannot be copied */
    copy_frames=0;
    /* and counts can not be created */
    if(copy_counts==2) copy_counts=1;
  }

  /* Check if count data needs to be created */
  if(copy_counts==2 && imgHasCounts(img)!=0) copy_counts=1;

  /* Verify that IMG contains isotope information */
  if(img->isot==ISOTOPE_UNKNOWN) {
    if(verbose>0) printf("  image does not contain isotope halflife.\n");
    /* not, then count data can not be created */
    if(copy_counts==2) copy_counts=1;
  }

  /* Allocate memory for SIF if necessary */
  int ret=0;
  if(sif->sampleNr<1 || sif->tacNr<1) {
    ret=tacAllocate(sif, img->dimt, 2);
    sif->sampleNr=img->dimt; sif->tacNr=2;
  } else if((copy_frames || copy_counts) && sif->sampleNr!=img->dimt) {
    if(sif->sampleNr<img->dimt)
      ret=tacAllocateMoreSamples(sif, img->dimt-sif->sampleNr);
    sif->sampleNr=img->dimt;
  }
  if((copy_frames || copy_counts) && sif->tacNr<2) {
    ret=tacAllocateMore(sif, 2-sif->tacNr);
    sif->tacNr=2;
  }
  if(ret!=0) return(TPCERROR_OUT_OF_MEMORY);

  /* Copy SIF header */
  if(copy_header) {
    if(verbose>1) printf("  copying header fields.\n");
    tacSetHeaderScanstarttime(&sif->h, img->scanStart);
    tacSetHeaderStudynr(&sif->h, img->studyNr);
    tacSetIsotope(sif, img->isot);
  }

  /* Copy frame times */
  if(copy_frames) {
    if(verbose>1) printf("  copying frame times.\n");
    for(int i=0; i<img->dimt; i++) {
      sif->x1[i]=img->x1[i]; sif->x2[i]=img->x2[i]; sif->x[i]=img->x[i];
    }
    sif->tunit=UNIT_SEC;
  }

  /* Copy counts */
  if(copy_counts==1) {
    if(verbose>1) printf("  copying count data.\n");
    for(int i=0; i<img->dimt; i++) {
      sif->c[0].y[i]=img->prompts[i];
      sif->c[1].y[i]=img->randoms[i];
    }
  }

  /* Create counts, if required and possible */
  if(copy_counts==2) {
    if(verbose>1) printf("  creating count data.\n");

    fprintf(stderr, "Warning: imgToSIF() does not yet guess counts.\n"); fflush(stderr);

    for(int i=0; i<img->dimt; i++) {
      sif->c[0].y[i]=0.0;
      sif->c[1].y[i]=0.0;
    }
  }

  return(TPCERROR_OK);
}
/*****************************************************************************/

/*****************************************************************************/
/** Copy information from SIF into IMG.
    @todo Add tests.
    @sa imgToSIF, imgInit, tacInit, imgWrite, imgRead, tacWriteSIF
    @return enum tpcerror (TPCERROR_OK when successful).
 */ 
int imgFromSIF(
  /** Pointer to IMG struct into which content is copied from SIF. */
  IMG *img,
  /** Pointer to SIF, which actually is stored as a TAC format. */
  TAC *sif,
  /** Select whether header contents are copied (1) or not copied (0) from SIF. */
  int copy_header,
  /** Select whether frame times are copied (1) or not copied (0) from SIF. */
  int copy_frames,
  /** Select whether counts are copied (1) or not copied (0) from SIF. */
  int copy_counts,
  /** Verbose level; if zero, then nothing is printed to stderr or stdout */
  int verbose
) {
  if(verbose>0) {
    printf("%s(img, sif, %d, %d, %d, ...)\n", __func__, copy_header, copy_frames, copy_counts);
    fflush(stdout);
  }

  if(sif==NULL || img==NULL) return(TPCERROR_FAIL);

  if(copy_header) {
    if(verbose>1) printf("  copying header.\n");
    int ret=tacGetHeaderScanstarttime(&sif->h, img->scanStart, NULL);
    if(ret!=TPCERROR_OK && verbose>0) fprintf(stderr, "Warning: cannot read scan start time in SIF\n");
    img->isot=tacGetIsotope(sif);
    if(img->isot==ISOTOPE_UNKNOWN && verbose>0) fprintf(stderr, "Warning: cannot read isotope in SIF\n");
    ret=tacGetHeaderStudynr(&sif->h, img->studyNr, NULL);
  }

  if(copy_frames) {
    if(verbose>1) printf("  copying frame times.\n");
    if(!sif->isframe || sif->sampleNr!=img->dimt) return(TPCERROR_INVALID_FORMAT);
    for(int fi=0; fi<img->dimt; fi++) {
      img->x1[fi]=sif->x1[fi];
      img->x2[fi]=sif->x2[fi];
      img->x[fi]=0.5*(img->x1[fi]+img->x2[fi]);
    }
    img->tunit=sif->tunit;
  }

  if(copy_counts) {
    if(verbose>1) printf("  copying count data.\n");
    if(sif->format!=TAC_FORMAT_SIF || !sif->isframe || sif->sampleNr!=img->dimt)
      return(TPCERROR_INVALID_FORMAT);
    if(sif->tacNr<4) return(TPCERROR_INVALID_FORMAT);
    for(int fi=0; fi<img->dimt; fi++) {
      img->prompts[fi]=sif->c[0].y[fi];
      img->randoms[fi]=sif->c[1].y[fi];
    }
    img->weighting=WEIGHTING_OFF;
    if(sif->weighting!=WEIGHTING_OFF && sif->weighting!=WEIGHTING_UNKNOWN) {
      for(int fi=0; fi<img->dimt; fi++)
        img->weight[fi]=sif->w[fi];
      img->weighting=sif->weighting;
    }
  }

  return(TPCERROR_OK);
}
/*****************************************************************************/

/*****************************************************************************/
