/** @file tpcimage.h
 *  @brief Header file for libtpcimage.
 *  @details Header file for image data processing library libtpcimage.
 *  @author Vesa Oikonen
 */
#ifndef _TPCIMAGE_H_
#define _TPCIMAGE_H_
/*****************************************************************************/

/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
/*****************************************************************************/
#include "tpcextensions.h"
#include "tpcift.h"
#include "tpcisotope.h"
#include "tpctac.h" // SIF is TAC
/*****************************************************************************/
#include "tpcecat.h"
#include "tpcdcm.h"
#include "tpcnifti.h"
/*****************************************************************************/

/*****************************************************************************/
/** @public Image file format codes for IMG data structure.
    @sa img_format, image.c
 */
typedef enum {
  IMG_FORMAT_UNKNOWN,     ///< Unknown format
  IMG_FORMAT_DICOM,       ///< DICOM
  IMG_FORMAT_E63,         ///< ECAT 6.3
  IMG_FORMAT_E7,          ///< ECAT 7.* 3D
  IMG_FORMAT_E7_2D,       ///< ECAT 7.* 2D
  IMG_FORMAT_POLARMAP,    ///< ECAT polar map
  IMG_FORMAT_ANA,         ///< Analyze big-endian
  IMG_FORMAT_ANA_L,       ///< Analyze little-endian
  IMG_FORMAT_INTERFILE,   ///< Interfile
  IMG_FORMAT_NIFTI_1D,    ///< NIfTI-1 dual-file format
  IMG_FORMAT_NIFTI_1S,    ///< NIfTI-1 single-file format
  IMG_FORMAT_NIFTI_2D,    ///< NIfTI-2 dual-file format
  IMG_FORMAT_NIFTI_2S,    ///< NIfTI-2 single-file format
  IMG_FORMAT_MICROPET,    ///< MicroPET
  IMG_FORMAT_FLAT,        ///< Flat format
  IMG_FORMAT_LAST         ///< End of list
} imgformat;
/*****************************************************************************/

/*****************************************************************************/
/** @public Image modality codes for IMG data structure.
    @sa img_modality, image.c
 */
typedef enum {
  IMG_MODALITY_UNKNOWN, ///< Unknown modality
  IMG_MODALITY_PET,     ///< PET
  IMG_MODALITY_SPECT,   ///< SPECT
  IMG_MODALITY_CT,      ///< CT
  IMG_MODALITY_MRI,     ///< MRI
  IMG_MODALITY_LAST     ///< End of list
} imgmodality;
/*****************************************************************************/

/*****************************************************************************/
/** @public Image data content codes for IMG data structure.
    @sa img_content, image.c
 */
typedef enum {
  IMG_CONTENT_UNKNOWN,   ///< Unknown data content
  IMG_CONTENT_IMAGE,     ///< Image data
  IMG_CONTENT_RAW,       ///< Sinogram, normalization
  IMG_CONTENT_ATTN,      ///< Attenuation data
  IMG_CONTENT_POLARMAP,  ///< Polarmap
  IMG_CONTENT_LAST       ///< End of list
} imgcontent;
/*****************************************************************************/

/*****************************************************************************/
/** 4D IMG data structure for dynamic image */
typedef struct {

  /** Study identification code, i.e. (consequential) study number */
  char studyNr[MAX_STUDYNR_LEN+1];

  /** Isotope. */
  isotope isot;

  /** Status of decay correction. */
  decaycorrection decayCorrection;

  /** Scan start date and time, YYYY-MM-DD hh:mm:ss */
  char scanStart[20];

  /** Data content @sa imgcontent, imgContentDescr */
  imgcontent content;

  /** Modality @sa imgmodality, imgModalityDescr */
  imgmodality modality;

  /** Original data format @sa imgformat, imgFormatDescr */
  imgformat format;
  /** Output data format @sa imgformat, imgFormatDescr */
  imgformat oformat;

  /* Coordination */

  /** Dimension of Time (f/t). */
  unsigned short int dimt;
  /** Dimension of Column (c/x). */
  unsigned short int dimx;
  /** Dimension of Row (r/y). */
  unsigned short int dimy;
  /** Dimension of Plane (p/z). */
  unsigned short int dimz;

  /** Pixel size (mm). */
  float sizex;
  /** Pixel size (mm). */
  float sizey;
  /** Pixel size (mm). */
  float sizez;

  /** Gaps between pixels in x direction (mm); negative value means overlap. */
  float gapx;
  /** Gaps between pixels in y direction (mm); negative value means overlap. */
  float gapy;
  /** Gaps between pixels in z direction (mm); negative value means overlap. */
  float gapz;

  /** NIfTI-1 XForm codes qform and sform. */
  short int xform[2];
  /** NIfTI-1 Quaternion parameters b, c, d, and x, y, z shift. */
  float quatern[6];
  /** NIfTI-1 affine transform parameters for the 1st, 2nd and 3rd row, x[4], y[4], and z[4]. */
  float srow[12];
  /** DICOM Image orientation patient (0x0020, 0x0037), [xa, xb, xc, ya, yb, yc]. */
  float iop[6];
  /** DICOM Image position patient (0x0020, 0x0032), [Xd, Yd, Zd]. */
  float ipp[3];
  /** ECAT 7 image matrix transformation parameters (1,1), (1,2), (1,3), (1,4), ... (3,4);
      or XFORM matrix. */
  float mt[12];

  /* Pixel data */

/// @cond
  /** 'Hidden' pointer to allocated data. */
  float *_t;
  /** 'Hidden' pointer to allocated data. */
  float **_x;
  /** 'Hidden' pointer to allocated data. */
  float ***_y;
  /** 'Hidden' pointer to allocated data. */
  float ****_z;
/// @endcond

  /* Pointers for data to be used */
  /** Pointer to image data in matrix format; use as m[plane][row][column][frame]. */
  float ****m;
  /** Pointer to image data in array format. 

      Example 1: to get pointer to the TAC of concentrations of pixel at 10th plane (z=9), 
      6th row from up (y=5), and 3rd column from left (x=2), use
      unsigned int z=9, y=5, x=2;
      float *c=img->p[z*dimy*dimx*dimt + y*dimx*dimt + x*dimt].

      Example 2: to get the value of pixel at 2nd plane, 3rd row, 4th column, and first time frame,
      use:
      unsigned int z=1, y=2, x=3, f=0;
      float c=img->p[z*dimy*dimx*dimt + y*dimx*dimt + x*dimt + f];
 */
  float *p;

  /*
   *  Frame times
   */
  /** Frame start time (sec) */
  float *x1;
  /** Frame end time (sec) */
  float *x2;
  /** Frame mid time (sec) */
  float *x;

  /*
   *  Frame weights
   */
  /** Weighting: WEIGHTING_UNKNOWN, WEIGHTING_OFF, WEIGHTING_ON_COUNTS, ...
      @sa imgHasWeights */
  weights weighting;
  /** Frame weight factor */
  float *weight;
  /** Prompts / frame */
  float *prompts;
  /** Randoms (delayed) / frame */
  float *randoms;

  /*
   *  Data units
   */
  /** Unit of pixel values (concentrations): UNIT_* @sa unitName */
  unit cunit;
  /** Unit of sample times: UNIT_* @sa unitName */
  unit tunit;

  /** Original header information. */
  IFT ih;
  /** Header information for output. */
  IFT oh;

} IMG;
/*****************************************************************************/

/*****************************************************************************/
/* image */
extern void imgInit(IMG *image);
extern void imgFree(IMG *image);
extern int imgAllocate(
  IMG *img, const unsigned int dimz, const unsigned int dimy,
  const unsigned int dimx, const unsigned int dimt, TPCSTATUS *status
);
extern char *imgContentDescr(imgcontent c);
extern char *imgModalityDescr(imgmodality c);
extern int imgHasData(IMG *img);
extern int imgHasTimes(IMG *img);
extern int imgHasCounts(IMG *img);
extern int imgHasWeights(IMG *img);
extern void imgContents(IMG *img, FILE *fp);
extern unsigned long long imgNaNs(IMG *img, int fix);
extern int imgMinMax(IMG *img, float *minvalue, float *maxvalue);
extern int imgXRange(IMG *img, double *xmin, double *xmax);
/*****************************************************************************/

/*****************************************************************************/
/* imageheader */
extern int imgCopyHeader(IMG *img1, IMG *img2);
extern int imgFillOHeader(IMG *img, TPCSTATUS *status);
/*****************************************************************************/

/*****************************************************************************/
/* imageio */
extern char *imgFormatDescr(imgformat c);
extern char *imgDefaultExtension(imgformat c);
extern int imgRead(IMG *img, const char *fname, TPCSTATUS *status);
extern int imgWrite(IMG *img, const char *fname, TPCSTATUS *status);
extern imgformat imgFormatFromFName(const char *fname, TPCSTATUS *status);
extern imgformat imgFormatIdentify(const char *s);
/*****************************************************************************/

/*****************************************************************************/
/* imagedcm */
extern int imgReadDICOM(IMG *img, const char *fname, TPCSTATUS *status);
extern int imgWriteDICOM(IMG *img, const char *fname, TPCSTATUS *status);
/*****************************************************************************/

/*****************************************************************************/
/* imagenii */
extern int niftiCreateFNames(
  const char *filename, char *hdrfile, char *imgfile, char *siffile, int fileformat
);
extern int imgReadNifti(IMG *img, const char *fname, TPCSTATUS *status);
extern int imgWriteNifti(IMG *img, const char *fname, TPCSTATUS *status);
extern int imgSetNiftiHeader(IMG *img, NIFTI_DSR *dsr, int verbose);
extern int imgGetNiftiHeader(IMG *img, NIFTI_DSR *dsr, int verbose);
/*****************************************************************************/

/*****************************************************************************/
/* imagesif */
extern int imgToSIF(IMG *img, TAC *sif, int copy_header, int copy_frames, int copy_counts, int verbose);
extern int imgFromSIF(IMG *img, TAC *sif, int copy_header, int copy_frames, int copy_counts, int verbose);
/*****************************************************************************/

/*****************************************************************************/
/* imagecomp */
extern int imgCompareMatrixSize(IMG *d1, IMG *d2);
extern int imgCompareUnit(IMG *d1, IMG *d2, TPCSTATUS *status);
extern int imgCompareConc(
  IMG *d1, IMG *d2, const float test_abs, const float test_rel, TPCSTATUS *status);
extern int imgCompareTimes(
  IMG *d1, IMG *d2, const float test_abs, const float test_rel, TPCSTATUS *status);
/*****************************************************************************/

/*****************************************************************************/
#endif /* TPCIMAGE */
