/** @file aic.c
 *  @brief Functions for using Akaike's information criteria.
 */
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <time.h>
#include <string.h>
/*****************************************************************************/
#include "tpcextensions.h"
/*****************************************************************************/
#include "tpcmodels.h"
/*****************************************************************************/

/*****************************************************************************/
/** @brief Calculate corrected AIC.

    Calculate AICc in the special case of sum-of-squares optimization from 
    the SS, number of fitted samples, and number of fitted parameters.

    If variance is different between the data points, weighted SS must be given.

    @sa parFreeNr, tacWSampleNr
    @return Returns the AIC value, or NaN in case of invalid input values.
*/
double aicSS(
  /** Sum-of-Squares of the fit (weighted, if necessary). */
  double ss,
  /** Sample size, i.e. number of fitted samples; do not include samples with zero weight. */
  const unsigned int n,
  /** Number of fitted model parameters; do not include fixed parameters;
      AICc calculation is valid only when (n-k)>1. */
  const unsigned int k
) {
  if(!(ss>=0.0) || n<1 || (n-k)<2) return(nan(""));

  int dr=n-k-1; 
  int dv=2*k*(k+1);
  double aic=0.0, bias_adj=0.0, css;
  if(dr>0) bias_adj=((double)dv)/((double)dr); else bias_adj=0.0;
  if(ss<1.0e-50) css=1.0e-50; else css=ss; /* Because log(0) is an error */
  aic= n*log(css/(double)n) + 2.0*(double)k + bias_adj;
  return(aic);
}
/*****************************************************************************/

/*****************************************************************************/
/** @brief Calculate the number of free parameters.

    Model parameters can be fixed by setting lower and upper limit to equal values.
    This function simply checks the limits for each parameter.
    
    @return Returns the number of free parameters.
    @author Vesa Oikonen
    @sa aicSS, tacWSampleNr
 */
unsigned int parFreeNr(
  /** Nr of parameters */
  const unsigned int n,
  /** Lower limits (array of length n) */
  double *pLower,
  /** Upper limits (array of length n) */
  double *pUpper
) {
  if(n==0 || pLower==NULL || pUpper==NULL) return(0);
  unsigned int nf=0;
  for(unsigned int i=0; i<n; i++) {
    double range=pUpper[i]-pLower[i];
    if(range>1.0E-10) nf++;
  }
  return(nf);
}
/*****************************************************************************/

/*****************************************************************************/
