/** @file optcrit.c
 *  @brief Optimality Criteria for other libraries.
 */
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <time.h>
#include <string.h>
/*****************************************************************************/
#include "tpcextensions.h"
/*****************************************************************************/
#include "tpcmodels.h"
/*****************************************************************************/

/*****************************************************************************/
/*! @cond PRIVATE */
/** One item for table of optimality criteria. */
typedef struct TPC_OPTCRIT {
  /** Human-readable name for optimality criterion. */
  char *code;
  /** Enumerated ID of the criterion. */
  optimality_criterion id;
  /** Description for humans. */
  char *desc;
} TPC_OPTCRIT;
/*****************************************************************************/
/** Table of optimality criteria. */
static TPC_OPTCRIT tpc_optcrit[]={
  {"unknown", OPTCRIT_UNKNOWN,  "Unknown optimality criterion"},
  {"OLS",     OPTCRIT_OLS,      "Ordinary Least Squares"},
  {"LMS",     OPTCRIT_LMS,      "Least Median of Squares"},
  {"LAD",     OPTCRIT_LAD,      "Least Absolute Deviations"},
  {"MAD",     OPTCRIT_MAD,      "Median Absolute Deviation"},
  {"ODR",     OPTCRIT_ODR,      "Orthogonal deviations"},
  {"",        0,                ""}
};
/*! @endcond */
/*****************************************************************************/

/*****************************************************************************/
/** Number of optimality criteria.
    @return Nr of criteria.
 */
unsigned int optcritNr()
{
  unsigned int i=0;
  while(tpc_optcrit[i].code[0]!='\0') i++;
  return(i);
}
/*****************************************************************************/

/*****************************************************************************/
/** Get string representation of an optimality criterion.
    @return pointer to string representation of the code, or NULL if not identified.
    @sa optcritDesc, optcritId, modelCode
 */
char *optcritCode(
  /** Optimality criterion. */
  optimality_criterion id
) {
  for(unsigned int i=0; i<optcritNr(); i++) if(id==tpc_optcrit[i].id) return(tpc_optcrit[i].code);
  return((char*)NULL);
}
/*****************************************************************************/

/*****************************************************************************/
/** Get description of an optimality criterion.
    @return pointer to string, or NULL if not identified.
    @sa optcritCode, optcritId
 */
char *optcritDesc(
  /** Optimality criterion. */
  optimality_criterion id
) {
  for(unsigned int i=0; i<optcritNr(); i++) if(id==tpc_optcrit[i].id) return(tpc_optcrit[i].desc);
  return((char*)NULL);
}
/*****************************************************************************/

/*****************************************************************************/
/** Get the Id for the string representation of optimality criterion code string.
    @return Enumerated optimality criterion id, OPTCRIT_UNKNOWN, if unknown code.
    @sa optcritCode, optcritDesc
 */
optimality_criterion optcritId(
  /** Optimality criterion code string; code is case-insensitive, but otherwise 
      exact match is required. */
  const char *s
) {
  if(s==NULL || *s=='\0') return(OPTCRIT_UNKNOWN);
  for(unsigned int i=0; i<optcritNr(); i++) 
    if(!strcasecmp(tpc_optcrit[i].code, s)) return(tpc_optcrit[i].id);
  /* Try also some alternative names */
  if(!strcasecmp("SS", s)) return(OPTCRIT_OLS);
  if(!strcasecmp("WSS", s)) return(OPTCRIT_OLS);
  if(!strcasecmp("SUMSQR", s)) return(OPTCRIT_OLS);
  if(!strcasecmp("LAE", s)) return(OPTCRIT_LAD);
  if(!strcasecmp("LAV", s)) return(OPTCRIT_LAD);
  if(!strcasecmp("LAR", s)) return(OPTCRIT_LAD);
  if(!strcasecmp("Deming", s)) return(OPTCRIT_ODR);
  return(OPTCRIT_UNKNOWN);
}
/*****************************************************************************/

/*****************************************************************************/
