/** @file analyzeio.c
 *  @brief Procedures for Analyze images.
 */
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include "tpcnifti.h"
/*****************************************************************************/

/*****************************************************************************/
/** Verify if specified file name is an Analyze 7.5 file.
   @todo Add tests.
   @sa anaReadHeader, niftiExists, micropetExists
   @return Returns 1 if it is Analyze, 0 if not.
 */
int anaExists(
  /** File name, either header file, image file, or base name without extensions. */
  const char *filename,
  /** If file name refers to an Analyze file, then header file name will be
      written in this char pointer (space needs to allocated by caller);
      If header and image are combined, then this will be the name of combined file;
      enter NULL if not needed. */
  char *hdrfile,
  /** If file name refers to an Analyze file, then image file name will be
      written in this char pointer (space needs to allocated by caller);
      If header and image are combined, then this will be the name of combined file;
      enter NULL if not needed. */
  char *imgfile,
  /** If file name refers to an Analyze file, and if SIF exists, then SIF file name
      will be written in this char pointer (space needs to allocated by caller);
      enter NULL if not needed. */
  char *siffile,
  /** Pointer to Analyze header, which is filled in this function; enter NULL, if not needed. */
  ANALYZE_DSR *header,
  /** Pointer to status data; enter NULL if not needed. */
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  if(verbose>0) {printf("%s(%s, ...)\n", __func__, filename); fflush(stdout);}
  statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_OK);

  /* Initiate output */
  if(hdrfile!=NULL) hdrfile[0]=(char)0;
  if(imgfile!=NULL) imgfile[0]=(char)0;
  if(siffile!=NULL) siffile[0]=(char)0;

  /* Empty file name means not Analyze file */
  if(strnlen(filename, 2)<1) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_INVALID_FILENAME);
    return(0);
  }

  /* Construct the base file name wo extensions */
  char basefile[FILENAME_MAX]; strlcpy(basefile, filename, FILENAME_MAX); 
  // If file exists it has extensions (similar than NIfTI-1)
  if(fileExist(basefile)) niftiBasename(basefile);
  if(verbose>1) printf("  basefile := %s\n", basefile);

  /* Combined header and image file exists? */
  int combined=0;
  char temp[FILENAME_MAX+10], localhdrfile[FILENAME_MAX];
  localhdrfile[0]=(char)0;
  strcpy(temp, basefile); strcat(temp, ".nii");
  if(fileExist(temp)) {
    if(verbose>1) printf("  %s exists.\n", temp);
    /* Preserve header and image file names */
    strcpy(localhdrfile, temp);
    if(hdrfile!=NULL) strlcpy(hdrfile, temp, FILENAME_MAX);
    if(imgfile!=NULL) strlcpy(imgfile, temp, FILENAME_MAX);
    combined=1;
  } else {
    if(verbose>1) printf("  %s does not exist.\n", temp);
    /* Not combined file, therefore check that header file exists */
    strcpy(temp, basefile); strcat(temp, ".hdr");
    if(!fileExist(temp)) {
      strcpy(temp, basefile); strcat(temp, ".img.hdr");
      if(!fileExist(temp)) {
        if(verbose>1) printf("  hdr file not found.\n");
        statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_MISSING_HEADER);
        return(0);
      }
    }
    /* Preserve file name of the header */
    strlcpy(localhdrfile, temp, FILENAME_MAX);
    if(hdrfile!=NULL) strlcpy(hdrfile, temp, FILENAME_MAX);
    if(verbose>1) printf("  %s is found.\n", localhdrfile);
    /* Not combined file, therefore check that image file exists */
    strcpy(temp, basefile); strcat(temp, ".img");
    if(!fileExist(temp)) {
      if(verbose>1) printf("  %s not found.\n", temp);
      statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_NO_FILE);
      return(0);
    }
    /* Preserve image filename */
    if(imgfile!=NULL) strlcpy(imgfile, temp, FILENAME_MAX);
    if(verbose>1) printf("  %s is found.\n", temp);
  }
  if(verbose>1) {printf("  combined := %d\n", combined); fflush(stdout);}

  /* Read header to check that this indeed is Analyze */
  {
    ANALYZE_DSR *dsr, local_dsr; if(header==NULL) dsr=&local_dsr; else dsr=header;
    int ret=anaReadHeader(localhdrfile, dsr, verbose-2);
    if(ret!=TPCERROR_OK) {
      if(verbose>1) {
        printf("  %s was not identified as Analyze header file.\n", localhdrfile); fflush(stdout);}
      statusSet(status, __func__, __FILE__, __LINE__, ret);
      return(0);
    }
    if(verbose>1) {printf("  %s is identified as Analyze.\n", localhdrfile); fflush(stdout);}
  }

  /* SIF exists? */
  strcpy(temp, basefile); strcat(temp, ".sif");
  if(verbose>3) printf("  checking if %s exists\n", temp);
  if(!fileExist(temp)) {
    strcpy(temp, basefile); strcat(temp, ".img.sif");
    if(verbose>3) printf("  checking if %s exists\n", temp);
    if(!fileExist(temp)) {
      strcpy(temp, basefile); strcat(temp, ".nii.sif");
      if(verbose>3) printf("  checking if %s exists\n", temp);
      if(!fileExist(temp)) {
        if(verbose>0) printf("  SIF not found or accessible.\n");
        statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_OK);
        return(1); // but otherwise ok NIfTI
      }
    }
  }
  /* Preserve SIF filename */
  if(siffile!=NULL) strcpy(siffile, temp);
  if(verbose>1) {printf("  %s is found.\n", temp); fflush(stdout);}

  statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_OK);
  return(1);
}
/*****************************************************************************/

/*****************************************************************************/
/** Read Analyze header contents. 
   @todo Add tests.
   @sa anaExists, micropetHeaderRead
   @return enum tpcerror (TPCERROR_OK when successful).
 */
int anaReadHeader(
  /** Name of file to read (including path and extension). */
  const char *filename,
  /** Pointer to previously allocated header structure. */
  ANALYZE_DSR *dsr,
  /** Verbose level; if zero, then nothing is printed to stderr or stdout */
  int verbose
) {
  if(verbose>0) {printf("%s(%s, ...)\n", __func__, filename); fflush(stdout);}

  /* Check arguments */
  if(strnlen(filename, 2)<1) return(TPCERROR_INVALID_FILENAME);
  if(dsr==NULL) return(TPCERROR_FAIL);

  /* Is current platform little endian (1) or not (0) ? */
  int little=endianLittle();
  if(verbose>2) {
    if(little) printf("little endian platform\n"); else printf("big endian platform\n");
  }

  /* Open file */
  FILE *fp=fopen(filename, "rb"); if(fp==NULL) return(TPCERROR_CANNOT_OPEN);

  /* Try to read the size of header, which should be stored in the beginning as
     4-byte integer in both NIfTI and Analyze header files. */
  int same_order=1;
  int hdrSize=0;
  {
    char buf[4];
    if(fread(buf, 4, 1, fp)<1) {fclose(fp); return(TPCERROR_CANNOT_READ);}
    memcpy(&hdrSize, buf, 4); 
    if(hdrSize!=ANALYZE_HEADER_SIZE && hdrSize!=NIFTI2_HEADER_SIZE) {
      swawbip(&hdrSize, 4); same_order=0;
    }
    if(verbose>1) {printf("  sizeof_hdr := %d\n", hdrSize); fflush(stdout);}
    if(hdrSize==NIFTI2_HEADER_SIZE) {
      if(verbose>1) {printf("  NIfTI-2 header size.\n"); fflush(stdout);}
      fclose(fp); return(TPCERROR_INVALID_FORMAT);
    } else if(hdrSize==ANALYZE_HEADER_SIZE) {
      if(verbose>1) {printf("  Analyze or NIfTI-1 header size.\n"); fflush(stdout);}
    } else {
      if(verbose>1) {printf("  invalid Analyze sizeof_hdr\n"); fflush(stdout);}
      return(TPCERROR_INVALID_FORMAT);
    }
  }


  /* Read file into Analyze header structure */
  if(verbose>1) {printf("  reading header as binary data\n"); fflush(stdout);}
  unsigned char buf[ANALYZE_HEADER_SIZE];
  if(fread(buf, ANALYZE_HEADER_SIZE, 1, fp)<1) {fclose(fp); return(TPCERROR_CANNOT_READ);}
  /* Close file */
  fclose(fp);

  /* Check that header does not have the Nifti Magic number */
  char magic[4]; 
  memcpy(magic, buf+344, 4); magic[3]=(char)0;
  if(strcasecmp(magic, "ni1")==0 || strcasecmp(magic, "n+1")==0) {
    if(verbose>1) {printf("  Nifti magic number was found"); fflush(stdout);}
    return(TPCERROR_INVALID_FORMAT);
  }

  /* Set the original byte order */
  if(same_order) dsr->byte_order=little;
  else {if(little==1) dsr->byte_order=0; else dsr->byte_order=1;}

  /* Set key header structure contents */
  dsr->h.sizeof_hdr=hdrSize;
  memcpy(dsr->h.data_type, buf+4, 10);
  memcpy(dsr->h.db_name, buf+14, 18);
  if(!same_order) swawbip(buf+32, 4);
  memcpy(&dsr->h.extents, buf+32, 4);
  if(!same_order) swabip(buf+36, 2);
  memcpy(&dsr->h.session_error, buf+36, 2);
  memcpy(&dsr->h.regular, buf+38, 1);
  memcpy(&dsr->h.hkey_un0, buf+39, 1);

  /* Set image dimension header structure contents */
  if(!same_order) swabip(buf+40, 16);
  memcpy(dsr->h.dim, buf+40, 16);
  if(!same_order) swabip(buf+56, 2);
  memcpy(&dsr->h.unused8, buf+56, 2);
  if(!same_order) swabip(buf+58, 2);
  memcpy(&dsr->h.unused9, buf+58, 2);
  if(!same_order) swabip(buf+60, 2);
  memcpy(&dsr->h.unused10, buf+60, 2);
  if(!same_order) swabip(buf+62, 2);
  memcpy(&dsr->h.unused11, buf+62, 2);
  if(!same_order) swabip(buf+64, 2);
  memcpy(&dsr->h.unused12, buf+64, 2);
  if(!same_order) swabip(buf+66, 2);
  memcpy(&dsr->h.unused13, buf+66, 2);
  if(!same_order) swabip(buf+68, 2);
  memcpy(&dsr->h.unused14, buf+68, 2);
  if(!same_order) swabip(buf+70, 2);
  memcpy(&dsr->h.datatype, buf+70, 2);
  if(!same_order) swabip(buf+72, 2);
  memcpy(&dsr->h.bitpix, buf+72, 2);
  if(!same_order) swabip(buf+74, 2);
  memcpy(&dsr->h.dim_un0, buf+74, 2);
  if(!same_order) swawbip(buf+76, 32);
  memcpy(dsr->h.pixdim, buf+76, 32);
  if(!same_order) swawbip(buf+108, 4);
  memcpy(&dsr->h.vox_offset, buf+108, 4);
  if(!same_order) swawbip(buf+112, 4);
  memcpy(&dsr->h.funused1, buf+112, 4);
  if(!same_order) swawbip(buf+116, 4);
  memcpy(&dsr->h.funused2, buf+116, 4);
  if(!same_order) swawbip(buf+120, 4);
  memcpy(&dsr->h.funused3, buf+120, 4);
  if(!same_order) swawbip(buf+124, 4);
  memcpy(&dsr->h.cal_max, buf+124, 4);
  if(!same_order) swawbip(buf+128, 4);
  memcpy(&dsr->h.cal_min, buf+128, 4);
  if(!same_order) swawbip(buf+132, 4);
  memcpy(&dsr->h.compressed, buf+132, 4);
  if(!same_order) swawbip(buf+136, 4);
  memcpy(&dsr->h.verified, buf+136, 4);
  if(!same_order) swawbip(buf+140, 4);
  memcpy(&dsr->h.glmax, buf+140, 4);
  if(!same_order) swawbip(buf+144, 4);
  memcpy(&dsr->h.glmin, buf+144, 4);

  /* Set data history header structure contents */
  memcpy(dsr->h.descrip, buf+148, 80);
  memcpy(dsr->h.aux_file, buf+228, 24);
  memcpy(&dsr->h.orient, buf+252, 1);
  memcpy(dsr->h.originator, buf+253, 10);
  memcpy(dsr->h.generated, buf+263, 10);
  memcpy(dsr->h.scannum, buf+273, 10);
  memcpy(dsr->h.patient_id, buf+283, 10);
  memcpy(dsr->h.exp_date, buf+293, 10);
  memcpy(dsr->h.exp_time, buf+303, 10);
  memcpy(dsr->h.hist_un0, buf+313, 3);
  if(!same_order) swawbip(buf+316, 4);
  memcpy(&dsr->h.views, buf+316, 4);
  if(!same_order) swawbip(buf+320, 4);
  memcpy(&dsr->h.vols_added, buf+320, 4);
  if(!same_order) swawbip(buf+324, 4);
  memcpy(&dsr->h.start_field, buf+324,4);
  if(!same_order) swawbip(buf+328, 4);
  memcpy(&dsr->h.field_skip, buf+328, 4);
  if(!same_order) swawbip(buf+332, 4);
  memcpy(&dsr->h.omax, buf+332, 4);
  if(!same_order) swawbip(buf+336, 4);
  memcpy(&dsr->h.omin, buf+336, 4);
  if(!same_order) swawbip(buf+340, 4);
  memcpy(&dsr->h.smax, buf+340, 4);
  if(!same_order) swawbip(buf+344, 4);
  memcpy(&dsr->h.smin, buf+344, 4);

  /* Check header contents */
  if(dsr->h.extents!=16384 && dsr->h.extents!=0) {
    if(verbose>0) printf("h.extents := %d\n", dsr->h.extents);
    return(TPCERROR_UNSUPPORTED);
  }
  if(dsr->h.regular!='r') {
    if(verbose>1) printf("h.regular := %c\n", dsr->h.regular);
    return(TPCERROR_UNSUPPORTED);
  }

  if(verbose>1) {printf("  complete Analyze header was read.\n"); fflush(stdout);}

  return(TPCERROR_OK);
}
/*****************************************************************************/

/*****************************************************************************/
