/// @file constraints.c
/// @brief Check nonlinear fitting parameters against constraints.
///
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <time.h>
#include <string.h>
/*****************************************************************************/
#include "tpcextensions.h"
#include "tpcrand.h"
/*****************************************************************************/
#include "tpcnlopt.h"
/*****************************************************************************/

/*****************************************************************************/
/** @brief Check that model parameters are within given limits.

    If one or more parameter(s) are outside of limits, then a penalty factor is optionally computed.
    Optionally, this function can make a list parameters that are inside the constraints.
    
    @return Return the number of parameters that are inside or at the limits;
    0 in case of error or if all are outside of limits.
    @author Vesa Oikonen
    @sa nloptForceLimits, aicSS
 */
unsigned int nloptCheckParameters(
  /** Nr of parameters. */
  unsigned int n,
  /** Lower limits (array of length par_nr). */
  double *pLower,
  /** Upper limits (array of length par_nr). */
  double *pUpper,
  /** Parameters to test (array of length par_nr). */
  double *p,
  /** Pointer to corrected parameters (array of length par_nr); NULL if not needed. */
  double *pAccept,
  /** Pointer to variable in which the possible penalty factor will be written;
      1.0 if no penalty was found is necessary, otherwise >1. Set to NULL if not needed. */
  double *penalty
) {
  unsigned int pi, nAccept=0;
  double range;

  if(penalty!=NULL) *penalty=1.0;
  if(n<1 || p==NULL || pLower==NULL || pUpper==NULL)
    return(nAccept);
  for(pi=0; pi<n; pi++) {
    range=pUpper[pi]-pLower[pi]; if(range<1.0E-30) range=1.0;
    if(p[pi]<pLower[pi]) {
      if(pAccept!=NULL) pAccept[pi]=pLower[pi];
      if(penalty!=NULL) *penalty += (pLower[pi]-p[pi])/range;
    } else if(p[pi]>pUpper[pi]) {
      if(pAccept!=NULL) pAccept[pi]=pUpper[pi];
      if(penalty!=NULL) *penalty += (p[pi]-pUpper[pi])/range;
    } else {
      if(pAccept!=NULL) pAccept[pi]=p[pi];
      nAccept++;
    }
  }
  return nAccept;
}
/*****************************************************************************/

/*****************************************************************************/
/** @brief Enforce the model parameters within given limits.
    
    @return Return the number of parameters that are inside or at the limits;
    0 in case of error or if all are outside of limits.
    @author Vesa Oikonen
    @sa nloptCheckParameters
 */
unsigned int nloptForceLimits(
  /** Number of parameters. */
  unsigned int n,
  /** Lower limits (array of length par_nr). */
  double *pLower,
  /** Upper limits (array of length par_nr). */
  double *pUpper,
  /** Parameters to enforce inside limits (array of length par_nr). */
  double *p
) {
  unsigned int pi, nAccept=0;

  if(n<1 || p==NULL || pLower==NULL || pUpper==NULL) return(nAccept);
  for(pi=0; pi<n; pi++) {
    if(!(p[pi]>=pLower[pi])) {
      p[pi]=pLower[pi];
    } else if(!(p[pi]<=pUpper[pi])) {
      p[pi]=pUpper[pi];
    } else {
      nAccept++;
    }
  }
  return(nAccept);
}
/*****************************************************************************/

/*****************************************************************************/
