/** @file tpcnlopt.h
 *  @brief Header file for library libtpcnlopt.
 *  @details Non-linear optimization.
 *  @author Vesa Oikonen
 */
#ifndef _TPCNLOPT_H_
#define _TPCNLOPT_H_
/*****************************************************************************/

/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
/*****************************************************************************/
#include "tpcextensions.h"
#include "tpcrand.h"
/*****************************************************************************/

/*****************************************************************************/
/** General settings for NLOPT routines; individual routines may not use all of these,
    and can require additional parameters. */
typedef struct NLOPT {
  /** Total nr of parameters (dimensions). */
  unsigned int totalNr;
  /** Array of all parameter values, including fixed ones. */
  double       *xfull;
  /** Array of lower limits for parameters, including fixed ones. */
  double       *xlower;
  /** Array of upper limits for parameters, including fixed ones. */
  double       *xupper;
  /** Array of (initial) steps for parameters, including fixed ones;
      set it to the expected distance from the solution. */
  double       *xdelta;
  /** Array of tolerances for parameters, including fixed ones; may be used
      as a stopping rule and for scaling. */
  double       *xtol;
  /** Pointer to objective function; parameters are optimized to minimize
      the return value of this function. */
  double (*_fun)(int, double*, void*);
  /** Pointer to data which is passed on to the function; NULL if not needed. */
  void *fundata;
  /** Maximum allowed number of function calls; set to zero for no limit. */
  unsigned int maxFunCalls;
  /** Number of function calls so far. */
  unsigned int funCalls;
  /** The return value of objective function with xfull[] and fundata. */
  double funval;
  /** Keep (1) or do not keep (0) plist (below). */
  unsigned int usePList;
  /** List of all visited parameters and objective function values;
      stored in order x[], funval, x[], funval, ... with length funCalls*(totalNr+1);
      updated only if usePList!=0 (above). */
  double       *plist;
} NLOPT;
/*****************************************************************************/

/*****************************************************************************/
/** Basic structure for data passed on to objective function via NLOPT structure.
    Actual implementations usually require more data, and instead of this structure,
    a dedicated local data structure should be created and used.
 */
typedef struct NLOPT_DATA {
  /** Sample number. */
  unsigned int  n;
  /** Pointer to an array of sample times. */
  double       *x;
  /** Pointer to an array of sample values. */
  double       *y;
  /** Pointer to an array of weights. */
  double       *w;
  /** Pointer to fitted/simulated sample values. */
  double       *sy;
  /** Verbose level. */
  int           verbose;
  /** Pointer to verbose output; usually stdout. Must be set if verbose>0. */
  FILE         *fp;
} NLOPT_DATA;
/*****************************************************************************/

/*****************************************************************************/
/* nlopt */
extern void nloptInit(NLOPT *nlo);
extern void nloptFree(NLOPT *nlo);
extern int nloptAllocate(NLOPT *nlo, unsigned int parNr);
extern int nloptDuplicate(NLOPT *nlo1, NLOPT *nlo2);
extern int nloptAddP(NLOPT *nlo, double *p, double funval);
extern int nloptSortP(NLOPT *nlo);
extern int nloptMeanP(NLOPT *nlo, unsigned int nr, double *meanp, double *sdp);
extern void nloptPrintP(NLOPT *nlo, unsigned int nr, FILE *fp);
extern void nloptWrite(NLOPT *d, FILE *fp);
extern unsigned int nloptLimitFixedNr(NLOPT *d);
extern unsigned int nloptFixedNr(NLOPT *d);
extern void nloptRemoveEmpties(NLOPT *d);
extern void nloptdataInit(NLOPT_DATA *d);
/*****************************************************************************/

/*****************************************************************************/
/* rndpoint */
extern int nloptRandomPoint(double *p, double *low, double *up, unsigned int n, MERTWI *mt);
extern int nloptGaussianPoint(
  double *p, double *mean, double *sd, double *low, double *up, unsigned int n, MERTWI *mt
);
/*****************************************************************************/

/*****************************************************************************/
/* constraints */
extern unsigned int nloptCheckParameters(
  unsigned int n, double *pLower, double *pUpper, double *p,
  double *pAccept, double *penalty
);
extern unsigned int nloptForceLimits(unsigned int n, double *pLower, double *pUpper, double *p);
/*****************************************************************************/

/*****************************************************************************/
/* nlopt1d */
extern int nlopt1D(NLOPT *nlo, unsigned int maxIter, TPCSTATUS *);
/*****************************************************************************/

/*****************************************************************************/
/* simplex */
extern int nloptSimplex(NLOPT *nlo, unsigned int maxIter, TPCSTATUS *);
extern int nloptSimplexARRS(NLOPT *nlo, unsigned int maxIter, TPCSTATUS *status);
/*****************************************************************************/

/*****************************************************************************/
/* praxis */
extern int nloptPowellBrent(NLOPT *nlo, unsigned int ktm, TPCSTATUS *);
/*****************************************************************************/

/*****************************************************************************/
/* mpso */
extern int nloptMPSO(
  NLOPT *nlo, unsigned int maxIter, unsigned int nSwarms, unsigned int nParticles, double wInertia,
  double wParticle, double wSwarm, double wGlobal, double pDeath, double pImmigration,
  const int doLocal, TPCSTATUS *
);
/*****************************************************************************/

/*****************************************************************************/
/* tgo */
extern int nloptITGO1(
  NLOPT *nlo, const int doLocal, unsigned int tgoNr, unsigned int sampleNr,
  unsigned int neighNr, TPCSTATUS *status
);
extern int nloptITGO2(
  NLOPT *nlo, const int doLocal, unsigned int tgoNr, unsigned int sampleNr,
  unsigned int neighNr, TPCSTATUS *status
);
extern int nloptIATGO(NLOPT *nlo, const int doLocal, unsigned int maxIterNr,
  double neighFract, TPCSTATUS *status
);
/*****************************************************************************/

/*****************************************************************************/
#endif /* TPCNLOPT */
