/** @file parcomb.c
 *  @brief Combinations of data in PAR structures.
 */
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <time.h>
#include <string.h>
/*****************************************************************************/
#include "tpcpar.h"
/*****************************************************************************/

/*****************************************************************************/
/** Select parameters and regions in a PAR structure that are available in another PAR structure.
    @return enum tpcerror (TPCERROR_OK when successful).
    @sa parSelectTACs, parEnsureNames, parCompareParameterNames, parCompareTacNames,
        parSelectedParameters, parSelectedTACs
 */
int parSelectByAnother(
  /** Pointer to target PAR structure, in which the PARN and PARR switches are set. */
  PAR *d1,
  /** Pointer to PAR structure to which the first is compared to; not modified. */
  PAR *d2,
  /** Pointer to store the number of available parameters; enter NULL, if not needed. */
  int *pn,
  /** Pointer to store the number of available regions; enter NULL, if not needed. */
  int *pr,
  /** Pointer to status data; enter NULL if not needed. */
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  if(verbose>0) printf("%s()\n", __func__);
  if(pn!=NULL) *pn=0;
  if(pr!=NULL) *pr=0;
  /* Check that required data exists */
  if(d1==NULL || d1->parNr<1 || d1->tacNr<1) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_NO_DATA);
    return TPCERROR_NO_DATA;
  }
  /* Unselect all */
  for(int i=0; i<d1->parNr; i++) d1->n[i].sw=0;
  for(int i=0; i<d1->tacNr; i++) d1->r[i].sw=0;
  /* If no data to compare to, then that is it */
  if(d2==NULL || d2->parNr<1 || d2->tacNr<1) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_OK);
    return TPCERROR_OK;
  }

  /* Search for parameter names */
  for(int i=0; i<d1->parNr; i++) {
    if(strnlen(d1->n[i].name, 1)<1) continue;
    for(int j=0; j<d2->parNr; j++) {
      if(strnlen(d2->n[j].name, 1)<1) continue;
      if(strcasecmp(d1->n[i].name, d2->n[j].name)==0) {d1->n[i].sw=1; continue;}
      /* Allow mixed '-' and '_' */
      char *buf1, *buf2;
      buf1=strdup(d1->n[i].name); strReplaceChar(buf1, '-', '_');
      buf2=strdup(d2->n[j].name); strReplaceChar(buf2, '-', '_');
      int ret=strcasecmp(buf1, buf2);
      free(buf1); free(buf2);
      if(ret==0) {d1->n[i].sw=1; continue;}
    }
  }

  /* Search for TAC names */
  for(int i=0; i<d1->tacNr; i++) {
    if(strnlen(d1->r[i].name, 1)<1) continue;
    for(int j=0; j<d2->tacNr; j++) {
      if(strnlen(d2->r[j].name, 1)<1) continue;
      if(strcasecmp(d1->r[i].name, d2->r[j].name)==0) {d1->r[i].sw=1; continue;}
      /* Allow mixed '-' and '_' */
      char *buf1, *buf2;
      buf1=strdup(d1->r[i].name); strReplaceChar(buf1, '-', '_');
      buf2=strdup(d2->r[j].name); strReplaceChar(buf2, '-', '_');
      int ret=strcasecmp(buf1, buf2);
      free(buf1); free(buf2);
      if(ret==0) {d1->r[i].sw=1; continue;}
    }
  }

  /* Count the selected numbers, if needed */
  if(pn!=NULL) *pn=parSelectedParameters(d1);
  if(pr!=NULL) *pr=parSelectedTACs(d1);

  statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_OK);
  return TPCERROR_OK;
}
/*****************************************************************************/

/*****************************************************************************/
/** Copy parameters from source PAR structure to target structure for TACs that are available
    in both structures.
    @return enum tpcerror (TPCERROR_OK when successful).
    @sa parSelectByAnother, parSelectTACs, parEnsureNames
 */
int parCombineTACs(
  /** Pointer to target PAR structure. */
  PAR *d1,
  /** Pointer to source PAR structure; not modified. */
  PAR *d2,
  /** Pointer to status data; enter NULL if not needed. */
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  if(verbose>0) printf("%s()\n", __func__);

  /* Check what there is to copy */
  int tacNr;
  int ret=parSelectByAnother(d1, d2, NULL, &tacNr, status);
  if(ret!=TPCERROR_OK) return(ret);
  if(tacNr<1) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_NO_DATA);
    return TPCERROR_NO_DATA;
  }
  int parNr=d2->parNr;
  if(verbose>1) printf("%d TACs and %d parameters to copy.\n", tacNr, parNr);

  /* Allocate memory for the new parameters */
  statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_OK);
  ret=parAllocateMore(d1, parNr, 0);
  if(ret!=TPCERROR_OK) return(ret);

  /* Copy new parameter names */
  for(int i=0; i<d2->parNr; i++) {
    strcpy(d1->n[i+d1->parNr].name, d2->n[i].name);
    d1->n[i+d1->parNr].unit=d2->n[i].unit;
    d1->n[i+d1->parNr].lim1=d2->n[i].lim1;
    d1->n[i+d1->parNr].lim2=d2->n[i].lim2;
    d1->n[i+d1->parNr].tol=d2->n[i].tol;
  }

  /* Copy contents for common TACs */
  for(int i=0; i<d1->tacNr; i++) {
    /* find matching TAC name */
    if(strnlen(d1->r[i].name, 1)<1) continue;
    int j;
    for(j=0; j<d2->tacNr; j++) {
      if(strnlen(d2->r[j].name, 1)<1) continue;
      if(strcasecmp(d1->r[i].name, d2->r[j].name)==0) break;
      /* Allow mixed '-' and '_' */
      char *buf1, *buf2;
      buf1=strdup(d1->r[i].name); strReplaceChar(buf1, '-', '_');
      buf2=strdup(d2->r[j].name); strReplaceChar(buf2, '-', '_');
      int ret=strcasecmp(buf1, buf2);
      free(buf1); free(buf2);
      if(ret==0) break;
    }
    if(j==d2->tacNr) continue; // matching name not found, leave values as NaN

    /* Copy new parameter values */
    for(int k=0; k<d2->parNr; k++) {
      d1->r[i].p[k+d1->parNr]=d2->r[j].p[k];
      d1->r[i].sd[k+d1->parNr]=d2->r[j].sd[k];
      d1->r[i].cl1[k+d1->parNr]=d2->r[j].cl1[k];
      d1->r[i].cl2[k+d1->parNr]=d2->r[j].cl2[k];
    }
  }

  d1->parNr+=d2->parNr;

  statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_OK);
  return TPCERROR_OK;
}
/*****************************************************************************/

/*****************************************************************************/
