/** @file parcomp.c
 *  @brief Comparison of PAR struct data.
 */
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include "tpcift.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <time.h>
#include <ctype.h>
#include <string.h>
#include <strings.h>
/*****************************************************************************/
#include "tpcpar.h"
/*****************************************************************************/

/*****************************************************************************/
/** Check whether parameter names are the same in two PAR data.

    Comparison is very strict, thus the names may need to be preprocessed if this is used for
    other purpose than SW testing.
    @return 0 in case of match, 1 if no match or error.
    @author Vesa Oikonen
 */
int parCompareParameterNames(
  /** Pointer to PAR struct. */
  PAR *d1,
  /** Pointer to PAR struct. */
  PAR *d2,
  /** Parameter index [0..parNr-1] to compare; enter <0 to verify all. */
  const int i,
  /** Case-sensitivity: 0=case does not have to match; <>0=case must match. */
  const int casens,
  /** Pointer to status data; enter NULL if not needed. */
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  if(verbose>1) printf("%s()\n", __func__);

  /* Check that required data exists */
  if(d1==NULL || d2==NULL || d1->parNr<1 || d2->parNr<1) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_NO_DATA);
    return 1;
  }
  /* If index is specified, then verify that that parameter is available */
  if(i>=0 && (i>=d1->parNr || i>=d2->parNr)) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL);
    return 1;
  }

  /* If index is not specified, then parNr must match */
  if(i<0 && (d1->parNr!=d2->parNr)) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_OK);
    if(verbose>0) printf("different parameter nr.\n");
    return(1);
  }

  /* Compare */
  statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_OK);
  int vr;
  for(int pi=0; pi<d1->parNr; pi++) if(i<0 || i==pi) {
    if(casens==0) vr=strcasecmp(d1->n[pi].name, d2->n[pi].name);
    else vr=strcmp(d1->n[pi].name, d2->n[pi].name);
    if(vr==0) continue;
    /* No problem, if difference is '_' versus '-' */
    if(strchr(d1->n[pi].name, '-') || strchr(d1->n[pi].name, '_')) {
      char *buf1, *buf2;
      buf1=strdup(d1->n[pi].name); strReplaceChar(buf1, '-', '_');
      buf2=strdup(d2->n[pi].name); strReplaceChar(buf2, '-', '_');
      if(verbose>100) printf("'%s' vs '%s'\n", buf1, buf2);
      if(casens==0) vr=strcasecmp(buf1, buf2); else vr=strcmp(buf1, buf2);
      free(buf1); free(buf2);
    }
    if(vr==0) continue;
    /* No problem, if difference is '_' versus ' ' */
    if(strchr(d1->n[pi].name, '_') || strchr(d2->n[pi].name, '_')) {
      char *buf1, *buf2;
      buf1=strdup(d1->n[pi].name); strReplaceChar(buf1, '_', ' ');
      buf2=strdup(d2->n[pi].name); strReplaceChar(buf2, '_', ' ');
      if(verbose>100) printf("'%s' vs '%s'\n", buf1, buf2);
      if(casens==0) vr=strcasecmp(buf1, buf2); else vr=strcmp(buf1, buf2);
      free(buf1); free(buf2);
    }
    if(vr==0) continue;
    if(verbose>0) {
      printf("par1.n[%d].name := '%s'\n", pi, d1->n[pi].name);
      printf("par2.n[%d].name := '%s'\n", pi, d2->n[pi].name);
    }
    return(1);
  }
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/** Check whether parameter units are the same in two PAR data.
    @return 0 in case of match, 1 if no match or error.
    @author Vesa Oikonen
 */
int parCompareParameterUnits(
  /** Pointer to PAR structure. */
  PAR *d1,
  /** Pointer to PAR structure. */
  PAR *d2,
  /** Parameter index [0..parNr-1] to compare; enter <0 to verify all. */
  const int i,
  /** Pointer to status data; enter NULL if not needed. */
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  if(verbose>1) printf("%s()\n", __func__);

  /* Check that required data exists */
  if(d1==NULL || d2==NULL || d1->parNr<1 || d2->parNr<1) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_NO_DATA);
    return 1;
  }
  /* If index is specified, then verify that parameter is available */
  if(i>=0 && (i>=d1->parNr || i>=d2->parNr)) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL);
    return 1;
  }

  /* If index is not specified, then parNr must match */
  if(i<0 && (d1->parNr!=d2->parNr)) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_OK);
    if(verbose>0) printf("different parameter nr.\n");
    return(1);
  }

  /* Compare */
  statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_OK);
  for(int pi=0; pi<d1->parNr; pi++) if(i<0 || i==pi) {
    if(d1->n[pi].unit!=d2->n[pi].unit) {
      if(verbose>0) {
        printf("par1.n[%d].unit := '%s'\n", pi, unitName(d1->n[pi].unit));
        printf("par2.n[%d].unit := '%s'\n", pi, unitName(d2->n[pi].unit));
      }
      return(1);
    }
  }
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/** Check whether TAC names are the same in two PAR data.

    Comparison is very strict, thus the names may need to be preprocessed if this is used for
    other purpose than SW testing.
    @return 0 in case of match, 1 if no match or error.
    @author Vesa Oikonen
 */
int parCompareTacNames(
  /** Pointer to PAR structure. */
  PAR *d1,
  /** Pointer to PAR structure. */
  PAR *d2,
  /** Parameter index [0..tacNr-1] to compare; enter <0 to verify all. */
  const int i,
  /** Case-sensitivity: 0=case does not have to match; <>0=case must match. */
  const int casens,
  /** Pointer to status data; enter NULL if not needed. */
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  if(verbose>1) printf("%s()\n", __func__);

  /* Check that required data exists */
  if(d1==NULL || d2==NULL || d1->tacNr<1 || d2->tacNr<1) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_NO_DATA);
    return 1;
  }
  /* If index is specified, then verify that TAC is available */
  if(i>=0 && (i>=d1->tacNr || i>=d2->tacNr)) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL);
    return 1;
  }

  /* If index is not specified, then tacNr must match */
  if(i<0 && (d1->tacNr!=d2->tacNr)) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_OK);
    if(verbose>0) printf("different TAC nr.\n");
    return(1);
  }

  /* Compare */
  statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_OK);
  int vr;
  for(int ri=0; ri<d1->tacNr; ri++) if(i<0 || i==ri) {
    if(casens==0) vr=strcasecmp(d1->r[ri].name, d2->r[ri].name);
    else vr=strcmp(d1->r[ri].name, d2->r[ri].name);
    if(vr==0) continue;
    /* No problem, if difference is '_' versus '-' */
    if(strchr(d1->r[ri].name, '-') || strchr(d1->r[ri].name, '_')) {
      char *buf1, *buf2;
      buf1=strdup(d1->r[ri].name); strReplaceChar(buf1, '-', '_');
      buf2=strdup(d2->r[ri].name); strReplaceChar(buf2, '-', '_');
      if(verbose>100) printf("'%s' vs '%s'\n", buf1, buf2);
      if(casens==0) vr=strcasecmp(buf1, buf2); else vr=strcmp(buf1, buf2);
      free(buf1); free(buf2);
    }
    if(vr==0) continue;
    /* No problem, if difference is '_' versus ' ' */
    if(strchr(d1->r[ri].name, '_') || strchr(d2->r[ri].name, '_')) {
      char *buf1, *buf2;
      buf1=strdup(d1->r[ri].name); strReplaceChar(buf1, '_', ' ');
      buf2=strdup(d2->r[ri].name); strReplaceChar(buf2, '_', ' ');
      if(verbose>100) printf("'%s' vs '%s'\n", buf1, buf2);
      if(casens==0) vr=strcasecmp(buf1, buf2); else vr=strcmp(buf1, buf2);
      free(buf1); free(buf2);
    }
    if(vr==0) continue;
    if(verbose>0) {
      printf("par1.r[%d].name := '%s'\n", ri, d1->r[ri].name);
      printf("par2.r[%d].name := '%s'\n", ri, d2->r[ri].name);
    }
    return(1);
  }
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/** Check whether parameter values are the same in two PAR data.

    Note that units are ignored here.
    If either absolute or relative difference is below the limit, the test is reported as passed.

    @sa parCompareParameterUnits.
    @return 0 in case of match, and >0 if no match or error.
    @author Vesa Oikonen
 */
int parCompareParameters(
  /** Pointer to PAR structure. */
  PAR *d1,
  /** Pointer to PAR structure. */
  PAR *d2,
  /** PAR index [0..parNr-1] to compare; enter <0 to verify all parameters. */
  const int pi,
  /** TAC index [0..tacNr-1] to compare; enter <0 to verify all TACs. */
  const int ti,
  /** Check parameter values (1) or do not check (0). */
  int checkpar,
  /** Check parameter SD values (1) or do not check (0). */
  int checksd,
  /** Check parameter CL values (1) or do not check (0). */
  int checkcl,
  /** Limit for accepted absolute difference; obligatory. */
  const double test_abs,
  /** Optional limit for accepted relative difference |2*(x1-x2)/(x1+x2)| ;
      set to negative value to not test this; in case of zero mean, this test
      is assumed to fail, but test for absolute difference may still pass. */
  const double test_rel,
  /** Pointer to status data; enter NULL if not needed. */
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  if(verbose>1) printf("%s()\n", __func__);

  /* Check that required data exists */
  if(d1==NULL || d2==NULL || d1->tacNr<1 || d2->tacNr<1 || d1->parNr<1 || d2->parNr<1) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_NO_DATA);
    return 1;
  }
  /* Check that at least something is tested */
  if(!checkpar && !checksd && !checkcl) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL);
    return 1;
  }
  /* If index is specified, then verify that TAC or PAR is available */
  if(pi>=0 && (pi>=d1->parNr || pi>=d2->parNr)) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL);
    return 2;
  }
  if(ti>=0 && (ti>=d1->tacNr || ti>=d2->tacNr)) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL);
    return 2;
  }

  /* If index is not specified, then parNr or tacNr must match */
  statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_OK);
  if(pi<0 && (d1->parNr!=d2->parNr)) {
    if(verbose>0) printf("different parameter nr.\n");
    return(3);
  }
  if(ti<0 && (d1->tacNr!=d2->tacNr)) {
    if(verbose>0) printf("different TAC nr.\n");
    return(3);
  }

  /* Compare */
  statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_OK);
  int pj, tj;
  if(checkpar) {
    for(tj=0; tj<d1->tacNr; tj++) if(ti<0 || tj==ti) {
      for(pj=0; pj<d1->parNr; pj++) if(pi<0 || pj==pi) {
        if(doubleMatch(d1->r[tj].p[pj], d2->r[tj].p[pj], test_abs)==1) continue;
        if(test_rel>0.0 && doubleMatchRel(d1->r[tj].p[pj], d2->r[tj].p[pj], test_rel)==1) continue;
        if(verbose>0) {
          double s=fabs(d1->r[tj].p[pj]-d2->r[tj].p[pj]);
          printf("par1.r[%d].p[%d] := %g\n", tj, pj, d1->r[tj].p[pj]);
          printf("par2.r[%d].p[%d] := %g\n", tj, pj, d2->r[tj].p[pj]);
          printf("|diff| := %g\n", s);
          printf("diff_limit := %g\n", test_abs);
          if(test_rel>0.0) printf("rel_diff_limit := %g\n", test_rel);
        }
        return(10);
      }
    }
  }
  if(checksd) {
    for(tj=0; tj<d1->tacNr; tj++) if(ti<0 || tj==ti) {
      for(pj=0; pj<d1->parNr; pj++) if(pi<0 || pj==pi) {
        if(doubleMatch(d1->r[tj].sd[pj], d2->r[tj].sd[pj], test_abs)==1) continue;
        if(test_rel>0.0 && doubleMatchRel(d1->r[tj].sd[pj], d2->r[tj].sd[pj], test_rel)==1) continue;
        if(verbose>0) {
          double s=fabs(d1->r[tj].sd[pj]-d2->r[tj].sd[pj]);
          printf("par1.r[%d].sd[%d] := %g\n", tj, pj, d1->r[tj].sd[pj]);
          printf("par2.r[%d].sd[%d] := %g\n", tj, pj, d2->r[tj].sd[pj]);
          printf("|diff| := %g\n", s);
          printf("diff_limit := %g\n", test_abs);
          if(test_rel>0.0) printf("rel_diff_limit := %g\n", test_rel);
        }
        return(10);
      }
    }
  }
  if(checkcl) {
    for(tj=0; tj<d1->tacNr; tj++) if(ti<0 || tj==ti) {
      for(pj=0; pj<d1->parNr; pj++) if(pi<0 || pj==pi) {
        if(doubleMatch(d1->r[tj].cl1[pj], d2->r[tj].cl1[pj], test_abs)==1) continue;
        if(test_rel>0.0 && doubleMatchRel(d1->r[tj].cl1[pj], d2->r[tj].cl1[pj], test_rel)==1) continue;
        if(verbose>0) {
          double s=fabs(d1->r[tj].cl1[pj]-d2->r[tj].cl1[pj]);
          printf("par1.r[%d].cl1[%d] := %g\n", tj, pj, d1->r[tj].cl1[pj]);
          printf("par2.r[%d].cl1[%d] := %g\n", tj, pj, d2->r[tj].cl1[pj]);
          printf("|diff| := %g\n", s);
          printf("diff_limit := %g\n", test_abs);
          if(test_rel>0.0) printf("rel_diff_limit := %g\n", test_rel);
        }
        return(10);
      }
    }
    for(tj=0; tj<d1->tacNr; tj++) if(ti<0 || tj==ti) {
      for(pj=0; pj<d1->parNr; pj++) if(pi<0 || pj==pi) {
        if(doubleMatch(d1->r[tj].cl2[pj], d2->r[tj].cl2[pj], test_abs)==1) continue;
        if(test_rel>0.0 && doubleMatchRel(d1->r[tj].cl2[pj], d2->r[tj].cl2[pj], test_rel)==1) continue;
        if(verbose>0) {
          double s=fabs(d1->r[tj].cl2[pj]-d2->r[tj].cl2[pj]);
          printf("par1.r[%d].cl2[%d] := %g\n", tj, pj, d1->r[tj].cl2[pj]);
          printf("par2.r[%d].cl2[%d] := %g\n", tj, pj, d2->r[tj].cl2[pj]);
          printf("|diff| := %g\n", s);
          printf("diff_limit := %g\n", test_abs);
          if(test_rel>0.0) printf("rel_diff_limit := %g\n", test_rel);
        }
        return(10);
      }
    }
  }

  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/** Check whether fit WSS values are the same in two PAR data.

    Note that units are ignored here.
    If either absolute or relative difference is below the limit, the test is reported as passed.
    @sa parCompareParameters.
    @return 0 in case of match, and >0 if no match or error.
    @author Vesa Oikonen
 */
int parCompareWSS(
  /** Pointer to PAR structure. */
  PAR *d1,
  /** Pointer to PAR structure. */
  PAR *d2,
  /** TAC index [0..tacNr-1] to compare; enter <0 to verify all TACs. */
  const int ti,
  /** Limit for accepted absolute difference; obligatory. */
  const double test_abs,
  /** Optional limit for accepted relative difference |2*(x1-x2)/(x1+x2)| ;
      set to negative value to not test this; in case of zero mean, this test
      is assumed to fail, but test for absolute difference may still pass. */
  const double test_rel,
  /** Pointer to status data; enter NULL if not needed. */
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  if(verbose>1) printf("%s()\n", __func__);

  /* Check that required data exists */
  if(d1==NULL || d2==NULL || d1->tacNr<1 || d2->tacNr<1) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_NO_DATA);
    return 1;
  }
  /* If index is specified, then verify that TAC is available */
  if(ti>=0 && (ti>=d1->tacNr || ti>=d2->tacNr)) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL);
    return 2;
  }

  /* If index is not specified, then tacNr must match */
  statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_OK);
  if(ti<0 && (d1->tacNr!=d2->tacNr)) {
    if(verbose>0) printf("different TAC nr.\n");
    return(3);
  }

  /* Compare */
  statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_OK);
  int tj;
  for(tj=0; tj<d1->tacNr; tj++) if(ti<0 || tj==ti) {
    if(doubleMatch(d1->r[tj].wss, d2->r[tj].wss, test_abs)==1) 
      continue;
    if(test_rel>0.0 && doubleMatchRel(d1->r[tj].wss, d2->r[tj].wss, test_rel)==1) 
      continue;
    if(verbose>0) {
      double s=fabs(d1->r[tj].wss-d2->r[tj].wss);
      printf("par1.r[%d].wss := %g\n", tj, d1->r[tj].wss);
      printf("par2.r[%d].wss := %g\n", tj, d2->r[tj].wss);
      printf("|diff| := %g\n", s);
      printf("diff_limit := %g\n", test_abs);
      if(test_rel>0.0) printf("rel_diff_limit := %g\n", test_rel);
    }
    return(10);
  }
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
