/** @file parselect.c
    @brief Selecting parameter(s) and/or TAC(s) in PAR struct.
 */
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include "tpcift.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <time.h>
#include <string.h>
/*****************************************************************************/
#include "tpcpar.h"
/*****************************************************************************/

/*****************************************************************************/
/** Select the TAC(s) in PAR structure that have matching ID name or number.
    @return the number of newly selected VOIs, or <=0 in case of an error.
    @author Vesa Oikonen
    @sa parSelectedTACs, parSelectParameters, parSortByName, parSelectByAnother, parEnsureNames
 */
int parSelectTACs(
  /** Pointer to PAR structure. */
  PAR *d,
  /** Name of TAC, or TAC numbers (as string, for example '1-3' or '1,2,3'); 
      string can be given with quotation marks. 
      Enter NULL to select all TACs, or empty string to select TACs with empty 
      name fields (which really should not exist). */
  const char *region_name,
  /** 1=Non-matching VOIs are deselected, 0=Old selections are preserved. */
  int reset,
  /** Pointer to status data; enter NULL if not needed. */
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  if(verbose>0) printf("%s()\n", __func__);
  /* Check that required data exists */
  if(d==NULL || d->tacNr<1) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_NO_DATA);
    return(0);
  }

  int ri, len, match_nr=0;

  /* Check if NULL was given as tac name; if, then select all */
  if(region_name==NULL) {
    for(ri=0; ri<d->tacNr; ri++) d->r[ri].sw=1;
    return(d->tacNr);
  }

  /* Reset all selections if required */
  if(reset!=0) for(ri=0; ri<d->tacNr; ri++) d->r[ri].sw=0;
  
  /* Remove any quotation marks from the name */
  len=strlen(region_name);
  char local_name[len+1];
  if(len>1) len=strncpyClean(local_name, region_name, len+1);
  else strcpy(local_name, region_name);

  /* If tac name is empty, then select all tacs with missing names */
  if(len<1 || strcmp(local_name, ".")==0) {
    for(ri=0; ri<d->tacNr; ri++) {
      if(strlen(d->r[ri].name)==0) {d->r[ri].sw=1; match_nr++; continue;}
      /* maybe here should also be checked for names consisting of only space characters */
    }
    return(match_nr);
  }

  /* Check each TAC name */
  for(ri=0; ri<d->tacNr; ri++) {
    /* does the name match? */
    if(roinameMatch(d->r[ri].name, local_name, status)!=0) {d->r[ri].sw=1; match_nr++; continue;}
  }
  /* If at least one match was found, then we are ready */
  if(match_nr>0) return(match_nr);

  /* Since there was no match with names, we'll try with numbers */
  INTLIST li; intlistInit(&li);
  if(intlistExpandFromString(local_name, ", ", &li, 1)<1) {intlistFree(&li); return(0);}
  intlistSort(&li);
  for(int i=0; i<li.nr; i++) {
    if(li.i[i]<1) continue;
    if(li.i[i]>d->tacNr) break;
    d->r[li.i[i]-1].sw=1; match_nr++;
  }
  intlistFree(&li);
  statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_OK);
  return(match_nr);
}
/*****************************************************************************/

/*****************************************************************************/
/** Get the number of selected TACs (sw!=0) in PAR structure.
    @return the number of selected TACs.
    @author Vesa Oikonen
    @sa parSelectTACs, parSortByName, parSelectByAnother
 */
int parSelectedTACs(
  /** Pointer to PAR data. */
  PAR *d
) {
  int ri, n;

  /* Check the input */
  if(d==NULL || d->tacNr<1) return(0);
  /* Check each VOI */
  for(ri=n=0; ri<d->tacNr; ri++) if(d->r[ri].sw!=0) n++;
  return(n);
}
/*****************************************************************************/

/*****************************************************************************/
/** Select the parameter(s) in PAR struct that have matching name or number.
    @return the number of newly selected parameters, or <=0 in case of an error.
    @author Vesa Oikonen
    @sa parSelectedParameters, parFindParameter, parSelectTACs, parSelectByAnother, parEnsureNames
 */
int parSelectParameters(
  /** Pointer to PAR structure. */
  PAR *d,
  /** Name of parameter, or parameter numbers (as string, for example '1-3' or '1,2,3'); 
      string can be given with quotation marks. 
      Enter NULL to select all parameters, or empty string to select parameters 
      with empty name (which really should not exist). */
  const char *par_name,
  /** 1=Non-matching parameters are deselected, 0=Old selections are preserved. */
  int reset,
  /** Pointer to status data; enter NULL if not needed. */
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  if(verbose>0) printf("%s()\n", __func__);
  /* Check that required data exists */
  if(d==NULL || d->parNr<1) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_NO_DATA);
    return(0);
  }

  int pi, len, match_nr=0;

  /* Check if NULL was given as parameter name; if, then select all */
  if(par_name==NULL) {
    for(pi=0; pi<d->parNr; pi++) d->n[pi].sw=1;
    return(d->parNr);
  }

  /* Reset all selections if required */
  if(reset!=0) for(pi=0; pi<d->parNr; pi++) d->n[pi].sw=0;
  
  /* Remove any quotation marks from the name */
  len=strlen(par_name);
  char local_name[len+1];
  if(len>1) len=strncpyClean(local_name, par_name, len+1);
  else strcpy(local_name, par_name);

  /* If parameter name is empty, then select all tacs with missing names */
  if(len<1 || strcmp(local_name, ".")==0) {
    for(pi=0; pi<d->parNr; pi++) {
      if(strlen(d->n[pi].name)==0) {d->n[pi].sw=1; match_nr++; continue;}
      /* maybe here should also be checked for names consisting of only space characters */
    }
    return(match_nr);
  }

  /* Check each parameter name */
  for(pi=0; pi<d->parNr; pi++) {
    /* does the name match? */
    if(strcasecmp(d->n[pi].name, local_name)==0) {
      d->n[pi].sw=1; match_nr++; continue;
    }
  }
  /* If at least one match was found, then we are ready */
  if(match_nr>0) return(match_nr);

  /* Since there was no match with names, we'll try with numbers */
  INTLIST li; intlistInit(&li);
  if(intlistExpandFromString(local_name, ", ", &li, 1)<1) {intlistFree(&li); return(0);}
  intlistSort(&li);
  for(int i=0; i<li.nr; i++) {
    if(li.i[i]<1) continue;
    if(li.i[i]>d->parNr) break;
    d->n[li.i[i]-1].sw=1; match_nr++;
  }
  intlistFree(&li);

  statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_OK);
  return(match_nr);
}
/*****************************************************************************/

/*****************************************************************************/
/** Get the number of selected parameters (sw!=0) in PAR structure.
    @return the number of selected parameters.
    @author Vesa Oikonen
    @sa parSelectParameters, parFindParameter, parSelectByAnother
 */
int parSelectedParameters(
  /** Pointer to PAR data */
  PAR *d
) {
  int pi, n;

  /* Check the input */
  if(d==NULL || d->parNr<1) return(0);
  /* Check each VOI */
  for(pi=n=0; pi<d->parNr; pi++) if(d->n[pi].sw!=0) n++;
  return(n);
}
/*****************************************************************************/

/*****************************************************************************/
/** Searches PAR structure for the given parameter name (case insensitive).
    @return the index [0..parNr-1] of the parameter, or <0 if not found.
    @author Vesa Oikonen
    @sa parGetParameter, parSelectParameters
 */
int parFindParameter(
  /** Pointer to PAR structure. */
  PAR *d,
  /** Name of the parameter. Comparison is case-insensitive, but otherwise
      strict match is required. */
  const char *par_name
) {
  /* Check that required data exists */
  if(d==NULL || d->parNr<1 || par_name==NULL) return(-1);
  /* Find the name */
  for(int i=0; i<d->parNr; i++) {
    if(strcasecmp(par_name, d->n[i].name)==0) return(i);
  }
  return(-1);
}
/*****************************************************************************/

/*****************************************************************************/
/** Searches PAR structure for the given parameter name (case insensitive).
    @return the value of the parameter for specified TAC, or NaN if not found.
    @author Vesa Oikonen
    @sa parGetParameterUnit, parFindParameter, parSelectParameters
 */
double parGetParameter(
  /** Pointer to PAR structure. */
  PAR *d,
  /** Name of the parameter. Comparison is case-insensitive, but otherwise
      strict match is required. */
  const char *par_name,
  /** TAC index [0..tacNr-1]. */
  const int ti
) {
  double v=nan("");
  /* Check that required data exists */
  if(d==NULL || d->parNr<1 || ti<0 || ti>=d->tacNr || par_name==NULL) return(v);
  /* Find the name */
  for(int pi=0; pi<d->parNr; pi++)
    if(strcasecmp(par_name, d->n[pi].name)==0) v=d->r[ti].p[pi];
  return(v);
}
/*****************************************************************************/

/*****************************************************************************/
/** Searches PAR structure for the given parameter name (case insensitive).
    @return the enum of the unit of the parameter, or UNIT_UNKNOWN if not found.
    @author Vesa Oikonen
    @sa parFindParameter, parGetParameter
 */
int parGetParameterUnit(
  /** Pointer to PAR structure. */
  PAR *d,
  /** Name of the parameter. Comparison is case-insensitive, but otherwise strict match is required. */
  const char *par_name
) {
  int u=UNIT_UNKNOWN;
  /* Check that required data exists */
  if(d==NULL || d->parNr<1 || par_name==NULL) return(u);
  /* Find the name */
  for(int pi=0; pi<d->parNr; pi++) if(strcasecmp(par_name, d->n[pi].name)==0) u=d->n[pi].unit;
  return(u);
}
/*****************************************************************************/

/*****************************************************************************/
/** Finds the min and max parameter value in PAR structure.
    @author Vesa Oikonen
    @sa parFindParameter, parGetParameter, parSelectParameters
 */
void parValueRange(
  /** Pointer to PAR structure. */
  PAR *d,
  /** Parameter index [0..parNr-1]. */
  const int i,
  /** Pointer to min parameter value; NULL if not needed. */
  double *pmin,
  /** Pointer to max parameter value; NULL if not needed. */
  double *pmax
) {
  if(pmin!=NULL) *pmin=nan("");
  if(pmax!=NULL) *pmax=nan("");
  if(d==NULL || i<0 || i>=d->parNr || d->tacNr<1) return;
  
  double mi, ma; mi=ma=d->r[0].p[i];
  for(int j=1; j<d->tacNr; j++) {
    if(d->r[j].p[i]<mi) mi=d->r[j].p[i];
    else if(d->r[j].p[i]>ma) ma=d->r[j].p[i];
  }
  if(pmin!=NULL) *pmin=mi;
  if(pmax!=NULL) *pmax=ma;
  return;
}
/*****************************************************************************/

/*****************************************************************************/
