/** @file gaussdev.c
 *  @brief Functions for creating random numbers.
 */
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <math.h>
#include <sys/time.h>
#include <time.h>
#include <string.h>
/*****************************************************************************/
#include "tpcextensions.h"
/*****************************************************************************/
#include "tpcrand.h"
/*****************************************************************************/

/*****************************************************************************/
/** @brief Make and optionally set the seed for rand(), drand, drandRange, and drandGaussian(). 
    @details Uses microseconds from the computer clock and process ID to reduce the chance of
    getting the same seed for simultaneously executing program threads and instances.
    @sa mertwiSeed64
    @return Returns the seed for srand(). 
 */ 
unsigned int drandSeed(
  /** Also sets seed with srand (1) or not (0). */
  short int seed
) {
  unsigned int li;
#if defined HAVE_TIMESPEC_GET
  struct timespec ts;
  timespec_get(&ts, TIME_UTC);
  //li=(unsigned int)ts.tv_sec+(unsigned int)ts.tv_nsec+(unsigned int)getpid();
  li=((ts.tv_sec % 10000)*523 ^ ts.tv_nsec*10) ^ ((getpid() % 1000)*983);
#elif defined HAVE_CLOCK_GETTIME
  struct timespec ts;
  clock_gettime(CLOCK_REALTIME, &ts);
  //li=(unsigned int)ts.tv_sec+(unsigned int)ts.tv_nsec+(unsigned int)getpid();
  li=((ts.tv_sec % 10000)*523 ^ ts.tv_nsec*10) ^ ((getpid() % 1000)*983);
#elif defined HAVE_GETTIMEOFDAY
  struct timeval tv;
  gettimeofday(&tv, 0);
  li=((tv.tv_sec % 10000)*523 ^ tv.tv_usec*13) ^ ((getpid() % 1000)*983);
#else
  li=(unsigned int)time(NULL)+(unsigned int)getpid();
#endif
  li+=(unsigned int)rand();
  if(seed) srand(li);
  return(li);
}
/*****************************************************************************/

/*****************************************************************************/
/** Alternative function to rand() which returns a double precision floating point number in 
    the range of [0,1] with uniform distribution. 

    @pre Uses rand(), therefore set seed for a new series of pseudo-random numbers; to produce truly
    random numbers (not just pseudo-random), do srand(time(0)) before calling this function.
    If no seed is set, then value 1 is used as default seed by rand().
    @sa mertwiRandomDouble1, drandRange, drandSeed
    @author Vesa Oikonen
    @return Random double value in the range [0,1]. 
 */
double drand()
{
  double d, s;
  s=1.0/(1.0+RAND_MAX);
  do {
    d = ( ( s*rand() + rand() )*s + rand() ) * s;
  } while(d>=1.0);
  return d;
}
/*****************************************************************************/

/*****************************************************************************/
/** Fill the given double array with random numbers with uniform distribution between
    the specified limits.

    With uniform distribution, the SD=(up-low)/sqrt(12), and CV=(up-low)/(sqrt(3)*(low+up)).

    @pre Set seed for a new series of pseudo-random numbers; to produce truly random numbers
    (not just pseudo-random), do srand(time(0)) before calling this function.
    If no seed is set, then value 1 is used as default seed by rand().
    @sa mertwiRandomBetween, drand, drandSeed
    @author Vesa Oikonen
    @return 0 when successful, otherwise <> 0.
 */
int drandRange(
  /** Nr of values in double array. */
  unsigned int nr,
  /** Pointer to pre-allocated double array. */
  double *d,
  /** Lower limit for random values. */
  double low,
  /** Upper limit for random values. */
  double up,
  /** Distribution: 0=even, 1=square-root transformation. */
  int type
) {
  unsigned int i;
  double dif, v, stl, stu;

  if(nr<1) return 0;
  if(d==NULL || type<0 || type>1) return 1;

  dif=up-low; if(dif<0.0) return 2;
  if(dif==0.0) {
    for(i=0; i<nr; i++) d[i]=low;
    return 0;
  }

  if(type==0) {
    for(i=0; i<nr; i++) d[i] = drand()*dif + low;
  } else if(type==1) {
    stl=copysign(sqrt(fabs(low)),low); if(!isnormal(stl)) stl=0.0;
    stu=copysign(sqrt(fabs(up)), up); if(!isnormal(stu)) stu=0.0;
    dif=stu-stl;
    for(i=0; i<nr; i++) {v=drand()*dif+stl; d[i]=copysign(v*v, v);}
  }

  return 0;
}
/*****************************************************************************/

/*****************************************************************************/
/** @brief Get pseudo-random number with normal (Gaussian) distribution with mean 0 and SD 1.

    Applies the polar form of Box-Müller transform to produce pseudo-random numbers with Gaussian
    (normal) distribution which has a zero mean and standard deviation of one.

    Box GEP, Muller ME. A note on the generation of random normal deviates.
    Annals of Mathematical Statistics, Volume 29, Issue 2, 1958, 610-611.
    Available from JSTOR http://www.jstor.org/

    @pre Set seed for a new series of pseudo-random numbers; to produce truly random numbers
    (not just pseudo-random), do srand(time(0)) before calling this function.
    If no seed is set, then value 1 is used as default seed by rand().
    @sa mertwiRandomGaussian, drandRange, drandSeed
    @return Returns the pseudo-random double value with normal distribution with zero mean and 
     standard deviation 1.
 */
double drandGaussian()
{
  static int ready=0;
  static double dev;
  double fac, rsq, a, b;

  /* If we don't have deviate already, then we'll have to make one */
  if(!ready) {
    do {
      a = 2.*drand() - 1.0; 
      b = 2.*drand() - 1.0; 
      rsq = a*a + b*b; 
    } while (rsq>1.0 || rsq==0.0);
   
    fac = sqrt(-2.0*log(rsq)/rsq);
    dev=a*fac; ready=1; 
    return(b*fac); 
  } else { /* dev is ready so return it */
    ready=0;
    return(dev);
  }
}
/*****************************************************************************/

/*****************************************************************************/
/** @brief Get pseudo-random number with exponential distribution.

    @pre Set seed for a new series of pseudo-random numbers; to produce truly random numbers
    (not just pseudo-random), do srand(time(0)) before calling this function.
    If no seed is set, then value 1 is used as default seed by rand().
    @sa mertwiRandomExponential, drandRange, drandSeed
    @return Returns the random non-negative double value with exponential distribution
     and given mean.
 */
double drandExponential(
  /** Mean of the exponential distribution. */
  double mean
) {
  double r;
  do {
    r=drand(); 
  } while(r==0.0);
  return(-mean*log(r)); 
}
/*****************************************************************************/

/*****************************************************************************/
