/** @file qviewio.c
 *  @brief I/O functions for QView TAC file format.
 */
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include "tpcift.h"
#include "tpcisotope.h"
#include "tpccsv.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <time.h>
#include <string.h>
/*****************************************************************************/
#include "tpctac.h"
/*****************************************************************************/

/*****************************************************************************/
/** Read 4DM CSV format from CSV structure into TAC structure.
    @return enum tpcerror (TPCERROR_OK when successful).
    @author Vesa Oikonen
    @sa tacRead, tacWrite, tacInit
 */
int tacRead4DM(
  /** Pointer to TAC structure, contents of which are to be written. */
  TAC *tac,
  /** Pointer to CSV from which data is read. */
  CSV *csv,
  /** Pointer to status data; enter NULL if not needed. */
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  if(tac==NULL) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL);
    return TPCERROR_FAIL;
  }
  if(verbose>0) {printf("%s()\n", __func__); fflush(stdout);}

  tacFree(tac);

  if(csv==NULL || csv->row_nr<4 || csv->col_nr<2) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_NO_DATA);
    return TPCERROR_NO_DATA;
  }

  int ret=0;

  /* Check from the first column that data indeed is 4DM data */
  if(strcasecmp(csv->c[0].content, "PatientName")!=0 || 
     strcasecmp(csv->c[2].content, "Patient MRN")!=0)
  {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_INVALID_FORMAT);
    return TPCERROR_INVALID_FORMAT;
  }

  /* Find the frame number from specific field */
  int frameNr=0;
  {
    int i=csvFindField(csv, "Number of Frames", 4);
    if(i<0 || i>=csv->nr-1) {
      statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_INVALID_FORMAT);
      return TPCERROR_INVALID_FORMAT;
    }
    i++;
    strClean(csv->c[i].content);
    frameNr=atoi(csv->c[i].content);
    if(verbose>1) printf("frameNr := %d\n", frameNr);
    if(frameNr<1) {
      statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_NO_DATA);
      return TPCERROR_NO_DATA;
    }
  }

  /* Estimate the number of ROIs */
  int maxTacNr=0;
  {
    int i=4;
    do {
      i=csvSearchField(csv, "TAC ROI ", ++i);
      if(i<0) break;
      if(csv->c[i].col==0) maxTacNr++;
    } while(i<csv->nr);
    if(verbose>1) printf("maxTacNr := %d\n", maxTacNr);
    if(maxTacNr<1) {
      statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_NO_DATA);
      return TPCERROR_NO_DATA;
    }
  }

  /* Allocate memory for TAC data */
  ret=tacAllocate(tac, frameNr, maxTacNr);
  statusSet(status, __func__, __FILE__, __LINE__, ret);
  if(ret!=TPCERROR_OK) return(ret);
  tac->isframe=1;
  tac->sampleNr=frameNr;

  /* Copy frame start times from CSV into TAC structure */
  {
    int i=csvSearchField(csv, "Frame Start Times", 4);
    char tmp[64];
    strncpyCleanSpaces(tmp, csv->c[i].content+17, 30); strCleanPars(tmp); 
    tac->tunit=unitIdentify(tmp);
    if(verbose>2) printf("tunit := %s\n", unitName(tac->tunit));
    int row=csv->c[i].row;
    int fails=0;
    for(int fi=0; fi<tac->sampleNr; fi++) {
      i++;
      if(csv->c[i].row!=row) {fails++; break;}
      strClean(csv->c[i].content);
      tac->x1[fi]=atofVerified(csv->c[i].content);
    }
    if(fails>0) {
      statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_INVALID_FORMAT);
      return TPCERROR_INVALID_FORMAT;
    }
  }

  /* Get frame durations from CSV to write frame end times into TAC structure */
  {
    int i=csvSearchField(csv, "Frame Durations", 4);
    int row=csv->c[i].row;
    int fails=0;
    for(int fi=0; fi<tac->sampleNr; fi++) {
      i++;
      if(csv->c[i].row!=row) {fails++; break;}
      strClean(csv->c[i].content);
      tac->x2[fi]=tac->x1[fi]+atofVerified(csv->c[i].content);
      tac->x[fi]=0.5*(tac->x1[fi]+tac->x2[fi]);
    }
    if(fails>0) {
      statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_INVALID_FORMAT);
      return TPCERROR_INVALID_FORMAT;
    }
  }

  /* Copy ROI data from CSV into TAC structure */
  tac->tacNr=0;
  {
    int i=4;
    while((i=csvSearchField(csv, "TAC ROI ", i))>0) {
      int row=csv->c[i].row;
      strlcpy(tac->c[tac->tacNr].name, csv->c[i].content+8, MAX_TACNAME_LEN+1);
      int fails=0, goods=0;
      for(int fi=0; fi<tac->sampleNr; fi++) {
        i++;
        if(csv->c[i].row!=row) {fails++; break;}
        strClean(csv->c[i].content);
        tac->c[tac->tacNr].y[fi]=atofVerified(csv->c[i].content);
        if(isnormal(tac->c[tac->tacNr].y[fi])) goods++; // zero is not accepted here on purpose 
      }
      if(fails==0 && goods>0) tac->tacNr++;
    }
    if(tac->tacNr<1) {
      statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_NO_DATA);
      return TPCERROR_NO_DATA;
    }
  }

  /* Get concentration units */
  {
    int i=csvSearchField(csv, "TAC FA Blood LV ", 4);
    if(i>=0) {
      char tmp[64];
      strncpyCleanSpaces(tmp, csv->c[i].content+16, 64); strCleanPars(tmp); 
      tac->cunit=unitIdentify(tmp);
      if(verbose>2) printf("cunit := %s\n", unitName(tac->cunit));
    }
  }

  /* Get isotope */
  {
    int isotope=ISOTOPE_UNKNOWN;
    int i=csvFindField(csv, "Radionuclide Name", 4);
    if(i>=0 && i<csv->nr-1 && csv->c[i].row==csv->c[i+1].row) {
      char *buf=strTokenDup(csv->c[++i].content, " ", NULL);
      isotope=isotopeIdentify(buf);
      free(buf);
    }
    if(isotope!=ISOTOPE_UNKNOWN) tacSetIsotope(tac, isotope);
    if(verbose>2) printf("isotope := %s\n", isotopeName(isotope));
  }

  if(tac->sampleNr<1 || tac->tacNr<1) {
    tacFree(tac);
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_INVALID_FORMAT);
    return TPCERROR_INVALID_FORMAT;
  }

  tac->format=TAC_FORMAT_4DM;

  statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_OK);
  return(TPCERROR_OK);
}
/*****************************************************************************/

/*****************************************************************************/
