/** @file qviewio.c
 *  @brief I/O functions for QView TAC file format.
 */
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include "tpcift.h"
#include "tpcisotope.h"
#include "tpccsv.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <time.h>
#include <string.h>
/*****************************************************************************/
#include "tpctac.h"
/*****************************************************************************/

/*****************************************************************************/
/** Read QView format from CSV struct into TAC struct.
    @details Note that QView TAC format does not contain frame times.
    @return enum tpcerror (TPCERROR_OK when successful).
    @author Vesa Oikonen
    @sa tacRead, tacWrite, tacInit
 */
int tacReadQView(
  /** Pointer to TAC structure, contents of which are to be written. */
  TAC *tac,
  /** Pointer to CSV from which data is read. */
  CSV *csv,
  /** Read TACs of all ROIs (0), or only group means (1). */
  const int grouponly,
  /** Pointer to status data; enter NULL if not needed. */
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  if(tac==NULL) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL);
    return TPCERROR_FAIL;
  }
  if(verbose>0) {printf("%s()\n", __func__); fflush(stdout);}

  tacFree(tac);

  if(csv==NULL || csv->row_nr<2 || csv->col_nr<12) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_NO_DATA);
    return TPCERROR_NO_DATA;
  }

  int ret=0;

  /* Check from the first line (title) that data indeed is QView data */
  if(strcasecmp(csv->c[0].content, "directory")!=0 || 
     strcasecmp(csv->c[11].content, "group")!=0)
  {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_INVALID_FORMAT);
    return TPCERROR_INVALID_FORMAT;
  }

  /* Find the highest frame number from parens in 'num' field */
  int maxFrameNr=0;
  {
    int fn;
    char *cptr, *cptr2;
    for(int ri=1; ri<csv->row_nr; ri++) {
      cptr=csvCell(csv, ri, 2); if(cptr==NULL) continue;
      if(verbose>50) printf("  '%s'\n", cptr);
      cptr2=strchr(cptr, '('); if(cptr2==NULL) continue;
      fn=atoi(cptr2+1); if(fn>maxFrameNr) maxFrameNr=fn;
    }
  }
  if(verbose>1) printf("maxFrameNr := %d\n", maxFrameNr);
  if(maxFrameNr<1) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_INVALID_FORMAT);
    return TPCERROR_INVALID_FORMAT;
  }
  /* Estimate the number of TACs */
  int maxTacNr=(csv->row_nr-1)/maxFrameNr; // row_nr includes the title line
  if(verbose>1) printf("maxTacNr := %d\n", maxTacNr);
  if(maxFrameNr<1) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_INVALID_FORMAT);
    return TPCERROR_INVALID_FORMAT;
  }

  /* Allocate memory for TAC data; and to be on the safe side, for one extra TAC */
  ret=tacAllocate(tac, maxFrameNr, maxTacNr);
  statusSet(status, __func__, __FILE__, __LINE__, ret);
  if(ret!=TPCERROR_OK) return(ret);
  tac->isframe=0;

  /* Copy data from CSV into TAC structure */
  ret=0;
  for(int i=1; i<csv->row_nr; i++) {
    /* Read the cells of this CSV line */
    char *cnum, *cslice, *cmean, *cpixel, *cgroup;
    cnum=csvCell(csv, i, 2); if(cnum==NULL) continue;
    cslice=csvCell(csv, i, 3); if(cslice==NULL) continue;
    cmean=csvCell(csv, i, 4); if(cmean==NULL) continue;
    cpixel=csvCell(csv, i, 9); if(cpixel==NULL) continue;
    cgroup=csvCell(csv, i, 11); if(cgroup==NULL) continue;
    /* Get slice number; if missing, then this is group ROI */
    int slice;
    ret=atoiCheck(cslice, &slice);
    if(ret!=0) {slice=-1; ret=0;}
    /* If only group ROIs were requested, and this is not group */
    if(slice>0 && grouponly!=0) continue;
    /* Depending on previous, get frame number and group name, or frame and ROI number */
    char *cptr=strchr(cnum, '('); if(cptr==NULL) continue;
    int fn=atoi(cptr+1); if(fn<1 || fn>maxFrameNr) {ret=1; break;}
    char tacname[MAX_TACNAME_LEN+1];
    if(slice<1) { // this is group
      /* Check group name, and use it as TAC name */
      if(strnlen(cgroup, 2)<1) {ret=2; break;}
      strlcpy(tacname, cgroup, MAX_TACNAME_LEN);
    } else { // this is one roi at one image slice
      /* Get ROI number */
      int roinr=atoi(cnum); if(roinr<1) {ret=3; break;}
      /* If group name exists, the use that and ROI number and slice number to construct TAC name */
      if(strnlen(cgroup, 2)>0 && strnlen(cgroup, 2)<12) {
        sprintf(tacname, "%s_roi%d__pl%d", cgroup, roinr, slice);
      } else {
        /* Otherwise use ROI number and slice number to construct TAC name */
        sprintf(tacname, "roi%d__pl%d", roinr, slice);
      }
    }
    /* Get the number of pixels */
    int pxlnr;
    ret=atoiCheck(cpixel, &pxlnr); if(ret) {pxlnr=0; ret=0;}
    /* Get the ROI concentration */
    double conc=atofVerified(cmean);
    if(verbose>20) printf("tacname='%s' pxlnr=%d fn=%d conc=%g\n", tacname, pxlnr, fn, conc);
    /* Determine the TAC index for this; first check if this TAC already is added */
    int ri;
    for(ri=0; ri<tac->tacNr; ri++) {
      if(strcmp(tac->c[ri].name, tacname)==0 && doubleMatch(tac->c[ri].size, (double)pxlnr, 0.1))
        break;
      /* if TAC is not yet added, then ri will be set to the first free index */
    }
    /* check that ri is not too big */
    if(ri>=maxTacNr) {ret=4; break;}
    /* Put value in place */
    tac->c[ri].y[fn-1]=conc;
    /* If this was first encounter of this TAC, then set other info, and increase tacNr */
    if(ri>=tac->tacNr) {
      strcpy(tac->c[ri].name, tacname);
      tac->c[ri].size=(double)pxlnr;
      tac->tacNr++;
    }
    /* Increase also sampleNr when necessary */
    if(fn>tac->sampleNr) tac->sampleNr=fn;
  }
  if(ret) {
    if(verbose>1) printf("  error %d\n", ret);
    tacFree(tac);
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_INVALID_FORMAT);
    return TPCERROR_INVALID_FORMAT;
  }

  if(tac->sampleNr<1 || tac->tacNr<1) {
    tacFree(tac);
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_INVALID_FORMAT);
    return TPCERROR_INVALID_FORMAT;
  }

  for(int i=0; i<tac->sampleNr; i++) tac->x[i]=tac->x1[i]=tac->x2[i]=0.0;

  tac->format=TAC_FORMAT_QVIEW;

  statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_OK);
  return(TPCERROR_OK);
}
/*****************************************************************************/

/*****************************************************************************/
