/** @file tacift.c
 *  @brief Get or set TAC header content from IFT struct stored
 *         inside TAC struct.
 */
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include "tpcift.h"
#include "tpcisotope.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <time.h>
#include <string.h>
/*****************************************************************************/
#include "tpctac.h"
/*****************************************************************************/

/*****************************************************************************/
/** Get study number from TAC struct header.
 *  @return enum tpcerror (TPCERROR_OK when successful).
 *  @author Vesa Oikonen
 *  @sa tacSetHeaderStudynr, parIsStudyNr
 */
int tacGetHeaderStudynr(
  /** Pointer to TAC header struct */
  IFT *h,
  /** Pointer to preallocated string of length MAX_STUDYNR_LEN+1 (at least);
      enter NULL if you only want to test whether header contains studynr */
  char *s,
  /** Pointer to status data; enter NULL if not needed */
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  if(s!=NULL) *s='\0';
  if(h==NULL) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_NO_DATA);
    return TPCERROR_NO_DATA;
  }
  if(verbose>0) printf("%s()\n", __func__);

  /* Find the item index */
  int i;
  i=iftFindKey(h, "studynr", 0);
  if(i<0) i=iftFindKey(h, "study_number", 0);
  if(i<0) i=iftFindKey(h, "study number", 0);
  if(i<0) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_NO_KEY);
    return TPCERROR_NO_KEY;
  }
  /* Check that item has content */
  size_t len;
  if(h->item[i].value==NULL) len=0;
  else len=strlen(h->item[i].value);
  if(len<1) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_NO_VALUE);
    return TPCERROR_NO_VALUE;
  }
  if(len>MAX_STUDYNR_LEN) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_INVALID_VALUE);
    return TPCERROR_INVALID_VALUE;
  }
  /* All fine, then copy studynr */
  if(s!=NULL) strlcpy(s, h->item[i].value, MAX_STUDYNR_LEN);
  
  statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_OK);
  return(TPCERROR_OK);
}
/*****************************************************************************/

/*****************************************************************************/
/** Set study number in TAC struct header.
 *  Any previous studynr items are deleted.
 *  @return enum tpcerror (TPCERROR_OK when successful).
 *  @author Vesa Oikonen
 *  @sa tacGetHeaderStudyNr, parSetStudyNr, studynrFromFilename
 */
int tacSetHeaderStudynr(
  /** Pointer to TAC header struct */
  IFT *h,
  /** Pointer to string containing the study number;
   *  enter NULL or "" if you only want to delete studynr in header */
  const char *s
) {
  if(h==NULL) return TPCERROR_FAIL;

  /* Find and delete previous studynr(s) */
  int i=0, start=0;
  while(1) {
    i=iftFindKey(h, "studynr", start);
    if(i<0) i=iftFindKey(h, "study_number", start);
    if(i<0) i=iftFindKey(h, "study number", start);
    if(i<0) break;
    iftDelete(h, i); if(i>start) start=i;
  }

  /* If new studynr is empty, then just return */
  if(s==NULL || *s=='\0') return(TPCERROR_OK);
  
  /* Otherwise create new key and value */
  return (iftPut(h, "studynr", s, (char)1, NULL));
}
/*****************************************************************************/

/*****************************************************************************/
/** Set concentration (y) unit in TAC struct header.
 *  Any previous y unit items are deleted.
 *  @return enum tpcerror (TPCERROR_OK when successful).
 *  @author Vesa Oikonen
 */
int tacSetHeaderUnit(
  /** Pointer to TAC header struct */
  IFT *h,
  /** Unit enum */
  int u
) {
  if(h==NULL) return TPCERROR_FAIL;

  /* Find and delete previous unit(s) */
  int i=0, start=0;
  while(1) {
    i=iftFindKey(h, "unit", start);
    if(i<0) i=iftFindKey(h, "calibration_unit", start);
    if(i<0) i=iftFindKey(h, "activity_units", start);
    if(i<0) i=iftFindKey(h, "activity units", start);
    if(i<0) break;
    iftDelete(h, i); if(i>start) start=i;
  }

  /* If new unit is unknown, then just return */
  if(u<=UNIT_UNKNOWN || u>=UNIT_LAST) return(TPCERROR_OK);
  
  /* Otherwise create new key and value */
  return (iftPut(h, "unit", unitName(u), (char)1, NULL));
}
/*****************************************************************************/

/*****************************************************************************/
/** Get concentration (y) unit from TAC struct header and write to main TAC
 *  struct.
 *  @return enum tpcerror (TPCERROR_OK when successful).
 *  @author Vesa Oikonen
 */
int tacGetHeaderUnit(
  /** Pointer to TAC struct */
  TAC *tac,
  /** Pointer to status data; enter NULL if not needed */
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  if(tac==NULL) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL);
    return TPCERROR_FAIL;
  }
  if(verbose>0) printf("%s()\n", __func__);
  tac->cunit=UNIT_UNKNOWN;

  /* Find the item index */
  int i;
  i=iftFindKey(&tac->h, "unit", 0);
  if(i<0) i=iftFindKey(&tac->h, "calibration_unit", 0);
  if(i<0) i=iftFindKey(&tac->h, "activity_units", 0);
  if(i<0) i=iftFindKey(&tac->h, "activity units", 0);
  if(i<0) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_NO_KEY);
    return TPCERROR_NO_KEY;
  }
  /* Check that item has content */
  size_t len;
  if(tac->h.item[i].value==NULL) len=0;
  else len=strlen(tac->h.item[i].value);
  if(len<1) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_NO_VALUE);
    return TPCERROR_NO_VALUE;
  }
  
  /* Identify unit string */
  tac->cunit=unitIdentify(tac->h.item[i].value);

  /* Delete unit strings from header */
  tacSetHeaderUnit(&tac->h, UNIT_UNKNOWN);

  /* If unit was unknown, then return error */
  if(tac->cunit==UNIT_UNKNOWN) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_INVALID_VALUE);
    return TPCERROR_INVALID_VALUE;
  }
  /* Otherwise ok */  
  statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_OK);
  return(TPCERROR_OK);
}
/*****************************************************************************/

/*****************************************************************************/
/** Set time (x) unit in TAC struct header.
 *  Any previous x units are deleted.
 *  @return enum tpcerror (TPCERROR_OK when successful).
 *  @author Vesa Oikonen
 */
int tacSetHeaderTimeunit(
  /** Pointer to TAC header struct */
  IFT *h,
  /** Unit enum */
  int u
) {
  if(h==NULL) return TPCERROR_FAIL;

  /* Find and delete previous time unit(s) */
  int i=0, start=0;
  while(1) {
    i=iftFindKey(h, "timeunit", start);
    if(i<0) i=iftFindKey(h, "time_unit", start);
    if(i<0) i=iftFindKey(h, "time unit", start);
    if(i<0) i=iftFindKey(h, "time units", start);
    if(i<0) break;
    iftDelete(h, i); if(i>start) start=i;
  }

  /* If new unit is unknown, then just return */
  if(u<=UNIT_UNKNOWN || u>=UNIT_LAST) return(TPCERROR_OK);
  
  /* Otherwise create new key and value */
  return (iftPut(h, "timeunit", unitName(u), (char)1, NULL));
}
/*****************************************************************************/

/*****************************************************************************/
/** Get time (x) unit from TAC struct header and write to main TAC
 *  struct.
 *  @return enum tpcerror (TPCERROR_OK when successful).
 *  @author Vesa Oikonen
 */
int tacGetHeaderTimeunit(
  /** Pointer to TAC struct */
  TAC *tac,
  /** Pointer to status data; enter NULL if not needed */
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  if(tac==NULL) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL);
    return TPCERROR_FAIL;
  }
  if(verbose>0) printf("%s()\n", __func__);
  tac->tunit=UNIT_UNKNOWN;

  /* Find the item index */
  int i;
  i=iftFindKey(&tac->h, "timeunit", 0);
  if(i<0) i=iftFindKey(&tac->h, "time_unit", 0);
  if(i<0) i=iftFindKey(&tac->h, "time unit", 0);
  if(i<0) i=iftFindKey(&tac->h, "time units", 0);
  if(i<0) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_NO_KEY);
    return TPCERROR_NO_KEY;
  }
  /* Check that item has content */
  size_t len;
  if(tac->h.item[i].value==NULL) len=0;
  else len=strlen(tac->h.item[i].value);
  if(len<1) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_NO_VALUE);
    return TPCERROR_NO_VALUE;
  }
  
  /* Identify unit string */
  tac->tunit=unitIdentify(tac->h.item[i].value);

  /* Delete unit strings from header */
  tacSetHeaderTimeunit(&tac->h, UNIT_UNKNOWN);

  /* If unit was unknown, then return error */
  if(tac->tunit==UNIT_UNKNOWN) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_INVALID_VALUE);
    return TPCERROR_INVALID_VALUE;
  }
  /* Otherwise ok */  
  statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_OK);
  return(TPCERROR_OK);
}
/*****************************************************************************/

/*****************************************************************************/
/** Get isotope from TAC struct header.
    @sa tacGetIsotope, tacDecayCorrection, isotopeIdentify, tacGetHeaderDecayCorrection
    @return enum tpcerror (TPCERROR_OK when successful).
    @author Vesa Oikonen
 */
int tacGetHeaderIsotope(
  /** Pointer to TAC header structure. */
  IFT *h,
  /** Pointer to preallocated string of length MAX_ISOTOPE_LEN+1 (at least);
      enter NULL if you only want to test if header contains isotope code. */
  char *s,
  /** Pointer to status data; enter NULL if not needed. */
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  if(s!=NULL) *s='\0';
  if(h==NULL) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_NO_DATA);
    return TPCERROR_NO_DATA;
  }
  if(verbose>0) printf("%s()\n", __func__);

  /* Find the item index */
  int i;
  i=iftFindKey(h, "isotope", 0);
  if(i<0) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_NO_KEY);
    return TPCERROR_NO_KEY;
  }
  /* Check that item has content */
  size_t len;
  if(h->item[i].value==NULL) len=0;
  else len=strlen(h->item[i].value);
  if(len<1) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_NO_VALUE);
    return TPCERROR_NO_VALUE;
  }
  if(len>MAX_ISOTOPE_LEN) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_INVALID_VALUE);
    return TPCERROR_INVALID_VALUE;
  }
  /* All fine, then copy isotope */
  if(s!=NULL) strlcpy(s, h->item[i].value, MAX_ISOTOPE_LEN);
  
  statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_OK);
  return(TPCERROR_OK);
}
/*****************************************************************************/

/*****************************************************************************/
/** Set isotope in TAC structure header.
    Any previous isotope items are deleted.
    @sa tacSetIsotope
    @return enum tpcerror (TPCERROR_OK when successful).
    @author Vesa Oikonen
 */
int tacSetHeaderIsotope(
  /** Pointer to TAC header structure. */
  IFT *h,
  /** Pointer to string containing the isotope;
      enter NULL or "" if you only want to delete isotope in header. */
  const char *s
) {
  if(h==NULL) return TPCERROR_FAIL;

  /* Find and delete previous isotope(s) */
  int i=0, start=0;
  while(1) {
    i=iftFindKey(h, "isotope", start);
    if(i<0) break;
    iftDelete(h, i); if(i>start) start=i;
  }

  /* If new isotope is empty, then just return */
  if(s==NULL || *s=='\0') return(TPCERROR_OK);
  
  /* Otherwise create new key and value */
  return (iftPut(h, "isotope", s, (char)1, NULL));
}
/*****************************************************************************/

/*****************************************************************************/
/** Get scan start time from TAC structure header.
    @return enum tpcerror (TPCERROR_OK when successful).
    @author Vesa Oikonen
    @sa tacSetHeaderScanstarttime, tacGetHeaderInjectiontime, tacDecayCorrection, tacGetHeaderIsotope
 */
int tacGetHeaderScanstarttime(
  /** Pointer to TAC header structure. */
  IFT *h,
  /** Pointer to preallocated string of length 19+1 (at least);
      enter NULL if you only want to test if header contains scan start time. */
  char *s,
  /** Pointer to status data; enter NULL if not needed. */
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  if(s!=NULL) *s='\0';
  if(h==NULL) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_NO_DATA);
    return TPCERROR_NO_DATA;
  }
  if(verbose>0) printf("%s()\n", __func__);

  /* Find the item index */
  int i;
  i=iftFindKey(h, "scan_start_time", 0);
  if(i<0) i=iftFindKey(h, "scan start time", 0);
  if(i<0) i=iftFindKey(h, "scan_start", 0);
  if(i<0) i=iftFindKey(h, "scan start", 0);
  if(i<0) i=iftFindKey(h, "scantime", 0);
  if(i<0) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_NO_KEY);
    return TPCERROR_NO_KEY;
  }
  /* Check that item has content */
  size_t len;
  if(h->item[i].value==NULL) len=0;
  else len=strlen(h->item[i].value);
  if(len<1) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_NO_VALUE);
    return TPCERROR_NO_VALUE;
  }
  if(len>19) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_INVALID_VALUE);
    return TPCERROR_INVALID_VALUE;
  }
  /* All fine, then copy it */
  if(s!=NULL) strcpy(s, h->item[i].value);
  
  statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_OK);
  return(TPCERROR_OK);
}
/*****************************************************************************/

/*****************************************************************************/
/** Set scan start time in TAC structure header.
    Any previous scan start items are deleted.
    @return enum tpcerror (TPCERROR_OK when successful).
    @sa tacGetHeaderScanstarttime, tacSetHeaderInjectiontime, tacDecayCorrection, tacSetHeaderIsotope
    @author Vesa Oikonen
 */
int tacSetHeaderScanstarttime(
  /** Pointer to TAC header structure. */
  IFT *h,
  /** Pointer to string containing the scan start time in international format YYYY-MM-DD hh:mm:ss ;
      enter NULL or "" if you only want to delete it in header. */
  const char *s
) {
  if(h==NULL) return TPCERROR_FAIL;

  /* Find and delete previous scan start time(s) */
  int i=0, start=0;
  while(1) {
    i=iftFindKey(h, "scan_start_time", start);
    if(i<0) i=iftFindKey(h, "scan start time", start);
    if(i<0) i=iftFindKey(h, "scan_start", start);
    if(i<0) i=iftFindKey(h, "scan start", start);
    if(i<0) i=iftFindKey(h, "scantime", start);
    if(i<0) break;
    iftDelete(h, i); if(i>start) start=i;
  }

  /* If new scan start time is empty, then just return */
  if(s==NULL || *s=='\0') return(TPCERROR_OK);
  
  /* Otherwise create new key and value */
  return (iftPut(h, "scan_start_time", s, (char)1, NULL));
}
/*****************************************************************************/

/*****************************************************************************/
/** Get injection time from TAC structure header.
    @return enum tpcerror (TPCERROR_OK when successful).
    @author Vesa Oikonen
    @sa tacSetHeaderInjectiontime, tacSetHeaderScanstarttime, tacDecayCorrection, tacGetHeaderIsotope
 */
int tacGetHeaderInjectiontime(
  /** Pointer to TAC header structure. */
  IFT *h,
  /** Pointer to preallocated string of length 19+1 (at least);
      enter NULL if you only want to test if header contains injection time. */
  char *s,
  /** Pointer to status data; enter NULL if not needed. */
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  if(s!=NULL) *s='\0';
  if(h==NULL) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_NO_DATA);
    return TPCERROR_NO_DATA;
  }
  if(verbose>0) printf("%s()\n", __func__);

  /* Find the item index */
  int i;
  i=iftFindKey(h, "injection_time", 0);
  if(i<0) i=iftFindKey(h, "injection time", 0);
  if(i<0) i=iftFindKey(h, "injectiontime", 0);
  if(i<0) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_NO_KEY);
    return TPCERROR_NO_KEY;
  }
  /* Check that item has content */
  size_t len;
  if(h->item[i].value==NULL) len=0;
  else len=strlen(h->item[i].value);
  if(len<1) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_NO_VALUE);
    return TPCERROR_NO_VALUE;
  }
  if(len>19) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_INVALID_VALUE);
    return TPCERROR_INVALID_VALUE;
  }
  /* All fine, then copy it */
  if(s!=NULL) strcpy(s, h->item[i].value);
  
  statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_OK);
  return(TPCERROR_OK);
}
/*****************************************************************************/

/*****************************************************************************/
/** Set injection time in TAC structure header.
    Any previous injection time items are deleted.
    @return enum tpcerror (TPCERROR_OK when successful).
    @sa tacGetHeaderInjectiontime, tacGetHeaderScanstarttime, tacDecayCorrection, tacSetHeaderIsotope
    @author Vesa Oikonen
 */
int tacSetHeaderInjectiontime(
  /** Pointer to TAC header structure. */
  IFT *h,
  /** Pointer to string containing the injection time in international format YYYY-MM-DD hh:mm:ss ;
      enter NULL or "" if you only want to delete it in header. */
  const char *s
) {
  if(h==NULL) return TPCERROR_FAIL;

  /* Find and delete previous scan start time(s) */
  int i=0, start=0;
  while(1) {
    i=iftFindKey(h, "injection_time", start);
    if(i<0) i=iftFindKey(h, "injection time", start);
    if(i<0) i=iftFindKey(h, "injectiontime", start);
    if(i<0) break;
    iftDelete(h, i); if(i>start) start=i;
  }

  /* If new scan start time is empty, then just return */
  if(s==NULL || *s=='\0') return(TPCERROR_OK);
  
  /* Otherwise create new key and value */
  return (iftPut(h, "injection_time", s, (char)1, NULL));
}
/*****************************************************************************/

/*****************************************************************************/
/** Get the status of decay correction from TAC struct header.
    @return enum DECAY_UNKNOWN, DECAY_NOTCORRECTED, or DECAY_CORRECTED.
    @author Vesa Oikonen
    @sa tacSetHeaderDecayCorrection, tacGetHeaderIsotope, tacDecayCorrection
 */
decaycorrection tacGetHeaderDecayCorrection(
  /** Pointer to TAC header structure. */
  IFT *h
) {
  if(h==NULL) return(DECAY_UNKNOWN);

  /* Find the item index */
  int i;
  i=iftFindKey(h, "decay_correction", 0);
  if(i<0) i=iftFindKey(h, "decay correction", 0);
  if(i<0) return(DECAY_UNKNOWN);

  /* Check the contents */
  if(h->item[i].value==NULL || strlen(h->item[i].value)<1) 
    return(DECAY_UNKNOWN);
  if(strncasecmp(h->item[i].value, "YES", 1)==0) return(DECAY_CORRECTED);
  if(strncasecmp(h->item[i].value, "NO", 1)==0) return(DECAY_NOTCORRECTED);
  if(strcasecmp(h->item[i].value, "ON")==0) return(DECAY_CORRECTED);
  if(strcasecmp(h->item[i].value, "OFF")==0) return(DECAY_NOTCORRECTED);
  return(DECAY_UNKNOWN);
}
/*****************************************************************************/

/*****************************************************************************/
/** Set the status of decay correction in TAC structure header.
    Any previous status is deleted.
    @return enum tpcerror (TPCERROR_OK when successful).
    @author Vesa Oikonen
    @sa tacGetHeaderDecayCorrection, tacSetHeaderIsotope, tacDecayCorrection
 */
int tacSetHeaderDecayCorrection(
  /** Pointer to TAC header structure. */
  IFT *h,
  /** Status of decay correction: DECAY_UNKNOWN, DECAY_NOTCORRECTED, or DECAY_CORRECTED. */
  decaycorrection dc 
) {
  if(h==NULL) return TPCERROR_FAIL;

  /* Find and delete previous status */
  int i=0, start=0;
  while(1) {
    i=iftFindKey(h, "decay_correction", start);
    if(i<0) i=iftFindKey(h, "decay correction", start);
    if(i<0) break;
    iftDelete(h, i); if(i>start) start=i;
  }

  /* Create new key and value */
  if(dc==DECAY_CORRECTED)
    return (iftPut(h, "decay_correction", "yes", (char)1, NULL));
  else if(dc==DECAY_NOTCORRECTED)
    return (iftPut(h, "decay_correction", "no", (char)1, NULL));

  /* If unknown, then just return */
  return(TPCERROR_OK);  
}
/*****************************************************************************/

/*****************************************************************************/
